﻿using System;
using System.Collections.Generic;
using System.Threading;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using DevMenuCommandTest;

namespace DevMenuCommandTestNuiShell
{
    [TestClass]
    public class TestVerifyApplication : TestBase
    {
        public override void DoCleanup()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            command.ResetAndExecute(new string[] {
                "application uninstall --all",
            });
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestVerifyApplicationNand()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);
            var aocv0 = m_Maker.MakeAddOnContent(id, version: 1);
            var patchv1 = m_Maker.MakePatch(id, 1, appv0.Path);

            var command = new DevMenuCommandSystem(this.TestContext);

            // コンテンツを NAND にインストールする
            Assert.IsTrue(command.Execute(new string[] {
                "application install " + appv0.Path + " -s builtin",
                "application install " + aocv0.Path + " -s builtin",
                "application install " + patchv1.Path + " -s builtin",
            }));

            Assert.IsTrue(command.Execute(new string[] {
                "application verify " + id.ToString("x16"),
            }, new string[] {
                "No corrupted contents.",
            }));

            // コンテンツを破損させる
            Assert.IsTrue(command.ResetAndExecute(new string[] {
                "application corrupt " + id.ToString("x16"),
            }));

            // 破損チェックで破損が検出される
            Assert.IsFalse(command.Execute(new string[] {
                "application verify " + id.ToString("x16"),
            }, new string[] {
                "The application is corrupted.",
            }));
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestVerifyApplicationSdCard()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);
            var aocv0 = m_Maker.MakeAddOnContent(id, version: 1);
            var patchv1 = m_Maker.MakePatch(id, 1, appv0.Path);

            var command = new DevMenuCommandSystem(this.TestContext);

            // コンテンツを SD カードにインストールする
            Assert.IsTrue(command.Execute(new string[] {
                "application install " + appv0.Path + " -s sdcard",
                "application install " + aocv0.Path + " -s sdcard",
                "application install " + patchv1.Path + " -s sdcard",
            }));

            Assert.IsTrue(command.Execute(new string[] {
                "application verify " + id.ToString("x16"),
            }, new string[] {
                "No corrupted contents.",
            }));

            // コンテンツを破損させる
            Assert.IsTrue(command.ResetAndExecute(new string[] {
                "application corrupt " + id.ToString("x16"),
            }));

            // 破損チェックで破損が検出される
            Assert.IsFalse(command.Execute(new string[] {
                "application verify " + id.ToString("x16"),
            }, new string[] {
                "The application is corrupted.",
            }));
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestVerifyApplicationWithCorruptedApplication()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);
            var aocv0 = m_Maker.MakeAddOnContent(id, version: 1);
            var patchv1 = m_Maker.MakePatch(id, 1, appv0.Path);

            var command = new DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.Execute(new string[] {
                "application install " + appv0.Path,
                "application install " + aocv0.Path,
                "application install " + patchv1.Path,
            }));

            Assert.IsTrue(command.Execute(new string[] {
                "application verify " + id.ToString("x16"),
            }, new string[] {
                "No corrupted contents.",
            }));

            // アプリケーションを破損させる
            Assert.IsTrue(command.Execute(new string[] {
                "application corrupt " + id.ToString("x16") + " --app",
            }));

            // 全コンテンツメタタイプ対象の破損チェックでは破損が検出される
            Assert.IsFalse(command.Execute(new string[] {
                "application verify " + id.ToString("x16"),
            }, new string[] {
                "The application is corrupted.",
            }));

            // アプリケーションが対象の破損チェックでは破損が検出される
            Assert.IsFalse(command.Execute(new string[] {
                "application verify " + id.ToString("x16") + " --app",
            }, new string[] {
                "The application is corrupted.",
            }));

            // パッチが対象の破損チェックでは破損が検出されない
            Assert.IsTrue(command.Execute(new string[] {
                "application verify " + id.ToString("x16") + " --patch",
            }, new string[] {
                "No corrupted contents.",
            }));

            // 追加コンテンツが対象の破損チェックでは破損が検出されない
            Assert.IsTrue(command.Execute(new string[] {
                "application verify " + id.ToString("x16") + " --aoc",
            }, new string[] {
                "No corrupted contents.",
            }));
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestVerifyApplicationWithCorruptedPatch()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);
            var aocv0 = m_Maker.MakeAddOnContent(id, version: 1);
            var patchv1 = m_Maker.MakePatch(id, 1, appv0.Path);

            var command = new DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.Execute(new string[] {
                "application install " + appv0.Path,
                "application install " + aocv0.Path,
                "application install " + patchv1.Path,
            }));

            Assert.IsTrue(command.Execute(new string[] {
                "application verify " + id.ToString("x16"),
            }, new string[] {
                "No corrupted contents.",
            }));

            // パッチを破損させる
            Assert.IsTrue(command.Execute(new string[] {
                "application corrupt " + id.ToString("x16") + " --patch",
            }));

            // 全コンテンツメタタイプ対象の破損チェックでは破損が検出される
            Assert.IsFalse(command.Execute(new string[] {
                "application verify " + id.ToString("x16"),
            }, new string[] {
                "The application is corrupted.",
            }));

            // アプリケーションが対象の破損チェックでは破損が検出されない
            Assert.IsTrue(command.Execute(new string[] {
                "application verify " + id.ToString("x16") + " --app",
            }, new string[] {
                "No corrupted contents.",
            }));

            // パッチが対象の破損チェックでは破損が検出される
            Assert.IsFalse(command.Execute(new string[] {
                "application verify " + id.ToString("x16") + " --patch",
            }, new string[] {
                "The application is corrupted.",
            }));

            // 追加コンテンツが対象の破損チェックでは破損が検出されない
            Assert.IsTrue(command.Execute(new string[] {
                "application verify " + id.ToString("x16") + " --aoc",
            }, new string[] {
                "No corrupted contents.",
            }));
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestVerifyApplicationWithCorruptedAddOnContent()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);
            var aocv0 = m_Maker.MakeAddOnContent(id, version: 1);
            var patchv1 = m_Maker.MakePatch(id, 1, appv0.Path);

            var command = new DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.Execute(new string[] {
                "application install " + appv0.Path,
                "application install " + aocv0.Path,
                "application install " + patchv1.Path,
            }));

            Assert.IsTrue(command.Execute(new string[] {
                "application verify " + id.ToString("x16"),
            }, new string[] {
                "No corrupted contents.",
            }));

            // 追加コンテンツを破損させる
            // 何故か fs::ResultTargetLocked で失敗する場合があるので Reset してから実行しておく
            Assert.IsTrue(command.ResetAndExecute(new string[] {
                "application corrupt " + id.ToString("x16") + " --aoc",
            }));

            // 全コンテンツメタタイプ対象の破損チェックでは破損が検出される
            Assert.IsFalse(command.Execute(new string[] {
                "application verify " + id.ToString("x16"),
            }, new string[] {
                "The application is corrupted.",
            }));

            // アプリケーションが対象の破損チェックでは破損が検出されない
            Assert.IsTrue(command.Execute(new string[] {
                "application verify " + id.ToString("x16") + " --app",
            }, new string[] {
                "No corrupted contents.",
            }));

            // パッチが対象の破損チェックでは破損が検出されない
            Assert.IsTrue(command.Execute(new string[] {
                "application verify " + id.ToString("x16") + " --patch",
            }, new string[] {
                "No corrupted contents.",
            }));

            // 追加コンテンツが対象の破損チェックでは破損が検出される
            Assert.IsFalse(command.Execute(new string[] {
                "application verify " + id.ToString("x16") + " --aoc",
            }, new string[] {
                "The application is corrupted.",
            }));
        }
    }
}
