﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Linq;
using System.Diagnostics;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Collections;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using DevMenuCommandTest;

namespace DevMenuCommandTestNuiShell
{
    [TestClass]
    public class TestApplicationView : TestBase
    {
        public override void DoCleanup()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            command.Execute(new string[] {
                "gamecard erase",
                "debug enable-jit-debug",
                "sdcard format",
            });
            command.ResetAndExecute(new string[] {
                "application uninstall --all",
                "application reset-required-version --all"
            });
        }


        [TestMethod]
        [Timeout(300 * 1000)]
        public void BasicTest()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);
            var appv1 = m_Maker.MakeApplication(id, 1);
            var aocv0 = m_Maker.MakeAddOnContent(id, 1, 0);
            var patchv2 = m_Maker.MakePatch(id, 2, appv0.Path);

            var command = new DevMenuCommandSystem(this.TestContext);

            // アプリケーションのインストール
            Assert.IsTrue(command.Execute(new string[]
            {
                "application install " + appv0.Path,
                "application list --detail",
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, ApplicationViewFlag.DefaultFlags | ApplicationViewFlag.ApplicationFlags)));

            // 追加コンテンツのインストール
            Assert.IsTrue(command.Execute(new string[]
            {
                "addoncontent install " + aocv0.Path,
                "application list --detail",
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, ApplicationViewFlag.DefaultFlags | ApplicationViewFlag.ApplicationFlags | ApplicationViewFlag.AocFlags)));

            // パッチのインストール
            Assert.IsTrue(command.Execute(new string[]
            {
                "patch install " + patchv2.Path,
                "application list --detail",
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, ApplicationViewFlag.AllInstalledFlags)));

            // ゲームカードの挿入
            // パッチの方が新しいが、アプリは同じなので、IsGameCard が立つ
            // GameCard のアタッチを促すため、コマンドを別々にする必要がある
            Assert.IsTrue(command.Execute(new string[]
            {
                "gamecard write " + appv0.Path,
            }));
            Assert.IsTrue(command.Execute(new string[]
            {
                "application list --detail",
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, ApplicationViewFlag.AllInstalledFlags | ApplicationViewFlag.GameCardFlags)));

            // ゲームカードの抜去
            Assert.IsTrue(command.Execute(new string[]
            {
                "gamecard erase",
            }));
            Assert.IsTrue(command.Execute(new string[]
            {
                "application list --detail",
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, ApplicationViewFlag.AllInstalledFlags)));

            // Patch だけでは起動できない
            Assert.IsTrue(command.Execute(new string[]
            {
                "application uninstall --all",
                "patch install " + patchv2.Path,
                "application list --detail"
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, ApplicationViewFlag.DefaultFlags | ApplicationViewFlag.PatchFlags)));

            // 追加コンテンツだけでは起動できない
            Assert.IsTrue(command.Execute(new string[]
            {
                "application uninstall --all",
                "addoncontent install " + aocv0.Path,
                "application list --detail"
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, ApplicationViewFlag.DefaultFlags | ApplicationViewFlag.AocFlags)));
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void PatchOnSdCard()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);
            var patchv2 = m_Maker.MakePatch(id, 2, appv0.Path);

            var command = new DevMenuCommandSystem(this.TestContext);
            var defaultFlags = ApplicationViewFlag.DefaultFlags | ApplicationViewFlag.ApplicationFlags;

            // アプリケーションのインストール
            Assert.IsTrue(command.Execute(new string[]
            {
                "application install " + appv0.Path,
                "application list --detail",
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, defaultFlags)));

            var patchedFlags = defaultFlags | ApplicationViewFlag.PatchFlags;

            // パッチのインストール
            Assert.IsTrue(command.Execute(new string[]
            {
                "patch install -s sdcard " + patchv2.Path,
                "application list --detail",
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, patchedFlags)));

            // SD カードの抜去によってパッチがなくなった
            Assert.IsTrue(command.Execute(new string[]
            {
                "sdcard detach"
            }));

            Assert.IsTrue(command.Execute(new string[]
            {
                "application list --detail",
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id,
                patchedFlags
                & ~ApplicationViewFlag.HasPatchEntity
                & ~ApplicationViewFlag.HasAllEntity
            )));

            // 再び挿入されていれば元に戻る
            Assert.IsTrue(command.ResetAndExecute(new string[]
            {
                "application list --detail",
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, patchedFlags)));
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void AocOnSdCard()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);
            var aocv0 = m_Maker.MakeAddOnContent(id, 1, 0);

            var command = new DevMenuCommandSystem(this.TestContext);
            var defaultFlags = ApplicationViewFlag.DefaultFlags | ApplicationViewFlag.ApplicationFlags;

            // アプリケーションのインストール
            Assert.IsTrue(command.Execute(new string[]
            {
                "application install " + appv0.Path,
                "application list --detail",
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, defaultFlags)));

            var aocFlags = defaultFlags | ApplicationViewFlag.AocFlags;
            // 追加コンテンツのインストール
            Assert.IsTrue(command.Execute(new string[]
            {
                "addoncontent install -s sdcard " + aocv0.Path,
                "application list --detail",
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, aocFlags)));

            // SD カードの抜去によって追加コンテンツがなくなった
            Assert.IsTrue(command.Execute(new string[]
            {
                "sdcard detach"
            }));

            Assert.IsTrue(command.Execute(new string[]
            {
                "application list --detail",
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id,
                aocFlags
                & ~ApplicationViewFlag.HasAllAddOnContentEntity
                & ~ApplicationViewFlag.HasAnyAddOnContentEntity
                & ~ApplicationViewFlag.HasAllEntity
            )));

            // 再び挿入されていれば元に戻る
            Assert.IsTrue(command.ResetAndExecute(new string[]
            {
                "application list --detail",
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, aocFlags)));
        }

        private void InvalidateFileCacheForWritingContents(DevMenuCommandSystem command, NspInfo dummy, string storage)
        {
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application install -s {0} {1}", storage, dummy.Path),
                "application uninstall " + dummy.Id
            }));
        }

        [TestMethod]
        [Timeout(1000 * 1000)]
        public void TestMaybeCorruptedAndVerification()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            ulong id = 0x0100394000059000;
            ulong dummyId = 0x010039400005a000;
            var appv0 = m_Maker.MakeApplication(id, 0);
            var aocv0 = m_Maker.MakeAddOnContent(id, 1, 0);
            var patchv1 = m_Maker.MakePatch(id, 1, appv0.Path);
            var dummy = m_Maker.MakeApplication(dummyId, 0);
            var viewFlags = ApplicationViewFlag.AllInstalledFlags;

            Assert.IsTrue(command.Execute(new string[] {
                "debug disable-jit-debug"
            }));
            Assert.IsTrue(command.ResetAndExecute(new string[] {
                "application install -s sdcard " + appv0.Path,
                "addoncontent install -s sdcard " + aocv0.Path,
                "patch install -s sdcard " + patchv1.Path,

                string.Format("application check-launch-version 0x{0:x16}", id),
                string.Format("application check-launch-rights 0x{0:x16}", id),
                string.Format("application verify 0x{0:x16}", id),
                "application list --detail"
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, viewFlags)));
            SuccessLaunchApplication(command, id);

            // 追加コンテンツは実際に使われるときまで影響しない
            InvalidateFileCacheForWritingContents(command, dummy, "sdcard");
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application corrupt 0x{0:x16} --aoc", id),
                string.Format("application check-launch-version 0x{0:x16}", id),
                string.Format("application check-launch-rights 0x{0:x16}", id),
            }));
            // 実行中に使われ DataCorrupted
            FailureLaunchApplication(command, id, true);
            Assert.IsTrue(command.Execute(new string[] {
                "application list --detail"
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, viewFlags | ApplicationViewFlag.MaybeCorrupted)));

            FailureExecuteCommands(command, new Tuple<string, string>[]
            {
                // ResultContentCorrupted
                new Tuple<string, string>(string.Format("application verify 0x{0:x16}", id), "Result: module = 16, description = 601"),
            });

            Assert.IsTrue(command.Execute(new string[]
            {
                string.Format("application clear-terminate-result 0x{0:x16}", id),
                "application list --detail"
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, viewFlags)));

            // 元に戻す
            Assert.IsTrue(command.Execute(new string[] {
                "addoncontent install -s sdcard " + aocv0.Path + " --force",
            }));

            // パッチが破壊されていると起動時に失敗する
            InvalidateFileCacheForWritingContents(command, dummy, "sdcard");
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application corrupt 0x{0:x16} --patch --code", id),
                string.Format("application check-launch-rights 0x{0:x16}", id),
                string.Format("application check-launch-version 0x{0:x16}", id),
            }));
            FailureLaunchApplication(command, id);

            // 実行中ではないので MaybeCorrupted が立つわけではない
            Assert.IsTrue(command.Execute(new string[] {
                "application list --detail"
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, viewFlags)));

            FailureExecuteCommands(command, new Tuple<string, string>[]
            {
                // ResultContentCorrupted
                new Tuple<string, string>(string.Format("application verify 0x{0:x16}", id), "Result: module = 16, description = 601")
            });

            // SD カードを抜いて再インストールする
            Assert.IsTrue(command.Execute(new string[] {
                "sdcard remove",
                "application install " + appv0.Path + " --force",
                "addoncontent install " + aocv0.Path + " --force",
                "patch install " + patchv1.Path + " --force",

                string.Format("application check-launch-version 0x{0:x16}", id),
                string.Format("application check-launch-rights 0x{0:x16}", id),
                string.Format("application verify 0x{0:x16}", id),
                "application list --detail"
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, viewFlags)));
            SuccessLaunchApplication(command, id);

            // アプリのコード領域破壊を破壊しても、実際にはパッチが使われるので起動出来てしまう
            InvalidateFileCacheForWritingContents(command, dummy, "builtin");
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application corrupt 0x{0:x16} --app --code", id),
                string.Format("application check-launch-rights 0x{0:x16}", id),
                string.Format("application check-launch-version 0x{0:x16}", id),
            }));
            SuccessLaunchApplication(command, id);
            Assert.IsTrue(command.Execute(new string[] {
                "application list --detail"
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, viewFlags)));

            // 検証すると破壊が確認できる
            FailureExecuteCommands(command, new Tuple<string, string>[]
            {
                // ResultContentCorrupted
                new Tuple<string, string>(string.Format("application verify 0x{0:x16}", id), "Result: module = 16, description = 601")
            });

            // SD カードを再度有効化して、ゲームカードに同じアプリケーションを書きこむ
            Assert.IsTrue(command.ResetAndExecute(new string[] {
                "gamecard write " + appv0.Path + " --on-card-patch " + patchv1.Path
            }));

            // ゲームカードが優先されるので、起動は出来る
            SuccessLaunchApplication(command, id);

            Assert.IsTrue(command.Execute(new string[] {
                "application list --detail"
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, viewFlags | ApplicationViewFlag.GameCardFlags)));

            // 破損チェックでは破損を見つけることが出来る
            FailureExecuteCommands(command, new Tuple<string, string>[]
            {
                // ResultContentCorrupted
                new Tuple<string, string>(string.Format("application verify 0x{0:x16}", id), "Result: module = 16, description = 601")
            });

            // ゲームカードを抜く
            Assert.IsTrue(command.Execute(new string[] {
                "gamecard erase"
            }));

            // 正常に戻す
            Assert.IsTrue(command.Execute(new string[] {
                "application install -s sdcard " + appv0.Path + " --force",
                "addoncontent install -s sdcard " + aocv0.Path + " --force",
                "patch install -s sdcard " + patchv1.Path + " --force",
            }));
            SuccessLaunchApplication(command, id);
            Assert.IsTrue(command.Execute(new string[] {
                "application list --detail"
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, viewFlags)));

            // SD カードのコンテンツを読めなくする
            InvalidateFileCacheForWritingContents(command, dummy, "sdcard");
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application corrupt 0x{0:x16} --sdcard", id),
                // バージョンチェックはストレージを読まないので成功する
                string.Format("application check-launch-version 0x{0:x16}", id)
            }));
            FailureExecuteCommands(command, new Tuple<string, string>[]
            {
                new Tuple<string, string>(string.Format("application check-launch-rights 0x{0:x16}", id), null),
                // ResultSdCardDataCorrupted
                new Tuple<string, string>(string.Format("application verify 0x{0:x16}", id), "Result: module = 16, description = 610")
            });

            FailureLaunchApplication(command, id);
            Assert.IsTrue(command.Execute(new string[] {
                "application list --detail"
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, viewFlags)));

            // 再起動すれば元に戻る
            // (DevMenuCommandSystem application corrupt --sdcard の仕様)
            Assert.IsTrue(command.ResetAndExecute(new string[] {
                string.Format("application check-launch-version 0x{0:x16}", id),
                string.Format("application check-launch-rights 0x{0:x16}", id),
                string.Format("application verify 0x{0:x16}", id),
            }));
            SuccessLaunchApplication(command, id);
            Assert.IsTrue(command.Execute(new string[] {
                "application list --detail"
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, viewFlags)));

            // 想定外のエラーの確認
            InvalidateFileCacheForWritingContents(command, dummy, "sdcard");
            Assert.IsTrue(command.Execute(new string[] {
                "filesystem delete-directory -s sdcard /Nintendo/Contents/registered",
            }));
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application check-launch-version 0x{0:x16}", id)
            }));
            FailureExecuteCommands(command, new Tuple<string, string>[] {
                new Tuple<string, string>(string.Format("application check-launch-rights 0x{0:x16}", id), null),
                // ResultVerifyContentUnexpected
                new Tuple<string, string>(string.Format("application verify 0x{0:x16}", id), "Result: module = 16, description = 602"),
            });

            FailureLaunchApplication(command, id);
            Assert.IsTrue(command.Execute(new string[] {
                "application list --detail"
            }, ApplicationViewFlag.MakeApplicationViewFlagTestString(id, viewFlags)));
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestApplicationViewCacheForSdCard()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);
            var patchv1 = m_Maker.MakePatch(id, 1, appv0.Path);

            Assert.IsTrue(command.Execute(new string[] {
                "application install " + appv0.Path,
                "patch install -s sdcard " + patchv1.Path,
            }));

            {
                Assert.IsTrue(command.Execute(new string[] {
                    "application list-view"
                }));
                var view = ApplicationViewForJson.Deserialize(command.LastOutput).Single(x => x.id == string.Format("0x{0:x16}", id));
                Assert.IsTrue(view.hasMainEntity);
                Assert.IsTrue(view.hasPatchEntity);
            }

            Assert.IsTrue(command.Execute(new string[] {
                "sdcard remove"
            }));

            {
                Assert.IsTrue(command.Execute(new string[] {
                    "application list-view"
                }));
                var view = ApplicationViewForJson.Deserialize(command.LastOutput).Single(x => x.id == string.Format("0x{0:x16}", id));
                Assert.IsTrue(view.hasMainEntity);
                Assert.IsTrue(!view.hasPatchEntity);
            }
        }
    }
}
