﻿using System;
using System.IO;
using System.Linq;
using System.Collections.Generic;
using System.Runtime.Serialization.Json;
using System.Threading;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using DevMenuCommandTest;

namespace DevMenuCommandTestNuiShell
{
    [TestClass]
    public class TestApplicationLaunch : TestBase
    {
        public const string MsgSystemUpdate = "System update required.";
        public const string MsgApplicationUpdateRequired = "Application update required.";
        public const string MsgApplicationUpdateRecommended = "Application update recommended.";

        public const string MsgRequiredVersion = "Required version is not satisfied.";
        public const string MsgRequiredApplicationVersion = "Required Application version is not satisfied.";

        public const string ResetVersionListFileName = "reset.json";

        public override void DoCleanup()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            command.Execute(new string[] {
                "debug disable-rid-mode",
                "systemupdate set-debug-id 0x0000000000000000-0",
            });
            command.ResetAndExecute(new string[] {
                "gamecard erase",
                "application uninstall --all",
                "ticket delete-all",
                "application reset-required-version --all",
            });
        }

        private string[] GetTestStorage()
        {
            return IsLongTest ? StoragesWithGc : Storages;
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void TestLaunchRights()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0, ticket: true);
            var aocv0 = m_Maker.MakeAddOnContent(id, version: 1, ticket: true);
            var patchv1 = m_Maker.MakePatch(id, 1, appv0.Path, ticket: true);

            var command = new DevMenuCommandSystem(this.TestContext);

            CheckFunc SuccessCheckLaunchRights = argId =>
            {
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("application check-launch-rights 0x{0:x16}", argId),
                }));
            };
            CheckFunc FailureByNoLaunchRights = argId =>
            {

                Assert.IsTrue(command.FailureExecute(string.Format("application check-launch-rights 0x{0:x16}", id), "RightsNotFound")); // nn::ns::ResultApplicationTicketNotFound
            };

            foreach (var storage in Storages)
            {
                //
                // アプリケーション
                //
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("application install {0} -s {1}", appv0.Path, storage),
                    string.Format("application check-launch-rights 0x{0:x16}", id),
                }));
                SuccessLaunchApplication(command, id);

                // チケットがない場合
                Assert.IsTrue(command.Execute(new string[] {
                    "ticket delete-all",
                }));
                FailureByNoLaunchRights(id);
                FailureLaunchApplication(command, id);

                //
                // 追加コンテンツ
                //
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("application install {0} -s {1} --force", appv0.Path, storage),
                    string.Format("addoncontent install {0} -s {1}", aocv0.Path, storage),
                }));
                SuccessCheckLaunchRights(id);
                SuccessLaunchApplication(command, id);

                // チケットがない場合
                Assert.IsTrue(command.Execute(new string[] {
                    "application uninstall --all",
                    string.Format("addoncontent install {0} -s {1}", aocv0.Path, storage),
                    "ticket delete-all",
                    string.Format("application install {0} -s {1}", appv0.Path, storage),
                }));
                SuccessCheckLaunchRights(id);
                // 起動に成功する (追加コンテンツは権利のあるものだけリストアップするという仕様なので、実行は出来る)
                SuccessLaunchApplication(command, id);

                //
                // パッチ
                //
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("addoncontent install {0} -s {1} --force", aocv0.Path, storage),
                    string.Format("patch install {0} -s {1}", patchv1.Path, storage),
                }));
                SuccessCheckLaunchRights(id);
                SuccessLaunchApplication(command, id);

                Assert.IsTrue(command.Execute(new string[] {
                    "application uninstall --all",
                    string.Format("patch install {0} -s {1}", patchv1.Path, storage),
                    "ticket delete-all",
                    string.Format("application install {0} -s {1}", appv0.Path, storage),
                    string.Format("addoncontent install {0} -s {1}", aocv0.Path, storage),
                }));
                FailureByNoLaunchRights(id);
                FailureLaunchApplication(command, id);

                // Cleanup
                Assert.IsTrue(command.Execute(new string[] {
                    "application uninstall --all",
                    "application reset-required-version --all",
                }));
            }

            // ゲームカード
            var appv1NoTicket = m_Maker.MakeApplication(id, 1);
            var patchv2 = m_Maker.MakePatch(id, 2, appv1NoTicket.Path, ticket: true);

            // アプリはカードに焼かれるときにチケットが参照できるパスがないので、テストしない
            // OnCardPach のみテストする
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("gamecard write {0} --on-card-patch {1}", appv1NoTicket.Path, patchv2.Path),
            }));
            SuccessCheckLaunchRights(id);
            SuccessLaunchApplication(command, id);

            // ゲームカードのチケットは消えない
            Assert.IsTrue(command.Execute(new string[] {
                "ticket delete-all",
            }));

            SuccessCheckLaunchRights(id);
            SuccessLaunchApplication(command, id);
        }

        private void TestCheckLaunchVersion(DevMenuCommandSystem command, ulong id, bool hasMsg, string[] msgList)
        {
            if (hasMsg)
            {
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("application check-launch-version 0x{0:x16}", id),
                },
                    msgList
                ));
            }
            else
            {
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("application check-launch-version 0x{0:x16}", id),
                }));
                Assert.IsFalse(command.Execute(new string[] {
                    string.Format("application check-launch-version 0x{0:x16}", id),
                },
                    msgList
                ));
            };
        }

        private void TestRequiredSystemVersionImpl(DevMenuCommandSystem command, NspInfo app, ulong id, string storage, bool requiresUpdate)
        {
            TestRequiredSystemVersionImpl(command, app, null, null, id, storage, null, requiresUpdate);
        }

        private void TestRequiredSystemVersionImpl(DevMenuCommandSystem command, NspInfo app, NspInfo patch, NspInfo dummy, ulong id, string appStorage, string patchStorage, bool requiresUpdate)
        {
            Install(command, app, patch, dummy, appStorage, patchStorage);

            TestCheckLaunchVersion(command, id, requiresUpdate, new string[] { MsgSystemUpdate });

            Uninstall(command, id, appStorage, patchStorage);
        }

        public void TestRequiredSystemVersionForRunnerToolImpl(TestContext context, NspInfo nspPath, bool isRequiredUpdateExpected)
        {
            string targetName = (string)this.TestContext.Properties["TargetName"];
            var command = new RunOnTarget(context, nspPath.Path, targetName);
            var requiresUpdateLog = "[ERROR] Command(LaunchApplicationCommand) failed(nn::Result=0x00015410)";
            var output = command.Execute(string.Empty, false, new string[] { " --no-wait " });

            bool requiredUpdate;
            if (output.Item1)
            {
                requiredUpdate = false;
            }
            else
            {
                if (output.Item3.ToString().Contains(requiresUpdateLog))
                {
                    requiredUpdate = true;
                }
                else
                {
                    Assert.Fail("RunOnTarget.exe was not executed by anathor reason of required system update:\n" + output.Item3.ToString());
                    requiredUpdate = false;
                }
            }

            Assert.IsTrue(requiredUpdate == isRequiredUpdateExpected);
        }

        [TestMethod, TestCategory("LongTest")]
        [Timeout(3600 * 1000)]
        public void TestRequiredSystemVersion()
        {
            ulong id = 0x0100394000059000;
            var command = new DevMenuCommandSystem(this.TestContext);

            // version がかぶると、同じアプリケーション名になって、上書きされてしまうので、バージョンをインクリメントして宣言する
            var dummyv0 = m_Maker.MakeApplication(id, 0);
            var appv0Required = m_Maker.MakeApplication(id, 1, requiredSystemVersion: 0);
            var appv1Required = m_Maker.MakeApplication(id, 2, requiredSystemVersion: 1);
            var appv2Required = m_Maker.MakeApplication(id, 3, requiredSystemVersion: 2);
            var appv65535Required = m_Maker.MakeApplication(id, 5, requiredSystemVersion: UInt16.MaxValue);
            var appv65536Required = m_Maker.MakeApplication(id, 6, requiredSystemVersion: UInt16.MaxValue + 1);
            var appv65537Required = m_Maker.MakeApplication(id, 7, requiredSystemVersion: UInt16.MaxValue + 2);
            var appv4294967295Required = m_Maker.MakeApplication(id, 8, requiredSystemVersion: UInt32.MaxValue);
            var patchv4294967295Required = m_Maker.MakePatch(id, 9, appv0Required.Path, requiredSystemVersion: UInt32.MaxValue);
            var patchv0Required = m_Maker.MakePatch(id, 10, appv4294967295Required.Path);

            foreach (var appStorage in GetTestStorage().OrderBy(i => Guid.NewGuid()).ToArray())
            {
                foreach (var patchStorage in GetTestStorage().OrderBy(i => Guid.NewGuid()).ToArray())
                {
                    // バージョン を 0 にする
                    Assert.IsTrue(command.Execute(new string[] {
                        "systemupdate set-debug-id 0x0000000000000001-0"
                    }));

                    TestRequiredSystemVersionImpl(command, appv0Required, id, appStorage, false);
                    TestRequiredSystemVersionImpl(command, appv1Required, id, appStorage, true);
                    TestRequiredSystemVersionImpl(command, appv65536Required, id, appStorage, true);
                    TestRequiredSystemVersionImpl(command, appv4294967295Required, id, appStorage, true);

                    // バージョン を 65536 にする (1 << 16)
                    Assert.IsTrue(command.Execute(new string[] {
                        "systemupdate set-debug-id 0x0000000000000001-65536"
                    }));
                    TestRequiredSystemVersionImpl(command, appv0Required, id, appStorage, false);
                    TestRequiredSystemVersionImpl(command, appv1Required, id, appStorage, false);
                    TestRequiredSystemVersionImpl(command, appv2Required, id, appStorage, false); // Release のような処理がされていない
                    TestRequiredSystemVersionImpl(command, appv65535Required, id, appStorage, false);
                    TestRequiredSystemVersionImpl(command, appv65536Required, id, appStorage, false);
                    TestRequiredSystemVersionImpl(command, appv65537Required, id, appStorage, true);

                    // バージョン を 4294901760 にする (0xFFFF << 16)
                    Assert.IsTrue(command.Execute(new string[] {
                        "systemupdate set-debug-id 0x0000000000000001-4294901760"
                    }));
                    TestRequiredSystemVersionImpl(command, appv0Required, id, appStorage, false);
                    TestRequiredSystemVersionImpl(command, appv65535Required, id, appStorage, false);
                    TestRequiredSystemVersionImpl(command, appv4294967295Required, id, appStorage, true);

                    // バージョン を 4294967295 にする (0xFFFFFFFF)
                    Assert.IsTrue(command.Execute(new string[] {
                        "systemupdate set-debug-id 0x0000000000000001-4294967295"
                    }));
                    TestRequiredSystemVersionImpl(command, appv0Required, id, appStorage, false);
                    TestRequiredSystemVersionImpl(command, appv65535Required, id, appStorage, false);
                    TestRequiredSystemVersionImpl(command, appv4294967295Required, id, appStorage, false);


                    Assert.IsTrue(command.Execute(new string[] {
                        "systemupdate set-debug-id 0x0000000000000000-0"
                    }));

                    // パッチ
                    TestRequiredSystemVersionImpl(command, appv0Required, patchv4294967295Required, dummyv0, id, appStorage, patchStorage, true);

                    // Patch には必須システムバージョンが書いていないけど、アプリには書いてある
                    // 実運用上ありえないはずだが、もしあった場合は、パッチの情報のみが使われるので、引っかからない
                    TestRequiredSystemVersionImpl(command, appv4294967295Required, patchv0Required, dummyv0, id, appStorage, patchStorage, false);

                    if (!IsLongTest)
                    {
                        return;
                    }
                }
            }
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestRequiredSystemVersionForRunnerTool()
        {
            ulong id = 0x0100394000059000;
            var devMenuCommandSystem = new DevMenuCommandSystem(this.TestContext);

            // version がかぶると、同じアプリケーション名になって、上書きされてしまうので、バージョンをインクリメントして宣言する
            var appv0Required = m_Maker.MakeApplication(id, 1, requiredSystemVersion: 0);
            var appv1Required = m_Maker.MakeApplication(id, 2, requiredSystemVersion: 1);
            var appv2Required = m_Maker.MakeApplication(id, 3, requiredSystemVersion: 2);
            var appv65535Required = m_Maker.MakeApplication(id, 5, requiredSystemVersion: UInt16.MaxValue);
            var appv65536Required = m_Maker.MakeApplication(id, 6, requiredSystemVersion: UInt16.MaxValue + 1);
            var appv65537Required = m_Maker.MakeApplication(id, 7, requiredSystemVersion: UInt16.MaxValue + 2);
            var appv4294967295Required = m_Maker.MakeApplication(id, 8, requiredSystemVersion: UInt32.MaxValue);

            // バージョン を 0 にする
            Assert.IsTrue(devMenuCommandSystem.Execute(new string[] {
                "systemupdate set-debug-id 0x0000000000000001-0"
            }));

            TestRequiredSystemVersionForRunnerToolImpl(this.TestContext, appv0Required, false);
            TestRequiredSystemVersionForRunnerToolImpl(this.TestContext, appv1Required, true);
            TestRequiredSystemVersionForRunnerToolImpl(this.TestContext, appv65535Required, true);
            TestRequiredSystemVersionForRunnerToolImpl(this.TestContext, appv4294967295Required, true);

            // バージョン を 65536 にする (1 << 16)
            Assert.IsTrue(devMenuCommandSystem.Execute(new string[] {
                "systemupdate set-debug-id 0x0000000000000001-65536"
            }));

            TestRequiredSystemVersionForRunnerToolImpl(this.TestContext, appv0Required, false);
            TestRequiredSystemVersionForRunnerToolImpl(this.TestContext, appv1Required, false);
            TestRequiredSystemVersionForRunnerToolImpl(this.TestContext, appv2Required, false);
            TestRequiredSystemVersionForRunnerToolImpl(this.TestContext, appv65535Required, false);
            TestRequiredSystemVersionForRunnerToolImpl(this.TestContext, appv65536Required, false);
            TestRequiredSystemVersionForRunnerToolImpl(this.TestContext, appv65537Required, true);

            // バージョン を 4294901760 にする (0xFFFF << 16)
            Assert.IsTrue(devMenuCommandSystem.Execute(new string[] {
                "systemupdate set-debug-id 0x0000000000000001-4294901760"
            }));

            TestRequiredSystemVersionForRunnerToolImpl(this.TestContext, appv0Required, false);
            TestRequiredSystemVersionForRunnerToolImpl(this.TestContext, appv65535Required, false);
            TestRequiredSystemVersionForRunnerToolImpl(this.TestContext, appv4294967295Required, true);

            // バージョン を 4294967295 にする (0xFFFFFFFF)
            Assert.IsTrue(devMenuCommandSystem.Execute(new string[] {
                "systemupdate set-debug-id 0x0000000000000001-4294967295"
            }));

            TestRequiredSystemVersionForRunnerToolImpl(this.TestContext, appv0Required, false);
            TestRequiredSystemVersionForRunnerToolImpl(this.TestContext, appv65535Required, false);
            TestRequiredSystemVersionForRunnerToolImpl(this.TestContext, appv4294967295Required, false);
        }

        [TestMethod, TestCategory("LongTest")]
        [Timeout(3600 * 1000)]
        public void TestRequiredVersion()
        {
            ulong id = 0x0100394000059000;
            var command = new DevMenuCommandSystem(this.TestContext);

            var dummyv0 = m_Maker.MakeApplication(id, 0);
            var appv1 = m_Maker.MakeApplication(id, 1);
            var patchv2 = m_Maker.MakePatch(id, 2, appv1.Path);

            var msgList = new string[]
            {
                MsgApplicationUpdateRequired,
                MsgRequiredVersion,
            };

            foreach (var appStorage in GetTestStorage().OrderBy(i => Guid.NewGuid()).ToArray())
            {
                foreach (var patchStorage in GetTestStorage().OrderBy(i => Guid.NewGuid()).ToArray())
                {
                    Install(command, appv1, appStorage);
                    // 要求されていない
                    TestCheckLaunchVersion(command, id, false, msgList);

                    // アプリを起動しても変更されないことを念のため確認
                    SuccessLaunchApplication(command, id);
                    TestCheckLaunchVersion(command, id, false, msgList);

                    if (!(patchStorage == GameCardName && appStorage == patchStorage))
                    {
                        // パッチをインストールするだけでは必須バージョンはあがらない
                        Install(command, null, patchv2, dummyv0, null, patchStorage);
                        RemovePatch(command, id, patchStorage);

                        TestCheckLaunchVersion(command, id, false, msgList);

                        // パッチをインストールして起動すると必須バージョンが上がっている
                        Install(command, null, patchv2, dummyv0, null, patchStorage);
                        SuccessLaunchApplication(command, id);
                        RemovePatch(command, id, patchStorage);
                        TestCheckLaunchVersion(command, id, true, msgList);
                    }

                    Uninstall(command, id, appStorage, patchStorage);

                    Assert.IsTrue(command.Execute(new string[] {
                        "application reset-required-version --all"
                    }));

                    if (!IsLongTest)
                    {
                        break;
                    }
                }

                // 境界値チェック
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("application set-required-version 0x{0:x16} -v 4294967295", id) // 0xFFFFFFFF
                }));

                Install(command, dummyv0, appStorage);

                TestCheckLaunchVersion(command, id, true, msgList);

                Assert.IsTrue(command.Execute(new string[] {
                    "application reset-required-version --all"
                }));

                Uninstall(command, id, appStorage, null);

                if (!IsLongTest)
                {
                    return;
                }
            }
        }

        public void TestRequiredApplicationVersionImpl(DevMenuCommandSystem command, ulong id, NspInfo app, NspInfo aoc, string appStorage, string aocStorage, bool requiresUpdate)
        {
            Install(command, app, appStorage);
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("addoncontent install -s {0} {1}", aocStorage, aoc.Path),
            }));

            var msgList = new string[]
            {
                MsgApplicationUpdateRequired,
                MsgRequiredApplicationVersion,
            };

            TestCheckLaunchVersion(command, id, requiresUpdate, msgList);

            Uninstall(command, id, appStorage, null);
        }

        [TestMethod, TestCategory("LongTest")]
        [Timeout(3600 * 1000)]
        public void TestRequiredApplicationVersion()
        {
            ulong id = 0x0100394000059000;
            var command = new DevMenuCommandSystem(this.TestContext);

            var appv0 = m_Maker.MakeApplication(id, 0);
            var appv65535 = m_Maker.MakeApplication(id, 65535);

            // 上書きされないようにバージョンをインクリメントしておく
            var aocv0Required = m_Maker.MakeAddOnContent(id, version: 0, requiredApplicationVersion: 0);
            var aocv1Required = m_Maker.MakeAddOnContent(id, version: 1, requiredApplicationVersion: 1);
            var aocv65535Required = m_Maker.MakeAddOnContent(id, version: 3, requiredApplicationVersion: UInt16.MaxValue);

            foreach (var appStorage in GetTestStorage())
            {
                foreach (var aocStorage in Storages)
                {
                    TestRequiredApplicationVersionImpl(command, id, appv0, aocv0Required, appStorage, aocStorage, false);
                    TestRequiredApplicationVersionImpl(command, id, appv0, aocv1Required, appStorage, aocStorage, true);
                    TestRequiredApplicationVersionImpl(command, id, appv0, aocv65535Required, appStorage, aocStorage, true);

                    TestRequiredApplicationVersionImpl(command, id, appv65535, aocv65535Required, appStorage, aocStorage, false);
                    TestRequiredApplicationVersionImpl(command, id, appv65535, aocv0Required, appStorage, aocStorage, false);

                    if (!IsLongTest)
                    {
                        return;
                    }
                }
            }
        }

        private static VersionList MakeVersionList(ulong id, UInt16 releaseVersion, UInt16 privateVersion = 0)
        {
            VersionList list = new VersionList();
            list.Titles = new TitleObject[1];
            list.Titles[0] = new TitleObject();
            list.Titles[0].Id = string.Format("0x{0:x16}", id);
            list.Titles[0].Version = (UInt32)(releaseVersion) << 16 | privateVersion;

            return list;
        }

        private static void WriteVersionList(string filePath, ulong id, UInt16 version)
        {
            VersionList list = MakeVersionList(id, version);
            if (File.Exists(filePath))
            {
                return;
            }
            using (var fs = File.Open(filePath, FileMode.Create))
            {
                var serializer = new DataContractJsonSerializer(typeof(VersionList));
                serializer.WriteObject(fs, list);
                fs.Flush();
            }
        }

        private void TestUpdateVersionListImpl(DevMenuCommandSystem command, ulong id, NspInfo app, string storage, bool recommendsUpdate)
        {
            Install(command, app, storage);

            var msgList = new string[]
            {
                MsgApplicationUpdateRecommended
            };

            TestCheckLaunchVersion(command, id, recommendsUpdate, msgList);

            Uninstall(command, id, storage, null);
        }

        public void TestUpdateVersionList(DevMenuCommandSystem command, MakeTestApplication m_Maker, ulong id, UInt16 version, string resetPath)
        {
            var outputDir = m_Maker.GetOutputDirectory().Replace("\"", "");
            var listPath = Path.Combine(outputDir, string.Format("listv{0}.json", version));
            WriteVersionList(listPath, id, version);

            int versionSmall = (UInt16)(version - 1);
            int versionEqual = version;
            int versionBig = (UInt16)(version + 1);
            var appSmall = m_Maker.MakeApplication(id, versionSmall);
            var appEqual = m_Maker.MakeApplication(id, versionEqual);
            var appBig = m_Maker.MakeApplication(id, versionBig);

            Assert.IsTrue(command.Execute(new string[] {
                "application update-version-list " + listPath
            }));

            foreach (var storage in GetTestStorage().OrderBy(i => Guid.NewGuid()).ToArray())
            {
                TestUpdateVersionListImpl(command, id, appSmall, storage, versionSmall < versionEqual);

                TestUpdateVersionListImpl(command, id, appEqual, storage, false);

                TestUpdateVersionListImpl(command, id, appBig, storage, versionBig < versionEqual);

                if (!IsLongTest)
                {
                    break;
                }
            }

            Assert.IsTrue(command.Execute(new string[] {
                "application update-version-list " + resetPath
            }));
            File.Delete(listPath);
        }

        private class ResetVersinonList : IDisposable
        {
            public string ResetPath;
            DevMenuCommandSystem Command;

            public ResetVersinonList(DevMenuCommandSystem command, MakeTestApplication maker, ulong id)
            {
                var outputDir = maker.GetOutputDirectory().Replace("\"", "");
                ResetPath = Path.Combine(outputDir, ResetVersionListFileName);
                Command = command;

                WriteVersionList(ResetPath, id, 0);
            }

            public void Dispose()
            {
                Reset();
            }

            public void Reset()
            {
                if (!string.IsNullOrEmpty(ResetPath))
                {
                    Command.Execute(new string[] {
                        "application update-version-list " + ResetPath
                    });
                }
            }
        }

        [TestMethod, TestCategory("LongTest")]
        [Timeout(3600 * 1000)]
        public void TestRecommendedVersion()
        {
            ulong id = 0x0100394000059000;
            var command = new DevMenuCommandSystem(this.TestContext);

            ResetVersinonList resetList = new ResetVersinonList(command, m_Maker, id);

            TestUpdateVersionList(command, m_Maker, id, 0, resetList.ResetPath);
            TestUpdateVersionList(command, m_Maker, id, 65535, resetList.ResetPath);
        }

        [TestMethod, TestCategory("LongTest")]
        [Timeout(3600 * 1000)]
        public void TestRequiredAndRecommendedVersion()
        {
            ulong id = 0x0100394000059000;
            var command = new DevMenuCommandSystem(this.TestContext);

            var dummyv0 = m_Maker.MakeApplication(id, 0);
            var appv1 = m_Maker.MakeApplication(id, requiredSystemVersion: 1);
            var patchv2 = m_Maker.MakePatch(id, 2, appv1.Path);
            var aocv1 = m_Maker.MakeAddOnContent(id, version: 1, requiredApplicationVersion: 2);
            var outputDir = m_Maker.GetOutputDirectory().Replace("\"", "");
            var listPath = Path.Combine(outputDir, string.Format("listv{0}.json", 1));
            WriteVersionList(listPath, id, 1);
            Assert.IsTrue(command.Execute(new string[] {
                "application update-version-list " + listPath
            }));
            var msgRequiredSystemUpdateList = new string[] { MsgSystemUpdate };
            var msgRequiredVersionList = new string[] { MsgApplicationUpdateRequired, MsgRequiredVersion };
            var msgRequiredApplicationVersionList = new string[] { MsgApplicationUpdateRequired, MsgRequiredApplicationVersion };
            var msgRecommendedVersion = new string[] { MsgApplicationUpdateRecommended };

            foreach (var storage in GetTestStorage().OrderBy(i => Guid.NewGuid()).ToArray())
            {
                // バージョン を 0 にする
                Assert.IsTrue(command.Execute(new string[] {
                    "systemupdate set-debug-id 0x0000000000000001-0"
                }));

                Install(command, appv1, storage);

                // 必須システムバージョンが優先される
                TestCheckLaunchVersion(command, id, true, msgRequiredSystemUpdateList);

                // バージョン を 1 にする
                Assert.IsTrue(command.Execute(new string[] {
                    "systemupdate set-debug-id 0x0000000000000001-1"
                }));

                TestCheckLaunchVersion(command, id, true, msgRecommendedVersion);
                Uninstall(command, id, storage, null);

                foreach (var extraStorage in GetTestStorage().OrderBy(i => Guid.NewGuid()).ToArray())
                {
                    // 必須バージョンが優先される
                    if (!(extraStorage == GameCardName && storage == extraStorage))
                    {
                        Install(command, appv1, patchv2, dummyv0, storage, extraStorage);
                        SuccessLaunchApplication(command, id);
                        RemovePatch(command, id, extraStorage);

                        TestCheckLaunchVersion(command, id, true, msgRequiredVersionList);

                        command.Execute(new string[] {
                            "application reset-required-version --all",
                        });

                        TestCheckLaunchVersion(command, id, true, msgRecommendedVersion);
                    }
                    else
                    {
                        Install(command, appv1, storage);
                    }

                    if (extraStorage != GameCardName)
                    {
                        // 必須アプリケーションバージョンが優先される

                        Assert.IsTrue(command.Execute(new string[] {
                            string.Format("addoncontent install -s {0} {1}", extraStorage, aocv1.Path),
                        }));

                        TestCheckLaunchVersion(command, id, true, msgRequiredApplicationVersionList);
                    }
                    Uninstall(command, id, storage, extraStorage);

                    if (!IsLongTest)
                    {
                        return;
                    }
                }
            }
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestRetailInteractiveDisplayApplication()
        {
            ulong id = 0x0100394000059000;
            var metaPath = Path.Combine(m_Maker.GetOutputDirectory().Replace("\"", ""), "quest.nmeta");
            // NXFP2-a64 であることを仮定する
            using (StreamWriter sw = new StreamWriter(metaPath, false, System.Text.Encoding.UTF8))
            {
                sw.WriteLine("<NintendoSdkMeta>");
                sw.WriteLine("  <Core>");
                sw.WriteLine("      <Name>Application</Name>");
                sw.WriteLine("      <ApplicationId>0x{0:X16}</ApplicationId>", id);
                sw.WriteLine("      <Is64BitInstruction>True</Is64BitInstruction>");
                sw.WriteLine("      <ProcessAddressSpace>AddressSpace64Bit</ProcessAddressSpace>");
                sw.WriteLine("  </Core>");
                sw.WriteLine("  <Application>");
                sw.WriteLine("      <Title>");
                sw.WriteLine("          <Language>AmericanEnglish</Language>");
                sw.WriteLine("          <Name>AmericanEnglish</Name>");
                sw.WriteLine("          <Publisher>Publisher</Publisher>");
                sw.WriteLine("      </Title>");
                sw.WriteLine("      <Icon>");
                sw.WriteLine("          <Language>AmericanEnglish</Language>");
                sw.WriteLine("          <IconPath>{0}/../../../Programs/Iris/Resources/SpecFiles/NintendoSDK_Application.bmp</IconPath>", m_Maker.GetOutputDirectory().Replace("\"", "").Replace("\\", "/"));
                sw.WriteLine("      </Icon>");
                sw.WriteLine("      <DisplayVersion>1.0.0</DisplayVersion>");
                sw.WriteLine("      <SupportedLanguage>AmericanEnglish</SupportedLanguage>");
                sw.WriteLine("      <Attribute>RetailInteractiveDisplay</Attribute>");
                sw.WriteLine("  </Application>");
                sw.WriteLine("</NintendoSdkMeta>");
            }

            var app = m_Maker.MakeApplication(id, metaFilePath: metaPath);

            var command = new DevMenuCommandSystem(this.TestContext);
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application install {0}", app.Path),
            }));

            FailureLaunchApplication(command, id, false, 0x0004ec10); // ResultRetailInteractiveDisplayApplicationNotPermitted

            Assert.IsTrue(command.Execute(new string[] {
                "debug enable-rid-mode",
                "power reboot"
            }));

            Thread.Sleep(10 * 1000);

            SuccessLaunchApplication(command, id);
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void TestResumeRights()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0, ticket: true);
            var aocv0 = m_Maker.MakeAddOnContent(id, version: 1, ticket: true);
            var patchv1 = m_Maker.MakePatch(id, 1, appv0.Path, ticket: true);

            var command = new DevMenuCommandSystem(this.TestContext);

            CheckFunc SuccessCheckResumeRights = argId =>
            {
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("application check-resume-rights 0x{0:x16}", argId),
                }));
            };
            CheckFunc FailureByNoResumeRights = argId =>
            {

                Assert.IsTrue(command.FailureExecute(string.Format("application check-resume-rights 0x{0:x16}", id), "RightsNotFound")); // nn::ns::ResultApplicationTicketNotFound
            };

            foreach (var storage in Storages)
            {
                //
                // アプリケーション
                //
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("application install {0} -s {1}", appv0.Path, storage),
                    string.Format("application check-launch-rights 0x{0:x16}", id),
                }));
                SuccessLaunchApplication(command, id);

                // チケットがない場合
                Assert.IsTrue(command.Execute(new string[] {
                    "ticket delete-all",
                }));
                FailureByNoResumeRights(id);
                FailureLaunchApplication(command, id);

                //
                // 追加コンテンツ
                //
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("application install {0} -s {1} --force", appv0.Path, storage),
                    string.Format("addoncontent install {0} -s {1}", aocv0.Path, storage),
                }));
                SuccessCheckResumeRights(id);
                SuccessLaunchApplication(command, id);

                // AOC のチケットがない場合
                Assert.IsTrue(command.Execute(new string[] {
                    "application uninstall --all",
                    string.Format("addoncontent install {0} -s {1}", aocv0.Path, storage),
                    "ticket delete-all",
                    string.Format("application install {0} -s {1}", appv0.Path, storage),
                }));
                SuccessCheckResumeRights(id);
                // 起動に成功する (追加コンテンツは権利のあるものだけリストアップするという仕様なので、実行は出来る)
                SuccessLaunchApplication(command, id);

                //
                // パッチ
                //
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("addoncontent install {0} -s {1} --force", aocv0.Path, storage),
                    string.Format("patch install {0} -s {1}", patchv1.Path, storage),
                }));
                SuccessCheckResumeRights(id);
                SuccessLaunchApplication(command, id);

                Assert.IsTrue(command.Execute(new string[] {
                    "application uninstall --all",
                    string.Format("patch install {0} -s {1}", patchv1.Path, storage),
                    "ticket delete-all",
                    string.Format("application install {0} -s {1}", appv0.Path, storage),
                    string.Format("addoncontent install {0} -s {1}", aocv0.Path, storage),
                }));
                // アプリの Program だけを見ているので、パッチだけ権限がないというケースは通過する
                // このようなケースは運用上はないはず
                SuccessCheckResumeRights(id);
                FailureLaunchApplication(command, id);

                // Cleanup
                Assert.IsTrue(command.Execute(new string[] {
                    "application uninstall --all",
                    "application reset-required-version --all",
                }));
            }

            // ゲームカード
            var appv1NoTicket = m_Maker.MakeApplication(id, 1);
            var patchv2 = m_Maker.MakePatch(id, 2, appv1NoTicket.Path, ticket: true);

            // アプリはカードに焼かれるときにチケットが参照できるパスがないので、テストしない
            // OnCardPach のみテストする
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("gamecard write {0} --on-card-patch {1}", appv1NoTicket.Path, patchv2.Path),
            }));
            SuccessCheckResumeRights(id);
            SuccessLaunchApplication(command, id);

            // ゲームカードのチケットは消えない
            Assert.IsTrue(command.Execute(new string[] {
                "ticket delete-all",
            }));

            SuccessCheckResumeRights(id);
            SuccessLaunchApplication(command, id);
        }
    }
}
