﻿using System;
using System.Linq;
using System.Threading;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using DevMenuCommandTest;

namespace DevMenuCommandTestNuiShell
{
    [TestClass]
    public class TestApplicationEntity : TestBase
    {
        public override void DoCleanup()
        {
            var command = new DevMenuCommandSystem(this.TestContext);

            command.Execute(new string[] {
                "debug enable-mount-sdcard",
                "sdcard format"
            });
            command.ResetAndExecute(new string[] {
                "gamecard erase",
                "application uninstall --all",
                "application reset-required-version --all",
                "application delete-redundant-entity",
            });
        }

        private void PrepareRedundantEnvironment(DevMenuCommandSystem command, NspInfo appv0, NspInfo appv1)
        {
            var commands = new string[] {
                "application uninstall --all",
                "application install " + appv1.Path + " -s sdcard",
                "application install-redundant-entity " + appv0.Path + " -s sdcard",
                "application install-redundant-entity " + appv1.Path + " -s builtin",
            };

            Assert.IsTrue(command.Execute(commands));

            IsRemainingRedundantEntities(command);
        }

        private void IsCleanupedRedundantEntities(DevMenuCommandSystem command)
        {
            Assert.IsTrue(command.Execute(new string[] {
                "application list-content-meta-database -s builtin",
            },
                @"\[\]"
            ));

            Assert.IsTrue(command.Execute(new string[] {
                "application list-content-meta-database -s sdcard",
            }, new string[] {
                @"\""id\"": \""0x0100394000059000\"",",
                @"\""version\"": 65536,"
            }));
            Assert.IsFalse(command.Execute(new string[] {
                "application list-content-meta-database -s sdcard",
            }, new string[] {
                @"\""version\"": 0,"
            }));

            Assert.IsTrue(command.Execute(new string[] {
                "application list-record 0x0100394000059000",
                "application verify 0x0100394000059000",
            }));
        }

        private void IsRemainingRedundantEntities(DevMenuCommandSystem command)
        {
            Assert.IsTrue(command.Execute(new string[] {
                "application list-content-meta-database -s builtin",
            }, new string[] {
                @"\""id\"": \""0x0100394000059000\"",",
                @"\""version\"": 65536,"
            }));
            Assert.IsTrue(command.Execute(new string[] {
                "application list-content-meta-database -s sdcard",
            }, new string[] {
                @"\""id\"": \""0x0100394000059000\"",",
                @"\""version\"": 0,",
                @"\""version\"": 65536,"
            }));
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void TestCleanupRedundant()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);
            var appv1 = m_Maker.MakeApplication(id, 1);

            var command = new DevMenuCommandSystem(this.TestContext);

            // 強制電源断すると、cleanup が走っている
            PrepareRedundantEnvironment(command, appv0, appv1);
            ForceReset(command);
            IsCleanupedRedundantEntities(command);

            // SD カードの状態が変わらないと、 cleanup が走らない
            // 正常再起動を DevMenuCommand などから行えるようになったら実施する

            PrepareRedundantEnvironment(command, appv0, appv1);
            Reboot(command);

            // DeveloperInterface の場合、先に実行されることがあるので、
            // アプリケーションを起動させることで、 ShutdownManager の起動を待つ
            SuccessLaunchApplication(command, id);

            IsRemainingRedundantEntities(command);

            // SD カードが抜き差しされると、 cleanup が実施される
            // 正常再起動を DevMenuCommand などから行えるようになったら実施する

            Assert.IsTrue(command.Execute(new string[] {
                "debug disable-mount-sdcard",
            }));
            Reboot(command);

            // DeveloperInterface の場合、先に実行されることがあるので、
            // アプリケーションを起動させることで、 SdCardManager の起動を待つ
            SuccessLaunchApplication(command, id);

            Assert.IsTrue(command.Execute(new string[] {
                "debug enable-mount-sdcard",
            }));

            Reboot(command);

            // DeveloperInterface の場合、先に実行されることがあるので、
            // アプリケーションを起動させることで、 ShutdownManager の実行を待つ
            SuccessLaunchApplication(command, id);

            IsCleanupedRedundantEntities(command);

            // アプリケーション記録にないコンテンツもクリーンナップされる
            Assert.IsTrue(command.Execute(new string[] {
                "application uninstall --all",
                "application install-redundant-entity " + appv0.Path
            }));

            Assert.IsTrue(command.FailureExecute(
                string.Format("application list-record 0x{0:x16}", id),
                "Not found."
            ));

            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application list-content-meta-database", id),
            },
                string.Format(@"\""id\"": \""0x{0:x16}\"",", id)
            ));

            Assert.IsTrue(command.Execute(new string[] {
                "application delete-redundant-entity"
            }));

            Assert.IsFalse(command.Execute(new string[] {
                string.Format("application list-content-meta-database", id),
            },
                string.Format(@"\""id\"": \""0x{0:x16}\"",", id)
            ));
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestDeleteApplicationOnRunning()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);

            var command = new DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.Execute(new string[] {
                "application install " + appv0.Path,
            }));

            var options = new DevMenuCommandBase.RunOnTargetOptions();
            options.NoWait = true;

            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application launch 0x{0:X16}", id)
            }, options: options));

            options.NoWait = false;
            options.SuppressAutoKill = true;

            // 起動するのを待つ
            Thread.Sleep(1 * 1000);

            Assert.IsTrue(command.FailureExecute(
                string.Format("application uninstall 0x{0:X16}", id),
                expect: "Unexpected error. result = 0x00071c10",
                options: options));

            // 同じアプリケーションをホスト経由で起動する
            var run = new RunOnTarget(this.TestContext, appv0.Path, command.TargetName);
            var execResult = run.Execute("", false, new string[] { "--no-wait" });
            Assert.IsTrue(execResult.Item1);

            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application uninstall 0x{0:X16} --ignore-running-check", id),
            }, options: options));
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestApplicationRecordOnDeleteEntity()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);

            var command = new DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.Execute(new string[] {
                "application install " + appv0.Path,
                string.Format("application enable-auto-update 0x{0:X16}", id)
            }));

            {
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("application list-record-detail 0x{0:X16}", id),
                }));

                var record = DetailApplicationRecordListForJson.Deserialize(command.LastOutput).Single(x => x.id == string.Format("0x{0:x16}", id));
                Assert.IsTrue(record.isAutoUpdateEnabled);
                Assert.AreEqual("AttributeUpdated", record.lastEvent);
            }
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application delete-entity 0x{0:X16}", id)
            }));

            {
                Assert.IsTrue(command.Execute(new string[] {
                    string.Format("application list-record-detail 0x{0:X16}", id)
                }));

                var record = DetailApplicationRecordListForJson.Deserialize(command.LastOutput).Single(x => x.id == string.Format("0x{0:x16}", id));
                Assert.IsTrue(!record.isAutoUpdateEnabled);
                Assert.AreEqual("Deleted", record.lastEvent);
            }
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestGameCardApplication()
        {
            ulong id = 0x0100394000059000;
            var appv0 = m_Maker.MakeApplication(id, 0);

            var command = new DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.Execute(new string[] {
                "gamecard write " + appv0.Path,
            }));

            // データの整理は行える
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application delete-entity 0x{0:X16}", id)
            }));

            // データの削除は行えない
            Assert.IsTrue(command.FailureExecute(
                string.Format("application uninstall 0x{0:X16}", id),
                expect: "Failed to execute the command. Game Card is inserted."
            ));

            // オプションを付ければ削除できる
            Assert.IsTrue(command.Execute(new string[] {
                string.Format("application uninstall 0x{0:X16} --ignore-gamecard-check", id),
            }));

            Assert.IsTrue(command.Execute(new string[] {
                "gamecard erase",
                "gamecard write " + appv0.Path,
            }));

            // --all の時は gamecard のアプリも削除できる
            Assert.IsTrue(command.Execute(new string[] {
                "application uninstall --all",
            }));
        }
    }
}
