﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using CsTestAssistants;
using System.Text.RegularExpressions;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Reflection;

namespace testNim_DebugResponse
{
    using ExpectEvaluator = ThrowEvaluator<UnexpectFailureException>;

    [TestClass]
    public class UnitMain : TestClassBase
    {
        public TestContext TestContext { get; set; }

        private const string LogPrefix = "[DevMenuCommandTest::NimDebugResponse]";

        //!----------------------------------------------------------------------------
        /// <summary>
        /// クラスが生成され、呼び出される前に一度だけ実施。
        /// </summary>
        /// <param name="context"></param>
        //!----------------------------------------------------------------------------
        [ClassInitialize]
        public static void TestClassInitialize(TestContext context)
        {
            ClassInitialize(context, LogPrefix);
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// クラスが破棄されるタイミングで一度だけ実施。
        /// </summary>
        //!----------------------------------------------------------------------------
        [ClassCleanup]
        public static void TestClassCleanup()
        {
            ClassCleanup();
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// `TestProperty属性の "JIRA" キーの値` と `Sigloコンフィギュレーションコンテキスト` に依存した作業用ワーキングディレクトリパスを生成します.
        /// </summary>
        /// <param name="context">Sigloコンフィギュレーションコンテキスト</param>
        /// <param name="methodName">
        /// 参照対象TestProperty属性を保持するメソッド名.
        /// 指定を省略した場合、呼び出されたカレントメソッド名が自動で選択されます.
        /// 明示的な null 指定の場合、もしくは "JIRA" キーの TestPropertyが見つからない場合には、TestPropertyに依存しないベース作業用ディレクトリが返却されます。
        /// </param>
        /// <returns>
        /// TestProperty( "JIRA", * ) に依存した作業用ディレクトリパス、もしくはベース作業用ディレクトリパス。
        /// </returns>
        //!----------------------------------------------------------------------------
        private string GenerateIntermediateDirectoryAsMethod(SigloHelper.Configuration.Context context,
            [System.Runtime.CompilerServices.CallerMemberName] string methodName = null)
        {
            var runAssembly = Assembly.GetExecutingAssembly();
            var asmTitle = (AssemblyTitleAttribute)System.Attribute.GetCustomAttribute(runAssembly, typeof(AssemblyTitleAttribute));
            return GenerateIntermediateDirectory(context, asmTitle.Title, methodName);
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        ///
        /// </summary>
        //!----------------------------------------------------------------------------
        [TestMethod]
        [TestProperty("JIRA", "SIGLO-83280")]
        public void TestForRegisterAndClear()
        {
            using (var scope = new TestMethodLog())
            {
                var account = NintendoAccount;
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                // 作業用ディレクトリ
                string intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // Enable dynamic-rights debug context.
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-debug-response-simulate --dynamic-rights"));

                // finally
                scope.AddDisposer(() => {
                    // Clear the registered debug response contents.
                    executor.RunDevMenuCommandSystem("debug clear-debug-response-simulate");
                    // Disable dynamic-rights debug context.
                    executor.RunDevMenuCommandSystem("debug disable-debug-response-simulate --dynamic-rights");
                });

                /// ログベリファイ。
                int maxKeyLength = 0;
                System.Func<string, string, int, uint, bool> verifyPerform = (executionLog, keyUrlPath, registeredIndex, bodySize) =>
                {
                    maxKeyLength = (maxKeyLength < keyUrlPath.Length) ? keyUrlPath.Length : maxKeyLength;
                    var patternHeader = $@"\[target\] \[ShopServiceAccess::Debug\] ResponseStore::WriteTo\(\) \=\> FileHeader {{registered: {registeredIndex}, keyLengthMax: {maxKeyLength}}}";
                    var r0 = Regex.Match(executionLog, patternHeader, RegexOptions.Multiline).Success;

                    var patternKeyLength = $@"\[target\]     keyLength: {keyUrlPath.Length}";
                    var r1 = Regex.Match(executionLog, patternKeyLength, RegexOptions.Multiline).Success;

                    var patternValueSize = $@"\[target\]     valueLength: {bodySize}";
                    var r2 = Regex.Match(executionLog, patternValueSize, RegexOptions.Multiline).Success;
                    return (r0 && r1 && r2);
                };

                /// 最小オプション
                {
                    string urlPath = "/v1/rights/test";
                    string command = $"debug register-debug-response-simulate";
                    string server = $"--server DynamicRights";
                    string url = $"--url-path \"{urlPath}\"";
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($@"{command} {server} {url} --verbose"));
                    ExpectEvaluator.IsTrue(verifyPerform(executor.OutputStream.Standard.ToString(), urlPath, 1, 1));
                }

                /// コマンドライン経由レスポンス
                {
                    string urlPath = "/v1/rights/available_elicenses";
                    string responseBodyForCommandLine =
                    "{" +
                        @"\""available_elicenses\"":[{" +
                            @"\""account_id\"":\""fd3e99e841f985a2\""," +
                            @"\""rights_id\"":\""0100000000000001\""," +
                            @"\""elicense_type\"":\""temporary\""" +
                        "}]" +
                    "}";
                    string command = $"debug register-debug-response-simulate";
                    string server = $"--server DynamicRights";
                    string url = $"--url-path \"{urlPath}\"";
                    string response = $"--response \"{responseBodyForCommandLine}\"";
                    string other = $"--result-inner-value 0x10 --rate 5000 --verbose";

                    // NOTE:    何故かコマンドライン上で二重引用符を含めたJSON指定しようとすると、オプション引数を `"\"` で囲む必要になる。
                    //          DevMenuCommand には空引数として渡るので影響はないが、コマンドラインの問題か、RunOnTarget の引数解釈の問題か。
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($@"{command} ""\"" {server} {url} {response} {other} ""\"""));
                    ExpectEvaluator.IsTrue(verifyPerform(executor.OutputStream.Standard.ToString(), urlPath, 2, (uint)Regex.Replace(responseBodyForCommandLine, @"\\", "", RegexOptions.Multiline).Length));
                }

                /// ファイル経由レスポンス
                {
                    string responseBodyForFile =
                    "{" +
                        "\"available_elicenses\": [{" +
                            "\"account_id\": \"fd3e99e841f985a2\"," +
                            "\"rights_id\": \"0100000000000001\"," +
                            "\"elicense_type\": \"temporary\"" +
                        "},{" +
                            "\"account_id\": \"fd3e99e841f985a2\"," +
                            "\"rights_id\": \"010000000000000b\"," +
                            "\"elicense_type\": \"unavailable\"," +
                            "\"reason\": \"limit_exceeded\"" +
                        "}]" +
                    "}";
                    string urlPath = "/v1/rights/publish_elicenses";
                    string responseFile = System.IO.Path.Combine(intermediate, "response.json");
                    FileHelper.MakeDirectory(System.IO.Path.GetDirectoryName(responseFile));
                    System.IO.File.WriteAllText(responseFile, responseBodyForFile, EncodeContext.UTF8_NoBOM);

                    string command = $"debug register-debug-response-simulate";
                    string server = $"--server DynamicRights";
                    string url = $"--url-path \"{urlPath}\"";
                    string response = $"--response-file \"{responseFile}\"";
                    string other = $"--result-inner-value 0x0 --rate 10000 --verbose";
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem($@"{command} {server} {url} {response} {other}"));
                    ExpectEvaluator.IsTrue(verifyPerform(executor.OutputStream.Standard.ToString(), urlPath, 3, (uint)responseBodyForFile.Length));
                }
            }
        }
    }
}
