﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Diagnostics;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace DevMenuCommandTest
{
    public class MakeTestApplication
    {
        private string m_ExePath;
        private string m_OutputDirectry;

        private struct MakeTestApplicationOption
        {
            public ulong Id { get; set; }
            public string Type { get; set; }
            public int Version { get; set; }
            public ulong Size { get; set; }
            public int Index { get; set; }
            public int LastIndex { get; set; }
            public string Original { get; set; }
            public string Previous { get; set; }
            public bool Ticket { get; set; }
            public UInt32 RequiredSystemVersion { get; set; }
            public UInt16 RequiredApplicationVersion { get; set; }
            public string MetaFilePath { get; set; }
            public string AddtionalOptions { get; set; }
        }

        public MakeTestApplication(TestContext context) : this(context, null) { }

        public MakeTestApplication(TestContext context, string postfix)
        {
            m_ExePath = new TestUtility.TestPath(context).GetSigloRoot() + "\\Tools\\CommandLineTools\\MakeTestApplication\\MakeTestApplication.exe";
            m_OutputDirectry = new TestUtility.TestPath(context).GetSigloRoot() + "\\Tests\\Outputs\\" + context.TestName;
            if (postfix != null)
            {
                m_OutputDirectry += "_" + postfix;
            }
            Directory.CreateDirectory(m_OutputDirectry);
        }

        public void Cleanup()
        {
            Directory.Delete(m_OutputDirectry, true);
            Directory.CreateDirectory(m_OutputDirectry);
        }

        public string GetOutputDirectory()
        {
            return @"""" + m_OutputDirectry + @"""";
        }

        public NspInfo MakeApplication(ulong id, int version = 0, ulong size = 4, bool ticket = false, UInt32 requiredSystemVersion = 0, string metaFilePath = null, string additionalOptions = null)
        {
            var option = new MakeTestApplicationOption();
            option.Id = id;
            option.Type = "Application";
            option.Version = version;
            option.Size = size;
            option.Ticket = ticket;
            option.RequiredSystemVersion = requiredSystemVersion;
            option.MetaFilePath = metaFilePath;
            option.AddtionalOptions = additionalOptions;
            return MakeImpl(option);
        }

        public NspInfo MakePatch(ulong id, int version, string original, string previous = null, ulong size = 4, bool ticket = false, UInt32 requiredSystemVersion = 0, string additionalOptions = null)
        {
            var option = new MakeTestApplicationOption();
            option.Id = id;
            option.Type = "Patch";
            option.Version = version;
            option.Size = size;
            option.Original = original;
            option.Previous = previous;
            option.Ticket = ticket;
            option.RequiredSystemVersion = requiredSystemVersion;
            option.AddtionalOptions = additionalOptions;
            return MakeImpl(option);
        }

        public NspInfo MakeAddOnContent(ulong id, int index = 1, int version = 0, ulong size = 4, bool ticket = false, UInt16 requiredApplicationVersion = 0, string additionalOptions = null)
        {
            var option = new MakeTestApplicationOption();
            option.Id = id;
            option.Type = "AddOnContent";
            option.Version = version;
            option.Size = size;
            option.Index = index;
            option.Ticket = ticket;
            option.RequiredApplicationVersion = requiredApplicationVersion;
            option.AddtionalOptions = additionalOptions;
            return MakeImpl(option);
        }

        public NspInfo MakeMultipleAddOnContent(ulong id, int begin, int end, int version = 0, ulong size = 4, string additionalOptions = null)
        {
            var option = new MakeTestApplicationOption();
            option.Id = id;
            option.Type = "AddOnContent";
            option.Version = version;
            option.Size = size;
            option.Index = begin;
            option.LastIndex = end;
            option.AddtionalOptions = additionalOptions;
            return MakeImpl(option);
        }

        public NspInfo MakeSystemProgram(ulong id, ulong size = 4, int version = 0, string additionalOptions = null)
        {
            var option = new MakeTestApplicationOption();
            option.Id = id;
            option.Type = "SystemProgram";
            option.Version = version;
            option.Size = size;
            option.AddtionalOptions = additionalOptions;
            return MakeImpl(option);
        }

        public NspInfo MakeSystemData(ulong id, ulong size = 4, int version = 0, string additionalOptions = null)
        {
            var option = new MakeTestApplicationOption();
            option.Id = id;
            option.Type = "SystemData";
            option.Version = version;
            option.Size = size;
            option.AddtionalOptions = additionalOptions;
            return MakeImpl(option);
        }

        private NspInfo MakeImpl(MakeTestApplicationOption option)
        {
            var filePath = MakeFilePath(option);
            using (var process = new Process())
            {
                string arguments;
                if (string.IsNullOrEmpty(option.MetaFilePath))
                {
                    arguments = "--id " + "0x" + option.Id.ToString("x16") + " " +
                                "--type " + option.Type + " " +
                                "--release-version " + option.Version + " " +
                                "--size " + option.Size + " " +
                                "--output-file-path " + filePath +
                                (option.Ticket ? " --ticket " : "") +
                                ((!option.Ticket && option.Type == "AddOnContent") ? " --no-ticket " : "") +
                                (option.RequiredSystemVersion > 0 ? " --required-version " + option.RequiredSystemVersion : "");
                }
                else
                {
                    arguments = "--meta \"" + option.MetaFilePath + "\" " +
                                "--output-file-path " + filePath +
                                (option.Ticket ? " --ticket " : "");
                }
                if (option.Type.Equals("Patch"))
                {
                    arguments += " --original-application " + option.Original;
                    if (option.Previous != null)
                    {
                        arguments += " --previous-patch " + option.Previous;
                    }
                }
                else if (option.Type.Equals("AddOnContent"))
                {
                    arguments += " --index " + option.Index;
                    if (option.LastIndex != 0)
                    {
                        arguments += " --index " + option.LastIndex;
                    }
                    arguments += option.RequiredApplicationVersion > 0 ? " --required-version " + option.RequiredApplicationVersion : "";
                }
                if (!string.IsNullOrEmpty(option.AddtionalOptions))
                {
                    arguments += " " + option.AddtionalOptions;
                }

                var outputs = ProcessManager.SetupProcess(process, m_ExePath, arguments);
                ProcessManager.RunAndWaitProcess(process);

                if (process.ExitCode != 0)
                {
                    throw new ApplicationException("Faile to execute MakeTestApplication: " + outputs.Item2);
                }
            }

            return new NspInfo(filePath, option.Id, option.Type);
        }

        private string MakeFilePath(MakeTestApplicationOption option)
        {
            var name = option.Type + "_" + option.Id.ToString("x16") + "_v" + option.Version;
            if (option.Index != 0)
            {
                name += "_i" + option.Index;
            }
            return @"""" + m_OutputDirectry + "\\" + name + ".nsp" + @"""";
        }
    }
}
