﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Linq;
using System.Threading;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using DevMenuCommandTest;

namespace DevMenuCommandTest
{
    [TestClass]
    public class DynamicRightsTest
    {
        public TestContext TestContext { get; set; }

        private static MakeTestApplication m_Maker;
        private static DevMenuCommandSystem m_Command;
        private static string NaId = "73ef8359f8ce37d1";
        private static ulong HavingRightsAppId = 0x0100d44001902000;
        private static ulong NoRightsAppId = 0x0100c90001976000;

        public static void ResetElicenseManagementAndReboot()
        {
            var options = new DevMenuCommandBase.RunOnTargetOptions();
            options.NoWait = true;
            m_Command.Execute(new string[] {
                "elicense delete-all-archives",
                "power reboot",
            },  options: options);
            Thread.Sleep(10 * 1000);
        }

        [ClassInitialize]
        public static void Initialize(TestContext context)
        {
            m_Maker = new MakeTestApplication(context);

            m_Command = new DevMenuCommandSystem(context);

            Assert.IsTrue(m_Command.Execute(new string[] {
                "servicediscovery import-all td1pass",
                "debug disable-dynamic-rights-check",
                "debug enable-debug-response-simulate --dynamic-rights",
                "debug clear-debug-response-simulate",
                "debug disable-debug-response-simulate",
            }));

            ResetElicenseManagementAndReboot();

            // コマンドが長いと解釈できない場合があるので、適度に分ける
            Assert.IsTrue(m_Command.Execute(new string[] {
                "application uninstall --all",
                "application reset-required-version --all",
                "ticket delete-all",
            }));
            Assert.IsTrue(m_Command.Execute(new string[] {
                "shop unlink-device-all",
                "shop unregister-device-account",
                "account clear_all",
            }));
            Assert.IsTrue(m_Command.Execute(new string[] {
                // NA を紐づけたアカウントを作成
                "account add",
                "account link --index 0 --id shimada_hiromasa+001-1td1@exmx.nintendo.co.jp --password shimada001-1td1test",
                "shop start",
                "dynamicrights revoke-all -f --account 0",
            }));
            Assert.IsTrue(m_Command.Execute(new string[] {
                // NA を紐づけたアカウントを作成
                "account add",
                "account link --index 1 --id shimada_hiromasa+002td1@exmx.nintendo.co.jp --password shimada002td1test",
                "shop start",
                "dynamicrights revoke-all -f --account 1",
            }));
            Assert.IsTrue(m_Command.Execute(new string[] {
                // NA を紐づけないアカウントを作成
                "account add",
            }));

        }

        [ClassCleanup]
        public static void Cleanup()
        {
            m_Command.Execute(new string[] {
                "debug disable-dynamic-rights-check",
                "debug enable-debug-response-simulate --dynamic-rights",
                "debug clear-debug-response-simulate",
                "debug disable-debug-response-simulate",
            });
            ResetElicenseManagementAndReboot();
            m_Command.Execute(new string[] {
                "shop unlink-device-all",
                "shop unregister-device-account",
                "account clear_all",
            });
            m_Command.Execute(new string[] {
                "application uninstall --all",
                "ticket delete-all",
                "application reset-required-version --all"
            });
            m_Maker.Cleanup();
        }

        public void SetDebugResponse(ulong id, string licenseType)
        {
            string fileName = Path.Combine(Path.GetTempPath(), Path.GetRandomFileName());

            try
            {
                string debugResponseString =
                    "{" +
                        @"""available_elicenses"":[{" +
                            $@"""account_id"":""{NaId}""," +
                            $@"""rights_id"":""{id:x16}""," +
                            $@"""elicense_type"":""{licenseType}""" +
                        "}]" +
                    "}";

                using (var fs = File.CreateText(fileName))
                {
                    fs.Write(debugResponseString);
                }

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    "debug enable-dynamic-rights-check",
                    "debug enable-debug-response-simulate --dynamic-rights",
                    "debug clear-debug-response-simulate",
                }));

                Assert.IsTrue(m_Command.ResetAndExecute(new string[]
                {
                    "debug register-debug-response-simulate " +
                    "--server DynamicRights " +
                    "--url-path /v1/rights/available_elicenses " +
                    "--response-file " + fileName + " " +
                    "--result-inner-value 0x0 --rate 10000 --verbose"
                }));
            }
            finally
            {
                if (File.Exists(fileName))
                {
                    File.Delete(fileName);
                }
            }
        }

        /* 現在未使用のためコメントアウト
        public void SetFailureDebugResponse(ulong id, string reason)
        {
            string fileName = Path.Combine(Path.GetTempPath(), Path.GetRandomFileName());

            try
            {
                string debugResponseString =
                    "{" +
                        @"""available_elicenses"":[{" +
                            $@"""account_id"":""{NaId}""," +
                            $@"""rights_id"":""{id:x16}""," +
                            @"""elicense_type"":""unavailable""," +
                            $@"\""reason\"":\""{reason}\""" +
                        "}]" +
                    "}";

                using (var fs = File.CreateText(fileName))
                {
                    fs.Write(debugResponseString);
                }

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    "debug enable-debug-response-simulate --dynamic-rights",
                    "debug clear-debug-response-simulate",
                    "debug register-debug-response-simulate " +
                    "--server DynamicRights " +
                    "--url-path /v1/rights/available_elicenses " +
                    "--response-file " + fileName + " " +
                    "--result-inner-value 0x0 --rate 10000 --verbose"
                }));
            }
            finally
            {
                if (File.Exists(fileName))
                {
                    File.Delete(fileName);
                }
            }
        }
        */

        public void CheckNoRights(ulong id, int accountIndex)
        {
            Assert.IsTrue(m_Command.Execute(new string[]
            {
                $"application get-application-rights-on-client 0x{id:x16}"
            }));
            {
                var rightsList = ApplicationRightsOnClientForJson.Deserialize(m_Command.LastOutput);
                var appRights = rightsList.Single(x => x.type == "Application");
                Assert.IsFalse(appRights.hasAvailableRights);
                Assert.IsTrue(appRights.hasUnavailableRights);
                Assert.IsFalse(appRights.hasAccountRestrictedRights);
                Assert.IsTrue(appRights.recommendInquireServer);
                Assert.IsFalse(appRights.hasPrepurchasedRights);
            }

            Assert.IsTrue(m_Command.Execute(new string[]
            {
                $"application get-application-rights-on-client 0x{id:x16} --account-index {accountIndex}"
            }));
            {
                var rightsList = ApplicationRightsOnClientForJson.Deserialize(m_Command.LastOutput);
                var appRights = rightsList.Single(x => x.type == "Application");
                Assert.IsFalse(appRights.hasAvailableRights);
                Assert.IsTrue(appRights.hasUnavailableRights);
                Assert.IsFalse(appRights.hasAccountRestrictedRights);
                Assert.IsTrue(appRights.recommendInquireServer);
                Assert.IsFalse(appRights.hasPrepurchasedRights);
            }
        }

        public void CheckDeviceLinkedPermanentRights(ulong id, int accountIndex)
        {
            Assert.IsTrue(m_Command.Execute(new string[]
            {
                $"application get-application-rights-on-client 0x{id:x16}"
            }));
            {
                var rightsList = ApplicationRightsOnClientForJson.Deserialize(m_Command.LastOutput);
                var appRights = rightsList.Single(x => x.type == "Application");
                Assert.IsTrue(appRights.hasAvailableRights);
                Assert.IsFalse(appRights.hasUnavailableRights);
                Assert.IsFalse(appRights.hasAccountRestrictedRights);
                Assert.IsFalse(appRights.recommendInquireServer);
                Assert.IsFalse(appRights.hasPrepurchasedRights);
            }

            Assert.IsTrue(m_Command.Execute(new string[]
            {
                $"application get-application-rights-on-client 0x{id:x16} --account-index {accountIndex}"
            }));
            {
                var rightsList = ApplicationRightsOnClientForJson.Deserialize(m_Command.LastOutput);
                var appRights = rightsList.Single(x => x.type == "Application");
                Assert.IsTrue(appRights.hasAvailableRights);
                Assert.IsFalse(appRights.hasUnavailableRights);
                Assert.IsFalse(appRights.hasAccountRestrictedRights);
                Assert.IsFalse(appRights.recommendInquireServer);
                Assert.IsFalse(appRights.hasPrepurchasedRights);
            }
        }

        public void CheckTemporaryRights(ulong id, int accountIndex)
        {
            Assert.IsTrue(m_Command.Execute(new string[]
            {
                $"application get-application-rights-on-client 0x{id:x16}"
            }));
            {
                var rightsList = ApplicationRightsOnClientForJson.Deserialize(m_Command.LastOutput);
                var appRights = rightsList.Single(x => x.type == "Application");
                Assert.IsFalse(appRights.hasAvailableRights);
                Assert.IsTrue(appRights.hasUnavailableRights);
                Assert.IsFalse(appRights.hasAccountRestrictedRights);
                Assert.IsTrue(appRights.recommendInquireServer);
                Assert.IsFalse(appRights.hasPrepurchasedRights);
            }

            Assert.IsTrue(m_Command.Execute(new string[]
            {
                $"application get-application-rights-on-client 0x{id:x16} --account-index {accountIndex}"
            }));
            {
                var rightsList = ApplicationRightsOnClientForJson.Deserialize(m_Command.LastOutput);
                var appRights = rightsList.Single(x => x.type == "Application");
                Assert.IsTrue(appRights.hasAvailableRights);
                Assert.IsFalse(appRights.hasUnavailableRights);
                Assert.IsTrue(appRights.hasAccountRestrictedRights);
                Assert.IsTrue(appRights.recommendInquireServer);
                Assert.IsFalse(appRights.hasPrepurchasedRights);
            }
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void LaunchByTemporaryRightsWithoutServerCommunication()
        {
            // 権利のあるアプリケーションの ID
            ulong id = HavingRightsAppId;

            try
            {
                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application download 0x{id:x16}"
                }));

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"debug enable-dynamic-rights-check --debug-license-type 4",
                }));

                CheckDeviceLinkedPermanentRights(id, 0);

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"ticket import-elicense 0",
                }));

                CheckNoRights(id, 0);

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application check-and-launch 0x{id:x16}",
                }));

                CheckTemporaryRights(id, 0);

                // 再度確認
                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application check-and-launch 0x{id:x16}",
                }));
            }
            finally
            {
                m_Command.Execute(new string[]
                {
                    "debug disable-dynamic-rights-check",
                });
                ResetElicenseManagementAndReboot();
                m_Command.Execute(new string[]
                {
                    "application uninstall --all",
                    "application reset-required-version --all",
                    "ticket delete-all",
                });
            }
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void LaunchByDeviceLinkedPermanentRightsWithoutServerCommunication()
        {
            // 権利のあるアプリケーションの ID
            ulong id = HavingRightsAppId;

            try
            {
                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application download 0x{id:x16}"
                }));

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"debug enable-dynamic-rights-check --debug-license-type 1",
                }));

                CheckDeviceLinkedPermanentRights(id, 0);

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"ticket import-elicense 0",
                }));

                CheckNoRights(id, 0);

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application check-and-launch 0x{id:x16}",
                }));

                CheckDeviceLinkedPermanentRights(id, 0);

                // 再度確認
                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application check-and-launch 0x{id:x16}",
                }));
            }
            finally
            {
                m_Command.Execute(new string[]
                {
                    "debug disable-dynamic-rights-check",
                });
                ResetElicenseManagementAndReboot();
                m_Command.Execute(new string[]
                {
                    "application uninstall --all",
                    "application reset-required-version --all",
                    "ticket delete-all",
                });
            }
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void LaunchByTemporaryRightsWithDebugResponse()
        {
            // 権利のあるアプリケーションの ID
            ulong id = HavingRightsAppId;

            try
            {
                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application download 0x{id:x16}"
                }));

                CheckDeviceLinkedPermanentRights(id, 0);

                SetDebugResponse(id, "temporary");

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"ticket import-elicense 0",
                }));

                CheckNoRights(id, 0);

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application check-and-launch 0x{id:x16}",
                }));

                CheckTemporaryRights(id, 0);

                // 再度確認
                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application check-and-launch 0x{id:x16}",
                }));
            }
            finally
            {
                m_Command.Execute(new string[]
                {
                    "debug disable-dynamic-rights-check",
                    "debug enable-debug-response-simulate --dynamic-rights",
                    "debug clear-debug-response-simulate",
                    "debug disable-debug-response-simulate",
                });
                ResetElicenseManagementAndReboot();
                m_Command.Execute(new string[]
                {
                    "application uninstall --all",
                    "application reset-required-version --all",
                    "ticket delete-all",
                });
            }
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void LaunchByDeviceLinkedPermanentRightsWithDebugResponse()
        {
            // 権利のあるアプリケーションの ID
            ulong id = HavingRightsAppId;

            try
            {
                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application download 0x{id:x16}"
                }));

                CheckDeviceLinkedPermanentRights(id, 0);

                SetDebugResponse(id, "device_linked_permanent");

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"ticket import-elicense 0",
                }));

                CheckNoRights(id, 0);

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application check-and-launch 0x{id:x16}",
                }));

                CheckDeviceLinkedPermanentRights(id, 0);

                // 再度確認
                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application check-and-launch 0x{id:x16}",
                }));
            }
            finally
            {
                m_Command.Execute(new string[]
                {
                    "debug disable-dynamic-rights-check",
                    "debug enable-debug-response-simulate --dynamic-rights",
                    "debug clear-debug-response-simulate",
                    "debug disable-debug-response-simulate",
                });
                ResetElicenseManagementAndReboot();
                m_Command.Execute(new string[]
                {
                    "application uninstall --all",
                    "application reset-required-version --all",
                    "ticket delete-all",
                });
            }
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void LaunchByTemporaryRights()
        {
            // 権利のあるアプリケーションの ID
            ulong id = HavingRightsAppId;

            try
            {
                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application download 0x{id:x16}"
                }));

                CheckDeviceLinkedPermanentRights(id, 0);

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"shop unlink-device"
                }));

                CheckNoRights(id, 0);

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application check-and-launch 0x{id:x16}",
                }));

                CheckTemporaryRights(id, 0);

                // 他のアカウントは起動出来ない
                Assert.IsTrue(m_Command.FailureExecute($"application check-and-launch 0x{id:x16} --account-index 1", "There is no rights of the content."));
                Assert.IsTrue(m_Command.FailureExecute($"application check-and-launch 0x{id:x16} --account-index 2", "There is no rights of the content."));

                // 自分は起動できることを再度確認
                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application check-and-launch 0x{id:x16}",
                }));
            }
            finally
            {
                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    "shop start",
                    "dynamicrights revoke-all -f --account 0",
                }));
                m_Command.Execute(new string[]
                {
                    "application uninstall --all",
                    "application reset-required-version --all",
                    "ticket delete-all",
                });
                ResetElicenseManagementAndReboot();
            }
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void LaunchByDeviceLinkedPermanentRights()
        {
            // 権利のあるアプリケーションの ID
            ulong id = HavingRightsAppId;

            try
            {
                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application download 0x{id:x16}"
                }));

                CheckDeviceLinkedPermanentRights(id, 0);

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"ticket delete 0x{id:x16}"
                }));

                CheckNoRights(id, 0);

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application check-and-launch 0x{id:x16}",
                }));

                CheckDeviceLinkedPermanentRights(id, 0);

                // 他のアカウントでも起動出来る
                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application check-and-launch 0x{id:x16} --account-index 1",
                }));
                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application check-and-launch 0x{id:x16} --account-index 2",
                }));

                // 自分は起動できることを再度確認
                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application check-and-launch 0x{id:x16}",
                }));
            }
            finally
            {
                m_Command.Execute(new string[]
                {
                    "dynamicrights revoke-all -f --account 0",
                    "application uninstall --all",
                    "application reset-required-version --all",
                    "ticket delete-all",
                });
                ResetElicenseManagementAndReboot();
            }
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void LaunchByDeviceLinkedPermanentRightsForOtherAccount()
        {
            // 権利のあるアプリケーションの ID
            ulong id = HavingRightsAppId;

            try
            {
                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application download 0x{id:x16}"
                }));

                CheckDeviceLinkedPermanentRights(id, 0);

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"ticket delete 0x{id:x16}"
                }));

                CheckNoRights(id, 0);
                CheckNoRights(id, 1);

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application check-and-launch 0x{id:x16} --account-index 1",
                }));

                CheckDeviceLinkedPermanentRights(id, 0);

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"ticket delete 0x{id:x16}"
                }));

                CheckNoRights(id, 0);

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application check-and-launch 0x{id:x16} --account-index 2",
                }));

                CheckDeviceLinkedPermanentRights(id, 0);
            }
            finally
            {
                m_Command.Execute(new string[]
                {
                    "dynamicrights revoke-all -f --account 0",
                    "application uninstall --all",
                    "application reset-required-version --all",
                    "ticket delete-all",
                });
                ResetElicenseManagementAndReboot();
            }
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void FailedToLaunchByTemporaryRightsWithDebugResponse()
        {
            // 権利のないアプリケーションの ID
            ulong id = NoRightsAppId;

            try
            {
                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"application download 0x{id:x16}"
                }));

                CheckNoRights(id, 0);

                Assert.IsTrue(m_Command.Execute(new string[]
                {
                    $"ticket import-elicense 0",
                }));

                CheckNoRights(id, 0);

                // TODO: 現在未対応
                // SetFailureDebugResponse(id, "no_rights");

                Assert.IsTrue(m_Command.FailureExecute($"application check-and-launch 0x{id:x16}", "There is no rights of the content."));

                CheckNoRights(id, 0);
            }
            finally
            {
                ResetElicenseManagementAndReboot();
                m_Command.Execute(new string[]
                {
                    "debug enable-debug-response-simulate --dynamic-rights",
                    "debug clear-debug-response-simulate",
                    "debug disable-debug-response-simulate",
                    "application uninstall --all",
                    "application reset-required-version --all",
                    "ticket delete-all",
                });
            }
        }
    }
}
