# coding: UTF-8

import codecs
import os
import re
import sys

# CppCodingRuleChecker をインポートします。
TEST_ROOT   = os.path.dirname( os.path.abspath(__file__) )
SIGLO_ROOT  = os.path.normpath( TEST_ROOT + '/../../../../..' )
APP_ROOT    = os.path.normpath( SIGLO_ROOT + '/Integrate/Sources/Tools/CppCodingRuleChecker' )
INPUT_ROOT  = os.path.normpath( TEST_ROOT + '/InputFiles' )
INCLUDE_DIR = os.path.normpath( TEST_ROOT + '/Headers' )
sys.path.append( APP_ROOT )
import CppCodingRuleChecker

# テストコードに埋め込まれたアサート取得に使用する正規表現です。
META_PATTERN = re.compile(r'//:\)\s*([A-Z][0-9A-Za-z]*)(.*)$')

class Test(object):
    def __init__(self, category, name, path):
        self.category  = category
        self.name      = name
        self.path      = path
        self.errors    = []
        self.dontcares = []

def setUp():
    CppCodingRuleChecker.syntax_error_ignored = False
    CppCodingRuleChecker.log_level            = CppCodingRuleChecker.LOG_LEVEL_WARNING
    CppCodingRuleChecker.error_level          = CppCodingRuleChecker.ERROR_LEVEL_INFORMATION
    CppCodingRuleChecker.ignores              = []
    CppCodingRuleChecker.only                 = []
    CppCodingRuleChecker.recursive            = False
    CppCodingRuleChecker.root                 = None
    CppCodingRuleChecker.checkers             = CppCodingRuleChecker.define_rules()
    CppCodingRuleChecker.build_options.extend([
        '-I',
        INCLUDE_DIR,
    ])

def tearDown():
    pass

def failed(linenum, message):
    print '  [FAILED] L.%d : %s' % (linenum, message)

def processCommand(test, command, args, linenum):
    if command == 'BAD':
        if len(args) == 0:
            test.errors.append(linenum)
        elif len(args) == 1:
            test.errors.append(int(args[0]))
        else:
            return False
    elif command.startswith('BADx'):
        if len(args) == 0:
            for i in xrange(int(command[4:])):
                test.errors.append(linenum)
        elif len(args) == 1:
            for i in xrange(int(command[4:])):
                test.errors.append(args[0])
        else:
            return False
    elif command.startswith('DONTCARE'):
        if len(args) == 0:
            test.dontcares.append(linenum)
        elif len(args) == 1:
            test.dontcares.append(int(args[0]))
        else:
            return False
    else:
        return False
    return True

def processArguments(text):
    args    = text.split(',')
    results = []
    for arg in args:
        arg = arg.strip()
        if len(arg) > 0:
            results.append(arg)
    return results

def processLine(test, line, linenum):
    match = META_PATTERN.search(line)
    if match:
        command = match.group(1)
        args    = processArguments(match.group(2))
        if not processCommand(test, command, args, linenum):
            failed(linenum, 'invalid command "%s"' % (command))
            return False
    return True

def processTest(test):
    source = codecs.open(test.path, 'r', 'utf_8', 'strict').read()
    lines  = source.splitlines()
    for i in xrange(len(lines)):
        if not processLine(test, lines[i], i + 1):
            return False
    return True

def doTest(test):
    print '(%s) %s' % (test.category, test.name)
    if not processTest(test):
        return 1

    result = CppCodingRuleChecker.check_file(test.path)
    errors = []
    for error in result.errors:
        if error.category == test.category:
            errors.append(error)
    errors = sorted(errors, cmp=lambda x, y:
                    cmp(x.file, y.file) or cmp(x.line, y.line))

    i = 0
    j = 0
    count = 0
    while i < len(test.errors) and j < len(errors):
        expect = test.errors[i]
        actual = errors[j].line
        if expect < actual:
            if expect not in test.dontcares:
                failed(expect, 'Could not found expected violations.')
                count += 1
            i += 1
        elif expect > actual:
            if actual not in test.dontcares:
                failed(actual, 'Unexpected violations.')
                count += 1
            j += 1
        else:
            i += 1
            j += 1
    while i < len(test.errors):
        expect = test.errors[i]
        if expect not in test.dontcares:
            failed(expect, 'Could not found expected violations.')
            count += 1
        i += 1
    while j < len(errors):
        actual = errors[j].line
        if actual not in test.dontcares:
            failed(actual, 'Unexpected violations.')
            count += 1
        j += 1

    return count

def doTestList(tests):
    count = 0
    for test in tests:
        count += doTest(test)
    return count

def createTestList(root_path, root_category):
    list  = []
    files = os.listdir(root_path)
    for file in files:
        path = os.path.normpath(root_path + '/' + file)
        if os.path.isdir(path):
            category = root_category + '/' + file if root_category else file
            list.extend(createTestList(path, category))
        else:
            name, ext = os.path.splitext(file)
            list.append(Test(root_category, name, path))
    return list

def main(path):
    setUp()
    tests = createTestList(path, None)
    print 'Found %d tests' % len(tests)
    error_count = doTestList(tests)
    tearDown()

if __name__ == '__main__':
    main(INPUT_ROOT)
