﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using CommandUtility;

namespace ContentsUploaderTest
{
    [TestClass]
    public class TestUnit
    {
        // 定数定義
        private const string NintendoProxy              = "http://proxy.nintendo.co.jp:8080";

        // upload するコンテンツ情報
        private const string UploadApplicationId        = "0x0100ce00023d8100";
        private const string UploadPatchId              = "0x0100ce00023d8900";
        private const string UploadAocIndex1Id          = "0x0100ce00023d9101";
        private const string UploadAocIndex2Id          = "0x0100ce00023d9102";

        // register-title するコンテンツ情報
        private const string RegisterTitleApplicationId = "0x0100ce00023d8200";
        private const string RegisterTitlePatchId       = "0x0100ce00023d8a00";
        private const string RegisterTitleAocIndex1Id   = "0x0100ce00023d9201";

        // register-demo するコンテンツ情報
        private const string RegisterDemoApplicationId  = "0x0100ce00023d8300";
        private const string RegisterDemoPatchId        = "0x0100ce00023d8b00";
        private const string RegisterDemoAocIndex1Id    = "0x0100ce00023d9301";

        // register-bundle するコンテンツ情報
        // ※ 0x0100ce00023d8400 ベースの ID は利用できなくなったので 0x0100ce00023d8500 に変更
        private const string RegisterBundleApplicationId = "0x0100ce00023d8500";
        private const string RegisterBundlePatchId      = "0x0100ce00023d8d00";
        private const string RegisterBundleAocIndex1Id  = "0x0100ce00023d9501";
        private const string RegisterBundleAocIndex2Id  = "0x0100ce00023d9502";

        // ns-uid
        private const ulong DemoNsUid = 70030000001841;
        private const ulong TitleNsUid = 70010000001631;
        private const ulong Aoc1NsUid = 70050000001231;
        private const ulong BundleNsUid = 70070000001991;

        // プロパティ
        private static TestUtility.TestPath TestPath { get; set; } = null;
        private static string TargetName { get; set; } = string.Empty;
        private static string CommonkOption { get; set; }  = $"--proxy {NintendoProxy} -e td1 --verbose";
        private static string UserOption { get; set; } = $"-u ssd_ci_tester -p OGVlYjBkN2MtNzI0";
        private static string UploadOption { get; set; } = $"{UserOption} --polling-timeout 10";
        private static Dictionary<string, string> CommandSuccessTable { get; set; } = null;
        private static Dictionary<string, UnitTestOutcome> TestResults { get; set; } = new Dictionary<string, UnitTestOutcome>();
        private static List<string> TestOutputs { get; set; } = new List<string>();
        private static bool IsUnitTestOnIDE { get; set; } = false;

        public TestContext TestContext { get; set; }

        private static string MakeTestApplicationPath
        {
            get { return GetSigloPath(@"Tools\CommandLineTools\MakeTestApplication\MakeTestApplication.exe"); }
        }
        private static string ContentsUploaderPath
        {
            get { return GetSigloPath(@"Tools\CommandLineTools\ContentsUploader\ContentsUploader.exe"); }
        }
        private static string RunOnTargetPath
        {
            get { return GetSigloPath(@"Tools\CommandLineTools\RunOnTarget.exe"); }
        }
        private static string DevMenuCommandSystemPath
        {
            get { return GetSigloPath(@"Programs\Eris\Outputs\NX-NXFP2-a64\TargetTools\DevMenuCommandSystem\Develop\DevMenuCommandSystem.nsp"); }
        }

        [ClassInitialize]
        public static void TestClassinitialize(TestContext context)
        {
            TestPath = new TestUtility.TestPath(context);

            // 実行環境で既定値を変更したいなら下記で行う
            TargetName = (string)context.Properties["TargetName"];
            if (IsTrue(context, "EnableRunTestOnIDE_FromPUX"))
            {
                // PUX 内からはプロキシを無効にする
                CommonkOption = "-e td1 --verbose";
            }
            if (IsTrue(context, "UnitTestOnIDE"))
            {
                // IDE でテストを単体実行するなら真に設定する
                IsUnitTestOnIDE = true;
            }

            // サブコマンドごとの成功表示テーブル
            CommandSuccessTable = new Dictionary<string, string>();
            CommandSuccessTable.Add("upload", "Contents Uploader Done.");
            CommandSuccessTable.Add("register-demo", "Register Demo Done.");
            CommandSuccessTable.Add("register-title", "Register Title Done.");
            CommandSuccessTable.Add("register-bundle", "Register Bundle Done.");
            CommandSuccessTable.Add("revoke-rom", "Revoke Rom Done.");
            CommandSuccessTable.Add("register-version", "Register Version Done.");
            CommandSuccessTable.Add("delete-version", "Delete Version Done.");
        }

        private static bool IsTrue(TestContext context, string paramKey, bool defaultValue = false)
        {
            var boolValue = defaultValue;
            if (context.Properties.Contains(paramKey))
            {
                var paramValue = (string)context.Properties[paramKey];
                if (!bool.TryParse(paramValue, out boolValue))
                {
                    boolValue = defaultValue;
                    Console.WriteLine($"[ContentsUploaderTest] Invalid TestRunParameters key:\"{paramKey}\" value:\"{paramValue}\".");
                }
            }
            return boolValue;
        }

        private static bool IsPassed(params string[] names)
        {
            if (names == null || names.Length == 0)
            {
                return false; // 引数なし
            }

            // IDE による単体実行時はテスト間の成否判定を無視する
            if (!IsUnitTestOnIDE)
            {
                foreach (var name in names)
                {
                    if (!TestResults.ContainsKey(name))
                    {
                        return false; // 未実行
                    }
                    if (TestResults[name] != UnitTestOutcome.Passed)
                    {
                        return false; // 成功以外
                    }
                }
            }
            return true;
        }

        private static string CombinePath(string path1, string path2)
        {
            return string.IsNullOrEmpty(path2) ? System.IO.Path.GetFullPath(path1) : System.IO.Path.Combine(System.IO.Path.GetFullPath(path1), path2);
        }

        private static string GetSigloPath(string path)
        {
            var root = TestPath.GetSigloRoot();
            return CombinePath(root, path);
        }

        private static string GetOutputtPath(string path)
        {
            var current = new System.IO.DirectoryInfo(System.Reflection.Assembly.GetExecutingAssembly().Location);
            var output = current.Parent.Parent.FullName;
            return CombinePath(output, path);
        }

        private static string RecreateOutputDirectory(string path)
        {
            var directory = GetOutputtPath(path);
            TestUtility.TestPath.DeleteDirectoryIfExisted(directory);
            Directory.CreateDirectory(directory);
            TestOutputs.Add(directory);
            return directory;
        }

        private Tuple<StringBuilder, StringBuilder> SetupProcess(Process process, string filename, string arguments)
        {
            var standardOutput = new StringBuilder();
            var standardError = new StringBuilder();

            process.StartInfo.FileName = filename;
            process.StartInfo.Arguments = arguments;
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardOutput = true;
            process.StartInfo.RedirectStandardError = true;
            process.StartInfo.StandardOutputEncoding = System.Text.Encoding.UTF8;
            process.OutputDataReceived += (object sender, DataReceivedEventArgs e) =>
            {
                standardOutput.AppendLine(e.Data);
            };
            process.ErrorDataReceived += (object sender, DataReceivedEventArgs e) =>
            {
                standardError.AppendLine(e.Data);
            };

            return new Tuple<StringBuilder, StringBuilder>(standardOutput, standardError);
        }

        private void RunAndWaitProcess(Process process)
        {
            Console.WriteLine($"[ContentsUploaderTest] process start.");
            Console.WriteLine($"[ContentsUploaderTest] command line: {process.StartInfo.FileName} {process.StartInfo.Arguments}");
            process.Start();
            process.BeginOutputReadLine();
            process.BeginErrorReadLine();
            Console.WriteLine($"[ContentsUploaderTest] process wait for exit.");
            process.WaitForExit();
        }

        private bool ExecuteMakeTestApplication(string arguments, string outputPath)
        {
            Console.WriteLine($"[ContentsUploaderTest] ExecuteMakeTestApplication start.");

            using (var process = new Process())
            {
                var outputs = SetupProcess(process, MakeTestApplicationPath, arguments);
                RunAndWaitProcess(process);

                var stdout = outputs.Item1.ToString();
                var stderr = outputs.Item2.ToString();
                if (process.ExitCode != 0)
                {
                    Console.WriteLine("[ContentsUploaderTest] Standard Output:");
                    Console.WriteLine(stdout);
                    Console.WriteLine("[ContentsUploaderTest] Standard Error:");
                    Console.WriteLine(stderr);
                    Environment.ExitCode = 1;
                    return false;
                }

                Console.WriteLine("[ContentsUploaderTest] Standard Output:");
                Console.WriteLine(stdout);
                return stdout.Contains("--ElapsedTime:");
            }
        }

        private bool ExecuteContentsUploader(string subcommand, string arguments, string outputPath, string[] expects = null)
        {
            Console.WriteLine($"[ContentsUploaderTest] ExecuteContentsUploader {subcommand} start.");

            using (var process = new Process())
            {
                var outputs = SetupProcess(process, ContentsUploaderPath, $"{subcommand} {CommonkOption} {arguments}");
                RunAndWaitProcess(process);

                var stdout = outputs.Item1.ToString();
                var stderr = outputs.Item2.ToString();
                if (process.ExitCode != 0)
                {
                    Console.WriteLine("[ContentsUploaderTest] Standard Output:");
                    Console.WriteLine(stdout);
                    Console.WriteLine("[ContentsUploaderTest] Standard Error:");
                    Console.WriteLine(stderr);
                    Environment.ExitCode = 1;
                    return false;
                }

                Console.WriteLine("[ContentsUploaderTest] Standard Output:");
                Console.WriteLine(stdout);

                var result = true;
                var list = new List<string>();
                if (CommandSuccessTable.ContainsKey(subcommand))
                {
                    list.Add(CommandSuccessTable[subcommand]);
                }
                if (expects != null)
                {
                    list.AddRange(expects);
                }
                foreach (var expect in list)
                {
                    if (!stdout.Contains(expect))
                    {
                        result = false;
                        Console.WriteLine($"[ContentsUploaderTest] Failure: Not found expected output \"{expect}\".");
                    }
                }
                return result;
            }
        }

        private bool ExecuteDevMenuCommandSystem(string exeArgs, string nspArgs)
        {
            Console.WriteLine("[ContentsUploaderTest] ExecuteDevMenuCommandSystem start.");

            using (var process = new Process())
            {
                var arguments = $"{DevMenuCommandSystemPath} -t {TargetName} {exeArgs} -- {nspArgs}";
                var outputs = SetupProcess(process, RunOnTargetPath, arguments);
                RunAndWaitProcess(process);

                var stdout = outputs.Item1.ToString();
                var stderr = outputs.Item2.ToString();
                if (process.ExitCode != 0)
                {
                    Console.WriteLine("[ContentsUploaderTest] Standard Output:");
                    Console.WriteLine(stdout);
                    Console.WriteLine("[ContentsUploaderTest] Standard Error:");
                    Console.WriteLine(stderr);
                    Environment.ExitCode = 1;
                    return false;
                }

                Console.WriteLine("[ContentsUploaderTest] Standard Output:");
                Console.WriteLine(stdout);
                return stdout.Contains("[SUCCESS]");
            }
        }

        [TestCleanup]
        public void TestCleanup()
        {
            var name = TestContext.TestName;
            var result = TestContext.CurrentTestOutcome;
            TestResults[name] = result;

            // IDE による単体実行時はテスト成果物を残しておく
            if (!IsUnitTestOnIDE)
            {
                foreach (var directory in TestOutputs)
                {
                    TestUtility.TestPath.DeleteDirectoryIfExisted(directory);
                }
            }
            TestOutputs.Clear();
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void UploadTestApplication()
        {
            var outputDirectory = RecreateOutputDirectory("logApplicationOutput");
            Assert.IsTrue(ExecuteMakeTestApplication($"--type Application --output-file-name app_v0 --size 40000 --id {UploadApplicationId} -o {outputDirectory}", outputDirectory));
            Assert.IsFalse(ExecuteContentsUploader("upload", $"{UploadOption} -s {outputDirectory}", outputDirectory)); // オプションなしアップロード、製品化処理が行われないから失敗
            Assert.IsTrue(ExecuteContentsUploader("upload", $"{UploadOption} -s {outputDirectory} --mastering-enabled", outputDirectory)); // サーバーで製品化処理を行うアップロード
            Assert.IsTrue(ExecuteContentsUploader("upload", $"{UploadOption} -s {outputDirectory} --allow-non-mastered", outputDirectory)); // 製品化処理を行わないことを許容するアップロード
            Assert.IsTrue(ExecuteContentsUploader("list-rom-id", $"--content-meta-id {UploadApplicationId}", outputDirectory));
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void UploadTestAddOnContent()
        {
            var outputDirectory = RecreateOutputDirectory("logAddOnContentOutput");
            var makeCommonArgs = $"--type AddOnContent --no-ticket --id {UploadApplicationId} -o {outputDirectory}";
            Assert.IsTrue(ExecuteMakeTestApplication($"--output-file-name aoc_i1 --index 1 {makeCommonArgs}", outputDirectory));
            Assert.IsTrue(ExecuteMakeTestApplication($"--output-file-name aoc_i2 --index 2 {makeCommonArgs}", outputDirectory));
            Assert.IsFalse(ExecuteContentsUploader("upload", $"{UploadOption} -s {outputDirectory} --use-index-for-aoc-archive-number", outputDirectory)); // オプションなしアップロード、製品化処理が行われないから失敗
            Assert.IsTrue(ExecuteContentsUploader("upload", $"{UploadOption} -s {outputDirectory} --use-index-for-aoc-archive-number --mastering-enabled", outputDirectory)); // サーバーで製品化処理を行うアップロード
            Assert.IsTrue(ExecuteContentsUploader("upload", $"{UploadOption} -s {outputDirectory} --use-index-for-aoc-archive-number --allow-non-mastered", outputDirectory)); // 製品化処理を行わないことを許容するアップロード
            Assert.IsTrue(ExecuteContentsUploader("list-rom-id", $"--content-meta-id {UploadAocIndex1Id}", outputDirectory));
            Assert.IsTrue(ExecuteContentsUploader("list-rom-id", $"--content-meta-id {UploadAocIndex2Id}", outputDirectory));
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void UploadTestPatch()
        {
            var outputAppDirectory = RecreateOutputDirectory("logAppOutput");
            var outputPatchDirectory = RecreateOutputDirectory("logPatchOutput");
            Assert.IsTrue(ExecuteMakeTestApplication($"--type Application --output-file-name app_v0 --size 40000 --id {UploadApplicationId} -o {outputAppDirectory}", outputAppDirectory));
            var patchCommonArgs = $"--type Patch --original-application {outputAppDirectory}\\app_v0.nsp --id {UploadApplicationId} -o {outputPatchDirectory}";
            Assert.IsTrue(ExecuteMakeTestApplication($"--output-file-name patch_v1 --ver 1 {patchCommonArgs}", outputPatchDirectory));
            Assert.IsTrue(ExecuteMakeTestApplication($"--output-file-name patch_v3 --ver 3 --previous-patch {outputPatchDirectory}\\patch_v1.nsp {patchCommonArgs}", outputPatchDirectory));
            Assert.IsTrue(ExecuteContentsUploader("delete-version", $"-s {outputPatchDirectory}", outputPatchDirectory));
            Assert.IsTrue(ExecuteContentsUploader("list-version", $"--content-meta-id {UploadPatchId}", outputPatchDirectory));
            Assert.IsTrue(ExecuteContentsUploader("upload", $"{UploadOption} -s {outputAppDirectory} --allow-non-mastered", outputAppDirectory)); // 製品化処理を行わないことを許容するアップロード
            Assert.IsTrue(ExecuteContentsUploader("upload", $"{UploadOption} -s {outputPatchDirectory}", outputPatchDirectory)); // パッチは必ず製品化済なので、オプションなしでアップロード可能
            Assert.IsTrue(ExecuteContentsUploader("register-version", $"-s {outputPatchDirectory}", outputPatchDirectory));
            Assert.IsTrue(ExecuteContentsUploader("list-version", $"--content-meta-id {UploadPatchId}", outputPatchDirectory));
            Assert.IsTrue(ExecuteContentsUploader("list-rom-id", $"--content-meta-id {UploadPatchId}", outputPatchDirectory));
        }

        [TestMethod]
        [Timeout(300 * 1000)]
        public void TestDownload()
        {
            if (!IsPassed("UploadTestApplication", "UploadTestAddOnContent", "UploadTestPatch"))
            {
                Console.WriteLine("[ContentsUploaderTest] Warning: TestDownload Skipped");
                return;
            }

            Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"servicediscovery import-all td1pass"));
            Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"application clear-task-status-list AND application uninstall --all AND sdcard format AND sdcard cleanup"));
            Assert.IsTrue(ExecuteDevMenuCommandSystem("--reset", $"application list --detail AND application list-record-detail AND application list-view"));

            Assert.IsTrue(ExecuteDevMenuCommandSystem("--failure-timeout 600",
                $"application create-download-task {UploadApplicationId} AND application wait-download {UploadApplicationId}"));

            Assert.IsTrue(ExecuteDevMenuCommandSystem("--failure-timeout 600",
                $"application create-download-task {UploadApplicationId} --type Patch --id {UploadPatchId} --version 65536 AND application wait-download {UploadApplicationId}"));

            Assert.IsTrue(ExecuteDevMenuCommandSystem("--failure-timeout 600",
                $"application create-download-task {UploadApplicationId} --type AddOnContent --id {UploadAocIndex1Id} --version 0 AND application wait-download {UploadApplicationId}"));

            Assert.IsTrue(ExecuteDevMenuCommandSystem("--failure-timeout 600",
                $"application create-download-task {UploadApplicationId} --type AddOnContent --id {UploadAocIndex2Id} --version 0 AND application wait-download {UploadApplicationId}"));

            Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"application list --detail AND application list-record-detail AND application list-view"));
            Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"application launch {UploadApplicationId}"));
            Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"application verify {UploadApplicationId} AND application list-record {UploadApplicationId} --installed"));
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void RegisterDemoTest()
        {
            var NsUids = new string[] { $"DemoNsUid: {DemoNsUid}" };

            // upload 前に明示的な revoke-rom を行う
            var outputDirectory = RecreateOutputDirectory("logRegisterDemoOutput");
            Assert.IsTrue(ExecuteMakeTestApplication($"--type Application --ticket --output-file-name app_v0 --size 40000 --id {RegisterDemoApplicationId} -o {outputDirectory}", outputDirectory));
            Assert.IsTrue(ExecuteMakeTestApplication($"--type AddOnContent --ticket --output-file-name aoc_i1 --index 1 --id {RegisterDemoApplicationId} -o {outputDirectory}", outputDirectory));
            var patchCommonArgs = $"--type Patch --original-application {outputDirectory}\\app_v0.nsp --id {RegisterDemoApplicationId} -o {outputDirectory}";
            Assert.IsTrue(ExecuteMakeTestApplication($"--output-file-name patch_v1 --ver 1 {patchCommonArgs}", outputDirectory));
            Assert.IsTrue(ExecuteContentsUploader("revoke-rom", $"{UserOption} --application-id {RegisterDemoApplicationId}", outputDirectory));
            Assert.IsTrue(ExecuteContentsUploader("upload", $"{UploadOption} -s {outputDirectory} --approve-accept-enabled --server-checking-enabled", outputDirectory)); // 全て製品化済なので、オプションなしでアップロード可能
            Assert.IsTrue(ExecuteContentsUploader("register-demo", $"--application-id {RegisterDemoApplicationId} --name \"demo %application_id% %language_name%\" --language=all", outputDirectory, NsUids));
            Assert.IsTrue(ExecuteContentsUploader("register-demo", $"-s {outputDirectory} --name \"demo %application_id% %language_name%\" --language=default", outputDirectory, NsUids));
            Assert.IsTrue(ExecuteContentsUploader("list-rom-id", $"--application-id {RegisterDemoApplicationId}", outputDirectory));
        }

        [TestMethod]
        [Timeout(600 * 1000)]
        public void RegisterTitleTest()
        {
            var NsUids = new string[] { $"ApplicationNsUid:  {TitleNsUid}", $"AddOnContentNsUid: {Aoc1NsUid}" };

            // upload 時に自動的に revoke-rom が行われる
            var outputDirectory = RecreateOutputDirectory("logRegisterTitleOutput");
            Assert.IsTrue(ExecuteMakeTestApplication($"--type Application --ticket --output-file-name app_v0 --size 40000 --id {RegisterTitleApplicationId} -o {outputDirectory}", outputDirectory));
            Assert.IsTrue(ExecuteMakeTestApplication($"--type AddOnContent --ticket --output-file-name aoc_i1 --index 1 --id {RegisterTitleApplicationId} -o {outputDirectory}", outputDirectory));
            var patchCommonArgs = $"--type Patch --original-application {outputDirectory}\\app_v0.nsp --id {RegisterTitleApplicationId} -o {outputDirectory}";
            Assert.IsTrue(ExecuteMakeTestApplication($"--output-file-name patch_v1 --ver 1 {patchCommonArgs}", outputDirectory));
            Assert.IsTrue(ExecuteContentsUploader("upload", $"{UploadOption} -s {outputDirectory} --approve-accept-enabled --server-checking-enabled", outputDirectory)); // 全て製品化済なので、オプションなしでアップロード可能
            Assert.IsTrue(ExecuteContentsUploader("register-title", $"{UserOption} --application-id {RegisterTitleApplicationId} --initial-code ko001 --name \"formal %initial_code% %language%\" --language=all", outputDirectory, NsUids));
            Assert.IsTrue(ExecuteContentsUploader("register-title", $"{UserOption} -s {outputDirectory} --initial-code ko002 --name \"formal %application_id%:%index% %language_name%\" --language=default", outputDirectory, NsUids));
            Assert.IsTrue(ExecuteContentsUploader("list-rom-id", $"--application-id {RegisterTitleApplicationId}", outputDirectory));
        }

        [TestMethod]
        [Timeout(900 * 1000)]
        public void RegisterBundleTest()
        {
            var NsUids = new string[] { $"BundleNsUid: {BundleNsUid}" };

            // upload 時に自動的に revoke-rom が行われる
            var outputDirectory = RecreateOutputDirectory("logRegisterBundleOutput");
            Assert.IsTrue(ExecuteMakeTestApplication($"--type Application --ticket --output-file-name app_v0 --size 40000 --id {RegisterBundleApplicationId} -o {outputDirectory}", outputDirectory));
            Assert.IsTrue(ExecuteMakeTestApplication($"--type AddOnContent --ticket --output-file-name aoc_i1 --index 1 --id {RegisterBundleApplicationId} -o {outputDirectory}", outputDirectory));
            Assert.IsTrue(ExecuteMakeTestApplication($"--type AddOnContent --ticket --output-file-name aoc_i2 --index 2 --id {RegisterBundleApplicationId} -o {outputDirectory}", outputDirectory));
            var patchCommonArgs = $"--type Patch --original-application {outputDirectory}\\app_v0.nsp --id {RegisterBundleApplicationId} -o {outputDirectory}";
            Assert.IsTrue(ExecuteMakeTestApplication($"--output-file-name patch_v1 --ver 1 {patchCommonArgs}", outputDirectory));
            Assert.IsTrue(ExecuteContentsUploader("upload", $"{UploadOption} -s {outputDirectory} --approve-accept-enabled --server-checking-enabled", outputDirectory)); // 全て製品化済なので、オプションなしでアップロード可能
            Assert.IsTrue(ExecuteContentsUploader("register-title", $"{UserOption} -s {outputDirectory} --initial-code ko005 --name \"title %application_id%:%index% %language_name%\" --language=default", outputDirectory));
            Assert.IsTrue(ExecuteContentsUploader("register-bundle", $"{UserOption} --ns-uid {BundleNsUid} --bundle-id {RegisterTitleApplicationId},{RegisterBundleApplicationId} --initial-code ko001", outputDirectory, NsUids));
            Assert.IsTrue(ExecuteContentsUploader("register-bundle", $"{UserOption} --ns-uid {BundleNsUid} --bundle-id {RegisterBundleAocIndex1Id},{RegisterBundleAocIndex2Id} --initial-code ko002  --name \"bundle name\"", outputDirectory, NsUids));
            Assert.IsTrue(ExecuteContentsUploader("register-bundle", $"{UserOption} --ns-uid {BundleNsUid} -s {outputDirectory} --initial-code ko003  --name \"bundle %language_name%\" --language=default", outputDirectory, NsUids));
            Assert.IsTrue(ExecuteContentsUploader("list-rom-id", $"--application-id {RegisterBundleApplicationId}", outputDirectory));
        }

        [TestMethod]
        [Timeout(1200 * 1000)]
        public void PurchaseTest()
        {
            if (!IsPassed("RegisterDemoTest", "RegisterTitleTest", "RegisterBundleTest"))
            {
                Console.WriteLine("[ContentsUploaderTest] Warning: PurchaseTest Skipped");
                return;
            }

            try
            {
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"servicediscovery import-all td1pass"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"application clear-task-status-list AND application uninstall --all AND sdcard format AND sdcard cleanup"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem("--reset", $"application list --detail AND application list-record-detail AND application list-view"));

                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"ticket delete-all"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"shop unlink-device-all"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"shop unregister-device-account"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"account clear_all"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"shop register-device-account"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"account add --register_nsa"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"account link --index 0 --id koshiyama_ssdtest_100@exmx.nintendo.co.jp --password 226t61te2t"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"shop delete-all-rights 0"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"shop shop-account-status 0"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"shop link-device 0"));

                // register-demo したコンテンツをダウンロード
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"shop download-demo 0 --id {RegisterDemoApplicationId}"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"shop download-demo 0 --id {RegisterDemoAocIndex1Id}"));
                System.Threading.Thread.Sleep(60 * 1000);
                Assert.IsTrue(ExecuteDevMenuCommandSystem("--failure-timeout 300", $"application wait-download {RegisterDemoApplicationId}"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"application list --detail AND application list-record-detail AND application list-view"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"application launch {RegisterDemoApplicationId}"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"application verify {RegisterDemoApplicationId} AND application list-record {RegisterDemoApplicationId} --installed"));

                // register-title したコンテンツをダウンロード
                // register-title の更新 API が正しく動作することを前提として、決め打ちの nsUid で購入を行う
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"shop purchase 0 --ns-uid {TitleNsUid} --auto"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"shop purchase 0 --ns-uid {Aoc1NsUid} --auto --type aocs"));
                System.Threading.Thread.Sleep(60 * 1000);//! DTL に載るまで処理を待つ
                Assert.IsTrue(ExecuteDevMenuCommandSystem("--failure-timeout 300", $"application wait-download {RegisterTitleApplicationId}"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"application list --detail AND application list-record-detail AND application list-view"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"application launch {RegisterTitleApplicationId}"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"application verify {RegisterTitleApplicationId} AND application list-record {RegisterTitleApplicationId} --installed"));

                // register-bundle したコンテンツをダウンロード
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"shop purchase 0 --ns-uid {BundleNsUid} --auto --type bundles"));
                System.Threading.Thread.Sleep(60 * 1000);//! DTL に載るまで処理を待つ
                Assert.IsTrue(ExecuteDevMenuCommandSystem("--failure-timeout 300", $"application wait-download {RegisterBundleApplicationId}"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"application list --detail AND application list-record-detail AND application list-view"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"application launch {RegisterBundleApplicationId}"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"application verify {RegisterBundleApplicationId} AND application list-record {RegisterBundleApplicationId} --installed"));

                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"shop delete-all-rights 0"));

                //! アカウント解除
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"shop unlink-device 0"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"account clear_all"));
                Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, $"shop unregister-device-account"));
            }
            finally
            {
                ExecuteDevMenuCommandSystem("--reset",    $"shop unlink-device-all");
                ExecuteDevMenuCommandSystem(string.Empty, $"account clear_all");
                ExecuteDevMenuCommandSystem(string.Empty, $"shop unregister-device-account");
            }
        }
    }
}
