﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using CodingCheckerUtil;
using System;
using System.Diagnostics;
using System.IO;
using TestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace GitHookTest
{
    public class GitHookTestContext : IDisposable
    {
        public string TestRemoteRepositoryDirectory => Path.Combine(m_TestRepositoryDirectory, "remote");
        public string TestLocalRepositoryDirectory => Path.Combine(m_TestRepositoryDirectory, "local");
        private string m_TestRepositoryDirectory { get; set; }
        private PrivateObject m_GitExecuterObject { get; set; } = new PrivateObject(new GitExecuter(TestUtils.GetGitExecutablePath()));
        private bool m_DisposedValue { get; set; } = false;

        public GitHookTestContext(TestContext testContext)
        {
            // TestRunner から実行した際の TestRunDirectory のパス長に余裕がないため、Outputs 以下にテスト用リポジトリを生成
            var testPath = new TestPath(testContext);
            m_TestRepositoryDirectory = Path.Combine(testPath.GetTestRoot(), @"Outputs\AnyCPU\Tools\CodingCheckerTest\GitHookTest\Release", "testgit", testContext.TestName);

            // 前回テストが中断された場合などでリポジトリが残っていれば削除
            if (Directory.Exists(m_TestRepositoryDirectory))
            {
                DeleteDirectory(m_TestRepositoryDirectory, true);
            }

            // テスト用リポジトリ作成
            Directory.CreateDirectory(m_TestRepositoryDirectory);
            ExecuteGit($"init --bare \"{TestRemoteRepositoryDirectory}\"");
            ExecuteGit($"clone \"{TestRemoteRepositoryDirectory}\" \"{TestLocalRepositoryDirectory}\"");

            // テスト用ローカルリポジトリへの PreCommitChecker のインストール
            InstallPreCommitChecker(testPath);
            // カレントディレクトリをテスト用ローカルリポジトリのルートに設定
            Environment.CurrentDirectory = TestLocalRepositoryDirectory;
        }

        public string ExecuteGit(string command)
        {
            return (string)m_GitExecuterObject.Invoke("Execute", command);
        }

        private static void DeleteDirectory(string directory, bool recursive)
        {
            foreach (var file in Directory.EnumerateFiles(directory, "*", SearchOption.AllDirectories))
            {
                var finfo = new FileInfo(file);
                if (finfo.Attributes.HasFlag(FileAttributes.ReadOnly))
                {
                    finfo.Attributes &= ~FileAttributes.ReadOnly;
                }
            }
            Directory.Delete(directory, recursive);
        }

        private void InstallPreCommitChecker(TestPath testPath)
        {
#if DEBUG
            // @"Integrate\Sources\Tools\PreCommitChecker\install.cmd" は Release のバイナリをインストールするので、
            // Debug ビルドでテストを実行することはできない
            // （厳密には、DEBUG マクロの有無ではなく、 構成が Release であるかどうかによる）
            throw new NotImplementedException();
#else
            string hooksDir = Path.Combine(TestLocalRepositoryDirectory, ".git", "hooks");
            string sigloCommonPreCommitScript = Path.Combine(testPath.GetSigloRoot(), @"Integrate\Repository\HookScripts\Common\pre-commit");
            string sigloCommonCommitMsgScript = Path.Combine(testPath.GetSigloRoot(), @"Integrate\Repository\HookScripts\Common\commit-msg");
            File.Copy(sigloCommonPreCommitScript, Path.Combine(hooksDir, "pre-commit"));
            File.Copy(sigloCommonCommitMsgScript, Path.Combine(hooksDir, "commit-msg"));

            string preCommitCheckerInstallScript = Path.Combine(testPath.GetSigloRoot(), @"Integrate\Sources\Tools\CodingChecker\install.cmd");

            var psi = new ProcessStartInfo()
            {
                FileName = preCommitCheckerInstallScript,
                Arguments = hooksDir,
                CreateNoWindow = true,
                UseShellExecute = false,
                RedirectStandardInput = true,
                RedirectStandardOutput = true,
                RedirectStandardError = true,
            };

            using (var p = Process.Start(psi))
            {
                p.StandardInput.Close();
                p.StandardOutput.Close();
                var errorTask = p.StandardError.ReadToEndAsync();
                p.WaitForExit();
                errorTask.Wait();

                if (p.ExitCode != 0)
                {
                    throw new Exception($"error: {preCommitCheckerInstallScript} {hooksDir} returned {p.ExitCode}. {errorTask.Result}");
                }
            }
#endif
        }

        #region IDisposable Support

        protected virtual void Dispose(bool disposing)
        {
            if (!m_DisposedValue)
            {
                if (disposing)
                {
                    // カレントディレクトリが削除対象のディレクトリ以下なら移動
                    if (Environment.CurrentDirectory.StartsWith(m_TestRepositoryDirectory, StringComparison.InvariantCultureIgnoreCase))
                    {
                        Environment.CurrentDirectory = Path.Combine(m_TestRepositoryDirectory, "..");
                    }

                    if (Directory.Exists(m_TestRepositoryDirectory))
                    {
                        DeleteDirectory(m_TestRepositoryDirectory, true);
                    }
                }

                m_DisposedValue = true;
            }
        }

        public void Dispose()
        {
            Dispose(true);
        }
        #endregion
    }
}
