﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace CommitLogCheckerTest
{
    [TestClass]
    public class CommitLogCheckerTest
    {
        [TestMethod]
        public void TestCheckCommitLog()
        {
            List<string> jiraKeys = new List<string> { "JIRAKEY1", "JIRAKEY2" };

            string[] goodLogs =
            {
                @"(JIRAKEY1-1) 表題",
                @"(JIRAKEY2-100) Title is here

* 項目１
* 項目２
* 項目３
* 項目４",
            };

            string[] badLogs =
            {
                @"(JIRA-1) 表題", // JIRAKEY 誤り
                @"(JIRAKEY-1) 表題
* 項目１", // 空行漏れ
                @"($(JIRA-KEY)) 表題", // $(JIRA-KEY) が残っていてはならない
                @"(JIRAKEY-1) 表題

* $(JIRA-KEY)", // $(JIRA-KEY) が残っていてはならない

                // 古い形式
                @"[JIRAKEY1-1] 表題",
                @"[JIRAKEY2-100] Title is here

* 項目１
* 項目２
* 項目３
* 項目４",
                @"other:[JIRAKEY1-1] (somemodule) 表題",
                @"product:[JIRAKEY2-100] (othermodule) Title is here

* 項目１
* 項目２
* 項目３
* 項目４",
                // 古い形式の誤り
                @"[JIRA-1] 表題", // JIRAKEY 誤り
                @"[JIRAKEY-1] 表題
* 項目１", // 空行漏れ
                @"[$(JIRA-KEY)] 表題", // $(JIRA-KEY) が残っていてはならない
                @"[JIRAKEY-1] 表題

* $(JIRA-KEY)", // $(JIRA-KEY) が残っていてはならない
                @"refactor:[JIRAKEY-1] (somemodule) 表題", // 種別誤り
                @"other[JIRAKEY-1] (somemodule) 表題", // コロン無し
                @"other:[JIRA-1] (somemodule) 表題", // JIRAKEY 誤り
                @"other:[JIRAKEY-1] 表題", // モジュール漏れ
                @"other:[JIRAKEY-1] (somemodule) 表題
* 項目１", // 空行漏れ
            };

            string[] badJiraKeyLogs =
            {
                @"(BADJIRA-1) 表題",
                @"(BADJIRAKEY-100) Title is here

* 項目１
* 項目２
* 項目３
* 項目４",
            };

            string[] badJiraKeyLogsButWip =
            {
                @"(BADJIRA-1) WIP:表題",
                @"(BADJIRAKEY-100) wip: Title is here

* 項目１
* 項目２
* 項目３
* 項目４",
            };

            string[] badLogsButFixup =
            {
                @"fixup!(JIRAKEY-1) 表題
* 項目１", // 空行漏れ
            };

            var commitLogChecker = new CommitLogChecker.CommitLogChecker(jiraKeys);

            foreach (var log in goodLogs)
            {
                Assert.IsTrue(commitLogChecker.CheckCommitLog(log));
            }
            foreach (var log in badJiraKeyLogsButWip)
            {
                Assert.IsTrue(commitLogChecker.CheckExceptionCommit(log));
            }
            foreach (var log in badLogsButFixup)
            {
                Assert.IsTrue(commitLogChecker.CheckExceptionCommit(log));
            }
            foreach (var log in badLogs)
            {
                Assert.IsFalse(commitLogChecker.CheckCommitLog(log));
            }

            var commitLogCheckerEmptyJiraKey = new CommitLogChecker.CommitLogChecker(new List<string>());
            foreach (var log in badJiraKeyLogs)
            {
                Assert.IsFalse(commitLogChecker.CheckCommitLog(log));
                Assert.IsTrue(commitLogCheckerEmptyJiraKey.CheckCommitLog(log)); // JIRA のキーが指定されてなければ通る
            }
        }
    }
}
