﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using CodingCheckerUtil;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace CodingCheckerUtilTest
{
    [TestClass]
    public class TargetExtensionListParserTest
    {
        public TestContext TestContext { get; set; }

        [TestMethod]
        public void ParseTargetExtensionListTest()
        {
            // テスト用の拡張子指定ファイル
            string tempDir = Path.Combine(TestContext.TestRunDirectory, nameof(ParseTargetExtensionListTest));
            Directory.CreateDirectory(tempDir);
            string targetExtensionListPath = Path.Combine(tempDir, "TestTargetExtensions.list");

            // すべての列挙値が拡張子ごとに認識できる
            var target = ".aaa\tAll\r\n.bbb\tCheckBOM\r\n.ccc\tAutoFormatContents\r\n.ddd\tNone\r\n";
            File.WriteAllText(targetExtensionListPath, target);
            var expected = new Dictionary<string, CheckOptions>()
            {
                { ".aaa", CheckOptions.All },
                { ".bbb", CheckOptions.CheckBOM },
                { ".ccc", CheckOptions.AutoFormatContents },
                { ".ddd", CheckOptions.None },
            };
            Assert.IsTrue(DictionaryEquals(new TargetExtensionListParser(targetExtensionListPath).TargetFileCheckOptionsForDebug, expected));

            // 列挙値を組み合わせて指定できる
            target = ".aaa\tCheckBOM | AutoFormatContents\n.bbb\t  CheckBOM|CheckBOM|CheckBOM  \n";
            File.WriteAllText(targetExtensionListPath, target);
            expected = new Dictionary<string, CheckOptions>()
            {
                { ".aaa", CheckOptions.CheckBOM | CheckOptions.AutoFormatContents },
                { ".bbb", CheckOptions.CheckBOM },
            };
            Assert.IsTrue(DictionaryEquals(new TargetExtensionListParser(targetExtensionListPath).TargetFileCheckOptionsForDebug, expected));

            // チェックオプションがなければ TargetExtensionListParser.CheckOptions.All が選択される
            target = ".aaa\n.bbb\n";
            File.WriteAllText(targetExtensionListPath, target);
            expected = new Dictionary<string, CheckOptions>()
            {
                { ".aaa", CheckOptions.All },
                { ".bbb", CheckOptions.All },
            };
            Assert.IsTrue(DictionaryEquals(new TargetExtensionListParser(targetExtensionListPath).TargetFileCheckOptionsForDebug, expected));

            // 空行は無視される
            target = ".aaa\tCheckBOM|AutoFormatContents\r\n\r\n.bbb\r\n";
            File.WriteAllText(targetExtensionListPath, target);
            expected = new Dictionary<string, CheckOptions>()
            {
                { ".aaa", CheckOptions.CheckBOM | CheckOptions.AutoFormatContents },
                { ".bbb", CheckOptions.All },
            };
            Assert.IsTrue(DictionaryEquals(new TargetExtensionListParser(targetExtensionListPath).TargetFileCheckOptionsForDebug, expected));

            // 不明なチェックオプション文字列に対しては例外が発生する
            target = ".aaa\tCheckBOM | UnknownOptions\r\n";
            File.WriteAllText(targetExtensionListPath, target);
            try
            {
                new TargetExtensionListParser(targetExtensionListPath);
                Assert.Fail();
            }
            catch (ArgumentException)
            {
            }
        }

        [TestMethod]
        public void GetFileCheckOptionsTest()
        {

            // テスト用の拡張子指定ファイルの生成
            string tempDir = Path.Combine(TestContext.TestRunDirectory, nameof(GetFileCheckOptionsTest));
            Directory.CreateDirectory(tempDir);
            string targetExtensionListPath = Path.Combine(tempDir, "TestTargetExtensions.list");
            // .cpp AutoFormatContents
            // .h   CheckBOM
            File.WriteAllText(targetExtensionListPath, ".cpp\tAutoFormatContents\r\n" + ".h\tCheckBOM\r\n");

            {
                var targetExtensionListParser = new TargetExtensionListParser(targetExtensionListPath);

                // 拡張子に対応するチェックオプションが取得される
                Assert.AreEqual(CheckOptions.CheckBOM, targetExtensionListParser.GetFileCheckOptions(@"Hogehoge.h"));
                Assert.AreEqual(CheckOptions.CheckBOM, targetExtensionListParser.GetFileCheckOptions(@"Hogehoge.hoge.h"));
                Assert.AreEqual(CheckOptions.AutoFormatContents, targetExtensionListParser.GetFileCheckOptions(@"c:\Hoge\Fuga\Hogehoge.cpp"));
                Assert.AreEqual(CheckOptions.AutoFormatContents, targetExtensionListParser.GetFileCheckOptions(@"c:\Hoge.dir\Fuga\Hogehoge.win.cpp"));

                // 対応する拡張子がなければ何もチェックしない
                Assert.AreEqual(CheckOptions.None, targetExtensionListParser.GetFileCheckOptions(@"Hogehoge.hpp"));
                Assert.AreEqual(CheckOptions.None, targetExtensionListParser.GetFileCheckOptions(@"Hogehoge.h.hpp"));
                Assert.AreEqual(CheckOptions.None, targetExtensionListParser.GetFileCheckOptions(@"c:\Hoge\Fuga\Hogehoge"));
                Assert.AreEqual(CheckOptions.None, targetExtensionListParser.GetFileCheckOptions(@"c:\Hoge.cpp\Fuga\Hogehoge.win"));
            }

            {
                var targetExtensionListParser = new TargetExtensionListParser(string.Empty);

                // 拡張子指定がなければすべてのファイルに全チェック項目を適用
                Assert.AreEqual(CheckOptions.All, targetExtensionListParser.GetFileCheckOptions(@"Hogehoge.h"));
                Assert.AreEqual(CheckOptions.All, targetExtensionListParser.GetFileCheckOptions(@"Hogehoge.hoge.h"));
                Assert.AreEqual(CheckOptions.All, targetExtensionListParser.GetFileCheckOptions(@"c:\Hoge\Fuga\Hogehoge.cpp"));
                Assert.AreEqual(CheckOptions.All, targetExtensionListParser.GetFileCheckOptions(@"c:\Hoge.dir\Fuga\Hogehoge.win.cpp"));
                Assert.AreEqual(CheckOptions.All, targetExtensionListParser.GetFileCheckOptions(@"Hogehoge.hpp"));
                Assert.AreEqual(CheckOptions.All, targetExtensionListParser.GetFileCheckOptions(@"Hogehoge.h.hpp"));
                Assert.AreEqual(CheckOptions.All, targetExtensionListParser.GetFileCheckOptions(@"c:\Hoge\Fuga\Hogehoge"));
                Assert.AreEqual(CheckOptions.All, targetExtensionListParser.GetFileCheckOptions(@"c:\Hoge.cpp\Fuga\Hogehoge.win"));
            }
        }

        private bool DictionaryEquals<TKey, TValue>(IDictionary<TKey, TValue> lhs, IDictionary<TKey, TValue> rhs)
        {
            if (lhs == null || rhs == null)
            {
                return false;
            }
            else
            {
                return lhs.Count == rhs.Count && !lhs.Except(rhs).Any();
            }
        }
    }
}
