# -*- coding: utf-8 -*-
from __future__ import division, print_function, absolute_import, unicode_literals

import os
import sys
import json
import codecs
from argparse import ArgumentParser

from kiki.util.log import file_logger as logger

# この時点で読み込み内容が固定化される
from lib import lib_path

# modified to adapt to example
from examples.res import res_path
from examples.res.res_path import RES_PATH as res_path_org

from lib.lib_path import LIB_PATH as lib_path_org
from kiki.util.log import LOG_DIR

"""
------------------------------------------------
How to use
------------------------------------------------
```
from config.config import CONF
from kiki.util.log import file_logger as logger

logger.info(CONF.RES_PATH)
logger.info(CONF.LIB_PATH)
logger.info(CONF.MCAT)
logger.info(CONF.WIN_POS)
CONF.show_all()
```

------------------------------------------------
load/reloadのタイミング
------------------------------------------------
初回にこのモジュールが呼び出されると、読み込みが発生する
それ以降はCONF.reload()で再読み込み可能

再読み込み時にはCONFのアトリビュートのみ動的に変更されるので注意


------------------------------------------------
コマンドライン引数でconfigのファイルパスを与える
------------------------------------------------
```
$ hoge.py -conf [conf_path]
```
のように-confのオプションで指定可能

------------------------------------------------
configのパラメータを増やしたくなったら
------------------------------------------------
ConfKey()
と
__init__()に値を追加すればOK

------------------------------------------------
config.jsonにあたるファイルを分割
------------------------------------------------
xx.jsonみたいなファイルを複数にして、互換性を保ちながら、
全部を認識するためのいい感じのクラスをかけなかったので、サポートしない

------------------------------------------------
これまでのバージョンとの互換性の問題
------------------------------------------------
* これまでのlib_path, res_pathも互換性を保てる
    * configの読み込みが必須
    * でも本当はCONF.LIB_PATHとかを使ってほしい
        * 変更されても動的反映される

------------------------------------------------
動作について
------------------------------------------------
* 値がおかしい系
    * keyが設定されてない時
        * デフォルト値が設定される
    * 値がnullのとき
        * デフォルト値が設定される
    * 値が空のとき
        * jsonのパースエラーになる
    * スペースが入ってるとき
        * 問題ない
    * ダメっぽい文字が入ってるとき
        * replace("\\" ,"\\\\")でOK
        * \\UとかになるのでOK
            * ただし、パス以外の値にも適応されるのがちょっとだけ心配
* パスの記法
    * 日本語パス
        * codecs.openでok
    * 相対パス
        * 絶対パスに直される
    * 絶対パス
        * そのまま使われる

"""


########################################################
# file path
########################################################

class ConfigManager(object):
    # キーのイテレータとPyCharmでの入力補完のため
    ###########################################################
    # ↓↓ If you want to add parameter, please edit here! ↓↓
    ###########################################################
    class ConfKey():
        class dic_item():
            def __init__( self, name, value):
                self.name = name
                self.value = value

        # 読み込まれるキーのリスト
        WIN_POS     = dic_item( "WIN_POS", "win_pos")
        MCAT        = dic_item( "MCAT", "mcat")
        RES_PATH    = dic_item( "RES_PATH", "res_path")
        LIB_PATH    = dic_item( "LIB_PATH", "lib_path")
        AMAREC      = dic_item( "AMAREC", "amarec")

        @classmethod
        def get_dic(cls):
            class dic_item():
                def __init__( self, name, value):
                    self.name = name
                    self.value = value
            return [
                dic_item( "WIN_POS", "win_pos"),
                dic_item( "MCAT", "mcat"),
                dic_item( "RES_PATH", "res_path"),
                dic_item( "LIB_PATH", "lib_path"),
                dic_item( "AMAREC", "amarec")
                ]

    ###########################################################
    # ↑↑ If you want to add parameter, please edit here! ↑↑
    ###########################################################

    def __init__(self, path, ):
        # 補完とデフォルト値のために初期化
        ###########################################################
        # ↓↓ If you want to add parameter, please edit here! ↓↓
        ###########################################################
        self.WIN_POS = (100, 100)
        self.MCAT = {
            "api_key": "dummy",
            "api_sec": "dummy"
        }
        self.RES_PATH = res_path_org
        self.LIB_PATH = lib_path_org
        self.AMAREC = {
            "saved_dir": "D:\WorkArea-z",
            "dest_dir": os.path.abspath(LOG_DIR),
            "rename_prefix": "movie_"
        }
        ###########################################################
        # ↑↑ If you want to add parameter, please edit here! ↑↑
        ###########################################################

        self.path = os.path.abspath(path)

        self.default_values = {}
        self._init_default_values()  # デフォルト値で初期化

        # 更新されたかどうか
        self.updated_flags = {}

    def _init_default_values(self):
        """
        デフォルト値を読み込む仕組み
        初期状態でメンバ変数に格納されているものをデフォルト値として保存
        イニシャライズされるときしか呼ばれない

        :return:
        """
        for key in self.ConfKey.get_dic():
            default_value = None
            # 次の処理で文字列かどうかで処理を変更する必要があるので、いったん変数に格納
            try:
                exec "default_value = self.{0}".format(key.name)
            except AttributeError:
                logger.warning("'{0}' has no default value.".format(key.value))

            # 文字列かどうかでコマンドが微妙に変わる
            if isinstance(default_value, str) or isinstance(default_value, unicode):
                cmd = "self.default_values['{0}'] = '{1}'".format(key.value, default_value)
            elif isinstance(default_value, dict):
                cmd = "self.default_values['{0}'] = {1}".format(key.value, unicode(default_value).replace("\\\\", "\\"))
            else:
                cmd = "self.default_values['{0}'] = {1}".format(key.value, default_value)

            # バックスラッシュが含まれているときにサニタイズ
            cmd = cmd.replace("\\", "\\\\")
            exec cmd

    ####################################################
    # main logic
    ####################################################
    def reload(self, encoding="utf-8"):
        """
        再読み込みを実行

        :param encoding:
        :return:
        """
        with codecs.open(self.path, encoding=encoding) as fi:
            text = fi.read().replace("\\", "\\\\")  # jsonの中にバックスラッシュがあっても大丈夫
            self.params = json.loads(text, "utf-8")

        # 読み込み後のデータの修正
        self.modify_params()

        # paramsの内容をメンバ変数に反映
        self.update_attr()

        logger.info("parameters are successfully loaded!")

    def modify_params(self):
        """
        読み込み後の修正処理
          * 配列→タプル
          * ファイルパス→絶対パス

        :return:
        """
        # タプル化
        key = self.ConfKey.WIN_POS.value
        if key in self.params:
            self.params[key] = tuple(self.params[key])

        # 絶対パス化
        key = self.ConfKey.RES_PATH.value
        if key in self.params:
            self.params[key] = self.default_values[key] if self.params[key] is None else os.path.abspath(
                self.params[key])

        key = self.ConfKey.LIB_PATH.value
        if key in self.params:
            self.params[key] = self.default_values[key] if self.params[key] is None else os.path.abspath(
                self.params[key])

        key = self.ConfKey.AMAREC.value
        if key in self.params:
            for key_detail in ["saved_dir", "dest_dir"]:
                if key_detail in self.params[key]:
                    if self.params[key][key_detail] is None:
                        self.params[key][key_detail] = os.path.abspath(self.default_values[key][key_detail])
                    else:
                        self.params[key][key_detail] = os.path.abspath(self.params[key][key_detail])

    def update_attr(self):
        """
        paramsの値をメンバ変数に反映

        :return:
        """
        # update attributes
        self.updated_flags = {}
        for key in self.ConfKey.get_dic():
            if key.value not in self.params:
                logger.warning("'{0}' parameter is not found, and initialized by default_value".format(key.value))
                self.params[key.value] = self.default_values[key.value]
                continue

            if isinstance(self.params[key.value], str) or isinstance(self.params[key.value], unicode):
                cmd = "self.{0} = '{1}'".format(key.name, self.params[key.value])
            elif isinstance(self.params[key.value], dict):
                cmd = "self.{0} = {1}".format(key.name, unicode(self.params[key.value]).replace("\\\\", "\\"))
            else:
                cmd = "self.{0} = {1}".format(key.name, self.params[key.value])

            # ファイルパスに対しては\ -> /でいいけど、ほか全部で大丈夫なのかは不明
            cmd = cmd.replace("\\", "\\\\")
            exec (cmd)

            self.updated_flags[key.value] = True

    ####################################################
    # util
    ####################################################
    def show_all(self):
        """
        読み込み済みの設定を表示

        :return:
        """
        logger.info("------------------------------------------------------------------------")
        logger.info("[attr of params : {0}]".format(self.path))
        for key in self.ConfKey.get_dic():
            logger.info("{0:20} -> {1:20} : {2}".format(key, key.value, self.params[key.value]))
        logger.info("------------------------------------------------------------------------")


########################################################
# Entry point
########################################################
# このモジュールが読み込まれたときにリロードが走る

# オプションを読み込む
_argparser = ArgumentParser()
_argparser.add_argument("-conf", "--confpath")

# confpath以外のオプションがもし来ていたら吸収
for _arg in sys.argv:
    if _arg.startswith("-"):
        if _arg != "-conf" and _arg != "--confpath":
            _argparser.add_argument(_arg, _arg)
_args, unknown = _argparser.parse_known_args()

# オプションで与えられてたらそれ、与えられてなかったら、デフォルトパス
CONF_PATH = _args.confpath if _args.confpath else os.path.dirname(__file__) + "/params.json"
CONF_PATH = os.path.abspath(CONF_PATH)

# 読み込み
CONF = ConfigManager(CONF_PATH)
CONF.reload()

# パスたちに更新繁栄
if hasattr(lib_path, "reflect_config"):
    lib_path._reflect_config_for_examples()  # modified to adapt to example
else:
    logger.warning("Such a named function was not found.")
    logger.warning("Please check your project version.")

if hasattr(res_path, "reflect_config"):
    res_path.reflect_config()
else:
    logger.warning("Such a named function was not found.")
    logger.warning("Please check your project version.")

# 値の確認
CONF.show_all()
