﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <iostream>
#include <cstdlib>
#include <chrono>
#include <locale>
#include <codecvt>
#include <nnt/base/testBase_Exit.h>
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include "..\Common\LogReaderWrapper.h"

/*
*   単体テスト
*   事前に Target Manager にターゲットに接続し、LogTestApp を起動しておく必要があります。
*   実行時には引数で対象ターゲットのシリアル番号を指定してください。
*/

namespace
{
    SerialNumberString g_SerialNumber;
    LogReaderWrapper g_LogReaderWrapper;

    class Basic : public ::testing::Test
    {
    protected:
        static void SetUpTestCase()
        {
            ASSERT_TRUE(g_LogReaderWrapper.TryLoadLibrary());
            ASSERT_TRUE(g_LogReaderWrapper.TryLoadFunctions());
        }

        virtual void SetUp()
        {
        }

        virtual void TearDown()
        {
        }
    };

    void PrintUtf8String(const char* str)
    {
        std::wcout.imbue(std::locale("japanese"));
        std::wstring_convert<std::codecvt_utf8<wchar_t>, wchar_t> converter;
        std::wcout << converter.from_bytes(std::string(str)) << std::endl;
    }

    void PrintCurrentString()
    {
        size_t size;
        EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_GetLogLineSizeFunction(&size));
        char* buffer = reinterpret_cast<char*>(malloc(size * sizeof(char)));
        EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_GetLogLineFunction(buffer, size));
        PrintUtf8String(buffer);
    }
}

TEST_F(Basic, WaitTestAppStart)
{
    ASSERT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_StartLogStoringFunction(g_SerialNumber, 100));

    const char* validRegexString = R"(Hello world \d+)";
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_WaitUntilMatchedFunction(validRegexString, 20000));

    g_LogReaderWrapper.m_StopLogStoringFunction();
}

TEST_F(Basic, StartStop)
{
    const SerialNumberString invalidSerial = { "XAL00000000000" };

    ASSERT_EQ(LogReaderResult_ConnectFailed, g_LogReaderWrapper.m_StartLogStoringFunction(invalidSerial, 100));
    ASSERT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_StartLogStoringFunction(g_SerialNumber, 100));
    ASSERT_EQ(LogReaderResult_AlreadyStarted, g_LogReaderWrapper.m_StartLogStoringFunction(g_SerialNumber, 100));
    g_LogReaderWrapper.m_StopLogStoringFunction();
    g_LogReaderWrapper.m_StopLogStoringFunction();
    ASSERT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_StartLogStoringFunction(g_SerialNumber, 100));
    g_LogReaderWrapper.m_StopLogStoringFunction();
}

TEST_F(Basic, GetLogSize)
{
    ASSERT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_StartLogStoringFunction(g_SerialNumber, 1000));
    size_t size1 = 0;
    size_t size2 = 0;

    // ログ取得を待機
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_WaitForNextLineFunction(1000));

    // timeout == 0 でも動作することを確認
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_WaitForNextLineFunction(0));

    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_GetLogLineSizeFunction(&size1));
    EXPECT_LT((size_t)0, size1);
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_GetLogLineSizeFunction(&size2));
    EXPECT_EQ(size1, size2);
    printf("size:%zu\n", size2);
    g_LogReaderWrapper.m_StopLogStoringFunction();
}

TEST_F(Basic, GetLog)
{
    const size_t bufferSize = 128;
    char buffer[bufferSize] = { 0 };
    char compareBuffer[bufferSize] = { 0 };

    ASSERT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_StartLogStoringFunction(g_SerialNumber, 1000));

    // ログ取得を待機
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_WaitForNextLineFunction(1000));

    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_GetLogLineFunction(buffer, bufferSize));
    printf("buffer:");
    PrintUtf8String(buffer);

    size_t size;
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_GetLogLineSizeFunction(&size));
    EXPECT_EQ(strlen(buffer) + 1, size);

    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_GetLogLineFunction(compareBuffer, bufferSize));
    EXPECT_EQ(0, strncmp(buffer, compareBuffer, bufferSize));
    printf("compareBuffer:");
    PrintUtf8String(compareBuffer);

    const size_t lowBufferSize = 4;
    char lowBuffer[lowBufferSize] = { 0 };
    ASSERT_LT(lowBufferSize, size);

    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_GetLogLineFunction(lowBuffer, lowBufferSize));
    EXPECT_EQ(0, strncmp(buffer, lowBuffer, lowBufferSize - 1)); // ヌル文字が入るため -1 している
    printf("lowBuffer:");
    PrintUtf8String(lowBuffer);

    g_LogReaderWrapper.m_StopLogStoringFunction();
}

TEST_F(Basic, MatchingLog)
{
    const char* validRegexString = R"(Hello world \d+)";
    const char* validUtf8RegexString = u8R"(こんにちは \d+)";
    const char* invalidRegexString = "xxxxxx";
    int waitTimeMs = 5000;

    ASSERT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_StartLogStoringFunction(g_SerialNumber, 1000));

    auto before = std::chrono::system_clock::now();
    EXPECT_EQ(LogReaderResult_Timeout, g_LogReaderWrapper.m_WaitUntilMatchedFunction(invalidRegexString, waitTimeMs));
    auto after = std::chrono::system_clock::now();
    auto diff = std::chrono::duration_cast<std::chrono::milliseconds>(after - before).count();
    EXPECT_LT(waitTimeMs - 1000, (int)diff);
    EXPECT_GT(waitTimeMs + 1000, (int)diff);

    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_WaitUntilMatchedFunction(validRegexString, 1000));
    printf("<matched String>\n");
    PrintCurrentString();
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_WaitUntilMatchedFunction(validUtf8RegexString, 1000));
    printf("<matched String>\n");
    PrintCurrentString();

    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_SearchFormerLogFunction(validRegexString));
    printf("<matched String>\n");
    PrintCurrentString();
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_SearchFormerLogFunction(validUtf8RegexString));
    printf("<matched String>\n");
    PrintCurrentString();

    EXPECT_EQ(LogReaderResult_NotFound, g_LogReaderWrapper.m_SearchFormerLogFunction(invalidRegexString));
    printf("<oldest String>\n");
    PrintCurrentString();

    g_LogReaderWrapper.m_StopLogStoringFunction();
}

TEST_F(Basic, MoveReference)
{
    const size_t bufferSize = 128;
    char buffer1[bufferSize] = { 0 };
    char buffer2[bufferSize] = { 0 };
    char buffer3[bufferSize] = { 0 };
    char buffer4[bufferSize] = { 0 };

    ASSERT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_StartLogStoringFunction(g_SerialNumber, 1000));

    // ログ取得を待機
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_WaitForNextLineFunction(1000));

    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_GetLogLineFunction(buffer1, bufferSize));

    ::Sleep(1000);
    g_LogReaderWrapper.m_MoveToNewestLineFunction();
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_GetLogLineFunction(buffer2, bufferSize));
    EXPECT_NE(0, strncmp(buffer1, buffer2, bufferSize));

    // MoveToFormerLineByTime で一定時間前のログが見つかるかを確認
    ::Sleep(100);
    g_LogReaderWrapper.m_MoveToNewestLineFunction();
    g_LogReaderWrapper.m_MoveToFormerLineByTimeFunction(1000);
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_WaitUntilMatchedFunction(buffer2, 1000));
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_GetLogLineFunction(buffer3, bufferSize));
    EXPECT_EQ(0, strncmp(buffer2, buffer3, bufferSize));

    // MoveToFormerLineByTime で最古のログまで戻れるかを確認
    g_LogReaderWrapper.m_MoveToFormerLineByTimeFunction(10000); // 十分大きな時間を指定
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_WaitUntilMatchedFunction(buffer1, 3000));
    g_LogReaderWrapper.m_MoveToFormerLineByTimeFunction(0);
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_GetLogLineFunction(buffer4, bufferSize));
    EXPECT_EQ(0, strncmp(buffer1, buffer4, bufferSize));

    printf("buffer1:");
    PrintUtf8String(buffer1);
    printf("buffer2:");
    PrintUtf8String(buffer2);
    printf("buffer3:");
    PrintUtf8String(buffer3);
    printf("buffer4:");
    PrintUtf8String(buffer4);

    g_LogReaderWrapper.m_StopLogStoringFunction();
}

TEST_F(Basic, LogFull)
{
    const size_t bufferSize = 128;
    char buffer1[bufferSize] = { 0 };
    char buffer2[bufferSize] = { 0 };

    ASSERT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_StartLogStoringFunction(g_SerialNumber, 100));
    ::Sleep(10000);
    g_LogReaderWrapper.m_MoveToNewestLineFunction();
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_GetLogLineFunction(buffer1, bufferSize));
    ::Sleep(1000);
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_GetLogLineFunction(buffer2, bufferSize));
    EXPECT_EQ(0, strncmp(buffer1, buffer2, bufferSize));
    printf("buffer2:");
    PrintUtf8String(buffer2);

    int waitTimeMs = 5000;
    const char* invalidRegexString = "xxxxxx";

    memset(buffer1, 0, bufferSize);
    memset(buffer2, 0, bufferSize);

    g_LogReaderWrapper.m_MoveToNewestLineFunction();
    auto before = std::chrono::system_clock::now();
    EXPECT_EQ(LogReaderResult_Timeout, g_LogReaderWrapper.m_WaitUntilMatchedFunction(invalidRegexString, waitTimeMs));
    auto after = std::chrono::system_clock::now();
    auto diff = std::chrono::duration_cast<std::chrono::milliseconds>(after - before).count();
    EXPECT_LT(waitTimeMs - 1000, (int)diff);
    EXPECT_GT(waitTimeMs + 1000, (int)diff);

    EXPECT_EQ(LogReaderResult_NotFound, g_LogReaderWrapper.m_SearchFormerLogFunction(invalidRegexString));
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_GetLogLineFunction(buffer1, bufferSize));
    ::Sleep(1000);
    EXPECT_EQ(LogReaderResult_Success, g_LogReaderWrapper.m_GetLogLineFunction(buffer2, bufferSize));
    EXPECT_NE(0, strncmp(buffer1, buffer2, bufferSize));
    printf("buffer2:");
    PrintUtf8String(buffer2);

    g_LogReaderWrapper.m_StopLogStoringFunction();
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    if (argc >= 2)
    {
        strncpy_s(g_SerialNumber.value, argv[1], sizeof(g_SerialNumber.value));
        printf("#######################################\n");
        printf("  Specified serial: %s\n", g_SerialNumber.value);
        printf("#######################################\n");
    }
    else
    {
        printf("Please specify serial number by argv[1]\n");
        nnt::Exit(1);
    }

    ::testing::InitGoogleTest(&argc, argv);

    const int exitCode = RUN_ALL_TESTS();

    nnt::Exit(exitCode);
}

