﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include "TestUtil.h"
#include "HidInputerWrapper.h"

using namespace hidInputer;


namespace
{
    // 乱数生成器
    std::random_device      g_RandomDevice;
    std::mt19937            g_Mersenne(g_RandomDevice());

    // center 値
    int g_CenterX;
    int g_CenterY;
    // 相対座標リスト
    float g_RelativePosX[g_MaxTouchNum];
    float g_RelativePosY[g_MaxTouchNum];
    // タッチ座標方向リスト
    float g_RelativePosDegree[g_MaxTouchNum];
    float g_Radius;
    // 移動方向
    float g_CenterMoveDegree;
    float g_CenterMoveLength;
    // 回転方向
    float g_StartDegree;
    float g_EndDegree;
    // 拡大
    float g_StartScale;
    float g_EndScale;
    // wait 値
    int g_WaitTouchBegin;
    int g_WaitTouchEnd;
    int g_WaitDuration;
    int g_WaitSpan;
    // 動作終了時のタッチ解除フラグ
    bool g_ReleaseAfter;
}

TEST(HidInputerSuite, SetTouchStateTest)
{
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));

    // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);

    const int touchNum = 3;
    int touchIdList[touchNum] = { 0, 1, 2 };
    int fingerIdList[touchNum] = { 100, 200, 300 };
    float xPosList[touchNum] = { 1000.0f, 2000.0f, 3000.0f };
    float yPosList[touchNum] = { 400.0f, 500.0f, 600.0f };

    // 正常な実行
    ASSERT_EQ(hidInputerWrapper.m_SetTouchStateFunction(touchIdList, fingerIdList, xPosList, yPosList, touchNum), TouchResult::TouchResult_Success);

    // 不正なタッチ ID での実行
    int invalidTouchIdList[touchNum] = { 10, 0, 1 };
    ASSERT_EQ(hidInputerWrapper.m_SetTouchStateFunction(invalidTouchIdList, fingerIdList, xPosList, yPosList, touchNum), TouchResult::TouchResult_InvalidTouchId);

    // 終了処理
    hidInputerWrapper.m_FinalizeFunction();
    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());
}

// マルチタッチ操作テスト
TEST( HidInputerSuite, MultiTouchTest )
{
    // ターゲットシリアル取得
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));

    // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);

    // タッチIDリスト
    int touchIdList[g_MaxTouchNum];
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        touchIdList[i] = i;
    }

    // 不正なタッチIDを含むリスト
    int invalidTouchIdList[g_MaxTouchNum];
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        invalidTouchIdList[i] = i;
    }
    invalidTouchIdList[g_Mersenne() % g_MaxTouchNum] = CreateInvalidTouchId();

    // タッチ座標リスト
    float touchPosX[g_MaxTouchNum];
    float touchPosY[g_MaxTouchNum];
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        touchPosX[i] = static_cast<float>(g_Mersenne() % g_ScreenWidth);
        touchPosY[i] = static_cast<float>(g_Mersenne() % g_ScreenHeight);
    }
    // 画面外のタッチ座標リスト
    float invalidTouchPosX[g_MaxTouchNum];
    float invalidTouchPosY[g_MaxTouchNum];
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        invalidTouchPosX[i] = static_cast<float>(((g_Mersenne() % g_ScreenWidth) + g_ScreenWidth) * CreateRandomSign());
        invalidTouchPosY[i] = static_cast<float>(((g_Mersenne() % g_ScreenHeight) + g_ScreenHeight) * CreateRandomSign());
    }

    //---------------------------------
    // 全タッチIDでタップ
    //---------------------------------
    EXPECT_EQ(hidInputerWrapper.m_MultiTapFunction(touchIdList, g_MaxTouchNum, touchPosX, touchPosY, g_DefaultWaitPress), TouchResult::TouchResult_Success);

    //---------------------------------
    // 全タッチIDで画面外をタップ
    //---------------------------------
    EXPECT_EQ(hidInputerWrapper.m_MultiTapFunction(touchIdList, g_MaxTouchNum, invalidTouchPosX, invalidTouchPosY, g_DefaultWaitPress), TouchResult::TouchResult_Success);

    //---------------------------------
    // 不正なタッチ ID でタップ
    //---------------------------------
    EXPECT_EQ(hidInputerWrapper.m_MultiTapFunction(invalidTouchIdList, g_MaxTouchNum, touchPosX, touchPosY, g_DefaultWaitPress), TouchResult::TouchResult_InvalidTouchId);


    // 終了処理
    hidInputerWrapper.m_FinalizeFunction();
    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());
}

TEST(HidInputerSuite, MultiTouchAnimatorTest)
{
    // ターゲットシリアル取得
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));

    // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);

    // タッチIDリスト
    int touchIdList[g_MaxTouchNum];
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        touchIdList[i] = i;
    }
    // 不正なタッチIDを含むリスト
    int invalidTouchIdList[g_MaxTouchNum];
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        invalidTouchIdList[i] = i;
    }
    invalidTouchIdList[g_Mersenne() % g_MaxTouchNum] = CreateInvalidTouchId();

    //---------------------------------
    // 正常呼び出しテスト
    //---------------------------------
    for (int i = 0; i < g_DefaultTryTimes; i++)
    {
        // ランダムパラメーター生成
        GenerateRandomMultitouchParameter(
            g_RelativePosX, g_RelativePosY,
            g_RelativePosDegree, &g_Radius,
            &g_CenterX, &g_CenterY,
            &g_CenterMoveDegree, &g_CenterMoveLength,
            &g_StartDegree, &g_EndDegree,
            &g_StartScale, &g_EndScale,
            &g_WaitDuration, &g_WaitSpan,
            &g_WaitTouchBegin, &g_WaitTouchEnd,
            &g_ReleaseAfter, g_MaxTouchNum, false);

        // MultiTouchAnimator
        EXPECT_EQ(hidInputerWrapper.m_MultiTouchAnimatorFunction(touchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_CenterMoveDegree, g_CenterMoveLength,
            g_RelativePosX, g_RelativePosY,
            g_StartDegree, g_EndDegree,
            g_StartScale, g_EndScale,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd, g_ReleaseAfter), TouchResult::TouchResult_Success);

    }
    //---------------------------------
    // 不正な ID 指定による呼び出し
    //---------------------------------
    {
        // ランダムパラメーター生成
        GenerateRandomMultitouchParameter(
            g_RelativePosX, g_RelativePosY,
            g_RelativePosDegree, &g_Radius,
            &g_CenterX, &g_CenterY,
            &g_CenterMoveDegree, &g_CenterMoveLength,
            &g_StartDegree, &g_EndDegree,
            &g_StartScale, &g_EndScale,
            &g_WaitDuration, &g_WaitSpan,
            &g_WaitTouchBegin, &g_WaitTouchEnd,
            &g_ReleaseAfter, g_MaxTouchNum, false);

        // MultiTouchAnimator
        EXPECT_EQ(hidInputerWrapper.m_MultiTouchAnimatorFunction(
            invalidTouchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_CenterMoveDegree, g_CenterMoveLength,
            g_RelativePosX, g_RelativePosY,
            g_StartDegree, g_EndDegree,
            g_StartScale, g_EndScale,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd, g_ReleaseAfter), TouchResult::TouchResult_InvalidTouchId);
    }
    //---------------------------------
    // 画面外での操作
    //---------------------------------
    {
        // ランダムパラメーター生成
        GenerateRandomMultitouchParameter(
            g_RelativePosX, g_RelativePosY,
            g_RelativePosDegree, &g_Radius,
            &g_CenterX, &g_CenterY,
            &g_CenterMoveDegree, &g_CenterMoveLength,
            &g_StartDegree, &g_EndDegree,
            &g_StartScale, &g_EndScale,
            &g_WaitDuration, &g_WaitSpan,
            &g_WaitTouchBegin, &g_WaitTouchEnd,
            &g_ReleaseAfter, g_MaxTouchNum, false);

        // MultiTouchAnimator
        EXPECT_EQ(hidInputerWrapper.m_MultiTouchAnimatorFunction(touchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_CenterMoveDegree, g_CenterMoveLength,
            g_RelativePosX, g_RelativePosY,
            g_StartDegree, g_EndDegree,
            g_StartScale, g_EndScale,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd, g_ReleaseAfter), TouchResult::TouchResult_Success);
    }

    // 終了処理
    hidInputerWrapper.m_FinalizeFunction();
    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());
}


// マルチタッチドラッグテスト
TEST(HidInputerSuite, MultiTouchDragTest)
{
    // ターゲットシリアル取得
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));

    // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;
    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());
    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);
    // タッチIDリスト
    int touchIdList[g_MaxTouchNum];
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        touchIdList[i] = i;
    }
    // 不正なタッチIDを含むリスト
    int invalidTouchIdList[g_MaxTouchNum];
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        invalidTouchIdList[i] = i;
    }
    invalidTouchIdList[g_Mersenne() % g_MaxTouchNum] = CreateInvalidTouchId();



    //---------------------------------
    // 正常呼び出しテスト
    //---------------------------------
    for (int i = 0; i < g_DefaultTryTimes; i++)
    {
        // ランダムパラメーター生成
        GenerateRandomMultitouchParameter(
            g_RelativePosX, g_RelativePosY,
            g_RelativePosDegree, &g_Radius,
            &g_CenterX, &g_CenterY,
            &g_CenterMoveDegree, &g_CenterMoveLength,
            &g_StartDegree, &g_EndDegree,
            &g_StartScale, &g_EndScale,
            &g_WaitDuration, &g_WaitSpan,
            &g_WaitTouchBegin, &g_WaitTouchEnd,
            &g_ReleaseAfter, g_MaxTouchNum, false);
        // MultiDragPos
        int endCenterX = g_Mersenne() % g_ScreenWidth;
        int endCenterY = g_Mersenne() % g_ScreenHeight;
        EXPECT_EQ(hidInputerWrapper.m_MultiDragPosFunction(
            touchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            endCenterX, endCenterY,
            g_RelativePosX, g_RelativePosY,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd, g_ReleaseAfter), TouchResult::TouchResult_Success);

        // MultiDragAngle
        EXPECT_EQ(hidInputerWrapper.m_MultiDragAngleFunction(
            touchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_CenterMoveDegree, g_CenterMoveLength,
            g_RelativePosX, g_RelativePosY,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd, g_ReleaseAfter), TouchResult::TouchResult_Success);
    }
    //---------------------------------
    // 不正な ID 指定による呼び出し
    //----------------------------------
    {
        // ランダムパラメーター生成
        GenerateRandomMultitouchParameter(
            g_RelativePosX, g_RelativePosY,
            g_RelativePosDegree, &g_Radius,
            &g_CenterX, &g_CenterY,
            &g_CenterMoveDegree, &g_CenterMoveLength,
            &g_StartDegree, &g_EndDegree,
            &g_StartScale, &g_EndScale,
            &g_WaitDuration, &g_WaitSpan,
            &g_WaitTouchBegin, &g_WaitTouchEnd,
            &g_ReleaseAfter, g_MaxTouchNum, false);
        // MultiDragPos
        int endCenterX = g_Mersenne() % g_ScreenWidth;
        int endCenterY = g_Mersenne() % g_ScreenHeight;
        EXPECT_EQ( hidInputerWrapper.m_MultiDragPosFunction(
            invalidTouchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            endCenterX, endCenterY,
            g_RelativePosX, g_RelativePosY,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd, g_ReleaseAfter), TouchResult::TouchResult_InvalidTouchId);


        // MultiDragAngle
        EXPECT_EQ(hidInputerWrapper.m_MultiDragAngleFunction(
            invalidTouchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_CenterMoveDegree, g_CenterMoveLength,
            g_RelativePosX, g_RelativePosY,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd, g_ReleaseAfter), TouchResult::TouchResult_InvalidTouchId);

    }

    //---------------------------------
    // 画面外での操作
    //---------------------------------
    {
        GenerateRandomMultitouchParameter(
            g_RelativePosX, g_RelativePosY,
            g_RelativePosDegree, &g_Radius,
            &g_CenterX, &g_CenterY,
            &g_CenterMoveDegree, &g_CenterMoveLength,
            &g_StartDegree, &g_EndDegree,
            &g_StartScale, &g_EndScale,
            &g_WaitDuration, &g_WaitSpan,
            &g_WaitTouchBegin, &g_WaitTouchEnd,
            &g_ReleaseAfter, g_MaxTouchNum, true);
        // MultiDragPos
        int endCenterX = g_Mersenne() % g_ScreenWidth;
        int endCenterY = g_Mersenne() % g_ScreenHeight;
        EXPECT_EQ( hidInputerWrapper.m_MultiDragPosFunction(
            touchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            endCenterX, endCenterY,
            g_RelativePosX, g_RelativePosY,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd, g_ReleaseAfter), TouchResult::TouchResult_Success);


        // MultiDragAngle
        EXPECT_EQ(hidInputerWrapper.m_MultiDragAngleFunction(
            touchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_CenterMoveDegree, g_CenterMoveLength,
            g_RelativePosX, g_RelativePosY,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd, g_ReleaseAfter), TouchResult::TouchResult_Success);
    }
    // 終了処理
    hidInputerWrapper.m_FinalizeFunction();
    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());
}

// マルチタッチピンチテスト
TEST(HidInputerSuite, MultiTouchPinchTest)
{
    // ターゲットシリアル取得
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));

    // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);

    // タッチIDリスト
    int touchIdList[g_MaxTouchNum];
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        touchIdList[i] = i;
    }
    // 不正なタッチIDを含むリスト
    int invalidTouchIdList[g_MaxTouchNum];
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        invalidTouchIdList[i] = i;
    }
    invalidTouchIdList[g_Mersenne() % g_MaxTouchNum] = CreateInvalidTouchId();

    //---------------------------------
    // 正常呼び出しテスト
    //---------------------------------
    for (int i = 0; i < g_DefaultTryTimes; i++)
    {
        // ランダムパラメーター生成
        GenerateRandomMultitouchParameter(
            g_RelativePosX, g_RelativePosY,
            g_RelativePosDegree, &g_Radius,
            &g_CenterX, &g_CenterY,
            &g_CenterMoveDegree, &g_CenterMoveLength,
            &g_StartDegree, &g_EndDegree,
            &g_StartScale, &g_EndScale,
            &g_WaitDuration, &g_WaitSpan,
            &g_WaitTouchBegin, &g_WaitTouchEnd,
            &g_ReleaseAfter, g_MaxTouchNum, false);
        // Pinch
        EXPECT_EQ(hidInputerWrapper.m_PinchFunction(
            touchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_RelativePosDegree,
            g_StartScale, g_EndScale,
            g_WaitDuration, g_WaitSpan,
            g_WaitTouchBegin, g_WaitTouchEnd,
            g_ReleaseAfter), TouchResult::TouchResult_Success);


        // CenterTouchPinch
        EXPECT_EQ(hidInputerWrapper.m_CenterTouchPinchFunction(
            touchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_RelativePosDegree,
            g_StartScale, g_EndScale,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd, g_ReleaseAfter), TouchResult::TouchResult_Success);

    }
    //---------------------------------
    // 不正な ID 指定による呼び出し
    //----------------------------------
    {
        GenerateRandomMultitouchParameter(
            g_RelativePosX, g_RelativePosY,
            g_RelativePosDegree, &g_Radius,
            &g_CenterX, &g_CenterY,
            &g_CenterMoveDegree, &g_CenterMoveLength,
            &g_StartDegree, &g_EndDegree,
            &g_StartScale, &g_EndScale,
            &g_WaitDuration, &g_WaitSpan,
            &g_WaitTouchBegin, &g_WaitTouchEnd,
            &g_ReleaseAfter, g_MaxTouchNum, false);

        // Pinch
        EXPECT_EQ(hidInputerWrapper.m_PinchFunction(
            invalidTouchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_RelativePosDegree,
            g_StartScale, g_EndScale,
            g_WaitDuration, g_WaitSpan,
            g_WaitTouchBegin, g_WaitTouchEnd,
            g_ReleaseAfter), TouchResult::TouchResult_InvalidTouchId);


        // CenterTouchPinch
        EXPECT_EQ(hidInputerWrapper.m_CenterTouchPinchFunction(
            invalidTouchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_RelativePosDegree,
            g_StartScale, g_EndScale,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd, g_ReleaseAfter), TouchResult::TouchResult_InvalidTouchId);
    }

    //---------------------------------
    // 画面外での操作
    //---------------------------------
    {
        GenerateRandomMultitouchParameter(
            g_RelativePosX, g_RelativePosY,
            g_RelativePosDegree, &g_Radius,
            &g_CenterX, &g_CenterY,
            &g_CenterMoveDegree, &g_CenterMoveLength,
            &g_StartDegree, &g_EndDegree,
            &g_StartScale, &g_EndScale,
            &g_WaitDuration, &g_WaitSpan,
            &g_WaitTouchBegin, &g_WaitTouchEnd,
            &g_ReleaseAfter, g_MaxTouchNum, true);

        // Pinch
        EXPECT_EQ(hidInputerWrapper.m_PinchFunction(
            touchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_RelativePosDegree,
            g_StartScale, g_EndScale,
            g_WaitDuration, g_WaitSpan,
            g_WaitTouchBegin, g_WaitTouchEnd,
            g_ReleaseAfter), TouchResult::TouchResult_Success);


        // CenterTouchPinch
        EXPECT_EQ(hidInputerWrapper.m_CenterTouchPinchFunction(
            touchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_RelativePosDegree,
            g_StartScale, g_EndScale,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd, g_ReleaseAfter), TouchResult::TouchResult_Success);
    }

    // 終了処理
    hidInputerWrapper.m_FinalizeFunction();
    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());
}

// マルチタッチ回転テスト
TEST( HidInputerSuite, MultiTouchRotateTest )
{
    // ターゲットシリアル取得
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));

    // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);

    // タッチIDリスト
    int touchIdList[g_MaxTouchNum];
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        touchIdList[i] = i;
    }
    // 不正なタッチIDを含むリスト
    int invalidTouchIdList[g_MaxTouchNum];
    for (int i = 0; i < g_MaxTouchNum; i++)
    {
        invalidTouchIdList[i] = i;
    }
    invalidTouchIdList[g_Mersenne() % g_MaxTouchNum] = CreateInvalidTouchId();



    //---------------------------------
    // 正常呼び出しテスト
    //---------------------------------
    for (int i = 0; i < g_DefaultTryTimes; i++)
    {
        GenerateRandomMultitouchParameter(
            g_RelativePosX, g_RelativePosY,
            g_RelativePosDegree, &g_Radius,
            &g_CenterX, &g_CenterY,
            &g_CenterMoveDegree, &g_CenterMoveLength,
            &g_StartDegree, &g_EndDegree,
            &g_StartScale, &g_EndScale,
            &g_WaitDuration, &g_WaitSpan,
            &g_WaitTouchBegin, &g_WaitTouchEnd,
            &g_ReleaseAfter, g_MaxTouchNum, false);
        // MultiRotate
        EXPECT_EQ(hidInputerWrapper.m_MultiRotateFunction(
            touchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_RelativePosDegree, g_Radius,
            g_StartDegree, g_EndDegree,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd,
            g_ReleaseAfter ), TouchResult::TouchResult_Success);


        // CenterTouchMultiRotate
        EXPECT_EQ(hidInputerWrapper.m_CenterTouchMultiRotateFunction(
            touchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_RelativePosDegree, g_Radius,
            g_StartDegree, g_EndDegree,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd,
            g_ReleaseAfter), TouchResult::TouchResult_Success);

    }
    //---------------------------------
    // 不正な ID 指定による呼び出し
    //----------------------------------
    {

        GenerateRandomMultitouchParameter(
            g_RelativePosX, g_RelativePosY,
            g_RelativePosDegree, &g_Radius,
            &g_CenterX, &g_CenterY,
            &g_CenterMoveDegree, &g_CenterMoveLength,
            &g_StartDegree, &g_EndDegree,
            &g_StartScale, &g_EndScale,
            &g_WaitDuration, &g_WaitSpan,
            &g_WaitTouchBegin, &g_WaitTouchEnd,
            &g_ReleaseAfter, g_MaxTouchNum, false);

        // MultiRotate
        EXPECT_EQ(hidInputerWrapper.m_MultiRotateFunction(
            invalidTouchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_RelativePosDegree, g_Radius,
            g_StartDegree, g_EndDegree,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd,
            g_ReleaseAfter), TouchResult::TouchResult_InvalidTouchId);


        // CenterTouchMultiRotate
        EXPECT_EQ(hidInputerWrapper.m_CenterTouchMultiRotateFunction(
            invalidTouchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_RelativePosDegree, g_Radius,
            g_StartDegree, g_EndDegree,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd,
            g_ReleaseAfter), TouchResult::TouchResult_InvalidTouchId);
    }

    //---------------------------------
    // 画面外での操作
    //---------------------------------
    {
        GenerateRandomMultitouchParameter(
            g_RelativePosX, g_RelativePosY,
            g_RelativePosDegree, &g_Radius,
            &g_CenterX, &g_CenterY,
            &g_CenterMoveDegree, &g_CenterMoveLength,
            &g_StartDegree, &g_EndDegree,
            &g_StartScale, &g_EndScale,
            &g_WaitDuration, &g_WaitSpan,
            &g_WaitTouchBegin, &g_WaitTouchEnd,
            &g_ReleaseAfter, g_MaxTouchNum, true);

        // MultiRotate
        EXPECT_EQ(hidInputerWrapper.m_MultiRotateFunction(
            touchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_RelativePosDegree, g_Radius,
            g_StartDegree, g_EndDegree,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd,
            g_ReleaseAfter
        ), TouchResult::TouchResult_Success);


        // CenterTouchMultiRotate
        EXPECT_EQ(hidInputerWrapper.m_CenterTouchMultiRotateFunction(
            touchIdList, g_MaxTouchNum,
            g_CenterX, g_CenterY,
            g_RelativePosDegree, g_Radius,
            g_StartDegree, g_EndDegree,
            g_WaitDuration, g_WaitSpan, g_WaitTouchBegin, g_WaitTouchEnd,
            g_ReleaseAfter), TouchResult::TouchResult_Success);

    }

    // 終了処理
    hidInputerWrapper.m_FinalizeFunction();
    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());
}

