﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include "TestUtil.h"
#include "HidInputerWrapper.h"

using namespace hidInputer;

// コントローラー接続
TEST(HidInputerSuite, ConnectControllerTest)
{
    // ターゲットシリアル取得
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));

    // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);

    int connectedControllerIdCount;
    int acquiredControllerIdCount;
    int controllerIds[g_ControllerIndexCount];

    //-------------------------------------
    // コントローラー接続数取得(未接続)
    //-------------------------------------
    EXPECT_EQ(ControllerResult::ControllerResult_Success,
        hidInputerWrapper.m_GetConnectedControllerCountFunction(&connectedControllerIdCount));
    EXPECT_EQ(0, connectedControllerIdCount);

    EXPECT_EQ(ControllerResult::ControllerResult_Success,
        hidInputerWrapper.m_GetConnectedControllerIdsFunction(&acquiredControllerIdCount,
            controllerIds, g_ControllerIndexCount));
    EXPECT_EQ(0, acquiredControllerIdCount);

    EXPECT_EQ(ControllerResult::ControllerResult_Success,
        hidInputerWrapper.m_GetConnectedDebugControllerCountFunction(&connectedControllerIdCount));
    EXPECT_EQ(0, connectedControllerIdCount);

    EXPECT_EQ(ControllerResult::ControllerResult_Success,
        hidInputerWrapper.m_GetConnectedDebugControllerIdsFunction(&acquiredControllerIdCount,
            controllerIds, g_ControllerIndexCount));
    EXPECT_EQ(0, acquiredControllerIdCount);


    //-------------------------------------
    // 未接続の全コントローラーへ接続します。
    //-------------------------------------
    for (int i = 0; i < g_ControllerIndexCount; i++)
    {
        EXPECT_EQ(hidInputerWrapper.m_AddControllerForIdFunction(i), ControllerResult::ControllerResult_Success);
    }

    //-------------------------------------
    // 未接続のデバッグコントローラーへ接続します。
    //-------------------------------------
    int debugControllerId;
    for( int i = 0; i < g_DebugControllerCount; i++ ){
        EXPECT_EQ(hidInputerWrapper.m_AddDebugControllerFunction(&debugControllerId), ControllerResult::ControllerResult_Success);
    }

    //-------------------------------------
    // コントローラー接続数取得(接続済み)
    //-------------------------------------
    EXPECT_EQ(ControllerResult::ControllerResult_Success,
        hidInputerWrapper.m_GetConnectedControllerCountFunction(&connectedControllerIdCount));
    EXPECT_EQ(g_ControllerIndexCount, connectedControllerIdCount);

    EXPECT_EQ(ControllerResult::ControllerResult_Success,
        hidInputerWrapper.m_GetConnectedControllerIdsFunction(&acquiredControllerIdCount, controllerIds, g_ControllerIndexCount / 2));
    EXPECT_EQ(g_ControllerIndexCount / 2, acquiredControllerIdCount);

    EXPECT_EQ(ControllerResult::ControllerResult_Success,
        hidInputerWrapper.m_GetConnectedControllerIdsFunction(&acquiredControllerIdCount, controllerIds, g_ControllerIndexCount));
    EXPECT_EQ(g_ControllerIndexCount, acquiredControllerIdCount);

    EXPECT_EQ(ControllerResult::ControllerResult_Success,
        hidInputerWrapper.m_GetConnectedDebugControllerCountFunction(&connectedControllerIdCount));
    EXPECT_EQ(g_DebugControllerCount, connectedControllerIdCount);

    EXPECT_EQ(ControllerResult::ControllerResult_Success,
        hidInputerWrapper.m_GetConnectedDebugControllerIdsFunction(&acquiredControllerIdCount,
            controllerIds, g_ControllerIndexCount));
    EXPECT_EQ(connectedControllerIdCount, acquiredControllerIdCount);

    //-------------------------------------
    // 接続済みのコントローラへ接続します。
    //-------------------------------------
    for (int i = 0; i < g_ControllerIndexCount; i++)
    {
        EXPECT_EQ(hidInputerWrapper.m_AddControllerForIdFunction(i), ControllerResult::ControllerResult_ControllerCountOver);
    }
    for (int i = 0; i < g_ControllerIndexCount; i++)
    {
        int controllerId = 0;
        EXPECT_EQ(hidInputerWrapper.m_AddControllerFunction(&controllerId), ControllerResult::ControllerResult_ControllerCountOver);
    }

    //-------------------------------------------------
    // 接続済みのデバッグコントローラーへ接続します。
    //-------------------------------------------------
    for( int i = 0; i < g_DebugControllerCount; i++ ){
        EXPECT_EQ(hidInputerWrapper.m_AddDebugControllerFunction(&debugControllerId), ControllerResult::ControllerResult_ConnectControllerFailed);
    }



    //-------------------------------------
    // 不正なコントローラIDへ接続を行います。
    //-------------------------------------
    ReleaseAllController( &hidInputerWrapper );
    for (int i = 0; i < g_DefaultTryTimes; i++)
    {
        int id = CreateInvalidControllerId();
        EXPECT_EQ(hidInputerWrapper.m_AddControllerForIdFunction(id), ControllerResult::ControllerResult_InvalidControllerId);
    }

    // 終了処理
    ReleaseAllController( &hidInputerWrapper );
    hidInputerWrapper.m_FinalizeFunction();

    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());
}

TEST(HidInputerSuite, ReleaseControllerTest)
{
    // ターゲットシリアル取得
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));

    // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);

    //-------------------------------------
    // 事前にすべてのコントローラーを接続しておく
    //-------------------------------------
    ConnectAllController(&hidInputerWrapper );


    //-------------------------------------
    // 接続済みのコントローラーID の解除
    //-------------------------------------
    for (int i = 0; i < g_AllControllerIndexRange; i++)
    {
        EXPECT_EQ(hidInputerWrapper.m_ReleaseControllerFunction(i), ControllerResult::ControllerResult_Success);
    }


    //-------------------------------------
    // 不正なコントローラー ID の解除
    //-------------------------------------
    for (int i = 0; i < g_DefaultTryTimes; i++)
    {
        // 不正な正負のランダムIDを交互に作成
        int id = CreateInvalidControllerId();
        EXPECT_EQ(hidInputerWrapper.m_ReleaseControllerFunction(id), ControllerResult::ControllerResult_InvalidControllerId);
    }


    //-------------------------------------
    // 解除済みのコントローラーID の解除
    //-------------------------------------
    for (int i = 0; i < g_AllControllerIndexRange; i++)
    {
        EXPECT_EQ(hidInputerWrapper.m_ReleaseControllerFunction(i), ControllerResult::ControllerResult_InvalidControllerId);
    }

    // 終了処理
    hidInputerWrapper.m_FinalizeFunction();

    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());

}


TEST(HidInputerSuite, SetControllerStateTest)
{
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));

    // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);

    const int controllerId = 3;
    const int buttonNum = 5;
    Button buttonList[buttonNum] = { Button::Button_Right, Button::Button_A, Button::Button_Up, Button::Button_Minus, Button::Button_X };

    int targetStickCount = 2;
    Stick stickList[] = { Stick::Stick_L, Stick::Stick_R };
    Stick invalidStickList[] = { static_cast<Stick>(CreateInvalidStick()), Stick::Stick_R };
    float stickDegreeList[] = { 90.0f, -90.0f };
    float stickPowerList[] = { 1.0, 1.0 };




    // 未接続のコントローラーに対して実行
    EXPECT_EQ(hidInputerWrapper.m_SetControllerStateFunction(controllerId, buttonList, buttonNum, stickList, stickDegreeList, stickPowerList, targetStickCount), ControllerResult::ControllerResult_InvalidControllerId);

    // 正常な実行
    EXPECT_EQ(hidInputerWrapper.m_AddControllerForIdFunction(controllerId), ControllerResult::ControllerResult_Success);
    EXPECT_EQ(hidInputerWrapper.m_SetControllerStateFunction(controllerId, buttonList, buttonNum, stickList, stickDegreeList, stickPowerList, targetStickCount), ControllerResult::ControllerResult_Success);

    // 不正なコントローラーID での実行
    EXPECT_EQ(hidInputerWrapper.m_SetControllerStateFunction(CreateInvalidControllerId(), buttonList, buttonNum, stickList, stickDegreeList, stickPowerList, targetStickCount), ControllerResult::ControllerResult_InvalidControllerId);

    // 不正なスティックを指定
    EXPECT_EQ(hidInputerWrapper.m_SetControllerStateFunction(controllerId, buttonList, buttonNum, invalidStickList, stickDegreeList, stickPowerList, targetStickCount), ControllerResult::ControllerResult_InvalidStick);

    // 不正なボタンの指定
    Button invalidButtonList[buttonNum] = { static_cast<Button>(CreateInvalidButton()), Button::Button_A, Button::Button_Up, Button::Button_Minus, Button::Button_X };
    EXPECT_EQ(hidInputerWrapper.m_SetControllerStateFunction(controllerId, invalidButtonList, buttonNum, stickList, stickDegreeList, stickPowerList, targetStickCount), ControllerResult::ControllerResult_InvalidButton);

    // 終了処理
    hidInputerWrapper.m_FinalizeFunction();
    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());
}
