﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include "TestUtil.h"
#include "HidInputerWrapper.h"

using namespace hidInputer;

// シングルボタン操作テスト
TEST( HidInputerSuite, SingleButtonControlTest )
{
    // ターゲットシリアル取得
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));

    // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);


    //-------------------------------------
    // 事前にすべてのコントローラーを接続しておく
    //-------------------------------------
    ConnectAllController(&hidInputerWrapper );

    //-------------------------------------
    // 接続済みのコントローラーで全てのボタン
    //-------------------------------------
    for (int controller = 0; controller < g_AllControllerIndexRange; controller++)
    {
        // ボタンを押す
        for (int button = g_XpadButtonBeginNum; button < g_XpadButtonEndNum; button++)
        {
            EXPECT_EQ(hidInputerWrapper.m_PressButtonFunction(controller, static_cast<Button>(button)), ControllerResult::ControllerResult_Success);
        }

        // ボタンを離す
        for (int button = g_XpadButtonBeginNum; button < g_XpadButtonEndNum; button++)
        {
            EXPECT_EQ(hidInputerWrapper.m_ReleaseButtonFunction(controller, static_cast<Button>(button)), ControllerResult::ControllerResult_Success);
        }

        // ボタンをプッシュ
        for (int button = g_XpadButtonBeginNum; button < g_XpadButtonEndNum; button++)
        {
            EXPECT_EQ(hidInputerWrapper.m_PushButtonFunction(controller, static_cast<Button>(button), g_DefaultWaitPress), ControllerResult::ControllerResult_Success);
        }

        // 未入力状態セット
        EXPECT_EQ(hidInputerWrapper.m_ResetControllerStateFunction(controller), ControllerResult::ControllerResult_Success);
    }

    //-------------------------------------
    // 不正なコントローラー
    //-------------------------------------
    for (int i = 0; i < g_DefaultTryTimes; i++)
    {
        // ボタンを押す
        int id = CreateInvalidControllerId();
        for (int button = g_XpadButtonBeginNum; button < g_XpadButtonEndNum; button++)
        {
            EXPECT_EQ(hidInputerWrapper.m_PressButtonFunction(id, static_cast<Button>(button)), ControllerResult::ControllerResult_InvalidControllerId);
        }

        // ボタンを離す
        for (int button = g_XpadButtonBeginNum; button < g_XpadButtonEndNum; button++)
        {
            EXPECT_EQ(hidInputerWrapper.m_ReleaseButtonFunction(id, static_cast<Button>(button)), ControllerResult::ControllerResult_InvalidControllerId);
        }

        // ボタンをプッシュする
        for (int button = g_XpadButtonBeginNum; button < g_XpadButtonEndNum; button++)
        {
            EXPECT_EQ(hidInputerWrapper.m_PushButtonFunction(id, static_cast<Button>(button), g_DefaultWaitPress), ControllerResult::ControllerResult_InvalidControllerId);
        }

        // 未入力状態セット
        EXPECT_EQ(hidInputerWrapper.m_ResetControllerStateFunction(id), ControllerResult::ControllerResult_InvalidControllerId);
    }

    //-------------------------------------
    // 不正なボタン
    //-------------------------------------
    for (int controller = 0; controller < g_AllControllerIndexRange; controller++)
    {
        // ボタンを押す
        for (int i = 0; i < g_DefaultTryTimes; i++ )
        {
            int button = CreateInvalidButton();

            EXPECT_EQ(hidInputerWrapper.m_PressButtonFunction(controller, static_cast<Button>(button)), ControllerResult::ControllerResult_InvalidButton);
        }

        // ボタンを離す
        for (int i = 0; i < g_DefaultTryTimes; i++ )
        {
            int button = CreateInvalidButton();

            EXPECT_EQ(hidInputerWrapper.m_ReleaseButtonFunction(controller, static_cast<Button>(button)), ControllerResult::ControllerResult_InvalidButton);
        }

        // ボタンをプッシュする
        for (int i = 0; i < g_DefaultTryTimes; i++ )
        {
            int button = CreateInvalidButton();

            EXPECT_EQ(hidInputerWrapper.m_PushButtonFunction(controller, static_cast<Button>(button), g_DefaultWaitPress), ControllerResult::ControllerResult_InvalidButton);
        }
    }

    //-------------------------------------
    // 接続済みのコントローラーID の解除
    //-------------------------------------
    ReleaseAllController(&hidInputerWrapper );

    //-------------------------------------
    // 未接続のコントローラーで全てのボタン
    //-------------------------------------
    for (int controller = 0; controller < g_AllControllerIndexRange; controller++)
    {
        // ボタンを押す
        for (int button = g_XpadButtonBeginNum; button < g_XpadButtonEndNum; button++)
        {
            EXPECT_EQ(hidInputerWrapper.m_PressButtonFunction(controller, static_cast<Button>(button)), ControllerResult::ControllerResult_InvalidControllerId);
        }


        // ボタンを離す
        for (int button = g_XpadButtonBeginNum; button < g_XpadButtonEndNum; button++)
        {
            EXPECT_EQ(hidInputerWrapper.m_ReleaseButtonFunction(controller, static_cast<Button>(button)), ControllerResult::ControllerResult_InvalidControllerId);
        }


        // ボタンをプッシュする
        for (int button = g_XpadButtonBeginNum; button < g_XpadButtonEndNum; button++)
        {
            EXPECT_EQ(hidInputerWrapper.m_PushButtonFunction(controller, static_cast<Button>(button), g_DefaultWaitPress), ControllerResult::ControllerResult_InvalidControllerId);
        }


        // 未入力状態セット
        EXPECT_EQ(hidInputerWrapper.m_ResetControllerStateFunction(controller), ControllerResult::ControllerResult_InvalidControllerId);
    }

    // 接続済みのコントローラーID の解除
    ReleaseAllController(&hidInputerWrapper );

    // 終了処理
    hidInputerWrapper.m_FinalizeFunction();
    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());
}

// 複数ボタン操作テスト
TEST( HidInputerSuite, MultiButtonControlTest )
{
    // ターゲットシリアル取得
    SerialNumberString serialNumber;
    ASSERT_TRUE(GetTargetSerial(&serialNumber));


    // Hid ライブラリのラッパークラス
    HidInputerWrapper       hidInputerWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadHidLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(hidInputerWrapper.LoadFunctiontions());
    hidInputerWrapper.m_InitializeFunction(serialNumber);

    //-------------------------------------
    // 事前にすべてのコントローラーを接続しておく
    //-------------------------------------
    ConnectAllController(&hidInputerWrapper );


    // 全ボタンリスト
    Button allButtonList[] =
    {
        Button_Home,
        Button_Capture,
        Button_A,
        Button_B,
        Button_X,
        Button_Y,
        Button_StickL,
        Button_StickR,
        Button_L,
        Button_R,
        Button_ZL,
        Button_ZR,
        Button_Plus,
        Button_Minus,
        Button_Left,
        Button_Up,
        Button_Right,
        Button_Down,
    };
    // 不正なボタンリスト
    Button invalidButtonList[] =
    {
        Button_A,
        Button_B,
        static_cast<Button>(CreateInvalidButton()),
        Button_X,
        Button_Y,
    };


    //-------------------------------------
    // 接続済みの全コントローラーで、全ボタン
    //-------------------------------------
    for (int i = 0; i < g_AllControllerIndexRange; i++)
    {
        EXPECT_EQ(hidInputerWrapper.m_PressMultiButtonFunction(i, allButtonList, sizeof(allButtonList) / sizeof(Button)), ControllerResult::ControllerResult_Success);

        EXPECT_EQ(hidInputerWrapper.m_ReleaseMultiButtonFunction(i, allButtonList, sizeof(allButtonList) / sizeof(Button)), ControllerResult::ControllerResult_Success);

        EXPECT_EQ(hidInputerWrapper.m_PushMultiButtonFunction(i, allButtonList, sizeof(allButtonList) / sizeof(Button), g_DefaultWaitPress), ControllerResult::ControllerResult_Success);

        EXPECT_EQ(hidInputerWrapper.m_PushMultiSeqFunction(i, allButtonList, sizeof(allButtonList) / sizeof(Button), g_DefaultWaitPress, g_DefaultWaitSpan), ControllerResult::ControllerResult_Success);
    }

    //-------------------------------------
    // 不正なコントローラー ID に対して操作
    //-------------------------------------
    for (int i = 0; i < g_DefaultTryTimes; i++)
    {
        int id = CreateInvalidControllerId();
        EXPECT_EQ(hidInputerWrapper.m_PressMultiButtonFunction(id, allButtonList, sizeof(allButtonList) / sizeof(Button)), ControllerResult::ControllerResult_InvalidControllerId);

        EXPECT_EQ(hidInputerWrapper.m_ReleaseMultiButtonFunction(id, allButtonList, sizeof(allButtonList) / sizeof(Button)), ControllerResult::ControllerResult_InvalidControllerId);

        EXPECT_EQ(hidInputerWrapper.m_PushMultiButtonFunction(id, allButtonList, sizeof(allButtonList) / sizeof(Button), g_DefaultWaitPress), ControllerResult::ControllerResult_InvalidControllerId);

        EXPECT_EQ(hidInputerWrapper.m_PushMultiSeqFunction(id, allButtonList, sizeof(allButtonList) / sizeof(Button), g_DefaultWaitPress, g_DefaultWaitSpan), ControllerResult::ControllerResult_InvalidControllerId);
    }

    //-------------------------------------
    // 不正なボタン操作
    //-------------------------------------
    for (int i = 0; i < g_AllControllerIndexRange; i++)
    {
        EXPECT_EQ(hidInputerWrapper.m_PressMultiButtonFunction(i, invalidButtonList, sizeof(invalidButtonList) / sizeof(Button)), ControllerResult::ControllerResult_InvalidButton);

        EXPECT_EQ(hidInputerWrapper.m_ReleaseMultiButtonFunction(i, invalidButtonList, sizeof(invalidButtonList) / sizeof(Button)), ControllerResult::ControllerResult_InvalidButton);

        EXPECT_EQ(hidInputerWrapper.m_PushMultiButtonFunction(i, invalidButtonList, sizeof(invalidButtonList) / sizeof(Button), g_DefaultWaitPress), ControllerResult::ControllerResult_InvalidButton);

        EXPECT_EQ(hidInputerWrapper.m_PushMultiSeqFunction(i, invalidButtonList, sizeof(invalidButtonList) / sizeof(Button), g_DefaultWaitPress, g_DefaultWaitSpan), ControllerResult::ControllerResult_InvalidButton);
    }

    //-------------------------------------
    // 接続済みのコントローラーID の解除
    //-------------------------------------
    ReleaseAllController(&hidInputerWrapper );


    //-------------------------------------
    // 未接続のコントローラーに対して
    //-------------------------------------
    for (int i = 0; i < g_AllControllerIndexRange; i++)
    {
        EXPECT_EQ(hidInputerWrapper.m_PressMultiButtonFunction(i, allButtonList, sizeof(allButtonList) / sizeof(Button)), ControllerResult::ControllerResult_InvalidControllerId);

        EXPECT_EQ(hidInputerWrapper.m_ReleaseMultiButtonFunction(i, allButtonList, sizeof(allButtonList) / sizeof(Button)), ControllerResult::ControllerResult_InvalidControllerId);

        EXPECT_EQ(hidInputerWrapper.m_PushMultiButtonFunction(i, allButtonList, sizeof(allButtonList) / sizeof(Button), g_DefaultWaitPress), ControllerResult::ControllerResult_InvalidControllerId);

        EXPECT_EQ(hidInputerWrapper.m_PushMultiSeqFunction(i, allButtonList, sizeof(allButtonList) / sizeof(Button), g_DefaultWaitPress, g_DefaultWaitSpan), ControllerResult::ControllerResult_InvalidControllerId);
    }

    // 全てのコントローラー接続を解除
    ReleaseAllController(&hidInputerWrapper );

    // 終了処理
    hidInputerWrapper.m_FinalizeFunction();
    // ライブラリ解放テスト
    ASSERT_TRUE(hidInputerWrapper.ReleaseHidLibrary());
}
