﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "TestUtil.h"
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>

using namespace hidInputer;

namespace
{
    // 乱数生成器
    std::random_device      g_RandomDevice;
    std::mt19937            g_Mersenne(g_RandomDevice());
}

bool GetTargetSerial( SerialNumberString* pOutSerial )
{
    static bool     isAlreadyAcquired = false;
    static int      argc = nnt::GetHostArgc();
    static char**   argv = nnt::GetHostArgv();

    // 引数チェック
    if (isAlreadyAcquired == false)
    {
        if (argc < 2)
        {
            return false;
        }

        isAlreadyAcquired = true;
    }

    strncpy_s(pOutSerial->value, argv[1], sizeof(pOutSerial->value));
    return true;
}

// 全コントローラー接続・解除
void ConnectAllController( HidInputerWrapper *wrapper )
{
    for (int i = 0; i < g_ControllerIndexCount; i++)
    {
        EXPECT_NE(wrapper->m_AddControllerForIdFunction(i), ControllerResult::ControllerResult_Unexpected);
    }
    for (int i = 0; i < g_DebugControllerCount; i++)
    {
        int debugControllerId;
        EXPECT_NE(wrapper->m_AddDebugControllerFunction(&debugControllerId), ControllerResult::ControllerResult_Unexpected);
    }
}

void ReleaseAllController( HidInputerWrapper *wrapper )
{
    // 接続済みコントローラー取得
    int controllerCount;
    EXPECT_EQ(ControllerResult::ControllerResult_Success,
        wrapper->m_GetConnectedControllerCountFunction(&controllerCount));

    int controllerIds[g_ControllerIndexCount];
    int connectedControllerCount;
    EXPECT_EQ(ControllerResult::ControllerResult_Success,
        wrapper->m_GetConnectedControllerIdsFunction(&connectedControllerCount, controllerIds, g_ControllerIndexCount));

    // コントローラー切断
    for (int i = 0; i < connectedControllerCount; i++)
    {
        EXPECT_EQ(ControllerResult::ControllerResult_Success,
            wrapper->m_ReleaseControllerFunction(controllerIds[i]));
    }

    // デバッグコントローラー切断
    int debugControllerCount;
    EXPECT_EQ(ControllerResult::ControllerResult_Success,
        wrapper->m_GetConnectedDebugControllerCountFunction(&debugControllerCount));

    int debugControllerIds[g_DebugControllerCount];
    int connectedDebugControllerCount;
    EXPECT_EQ(ControllerResult::ControllerResult_Success,
        wrapper->m_GetConnectedDebugControllerIdsFunction(&connectedDebugControllerCount, debugControllerIds, g_DebugControllerCount));
    for (int i = 0; i < connectedDebugControllerCount; i++)
    {
        EXPECT_EQ(ControllerResult::ControllerResult_Success,
            wrapper->m_ReleaseControllerFunction(debugControllerIds[i]));
    }
}

// 不正なコントローラー ID 作成
int CreateInvalidControllerId()
{
    int id = 0;
    while (true){
        id = static_cast<int>(g_Mersenne() % g_RandomInvalidIdRange);
        if (id >= g_ControllerMinId && id <= g_ControllerMaxId)
        {
            continue;
        }

        break;
    }
    return id;
}

// 不正なボタンタイプ作成
int CreateInvalidButton()
{
    return static_cast<int>((g_Mersenne() % g_RandomInvalidIdRange) + Button::Button_MaxCount) * CreateRandomSign();
}

// 不正なボタンタイプ作成
int CreateInvalidStick()
{
    return static_cast<int>((g_Mersenne() % g_RandomInvalidIdRange) + Stick::Stick_MaxCount) * CreateRandomSign();
}

// 不正なタッチID作成
int CreateInvalidTouchId()
{
    return static_cast<int>((g_Mersenne() % g_RandomInvalidIdRange) + g_MaxTouchNum) * CreateRandomSign();
}

// ランダムの正負符号作成
int CreateRandomSign()
{
    return (((g_Mersenne() % 2) * 2) - 1);
}

float CreateDegree(bool isBigValue)
{
    if (isBigValue)
    {
        // 通常より絶対値の大きなパラメータを生成する
        return ((static_cast<float>(g_Mersenne()) / g_Mersenne._Max) * g_RandomDegreeMaxRange * CreateRandomSign() * g_RandomBigParameterRate) + g_RandomDegreeMaxRange;
    }
    else
    {
        return (static_cast<float>(g_Mersenne()) / g_Mersenne._Max) * g_RandomDegreeMaxRange * CreateRandomSign();
    }
}
float CreatePower(bool isBigValue)
{
    if (isBigValue)
    {
        // 通常より絶対値の大きなパラメータを生成する
        return (static_cast<float>(g_Mersenne()) / g_Mersenne._Max) * CreateRandomSign() * g_RandomBigParameterRate;
    }
    else
    {
        return (static_cast<float>(g_Mersenne()) / g_Mersenne._Max) * CreateRandomSign();
    }
}


void CreateWaitTime(int *pOutWaitTime)
{
    if( pOutWaitTime != NULL )
    {
        (*pOutWaitTime) = g_Mersenne() % g_RandomWaitMaxRange;
    }
}

void GenerateRandomMultitouchParameter(
    float pOutRelativePosX[], float pOutRelativePosY[],
    float pOutTouchDegreeList[], float* radius,
    int* pOutCenterX, int* pOutCenterY,
    float* pOutCenterMoveDeg, float* pOutCenterMoveLength,
    float* pOutStartDegree, float* pOutEndDegree,
    float* pOutStartScale, float* pOutEndScale,
    int* pOutWaitDuration, int* pOutWaitSpan, int* pOutWaitTouchBegin,
    int* pOutWaitTouchEnd, bool* pOutReleaseAfter,
    int touchNum,
    bool isOutOfScreen)
{
    if (pOutRelativePosX != NULL)
    {
        for (int i = 0; i < touchNum; i++)
        {
            if (isOutOfScreen)
            {
                // 画面外での操作用パラメータ
                pOutRelativePosX[i] = (static_cast<float>(g_Mersenne() % g_ScreenWidth) + g_ScreenWidth) * CreateRandomSign();
            }
            else
            {
                // 画面内での操作用パラメータ
                pOutRelativePosX[i] = static_cast<float>(g_Mersenne() % g_ScreenWidth) * CreateRandomSign();
            }
        }
    }
    if (pOutRelativePosY != NULL)
    {
        for (int i = 0; i < touchNum; i++)
        {
            if (isOutOfScreen)
            {
                // 大きな値を代入し、画面外での操作パラメータを作成
                pOutRelativePosX[i] = (static_cast<float>(g_Mersenne() % g_ScreenHeight) + g_ScreenHeight) * CreateRandomSign();
            }
            else
            {
                // 画面内での操作パラメータ
                pOutRelativePosY[i] = static_cast<float>(g_Mersenne() % g_ScreenHeight) * CreateRandomSign();
            }
        }
    }

    // touchDegreeList
    if (pOutTouchDegreeList != NULL)
    {
        for (int i = 0; i < touchNum; i++)
        {
            pOutTouchDegreeList[i] = (static_cast<float>(g_Mersenne()) / g_Mersenne._Max) * g_RandomDegreeMaxRange * CreateRandomSign();
        }
    }

    if (radius != NULL)
    {
        if (isOutOfScreen)
        {
            // 大きな値を代入し、画面外での操作パラメータを作成
            (*radius) = ((static_cast<float>(g_Mersenne()) / g_Mersenne._Max)  * g_RandomRadiusMaxRange) + (g_ScreenWidth * g_RandomBigParameterRate) * CreateRandomSign();
        }
        else
        {
            // 画面内での操作パラメータ
            (*radius) = ((static_cast<float>(g_Mersenne()) / g_Mersenne._Max)  * g_RandomRadiusMaxRange);
        }
    }
    // centerPos
    if (pOutCenterX != NULL)
    {
        (*pOutCenterX) = g_Mersenne() % g_ScreenWidth;
    }
    if (pOutCenterY != NULL)
    {
        (*pOutCenterY) = g_Mersenne() % g_ScreenHeight;
    }
    // moveDeg
    if (pOutCenterMoveDeg != NULL)
    {
        (*pOutCenterMoveDeg) = CreateDegree( false );
    }
    if (pOutCenterMoveLength != NULL)
    {
        (*pOutCenterMoveLength) = (static_cast<float>(g_Mersenne()) / g_Mersenne._Max) * g_ScreenWidth;
    }
    // degree
    if (pOutStartDegree != NULL)
    {
        (*pOutStartDegree) = CreateDegree( false );
    }
    if (pOutEndDegree != NULL)
    {
        (*pOutEndDegree) = CreateDegree( false );
    }
    // scale
    if (pOutStartScale != NULL)
    {
        (*pOutStartScale) = (static_cast<float>(g_Mersenne()) / g_Mersenne._Max) * g_RandomScaleMaxRange;
    }
    if (pOutEndScale)
    {
        (*pOutEndScale) = (static_cast<float>(g_Mersenne()) / g_Mersenne._Max) * g_RandomScaleMaxRange;
    }
    // waitSpan
    CreateWaitTime(pOutWaitTouchBegin);
    CreateWaitTime(pOutWaitTouchEnd);
    CreateWaitTime(pOutWaitSpan);
    CreateWaitTime(pOutWaitDuration);
    if (pOutWaitSpan != NULL)
    {
        (*pOutWaitDuration) = (*pOutWaitDuration) + (*pOutWaitSpan);
    }

    // release After
    if (pOutReleaseAfter != NULL)
    {
        (*pOutReleaseAfter) = (g_Mersenne() % 2) == 0 ? false : true;;
    }
}

