﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include "CaptureProcessorWrapper.h"
#include "testCaptureProcessor_TestDefinition.h"
#include "../Common/testAutoTestAssistTool_Util.h"

using namespace captureProcessor;


TEST(CaptureProcessorSuite, LoadImageTestInUninitialized)
{
     // Capture ライブラリのラッパークラス
    CaptureProcessorWrapper       captureProcessorWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadCaptureLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadFunctiontions());


    ImageIdType capturedImageId;
    ImageIdType loadedImageId;

    auto templateImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
        "\\Externals\\TestBinaries\\AutoTestAssistTool\\CaptureProcessorTest\\template_image720p.png");
    auto saveImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
        "\\save_image.png");

    //-------------------------------------
    // 未初期化での実行
    //-------------------------------------
    // キャプチャ画像取得
    EXPECT_EQ(captureProcessorWrapper.m_GetImageFromCaptureDeviceFunction(&capturedImageId),
        CaptureResult::CaptureResult_NotInitialized);

    // 画像読み込み
    EXPECT_EQ(captureProcessorWrapper.m_LoadImageFileFunction(
        &loadedImageId, templateImagePath.c_str()), CaptureResult::CaptureResult_NotInitialized);

    // 画像解放
    EXPECT_EQ(captureProcessorWrapper.m_ReleaseImageFunction(capturedImageId),
        CaptureResult::CaptureResult_NotInitialized);

    // 全画像解放
    EXPECT_EQ(captureProcessorWrapper.m_ReleaseAllImagesFunction(),
        CaptureResult::CaptureResult_NotInitialized);

    // 画像保存
    EXPECT_EQ(captureProcessorWrapper.m_SaveImageFileFunction(
        saveImagePath.c_str(), loadedImageId ), CaptureResult::CaptureResult_NotInitialized);

    // ライブラリ解放テスト
    ASSERT_TRUE(captureProcessorWrapper.ReleaseCaptureLibrary());

}

// ライブラリ読み込み、解放確認
TEST_P(CaptureModeTest, LoadImageTest)
{
     // Capture ライブラリのラッパークラス
    CaptureProcessorWrapper       captureProcessorWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadCaptureLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadFunctiontions());

    ImageIdType capturedImageId;
    ImageIdType loadedImageId;
    ImageIdType invalidImageId = { 0 };

    auto saveImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
        "\\save_image.png");
    auto invalidImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
        "\\Externals\\TestBinaries\\AutoTestAssistTool\\CaptureProcessorTest\\invalidImage.png");

    //-------------------------------------
    // 初期化済みでの実行
    //-------------------------------------
    auto mode = GetParam();
    std::string templateImagePath;
    if (mode == CaptureMode::CaptureMode_720p)
    {
        templateImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
            "\\Externals\\TestBinaries\\AutoTestAssistTool\\CaptureProcessorTest\\template_image720p.png");
        captureProcessorWrapper.m_InitializeForCaptureDeviceFunction(
            0, false, CaptureResolution::CaptureResolution_720p);
    }
    else if (mode == CaptureMode::CaptureMode_1080p)
    {
        templateImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
            "\\Externals\\TestBinaries\\AutoTestAssistTool\\CaptureProcessorTest\\template_image1080p.png");
        captureProcessorWrapper.m_InitializeForCaptureDeviceFunction(
            0, false, CaptureResolution::CaptureResolution_1080p);
    }
    else if (mode == CaptureMode::CaptureMode_DevKit)
    {
        int     argc = nnt::GetHostArgc();
        char**  argv = nnt::GetHostArgv();
        SerialNumberString serialNumber;
        ASSERT_GE(argc, 2);
        strncpy_s(serialNumber.value, argv[1], sizeof(serialNumber.value));

        templateImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
            "\\Externals\\TestBinaries\\AutoTestAssistTool\\CaptureProcessorTest\\template_image720p.png");
        ASSERT_EQ(captureProcessorWrapper.m_InitializeForDevkitCapture(
            serialNumber, false), CaptureResult::CaptureResult_Success);
    }
    else
    {
        ASSERT_TRUE(false);
    }

    // キャプチャ画像取得
    EXPECT_EQ(captureProcessorWrapper.m_GetImageFromCaptureDeviceFunction(&capturedImageId),
        CaptureResult::CaptureResult_Success);

    // 画像読み込み
    EXPECT_EQ(captureProcessorWrapper.m_LoadImageFileFunction(
        &loadedImageId, templateImagePath.c_str()), CaptureResult::CaptureResult_Success);

    // 画像保存
    EXPECT_EQ(captureProcessorWrapper.m_SaveImageFileFunction(
        saveImagePath.c_str(), loadedImageId ), CaptureResult::CaptureResult_Success);

    // 不正なファイルパスでの画像保存
    EXPECT_EQ(captureProcessorWrapper.m_SaveImageFileFunction("??/??.png", capturedImageId),
        CaptureResult::CaptureResult_FileCreationFailed);

    // 画像解放
    EXPECT_EQ(captureProcessorWrapper.m_ReleaseImageFunction(capturedImageId),
        CaptureResult::CaptureResult_Success);

    // 全画像解放
    EXPECT_EQ(captureProcessorWrapper.m_ReleaseAllImagesFunction(),
        CaptureResult::CaptureResult_Success);


    //-------------------------------------
    // 不正なパラメーターでの実行
    //-------------------------------------
    // 不正な画像読み込み
    EXPECT_EQ(captureProcessorWrapper.m_LoadImageFileFunction(
        &loadedImageId, invalidImagePath.c_str()), CaptureResult::CaptureResult_Unexpected);

    // 不正な画像 ID での画像解放
    EXPECT_EQ(captureProcessorWrapper.m_ReleaseImageFunction(invalidImageId),
        CaptureResult::CaptureResult_InvalidImageId);

    // 不正な画像 ID での画像出力
    EXPECT_EQ(captureProcessorWrapper.m_SaveImageFileFunction(
        saveImagePath.c_str(), invalidImageId ), CaptureResult::CaptureResult_InvalidImageId);


    // ライブラリ解放テスト
    ASSERT_TRUE(captureProcessorWrapper.ReleaseCaptureLibrary());
}
