﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>

#include "CaptureProcessorWrapper.h"
#include "testCaptureProcessor_TestDefinition.h"
#include "../Common/testAutoTestAssistTool_Util.h"

using namespace captureProcessor;

namespace
{
    const double   g_TemplateMatchingThreshold = 0.75;
}

TEST(CaptureProcessorSuite, CreateFilterTest)
{
    CaptureProcessorWrapper       captureProcessorWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadCaptureLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadFunctiontions());

    //-------------------------------------
    // 未初期化での実行
    //-------------------------------------
    FilterIdType filterId;
    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_CreateFilterFunction(&filterId));
    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_DeleteFilterFunction(filterId));

    //-------------------------------------
    // 初期化済みでの実行
    //-------------------------------------
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();
    SerialNumberString serialNumber;
    ASSERT_GE(argc, 2);
    strncpy_s(serialNumber.value, argv[1], sizeof(serialNumber.value));

    ASSERT_EQ(captureProcessorWrapper.m_InitializeForDevkitCapture(
        serialNumber, false), CaptureResult::CaptureResult_Success);

    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_CreateFilterFunction(&filterId));
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_DeleteFilterFunction(filterId));

    // 解放後のフィルターを解放
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidFilterId,
        captureProcessorWrapper.m_DeleteFilterFunction(filterId));

    // ライブラリ解放テスト
    ASSERT_TRUE(captureProcessorWrapper.ReleaseCaptureLibrary());
}

TEST(CaptureProcessorSuite, SetEachFilterTest)
{
    CaptureProcessorWrapper       captureProcessorWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadCaptureLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadFunctiontions());

    //-------------------------------------
    // 未初期化での実行
    //-------------------------------------
    FilterIdType filterId;
    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_CreateFilterFunction(&filterId));

    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_SetBinarizationFilterFunction(filterId, 128));

    Rgb24Color lowerColor = { 0,0,0 };
    Rgb24Color upperColor = { 255,255,255 };
    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_SetColorRangeFilterFunction(filterId, lowerColor, upperColor));

    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_SetMorphologyFilterFunction(filterId, MorphologyConversionType::MorphologyConversionType_Closing, 3));

    ImageIdType filteredImageId = { 0 };
    ImageIdType captureImageId = { 0 };
    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_ApplyFilterFunction(&filteredImageId, captureImageId, filterId));

    //-------------------------------------
    // 初期化済みでの実行
    //-------------------------------------
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();
    SerialNumberString serialNumber;
    ASSERT_GE(argc, 2);
    strncpy_s(serialNumber.value, argv[1], sizeof(serialNumber.value));

    ASSERT_EQ(captureProcessorWrapper.m_InitializeForDevkitCapture(
        serialNumber, false), CaptureResult::CaptureResult_Success);

    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_CreateFilterFunction(&filterId));
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_SetBinarizationFilterFunction(filterId, 128));
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_SetColorRangeFilterFunction(filterId, lowerColor, upperColor));
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_SetMorphologyFilterFunction(filterId, MorphologyConversionType::MorphologyConversionType_Closing, 3));

    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_GetImageFromCaptureDeviceFunction(&captureImageId));
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_ApplyFilterFunction(&filteredImageId, captureImageId, filterId));

    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_DeleteFilterFunction(filterId));

    //-------------------------------------
    // 不正なフィルター ID に対して実行
    //-------------------------------------
    srand(static_cast<unsigned int>(time(NULL)));
    filterId.id = static_cast<int>(rand());
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidFilterId,
        captureProcessorWrapper.m_SetBinarizationFilterFunction(filterId, 128));
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidFilterId,
        captureProcessorWrapper.m_SetColorRangeFilterFunction(filterId, lowerColor, upperColor));
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidFilterId,
        captureProcessorWrapper.m_SetMorphologyFilterFunction(filterId, MorphologyConversionType::MorphologyConversionType_Closing, 3));
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidFilterId,
        captureProcessorWrapper.m_ApplyFilterFunction(&filteredImageId, captureImageId, filterId));

    //-------------------------------------
    // 不正なフィルターパラメーターで実行
    //-------------------------------------
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_CreateFilterFunction(&filterId));
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidParameter,
        captureProcessorWrapper.m_SetBinarizationFilterFunction(filterId, 256));
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidParameter,
        captureProcessorWrapper.m_SetBinarizationFilterFunction(filterId, -1));
    Rgb24Color invalidLowerColor = { -1, 0, 0 };
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidParameter,
        captureProcessorWrapper.m_SetColorRangeFilterFunction(filterId, invalidLowerColor, upperColor));
    Rgb24Color invalidUpperColor = { 256, 0, 0 };
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidParameter,
        captureProcessorWrapper.m_SetColorRangeFilterFunction(filterId, lowerColor, invalidUpperColor));
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidParameter,
        captureProcessorWrapper.m_SetMorphologyFilterFunction(filterId, static_cast<MorphologyConversionType>(100), 3));
    EXPECT_EQ(CaptureResult::CaptureResult_InvalidParameter,
        captureProcessorWrapper.m_SetMorphologyFilterFunction(filterId, static_cast<MorphologyConversionType>(-1), 3));

    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_DeleteFilterFunction(filterId));


    ASSERT_TRUE(captureProcessorWrapper.ReleaseCaptureLibrary());
}

TEST(CaptureProcessorSuite, DetectObjectWithFilterTestInUninitialized)
{
    CaptureProcessorWrapper       captureProcessorWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadCaptureLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadFunctiontions());

    int positionX;
    int positionY;
    ImageIdType imageId = { 0 };
    FilterIdType filterId = { 0 };

    //-------------------------------------
    // 未初期化での実行
    //-------------------------------------
    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_GetImageFromCaptureDeviceFunction(&imageId));

    EXPECT_EQ(CaptureResult::CaptureResult_NotInitialized,
        captureProcessorWrapper.m_CreateFilterFunction(&filterId));

    EXPECT_EQ(MatchingResult::MatchingResult_NotInitialized,
        captureProcessorWrapper.m_DetectObjectWithFilterFunction(&positionX, &positionY, imageId, 0.5, filterId));

    // ライブラリ解放テスト
    ASSERT_TRUE(captureProcessorWrapper.ReleaseCaptureLibrary());
}

TEST_P(CaptureModeTest, DetectObjectWithFilterTest)
{
     // Capture ライブラリのラッパークラス
    CaptureProcessorWrapper       captureProcessorWrapper;

    // ライブラリ読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadCaptureLibrary());

    // 関数読み込みテスト
    ASSERT_TRUE(captureProcessorWrapper.LoadFunctiontions());

    int positionX;
    int positionY;
    ImageIdType correctImageId = { 0 };
    ImageIdType wrongImageId = { 0 };

    auto mode = GetParam();
    std::string correctImagePath;
    std::string wrongImagePath;
    if (mode == CaptureMode::CaptureMode_720p)
    {
        correctImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
            "\\Externals\\TestBinaries\\AutoTestAssistTool\\CaptureProcessorTest\\template_image720p.png");
        wrongImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
            "\\Externals\\TestBinaries\\AutoTestAssistTool\\CaptureProcessorTest\\template_image1080p.png");
        captureProcessorWrapper.m_InitializeForCaptureDeviceFunction(
            0, false, CaptureResolution::CaptureResolution_720p);
    }
    else if (mode == CaptureMode::CaptureMode_1080p)
    {
        correctImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
            "\\Externals\\TestBinaries\\AutoTestAssistTool\\CaptureProcessorTest\\template_image1080p.png");
        wrongImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
            "\\Externals\\TestBinaries\\AutoTestAssistTool\\CaptureProcessorTest\\template_image720p.png");
        captureProcessorWrapper.m_InitializeForCaptureDeviceFunction(
            0, false, CaptureResolution::CaptureResolution_1080p);
    }
    else if (mode == CaptureMode::CaptureMode_DevKit)
    {
        int     argc = nnt::GetHostArgc();
        char**  argv = nnt::GetHostArgv();
        SerialNumberString serialNumber;
        ASSERT_GE(argc, 2);
        strncpy_s(serialNumber.value, argv[1], sizeof(serialNumber.value));

        correctImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
            "\\Externals\\TestBinaries\\AutoTestAssistTool\\CaptureProcessorTest\\template_image720p.png");
        wrongImagePath = autoTestAssistTool::util::GetSdkRootPath().append(
            "\\Externals\\TestBinaries\\AutoTestAssistTool\\CaptureProcessorTest\\template_image1080p.png");
        ASSERT_EQ(captureProcessorWrapper.m_InitializeForDevkitCapture(
            serialNumber, false), CaptureResult::CaptureResult_Success);
    }
    else
    {
        ASSERT_TRUE(false);
    }

    //-------------------------------------
    // フィルター準備
    //-------------------------------------
    // [0]: Blank filter, [1]: Normal filter
    FilterIdType filterId[2] = { 0 };
    int filterCount = sizeof(filterId) / sizeof(FilterIdType);
    // Blank filter
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_CreateFilterFunction(&filterId[0]));
    // Normal filter
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_CreateFilterFunction(&filterId[1]));
    EXPECT_EQ(CaptureResult::CaptureResult_Success,
        captureProcessorWrapper.m_SetBinarizationFilterFunction(filterId[1], 128));

    //-------------------------------------
    // 初期化済みでの実行(720p)
    //-------------------------------------
    // 画像読み込み
    EXPECT_EQ(captureProcessorWrapper.m_LoadImageFileFunction(
        &correctImageId, correctImagePath.c_str()), CaptureResult::CaptureResult_Success);
    EXPECT_EQ(captureProcessorWrapper.m_LoadImageFileFunction(
        &wrongImageId, wrongImagePath.c_str()), CaptureResult::CaptureResult_Success);

    for (int i = 0; i < filterCount; i++)
    {
        FilterIdType filter = filterId[i];

        // テンプレートマッチング(成功)
        EXPECT_EQ(captureProcessorWrapper.m_DetectObjectWithFilterFunction(
            &positionX, &positionY, correctImageId, g_TemplateMatchingThreshold, filter),
            MatchingResult::MatchingResult_Success);

        EXPECT_EQ(captureProcessorWrapper.m_WaitDetectObjectWithFilterFunction(
            &positionX, &positionY, correctImageId, g_TemplateMatchingThreshold, 1000, filter),
            MatchingResult::MatchingResult_Success);


        // テンプレートマッチング(失敗)
        EXPECT_EQ(captureProcessorWrapper.m_DetectObjectWithFilterFunction(
            &positionX, &positionY, wrongImageId, g_TemplateMatchingThreshold, filter),
            MatchingResult::MatchingResult_Failed);

        EXPECT_EQ(captureProcessorWrapper.m_WaitDetectObjectWithFilterFunction(
            &positionX, &positionY, wrongImageId, g_TemplateMatchingThreshold, 1000, filter),
            MatchingResult::MatchingResult_Failed);
    }

    //-------------------------------------
    // 不正なパラメーターでの実行
    //-------------------------------------
    // 不正な画像 ID に対して実行
    ImageIdType invalidImageId = { 0 };
    EXPECT_EQ(captureProcessorWrapper.m_DetectObjectWithFilterFunction(
        &positionX, &positionY, invalidImageId, g_TemplateMatchingThreshold, filterId[0]),
        MatchingResult::MatchingResult_InvalidImageId);

    EXPECT_EQ(captureProcessorWrapper.m_WaitDetectObjectWithFilterFunction(
        &positionX, &positionY, invalidImageId, g_TemplateMatchingThreshold, 1000, filterId[0]),
        MatchingResult::MatchingResult_InvalidImageId);

    //-------------------------------------
    // 不正なフィルターでの実行
    //-------------------------------------
    FilterIdType invalidFilter = { 0 };
    EXPECT_EQ(captureProcessorWrapper.m_DetectObjectWithFilterFunction(
        &positionX, &positionY, correctImageId, g_TemplateMatchingThreshold, invalidFilter),
        MatchingResult::MatchingResult_InvalidFilterId);

    EXPECT_EQ(captureProcessorWrapper.m_WaitDetectObjectWithFilterFunction(
        &positionX, &positionY, correctImageId, g_TemplateMatchingThreshold, 1000, invalidFilter),
        MatchingResult::MatchingResult_InvalidFilterId);

    // ライブラリ解放テスト
    ASSERT_TRUE(captureProcessorWrapper.ReleaseCaptureLibrary());
}
