﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading.Tasks;

namespace ProdTestTool
{
    public class ProdEncrypt
    {
        private static void ExecuteProdEncrypt(string devNspPath, string prodNspPath, string keyConfigXmlPath = null)
        {
            if (File.Exists(prodNspPath))
            {
                Console.WriteLine("Already prod encrypted. Skipped. ({0})", prodNspPath);
                return;
            }
            var outDirPath = Path.GetDirectoryName(prodNspPath);
            Directory.CreateDirectory(outDirPath);
            Func<string, string> getKeyConfigOption = (path) =>
            {
                if (string.IsNullOrEmpty(path))
                {
                    return string.Empty;
                }
                return string.Format("--keyconfig \"{0}\"", path);
            };
            var args = string.Format("prodencryption -o \"{0}\" --no-nspu {1} \"{2}\"", outDirPath, getKeyConfigOption(keyConfigXmlPath), devNspPath);
            Common.ExecuteCommand(Constant.Authoring, args);
        }

        private static void ExecuteProdEncryptPatch(string devPatchPath, string prodPatchPath, string previousProdPatchPath, string originalPath, string originalProdPath)
        {
            if (File.Exists(prodPatchPath))
            {
                Console.WriteLine("Already prod encrypted. Skipped. ({0})", prodPatchPath);
                return;
            }
            var outDirPath = Path.GetDirectoryName(prodPatchPath);
            Directory.CreateDirectory(outDirPath);
            Func<string, string> getPreviousProdOption = (path) =>
            {
                if (string.IsNullOrEmpty(path))
                {
                    return string.Empty;
                }
                return string.Format("--previous-prod \"{0}\"", path);
            };
            var args = string.Format("prodencryption-patch -o \"{0}\" --no-nspu --original \"{1}\" --original-prod \"{2}\" {3} \"{4}\"",
                                     outDirPath, originalPath, originalProdPath, getPreviousProdOption(previousProdPatchPath), devPatchPath);
            Common.ExecuteCommand(Constant.Authoring, args);
        }

        private static void ExecuteProdEncryptGameCard(string devAppPath, string prodXciPath, string prodUppPath, string prodPatchPath, string aocPath)
        {
            if (File.Exists(prodXciPath))
            {
                Console.WriteLine("Already prod encrypted. Skipped. ({0})", prodXciPath);
                return;
            }
            var outDirPath = Path.GetDirectoryName(prodXciPath);
            Directory.CreateDirectory(outDirPath);
            Func<string, string, string> getOption = (prefix, path) =>
            {
                if (string.IsNullOrEmpty(path))
                {
                    return string.Empty;
                }
                return string.Format("{0} \"{1}\"", prefix, path);
            };
            {
                var args = string.Format("prodencryption -o \"{0}\" --gamecard --no-xcie --no-padding {1} {2} {3} \"{4}\"", outDirPath, getOption("--upp", prodUppPath), getOption("--patch", prodPatchPath), getOption("--aoc", aocPath), devAppPath);
                Common.ExecuteCommand(Constant.Authoring, args);
            }
        }

        public static void ProdEncryptSystemNsp(string[] targetDirPaths)
        {
            foreach (var targetDirPath in targetDirPaths)
            {
                var outDirPath = Constant.OutDir + "/prod/system";
                try
                {
                    Parallel.ForEach(Directory.EnumerateFiles(targetDirPath).Where(x => x.EndsWith(".nsp")), nsp =>
                    {
                        var outNspPath = outDirPath + string.Format("/{0}", Path.GetFileName(nsp).Replace(".nsp", "_prod.nsp"));
                        ExecuteProdEncrypt(nsp, outNspPath, Constant.KeyConfig);
                    });
                }
                catch (AggregateException e)
                {
                    throw e.InnerException;
                }
            }
        }

        public static void ProdEncryptNsp(string[] appIds)
        {
            var romInfos = Download.GetRomInfoList();
            var groupedVersionInfos = Download.GroupRomInfoList(romInfos);

            Func<string, Download.VersionInfoWithRomId, string> getDevNspPath = (appId, versionInfo) =>
            {
                Console.WriteLine("getDevNspPath: {0}, {1}", appId, versionInfo.Type);
                var type = (new Func<string, string>((typeStr) =>
                {
                    switch (typeStr)
                    {
                        case "Application" : return "app";
                        case "Patch" : return "patch";
                        case "AddOnContent" : return "aoc";
                        case "OnCardPatch" : return "oncardpatch";
                        case "OnCardAoc" : return "oncardaoc";
                        default: throw new NotImplementedException();
                    }
                }) (versionInfo.Type));
                var prefix = (new Func<string, string>((typeStr) =>
                {
                    switch (typeStr)
                    {
                        case "Application" : return "HAC-P";
                        case "Patch" : return "HAC-U";
                        case "AddOnContent" : return "HAC-M";
                        case "OnCardPatch" : return "HAC-U";
                        case "OnCardAoc" : return "HAC-P";
                        default: throw new NotImplementedException();
                    }
                }) (versionInfo.Type));
                var targetPath = Constant.OutDir + string.Format("/dev/{0}/{1}", appId, type);
                var nsp = Directory.EnumerateFiles(targetPath).Where(x => x.EndsWith(".nsp") && x.Contains(versionInfo.RomId) && x.Contains(prefix));
                if (!nsp.Any())
                {
                    return null;
                }
                return nsp.Single();
            };

            Func<string, string> getProdNspPath = (devNspPath) =>
            {
                return devNspPath.Replace("/dev/", "/prod/").Replace(".nsp", "_prod.nsp");
            };

            Func<string, string> getProdXciPath = (devNspPath) =>
            {
                return devNspPath.Replace("/dev/", "/prod/").Replace(".nsp", "_prod.xci");
            };

            try
            {
                Parallel.ForEach(appIds, appId =>
                {
                    var groupedVersionInfo = groupedVersionInfos[appId];

                    if (groupedVersionInfo.Application.Count == 0)
                    {
                        // テストアプリと思われるので Skip
                        Console.WriteLine("Skip {0} because it has no application.", appId);
                        return;
                    }

                    var donePathList = new List<string>();

                    var app = groupedVersionInfo.Application.First();
                    var devAppPath = getDevNspPath(appId, app);
                    var prodAppPath = string.Empty;
                    if (devAppPath != null && !donePathList.Contains(devAppPath))
                    {
                        prodAppPath = getProdNspPath(devAppPath);
                        ExecuteProdEncrypt(devAppPath, prodAppPath);
                        donePathList.Add(devAppPath);
                    }

                    string previousProdPatchPath = null;
                    foreach (var patch in groupedVersionInfo.Patch.OrderBy(x => Convert.ToInt32(x.Version)))
                    {
                        var devPatchPath = getDevNspPath(appId, patch);
                        if (devPatchPath != null && devAppPath != null && !donePathList.Contains(devPatchPath))
                        {
                            var prodPatchPath = getProdNspPath(devPatchPath);
                            ExecuteProdEncryptPatch(devPatchPath, prodPatchPath, previousProdPatchPath, devAppPath, prodAppPath);
                            previousProdPatchPath = prodPatchPath;
                            donePathList.Add(devPatchPath);
                        }
                    }

                    foreach (var aoc in groupedVersionInfo.AddOnContent)
                    {
                        var devAocPath = getDevNspPath(appId, aoc);
                        if (devAocPath != null && !donePathList.Contains(devAocPath))
                        {
                            var prodAocPath = getProdNspPath(devAocPath);
                            ExecuteProdEncrypt(devAocPath, prodAocPath);
                            donePathList.Add(devAocPath);
                        }
                    }

                    var prodXciPath = getProdXciPath(devAppPath);

                    foreach (var patch in groupedVersionInfo.OnCardPatch)
                    {
                        var devPatchPath = getDevNspPath(appId, patch);
                        if (devPatchPath != null && !donePathList.Contains(devPatchPath))
                        {
                            if (File.Exists(prodXciPath))
                            {
                                File.Delete(prodXciPath);
                            }
                            var targetPatch = groupedVersionInfo.Patch.Single(x => x.Version == patch.Version);
                            var devTargetPatchPath = getDevNspPath(appId, targetPatch);
                            var prodTargetPatchPath = getProdNspPath(devTargetPatchPath);
                            ExecuteProdEncryptGameCard(devAppPath, prodXciPath, null, prodTargetPatchPath, null); // TODO: UPP
                            donePathList.Add(devPatchPath);
                        }
                    }

                    foreach (var aoc in groupedVersionInfo.OnCardAoc)
                    {
                        var devAocPath = getDevNspPath(appId, aoc);
                        if (devAocPath != null && !donePathList.Contains(devAocPath))
                        {
                            if (File.Exists(prodXciPath))
                            {
                                File.Delete(prodXciPath);
                            }
                            string prodPatchPath = null;
                            if (aoc.LatestVersion != 0)
                            {
                                var targetPatch = groupedVersionInfo.Patch.Single(x => Convert.ToInt32(x.Version) == aoc.LatestVersion);
                                var devPatchPath = getDevNspPath(appId, targetPatch);
                                prodPatchPath = getProdNspPath(devPatchPath);
                            }
                            ExecuteProdEncryptGameCard(devAppPath, prodXciPath, null, prodPatchPath, devAocPath); // TODO: UPP
                            donePathList.Add(devAocPath);
                        }
                    }

                    if (devAppPath != null)
                    {
                        // OnCardPatch, OnCardAoc のいずれかがある場合は省略される
                        ExecuteProdEncryptGameCard(devAppPath, prodXciPath, null, null, null); // TODO: UPP
                    }
                });
            }
            catch (AggregateException e)
            {
                throw e.InnerException;
            }
        }
    }
}

