﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.Authoring.AuthoringLibrary;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using Nintendo.Authoring.AuthoringTool;

namespace AuthoringToolsTest
{
    [TestClass]
    public class ExcecutionCreateFsTest : ExcecutionTestBase
    {
        [TestInitialize]
        public void Initialize()
        {
            System.Runtime.GCSettings.LargeObjectHeapCompactionMode = System.Runtime.GCLargeObjectHeapCompactionMode.CompactOnce;
            GC.Collect();
        }

        //
        [TestMethod]
        public void TestExecutionCreateFsAlignment()
        {
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(outputDir);

            var testPath = new TestUtility.TestPath(this.TestContext);
            string outputFile = outputDir + "/test.bin";
            string outputRomAdfFile = outputDir + "/test.bin.adf";

            {
                File.Delete(outputFile);
                File.Delete(outputRomAdfFile);

                {
                    var sampleFile = outputDir + "\\test1.bin";
                    using (var file = File.OpenWrite(sampleFile))
                    {
                        file.WriteByte(0xAA);
                    }
                }

                {
                    var sampleFile = outputDir + "\\test2.bin";
                    using (var file = File.OpenWrite(sampleFile))
                    {
                        file.WriteByte(0x55);
                    }
                }

                Program.Main(string.Format("createfs --format romfs -o {0} --save-adf {1}",
                    outputFile, outputDir).Split());
                Assert.IsTrue(File.Exists(outputFile));
                Assert.IsTrue(File.Exists(outputRomAdfFile));

                using (var file = File.OpenRead(outputFile))
                {
                    Assert.AreEqual(512 + 16 + 8 + 136, file.Length);

                    file.Seek(512, SeekOrigin.Begin);
                    Assert.AreEqual(0xAA, file.ReadByte());

                    file.Seek(512 + 16, SeekOrigin.Begin);
                    Assert.AreEqual(0x55, file.ReadByte());
                }
            }
        }

        void TestExecutionCreateFsFromAdf(string outputDir, Func<int, Pair<string, string>> fileGenerator)
        {
            const int AlignmentSize = 16;

            var testPath = new TestUtility.TestPath(this.TestContext);
            string outputFile = outputDir + "/test.bin";
            string inputAdf = outputDir + "/test.bin.adf";

            {
                File.Delete(outputFile);
                File.Delete(inputAdf);

                using (var adf = new StreamWriter(File.Create(inputAdf)))
                {
                    adf.WriteLine("formatType : RomFs");
                    adf.WriteLine("version : 0");
                    adf.WriteLine("entries :");

                    int index = 0;
                    long offset = 0;
                    Pair<string, string> pair;
                    while ((pair = fileGenerator(index)) != null)
                    {
                        adf.WriteLine("  - type : file");
                        adf.WriteLine("    name : " + pair.first);
                        adf.WriteLine("    offset : " + offset);
                        adf.WriteLine("    path : " + pair.second);

                        FileInfo fi = new FileInfo(pair.second);
                        offset += fi.Length;
                        offset = (offset + (AlignmentSize - 1)) & ~(AlignmentSize - 1);

                        ++index;
                    }
                }

                var error = ExecuteProgram(string.Format("createfs -o {0} {1}", outputFile, inputAdf));
                Assert.IsTrue(error == string.Empty);
                Assert.IsTrue(File.Exists(outputFile));

                File.Delete(outputFile);
                File.Delete(inputAdf);
            }
        }

        [TestMethod]
        public void TestExecutionCreateFsDeepFilePath()
        {
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(outputDir);

            string dataBin = Path.Combine(outputDir, "data.bin");
            File.Delete(dataBin);
            using (var data = File.Create(dataBin))
            {
                data.WriteByte(0xFE);
            }

            // 後で / が足されるので 767 文字にする
            string sampleFile = "";
            for (int i = 0; i < 768 / 2 - 1; ++i)
            {
                sampleFile += "a/";
            }
            sampleFile += "a";

            TestExecutionCreateFsFromAdf(outputDir, delegate (int index)
            {
                return index == 0 ? new Pair<string, string>(sampleFile, dataBin) : null;
            });

            sampleFile += "a";

            try
            {
                TestExecutionCreateFsFromAdf(outputDir, delegate (int index)
                {
                    return index == 0 ? new Pair<string, string>(sampleFile, dataBin) : null;
                });
                Assert.Fail();
            }
            catch
            {
                // 失敗することを期待する
            }

            File.Delete(dataBin);
        }

        [TestMethod]
        public void TestExecutionCreateFsManyFiles()
        {
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(outputDir);

            string dataBin = Path.Combine(outputDir, "data.bin");
            File.Delete(dataBin);
            using (var data = File.Create(dataBin))
            {
                data.WriteByte(0xFE);
            }

            TestExecutionCreateFsFromAdf(outputDir, delegate (int index)
            {
                return index < 50 * 10000
                    ? new Pair<string, string>(
                        string.Format(
                            @"{0}/{1}/{2}.bin",
                            index >> 16,
                            (index & 0xFF00) >> 8,
                            index & 0xFF),
                        dataBin)
                    : null;
            });

            File.Delete(dataBin);
        }

        [TestMethod]
        public void TestExecutionCreateFsMultiByteNameEntry()
        {
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(outputDir);

            string dataBin = Path.Combine(outputDir, "data.bin");
            File.Delete(dataBin);
            using (var data = File.Create(dataBin))
            {
                data.WriteByte(0xFE);
            }

            TestExecutionCreateFsFromAdf(outputDir, delegate (int index)
            {
                return index < 10 ? new Pair<string, string>(string.Format("マルチバイト{0}.bin", index), dataBin) :
                       index < 20 ? new Pair<string, string>(string.Format("マルチバイトDir{0}/マルチバイト{0}.bin", index), dataBin) : null;
            });

            File.Delete(dataBin);
        }

        [TestMethod]
        public void TestExecutionCreateFsRomFsIsCaseSensitive()
        {
            var outputDir = TestEnvironment.GetOutputPath(MethodBase.GetCurrentMethod().Name);
            SafeDeleteDirectory(outputDir);
            Directory.CreateDirectory(outputDir);

            string dataBin = Path.Combine(outputDir, "data.bin");
            File.Delete(dataBin);
            using (var data = File.Create(dataBin))
            {
                data.WriteByte(0xFE);
            }

            TestExecutionCreateFsFromAdf(outputDir, delegate (int index)
            {
                switch (index)
                {
                    case 0:
                        return new Pair<string, string>("a.txt", dataBin);
                    case 1:
                        return new Pair<string, string>("A.txt", dataBin);
                    default:
                        return null;
                }
            });

            File.Delete(dataBin);
        }
    }
}
