﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Collections.Generic;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Security.Cryptography;
using System.Xml.Serialization;
using System.Xml;
using System.Text.RegularExpressions;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.Authoring.AuthoringLibrary;
using Nintendo.Authoring.AuthoringTool;
using Nintendo.Authoring.ETicketLibrary;
using Nintendo.Authoring.FileSystemMetaLibrary;
using TestUtility;

namespace AuthoringToolsTest
{
    [TestClass]
    public class ExcecutionProdTest : ExcecutionTestBase
    {
        [TestInitialize]
        protected void ForceGC()
        {
            ForceGCImpl();
            Environment.SetEnvironmentVariable("AOC_TEST_DATA_PATH", "data", EnvironmentVariableTarget.Process);
        }

        private void CheckTicketCommonKeyId(string nspPath, byte expectedCommonKeyId)
        {
            ForceGC();

            byte commonKeyId = 0;
            using (var nspReader = new NintendoSubmissionPackageReader(nspPath))
            {
                foreach (var info in nspReader.ListFileInfo().FindAll(x => Path.GetExtension(x.Item1) == ".tik"))
                {
                    var ticketName = info.Item1;

                    TicketDataReader ticketReader = new TicketDataReader(nspReader.ReadFile(ticketName, 0, nspReader.GetFileSize(ticketName)), (UInt32)nspReader.GetFileSize(ticketName));

                    commonKeyId = ticketReader.GetCommonKeyId();
                }
            }

            Assert.AreEqual(expectedCommonKeyId, commonKeyId);
        }

        [TestMethod]
        public void TestExecutionProdEncryptionWithUpdatePartition()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            var uppV450 = env.SourceDir + "\\UppTest\\dummyUpp_v450.nsp";
            var uppV262594 = env.SourceDir + "\\UppTest\\dummyUpp_v262594.nsp";
            var uppV201327042 = env.SourceDir + "\\UppTest\\dummyUpp_v201327042.nsp";
            var uppV268435906 = env.SourceDir + "\\UppTest\\dummyUpp_v268435906.nsp";

            // RequiredSystemVersion: 65536, CardSize: 16 GB
            var metaFileV0 = env.SourceDir + "\\UppTest\\describe_all_required_system_version.nmeta";
            var metaFileV1 = env.OutputDir + "\\" + Path.GetFileName(metaFileV0).Replace(".nmeta", "_v1.nmeta");
            var metaFileV2 = env.OutputDir + "\\" + Path.GetFileName(metaFileV0).Replace(".nmeta", "_v2.nmeta");
            MakeSpecifiedVersionMetaFile(metaFileV0, metaFileV1, 1);
            MakeSpecifiedVersionMetaFile(metaFileV0, metaFileV2, 2);

            var metaFileV0KeepGen = env.SourceDir + "\\UppTest\\describe_all_keep_gen.nmeta";
            var metaFileV2KeepGen = env.OutputDir + "\\" + Path.GetFileName(metaFileV0KeepGen).Replace(".nmeta", "_v2.nmeta");
            MakeSpecifiedVersionMetaFile(metaFileV0KeepGen, metaFileV2KeepGen, 2);

            var iconPath = env.SourceDir + "\\Icon\\describe_all.bmp";
            var legalInformationZip = MakeLegalInfoZipfile(env.OutputDir);
            var accessibleUrlsDir = Path.Combine(env.OutputDir, "accessible-urls");
            var accessibleUrlsFilePath = Path.Combine(accessibleUrlsDir, "accessible-urls.txt");
            MakeFileImpl(accessibleUrlsFilePath, "http://test0.com\n");

            Action<string, string, string> createNsp = (nsp, metaFile, addCmd) =>
            {
                var cmd = string.Format("creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} {3} --icon AmericanEnglish {4} Japanese {4}", nsp, metaFile, env.DefaultDescFile, env.TestCodeDir, iconPath) + addCmd;
                MakeNpdm(env.NpdmFile, metaFile, env.DefaultDescFile);
                var error = ExecuteProgram(cmd);
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(nsp, null);
            };

            Action<string, string, string, string> createPatch = (nsp, orig, prev, cur) =>
            {
                var cmd = string.Format("makepatch -o {0} --desc {1} --original {2} --current {3}", nsp, env.DefaultDescFile, orig, cur);
                if (!string.IsNullOrEmpty(prev))
                {
                    cmd += string.Format(" --previous {0}", prev);
                }
                var error = ExecuteProgram(cmd);
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(nsp, null);
            };

            Action<string, string, string, string> createProdPatch = (nsp, orig, prevProd, addCmd) =>
            {
                // upp v201327042 で製品化
                var cmd = string.Format("prodencryption-patch -o {0} --no-nspu --no-check --keyconfig {1} --upp {2} --original {3} {4}", env.OutputDir, env.DefaultKeyConfigFile, uppV201327042, orig, nsp) + addCmd;
                if (!string.IsNullOrEmpty(prevProd))
                {
                    cmd += string.Format(" --previous-prod {0}", prevProd);
                }
                var error = ExecuteProgram(cmd);
                Assert.IsTrue(error == string.Empty, error);
                CheckResultXml(env.OutputDir, nsp);
            };

            Func<string, string, string, string> createXci = (nsp, upp, patch) =>
            {
                var cmd = string.Format("prodencryption --no-check --keyconfig {0} --gamecard --includes-cnmt --no-padding --no-xcie -o {1} {2}", env.DefaultKeyConfigFile, env.OutputDir, nsp);
                if (!string.IsNullOrEmpty(upp))
                {
                    cmd += string.Format(" --upp {0}", upp);
                }
                if (!string.IsNullOrEmpty(patch))
                {
                    cmd += string.Format(" --patch {0}", patch);
                }
                return ExecuteProgram(cmd);
            };

            Action<string, int, int, int, int, byte, ulong> checkXciResultXml = (nsp, legalInfoCount, htmlDocCount, toolInfoCount, keyGeneration, launchFlags, fwVersion) =>
            {
                var xml = nsp.Replace(".nsp", "_prod.xci.result.xml");
                CheckXmlEncoding(xml);
                ResultModel model;
                using (var fs = new FileStream(xml, FileMode.Open, FileAccess.Read))
                {
                    var serializer = new XmlSerializer(typeof(ResultModel));
                    model = (ResultModel)serializer.Deserialize(fs);
                    foreach (var contentMeta in model.ContentMetaList.Where(x => x.Type == NintendoContentMetaConstant.ContentMetaTypeApplication))
                    {
                        Assert.IsTrue(!contentMeta.ContentList.Any(x => x.KeyGeneration != keyGeneration));
                    }
                }
                // ゲームカード製品化時にカードサイズが再評価されていないことのテスト
                Assert.IsTrue(model.CardHeader != null);
                Assert.IsTrue(model.CardHeader.RomSize == XciUtils.GetBytesString(XciInfo.RomSize16GB));
                Assert.IsTrue(model.CardHeader.Flags == XciUtils.GetBytesString(launchFlags));
                Assert.IsTrue(model.CardHeader.FwVersion == XciUtils.GetBytesString(fwVersion));
                // 各種 xml 情報のテスト
                Assert.AreEqual(model.LegalInformationList.Count, legalInfoCount);
                Assert.AreEqual(model.HtmlDocumentList.Count, htmlDocCount);
                Assert.AreEqual(model.AuthoringToolInfoList.Count, toolInfoCount);
                foreach (var cnmtModel in model.ContentMetaList)
                {
                    if (cnmtModel.Type == NintendoContentMetaConstant.ContentMetaTypePatch)
                    {
                        Assert.IsTrue(!string.IsNullOrEmpty((cnmtModel as PatchContentMetaModel)?.ApplicationId));
                    }
                    else if (cnmtModel.Type == NintendoContentMetaConstant.ContentMetaTypeApplication)
                    {
                        Assert.IsTrue(!string.IsNullOrEmpty((cnmtModel as ApplicationContentMetaModel)?.PatchId));
                    }
                    else
                    {
                        Assert.IsTrue(false);
                    }
                }
            };

            Action<string, uint, bool> checkCnmtXml = (nsp, requiredSystemVersion, isPatch) =>
            {
                var xml = nsp.Replace(".nsp", "_prod.xci.debug.cnmt.xml");
                CheckXmlEncoding(xml);
                using (var fs = new FileStream(xml, FileMode.Open, FileAccess.Read))
                {
                    if (isPatch)
                    {
                        var serializer = new XmlSerializer(typeof(PatchContentMetaModel));
                        var model = (PatchContentMetaModel)serializer.Deserialize(fs);
                        Assert.AreEqual(model.RequiredSystemVersion, requiredSystemVersion);
                    }
                    else
                    {
                        var serializer = new XmlSerializer(typeof(ApplicationContentMetaModel));
                        var model = (ApplicationContentMetaModel)serializer.Deserialize(fs);
                        Assert.AreEqual(model.RequiredSystemVersion, requiredSystemVersion);
                    }
                }
            };

            Action<string, int> checkProdPatchKeyGeneration = (nsp, keyGeneration) =>
            {
                var xml = nsp.Replace(".nsp", "_prod.nsp.result.xml");
                CheckXmlEncoding(xml);
                using (var fs = new FileStream(xml, FileMode.Open, FileAccess.Read))
                {
                    var serializer = new XmlSerializer(typeof(ResultModel));
                    var model = (ResultModel)serializer.Deserialize(fs);
                    foreach (var contentMeta in model.ContentMetaList)
                    {
                        Assert.IsTrue(!contentMeta.ContentList.Any(x => x.KeyGeneration != keyGeneration));
                    }
                }
            };

            // app v0 (program, nacp, gen0)
            var v0 = env.OutputDir + "\\v0.nsp";
            createNsp(v0, metaFileV0, " --keygeneration 0");

            // app v1 (program, nacp, legal, gen0)
            var v1 = env.OutputDir + "\\v1.nsp";
            createNsp(v1, metaFileV1, string.Format(" --keygeneration 0 --legal-information {0}", legalInformationZip));

            // app v2 (program, nacp, legal, html, gen2)
            // TODO: htmldocument をパッチで削除するケース
            var v2 = env.OutputDir + "\\v2.nsp";
            createNsp(v2, metaFileV2, string.Format(" --keygeneration 2 --legal-information {0} --accessible-urls {1}", legalInformationZip, accessibleUrlsDir));

            // app v2 (program, nacp, legal, html, gen2, keep-gen)
            var v2KeepGen = env.OutputDir + "\\v2KeepGen.nsp";
            createNsp(v2KeepGen, metaFileV2KeepGen, string.Format(" --keygeneration 2 --legal-information {0} --accessible-urls {1}", legalInformationZip, accessibleUrlsDir));

            // app v0 (program, nacp, gen3)
            var v0gen3 = env.OutputDir + "\\v0gen3.nsp";
            createNsp(v0gen3, metaFileV0, " --keygeneration 3");

            // patch v1 (gen0)
            var v1patch = env.OutputDir + "\\v1patch.nsp";
            createPatch(v1patch, v0, null, v1);

            // patch v2 (gen0)
            var v2patch = env.OutputDir + "\\v2patch.nsp";
            createPatch(v2patch, v0, v1patch, v2);

            // patch v2 (gen0, keep-gen)
            var v2patchKeepGen = env.OutputDir + "\\v2patchKeepGen.nsp";
            createPatch(v2patchKeepGen, v0, v1patch, v2KeepGen);

            // patch v2 (as v1, gen2)
            var v2dpatch = env.OutputDir + "\\v2dpatch.nsp";
            createPatch(v2dpatch, v0, null, v2);

            {
                // 初回パッチは世代を上げられる
                createProdPatch(v1patch, v0, null, string.Format(" --upp-for-desired-safe-keygen {0}", uppV262594));
                checkProdPatchKeyGeneration(v1patch, 2);

                // 初回パッチは threshold を超えていても世代を上げられる
                createProdPatch(v2dpatch, v0, null, string.Format(" --upp-for-desired-safe-keygen {0} --size-threshold-for-keygen-update 1024", uppV201327042));
                checkProdPatchKeyGeneration(v2dpatch, 3);

                // 初回パッチ以外では threshold を超えると鍵世代が更新されず、previous-prod の鍵世代を引き継ぐ（デルタ含む）
                createProdPatch(v2patch, v0, v1patch.Replace(".nsp", "_prod.nsp"), string.Format(" --upp-for-desired-safe-keygen {0} --size-threshold-for-keygen-update 1024", uppV201327042));
                checkProdPatchKeyGeneration(v2patch, 2);
            }

            {
                createProdPatch(v1patch, v0, null, string.Empty);
                checkProdPatchKeyGeneration(v1patch, 0);

                // threshold を超えない場合は初回パッチ以外でも鍵世代が更新できる（デルタ含む）
                createProdPatch(v2patch, v0, v1patch.Replace(".nsp", "_prod.nsp"), string.Format(" --upp-for-desired-safe-keygen {0} --use-safe-keygen-forcibly", uppV201327042));
                checkProdPatchKeyGeneration(v2patch, 3);

                // 初回パッチ以外では 0 -> 2 への世代更新はできない
                createProdPatch(v2patch, v0, v1patch.Replace(".nsp", "_prod.nsp"), string.Format(" --upp-for-desired-safe-keygen {0}", uppV262594));
                checkProdPatchKeyGeneration(v2patch, 0);

                // KeepGeneration が設定されていると鍵世代を更新しない
                createProdPatch(v2patchKeepGen, v0, v1patch.Replace(".nsp", "_prod.nsp"), string.Format(" --upp-for-desired-safe-keygen {0} --use-safe-keygen-forcibly", uppV201327042));
                checkProdPatchKeyGeneration(v2patchKeepGen, 0);
            }

            {
                // xci には app の RequiredSystemVersion を下回る UPP を搭載できない
                var error = createXci(v0, uppV450, null);
                Assert.IsTrue(error.IndexOf("The version of specified update partition (= 450) is older than that of application or patch being included (= 65536).") >= 0, error);

                // xci には app の鍵世代に対応していない UPP を搭載できない
                error = createXci(v0gen3, uppV262594, null);
                Assert.IsTrue(error.IndexOf("Target nsp includes content has newer key generation than upp has.") >= 0, error);
                error = createXci(v0gen3, uppV201327042, null);
                Assert.IsTrue(error == string.Empty, error);
                checkXciResultXml(v0gen3, 0, 0, 1, 3, 0, 1);
                checkCnmtXml(v0gen3, 65536, false);
                // 4 NUP からは FwVersion 更新
                error = createXci(v0gen3, uppV268435906, null);
                Assert.IsTrue(error == string.Empty, error);
                checkXciResultXml(v0gen3, 0, 0, 1, 4, 0, 2);
                checkCnmtXml(v0gen3, 65536, false);

                // 製品化後も app の RequiredSystemVersion は据え置き
                error = createXci(v0, uppV262594, null);
                Assert.IsTrue(error == string.Empty, error);
                checkXciResultXml(v0, 0, 0, 1, 2, 0, 1);
                checkCnmtXml(v0, 65536, false);

                error = createXci(v1, uppV262594, null);
                Assert.IsTrue(error == string.Empty, error);
                checkXciResultXml(v1, 1, 0, 1, 2, 0, 1);
                checkCnmtXml(v1, 65536, false);

                error = createXci(v2, null, null);
                Assert.IsTrue(error == string.Empty, error);
                checkXciResultXml(v2, 1, 1, 1, 2, 0, 1);
                checkCnmtXml(v2, 65536, false);

                // KeepGeneration が設定されていると鍵世代を更新しない
                // nmeta で指定した LaunchFlags が使われる
                error = createXci(v2KeepGen, uppV268435906, null);
                Assert.IsTrue(error == string.Empty, error);
                checkXciResultXml(v2KeepGen, 1, 1, 1, 2, 3, 1);
                checkCnmtXml(v2KeepGen, 65536, false);
            }

            // on-card-patch
            {
                // xci (on-card-patch) には "製品化前の" パッチの RequiredSystemVersion を下回る UPP を搭載できない
                var error = createXci(v0, uppV450, v2patch.Replace(".nsp", "_prod.nsp"));
                Assert.IsTrue(error.IndexOf("The version of specified update partition (= 450) is older than that of application or patch being included (= 65536).") >= 0, error);

                // xci (on-card-patch) には製品化後のパッチの鍵世代に対応していない UPP を搭載できない
                error = createXci(v0, uppV262594, v2dpatch.Replace(".nsp", "_prod.nsp"));
                Assert.IsTrue(error.IndexOf("The patch includes content has newer key generation than upp has.") >= 0, error);
                error = createXci(v0gen3, uppV262594, v2patch.Replace(".nsp", "_prod.nsp"));
                Assert.IsTrue(error.IndexOf("Target nsp includes content has newer key generation than upp has.") >= 0, error);
                error = createXci(v0, uppV201327042, v2dpatch.Replace(".nsp", "_prod.nsp"));
                Assert.IsTrue(error == string.Empty, error);
                checkXciResultXml(v0, 1, 1, 2, 3, 0, 1);
                // UPP の RequiredSystemVersion が製品化時の RequiredSystemVersion より高い場合は製品化時点の値を据え置き
                checkCnmtXml(v0, 201327042 & 0xFFFF0000, true);
                // 4 NUP からは FwVersion 更新
                error = createXci(v0, uppV268435906, v2dpatch.Replace(".nsp", "_prod.nsp"));
                Assert.IsTrue(error == string.Empty, error);
                checkXciResultXml(v0, 1, 1, 2, 4, 0, 2);
                checkCnmtXml(v0, 201327042 & 0xFFFF0000, false);

                // 製品化時の RequiredSystemVersion が UPP の RequiredSystemVersion より大きい場合は UPP の値で上書きされる
                error = createXci(v0, uppV262594, v2patch.Replace(".nsp", "_prod.nsp"));
                Assert.IsTrue(error == string.Empty, error);
                checkXciResultXml(v0, 1, 1, 2, 2, 0, 1);
                checkCnmtXml(v0, 262594, true);

                // UPP を指定しなかった場合 "製品化前の" パッチの RequiredSystemVersion が継承される
                error = createXci(v2, null, v2patch.Replace(".nsp", "_prod.nsp")); // XCI としては不正
                Assert.IsTrue(error == string.Empty, error);
                checkXciResultXml(v2, 2, 2, 2, 2, 0, 1);
                checkCnmtXml(v2, 65536, true);

                // Patch でも KeepGeneration、LaunchFlags が設定できる
                error = createXci(v0, uppV268435906, v2patchKeepGen.Replace(".nsp", "_prod.nsp"));
                Assert.IsTrue(error == string.Empty, error);
                checkXciResultXml(v0, 1, 1, 2, 0, 3, 1);
                checkCnmtXml(v0, 201327042 & 0xFFFF0000, true);
            }
        }

        [TestMethod]
        public void TestNcaKeyGenerationWithMetaType()
        {
            TestEnvironment env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);
            var metaTypes = new string[] {
                NintendoContentMetaConstant.ContentMetaTypeSystemProgram,
                NintendoContentMetaConstant.ContentMetaTypeSystemData,
                NintendoContentMetaConstant.ContentMetaTypeSystemUpdate,
                NintendoContentMetaConstant.ContentMetaTypeBootImagePackage,
                NintendoContentMetaConstant.ContentMetaTypeBootImagePackageSafe,
            };

            var keyConfigFile = env.SourceDir + "\\AuthoringToolsTest.keyconfig.xml";
            var config = new AuthoringConfiguration();
            config.KeyConfigFilePath = keyConfigFile;

            var defaultMetaFilePath = env.SourceDir + "\\SystemDataMeta\\default.nmeta";
            var keepGenMetaFilePath = env.SourceDir + "\\SystemDataMeta\\keepGen.nmeta";
            var systemUpdateMetaFilePath = env.SourceDir + "\\SystemUpdateMeta\\basic.nmeta";

            foreach (var metaType in metaTypes)
            {
                var nspPath = env.OutputDir + "\\systemMetaType.nsp";
                if (metaType == NintendoContentMetaConstant.ContentMetaTypeSystemUpdate)
                {
                    var error = ExecuteProgram(string.Format("creatensp -o {0} --keygeneration 2 --type {1} --meta {2} --keyconfig {3}", nspPath, metaType, systemUpdateMetaFilePath, keyConfigFile));
                    Assert.IsTrue(error == string.Empty, error);
                    CheckNcaKeyGeneration(nspPath, new List<byte>() { 0 }, config.GetKeyConfiguration());
                    error = ExecuteProgram(string.Format("prodencryption -o {0} --no-nspu --no-check --keyconfig {1} {2}", env.OutputDir, env.DefaultKeyConfigFile, nspPath));
                    Assert.IsTrue(error == string.Empty, error);
                    CheckNcaKeyGeneration(nspPath.Replace(".nsp", "_prod.nsp"), new List<byte>() { 0 });
                }
                else
                {
                    var error = ExecuteProgram(string.Format("creatensp -o {0} --keygeneration 2 --type {1} --meta {2} --data {3} --keyconfig {4}", nspPath, metaType, defaultMetaFilePath, env.TestCodeDir, keyConfigFile));
                    Assert.IsTrue(error == string.Empty, error);
                    CheckNcaKeyGeneration(nspPath, new List<byte>() { 2, 0 }, config.GetKeyConfiguration());
                    // システムタイトルは製品化時にツールが対応する最新の鍵世代になる
                    error = ExecuteProgram(string.Format("prodencryption -o {0} --no-nspu --no-check --keyconfig {1} {2}", env.OutputDir, env.DefaultKeyConfigFile, nspPath));
                    Assert.IsTrue(error == string.Empty, error);
                    CheckNcaKeyGeneration(nspPath.Replace(".nsp", "_prod.nsp"), new List<byte>() { NintendoContentFileSystemMetaConstant.CurrentKeyGeneration, 0 });

                    // keepGeneration が指定されている場合は更新されない
                    error = ExecuteProgram(string.Format("creatensp -o {0} --keygeneration 2 --type {1} --meta {2} --data {3} --keyconfig {4}", nspPath, metaType, keepGenMetaFilePath, env.TestCodeDir, keyConfigFile));
                    Assert.IsTrue(error == string.Empty, error);
                    CheckNcaKeyGeneration(nspPath, new List<byte>() { 2, 0 }, config.GetKeyConfiguration());
                    error = ExecuteProgram(string.Format("prodencryption -o {0} --no-nspu --no-check --keyconfig {1} {2}", env.OutputDir, env.DefaultKeyConfigFile, nspPath));
                    Assert.IsTrue(error == string.Empty, error);
                    CheckNcaKeyGeneration(nspPath.Replace(".nsp", "_prod.nsp"), new List<byte>() { 2, 0 });
                }
            }
        }

        [TestMethod]
        public void TestNcaKeyGeneration()
        {
            TestEnvironment env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var testExtractDir = Path.Combine(env.OutputDir, "extract");
            var config = new AuthoringConfiguration();

            var accessibleUrlsDir = Path.Combine(env.OutputDir, "accessible-urls");
            var accessibleUrlsFilePath = Path.Combine(accessibleUrlsDir, "accessible-urls.txt");

            var metaFile = env.SourceDir + "\\ApplicationMeta\\describe_all.nmeta";
            var metaFileV1 = env.OutputDir + "\\describe_all_v1.nmeta";
            var metaFileV2 = env.OutputDir + "\\describe_all_v2.nmeta";
            var metaFileV3 = env.OutputDir + "\\describe_all_v3.nmeta";
            var metaFileV4 = env.OutputDir + "\\describe_all_v4.nmeta";
            var iconPath = testSourceDir + "\\Icon\\describe_all.bmp";

            MakeSpecifiedVersionMetaFile(metaFile, metaFileV1, 1);
            MakeSpecifiedVersionMetaFile(metaFile, metaFileV2, 2);
            MakeSpecifiedVersionMetaFile(metaFile, metaFileV3, 3);
            MakeSpecifiedVersionMetaFile(metaFile, metaFileV4, 4);

            var metaFileV0KeepGen = env.SourceDir + "\\UppTest\\describe_all_keep_gen.nmeta";
            var metaFileV2KeepGen = env.OutputDir + "\\" + Path.GetFileName(metaFileV0KeepGen).Replace(".nmeta", "_v2.nmeta");
            MakeSpecifiedVersionMetaFile(metaFileV0KeepGen, metaFileV2KeepGen, 2);

            var latestKeyGen = NintendoContentFileSystemMetaConstant.SupportedKeyGenerationMax;

            // v0 (generation 0)
            var v0gen0 = env.OutputDir + "\\v0gen0.nsp";
            {
                MakeNpdm(env.NpdmFile, metaFile, env.DefaultDescFile);
                MakeFileImpl(accessibleUrlsFilePath, "http://test0.com\n");

                var error = ExecuteProgram(string.Format("creatensp -o {0} --keygeneration 0 --type Application --meta {1} --desc {2} --program {3} {3} --accessible-urls {4} --icon AmericanEnglish {5} Japanese {5}", v0gen0, metaFile, env.DefaultDescFile, env.TestCodeDir, accessibleUrlsDir, iconPath));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v0gen0, null);
                CheckNcaKeyGeneration(v0gen0, new List<byte>() { 0, 0, 0, 0 }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption -o {0} --no-nspu --no-check --keyconfig {1} {2}", env.OutputDir, env.DefaultKeyConfigFile, v0gen0));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v0gen0.Replace(".nsp", "_prod.nsp"), new List<byte>() { 0, 0, 0, 0 });
            }

            // v0 (generation 2)
            var v0gen2 = env.OutputDir + "\\v0gen2.nsp";
            {
                MakeNpdm(env.NpdmFile, metaFile, env.DefaultDescFile);
                MakeFileImpl(accessibleUrlsFilePath, "http://test0.com\n");

                var error = ExecuteProgram(string.Format("creatensp -o {0} --keygeneration 2 --type Application --meta {1} --desc {2} --program {3} {3} --accessible-urls {4} --icon AmericanEnglish {5} Japanese {5}", v0gen2, metaFile, env.DefaultDescFile, env.TestCodeDir, accessibleUrlsDir, iconPath));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v0gen2, null);
                CheckNcaKeyGeneration(v0gen2, new List<byte>() { 2, 2, 2, 2 }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption -o {0} --no-nspu --no-check --keyconfig {1} {2}", env.OutputDir, env.DefaultKeyConfigFile, v0gen2));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v0gen2.Replace(".nsp", "_prod.nsp"), new List<byte>() { 2, 2, 2, 2 });
            }

            // v0 (generation 2 file replaced)
            var v0gen2mod = env.OutputDir + "\\v0gen2mod.nsp";
            {
                MakeFileImpl(accessibleUrlsFilePath, "http://test1.com\n");
                var replacedPath = string.Empty;
                {
                    var lines = ExecuteProgram(string.Format("list {0}", v0gen2), true).Split('\n');
                    foreach (var line in lines)
                    {
                        var pattern = @"(?<prefix>.*)(accessible-urls\.txt)(.*)";
                        if (Regex.IsMatch(line, pattern))
                        {
                            replacedPath = Regex.Match(line, pattern).Groups["prefix"].Value + "accessible-urls.txt";
                        }
                    }
                }
                var error = ExecuteProgram(string.Format("replace -o {0} {1} {2} {3}", env.OutputDir, v0gen2, replacedPath, accessibleUrlsFilePath));
                Assert.IsTrue(error == string.Empty, error);
                File.Move(v0gen2.Replace(".nsp", "_replaced.nsp"), v0gen2mod);
                VerifyNsp(v0gen2mod, null);
                CheckNcaKeyGeneration(v0gen2mod, new List<byte>() { 2, 2, 2, 2 }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption -o {0} --no-nspu --no-check --keyconfig {1} {2}", env.OutputDir, env.DefaultKeyConfigFile, v0gen2mod));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v0gen2mod.Replace(".nsp", "_prod.nsp"), new List<byte>() { 2, 2, 2, 2 });
            }

            // v0 (fail: program: generation 2, other: gen 0)
            var v0gen2onlyProgram = env.OutputDir + "\\v0gen2onlyProgram.nsp";
            {
                var ncaPath = ArchiveReconstructionUtils.GetSpecifiedContentId(new NintendoSubmissionPackageReader(v0gen2), 0x0005000C10000001, NintendoContentMetaConstant.ContentTypeProgram, 0) + ".nca";
                var error = ExecuteProgram(string.Format("extract -o {0} {1} --target {2}", env.OutputDir, v0gen2, ncaPath));
                Assert.IsTrue(error == string.Empty, error);

                var ncaPathOrig = ArchiveReconstructionUtils.GetSpecifiedContentId(new NintendoSubmissionPackageReader(v0gen0), 0x0005000C10000001, NintendoContentMetaConstant.ContentTypeProgram, 0) + ".nca";
                error = ExecuteProgram(string.Format("replace -o {0} {1} {2} {3}", env.OutputDir, v0gen0, ncaPathOrig, Path.Combine(env.OutputDir, ncaPath)));
                Assert.IsTrue(error.IndexOf("has no compatibility with target nca.") >= 0);
            }

            // v0 (fail: legalinfo: generation 2, other: gen 0)
            var v0gen2onlyAddedLegal = env.OutputDir + "\\v0gen2onlyAddedLegal.nsp";
            {
                var ncaPath = env.OutputDir + "\\legal.nca";
                var legalInformationZip = MakeLegalInfoZipfile(env.OutputDir);
                var error = ExecuteProgram(string.Format("createnca -o {0} --keygeneration 2 --meta {1} --meta-type Application --legal-information {2} --icon AmericanEnglish {3} Japanese {3}", ncaPath, metaFile, legalInformationZip, iconPath));
                Assert.IsTrue(error == string.Empty, error);

                error = ExecuteProgram(string.Format("replace -o {0} {1} null.nca@LegalInformation {2}", env.OutputDir, v0gen0, ncaPath));
                Assert.IsTrue(error.IndexOf("Adding nca has no compatibility with target nsp. Please recreate the nca by tool has same version as one used for target nca.") >= 0);
            }

            // v1 (generation 0)
            var v1gen0 = env.OutputDir + "\\v1gen0.nsp";
            {
                MakeNpdm(env.NpdmFile, metaFileV1, env.DefaultDescFile);
                MakeFileImpl(accessibleUrlsFilePath, "http://test1.com\n");

                var error = ExecuteProgram(string.Format("creatensp -o {0} --keygeneration 0 --type Application --meta {1} --desc {2} --program {3} {3} --accessible-urls {4} --icon AmericanEnglish {5} Japanese {5}", v1gen0, metaFileV1, env.DefaultDescFile, env.TestCodeDir, accessibleUrlsDir, iconPath));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v1gen0, null);
                CheckNcaKeyGeneration(v1gen0, new List<byte>() { 0, 0, 0, 0 }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption -o {0} --no-nspu --no-check --keyconfig {1} {2}", env.OutputDir, env.DefaultKeyConfigFile, v1gen0));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v1gen0.Replace(".nsp", "_prod.nsp"), new List<byte>() { 0, 0, 0, 0 });
            }

            // v1 (generation 2)
            var v1gen2 = env.OutputDir + "\\v1gen2.nsp";
            {
                MakeNpdm(env.NpdmFile, metaFileV1, env.DefaultDescFile);
                MakeFileImpl(accessibleUrlsFilePath, "http://test1.com\n");

                var error = ExecuteProgram(string.Format("creatensp -o {0} --keygeneration 2 --type Application --meta {1} --desc {2} --program {3} {3} --accessible-urls {4} --icon AmericanEnglish {5} Japanese {5}", v1gen2, metaFileV1, env.DefaultDescFile, env.TestCodeDir, accessibleUrlsDir, iconPath));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v1gen0, null);
                CheckNcaKeyGeneration(v1gen2, new List<byte>() { 2, 2, 2, 2 }, config.GetKeyConfiguration());
            }

            // v1 (generation 0 no html)
            var v1gen0noHtml = env.OutputDir + "\\v1gen0noHtml.nsp";
            {
                MakeNpdm(env.NpdmFile, metaFileV1, env.DefaultDescFile);
                MakeFileImpl(accessibleUrlsFilePath, "http://test1.com\n");

                var error = ExecuteProgram(string.Format("creatensp -o {0} --keygeneration 0 --type Application --meta {1} --desc {2} --program {3} {3} --icon AmericanEnglish {5} Japanese {5}", v1gen0noHtml, metaFileV1, env.DefaultDescFile, env.TestCodeDir, accessibleUrlsDir, iconPath));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v1gen0noHtml, null);
                CheckNcaKeyGeneration(v1gen0noHtml, new List<byte>() { 0, 0, 0 }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption -o {0} --no-nspu --no-check --keyconfig {1} {2}", env.OutputDir, env.DefaultKeyConfigFile, v1gen0noHtml));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v1gen0noHtml.Replace(".nsp", "_prod.nsp"), new List<byte>() { 0, 0, 0 });
            }

            // v2 (generation 2)
            var v2gen2 = env.OutputDir + "\\v2gen2.nsp";
            {
                MakeNpdm(env.NpdmFile, metaFileV2, env.DefaultDescFile);
                MakeFileImpl(accessibleUrlsFilePath, "http://test2.com\n");

                var error = ExecuteProgram(string.Format("creatensp -o {0} --keygeneration 2 --type Application --meta {1} --desc {2} --program {3} {3} --accessible-urls {4} --icon AmericanEnglish {5} Japanese {5}", v2gen2, metaFileV2, env.DefaultDescFile, env.TestCodeDir, accessibleUrlsDir, iconPath));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v2gen2, null);
                CheckNcaKeyGeneration(v2gen2, new List<byte>() { 2, 2, 2, 2 }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption -o {0} --no-nspu --no-check --keyconfig {1} {2}", env.OutputDir, env.DefaultKeyConfigFile, v2gen2));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v2gen2.Replace(".nsp", "_prod.nsp"), new List<byte>() { 2, 2, 2, 2 });

                error = ExecuteProgram(string.Format("prodencryption -o {0} --desired-safe-keygen 3 --no-nspu --no-check --keyconfig {1} {2}", env.OutputDir, env.DefaultKeyConfigFile, v2gen2));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v2gen2.Replace(".nsp", "_prod.nsp"), new List<byte>() { 3, 3, 3, 3 });
            }

            // v2 (generation 2, keep-gen)
            var v2gen2KeepGen = env.OutputDir + "\\v2gen2KeepGen.nsp";
            {
                MakeNpdm(env.NpdmFile, metaFileV2KeepGen, env.DefaultDescFile);
                MakeFileImpl(accessibleUrlsFilePath, "http://test2.com\n");

                var error = ExecuteProgram(string.Format("creatensp -o {0} --keygeneration 2 --type Application --meta {1} --desc {2} --program {3} {3} --accessible-urls {4} --icon AmericanEnglish {5} Japanese {5}", v2gen2KeepGen, metaFileV2KeepGen, env.DefaultDescFile, env.TestCodeDir, accessibleUrlsDir, iconPath));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v2gen2KeepGen, null);
                CheckNcaKeyGeneration(v2gen2KeepGen, new List<byte>() { 2, 2, 2, 2 }, config.GetKeyConfiguration());
                // KeepGeneration が指定されている場合、更新されない
                error = ExecuteProgram(string.Format("prodencryption -o {0} --desired-safe-keygen 3 --no-nspu --no-check --keyconfig {1} {2}", env.OutputDir, env.DefaultKeyConfigFile, v2gen2KeepGen));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v2gen2KeepGen.Replace(".nsp", "_prod.nsp"), new List<byte>() { 2, 2, 2, 2 });
            }

            // v3 (generation 3)
            var v3gen3 = env.OutputDir + "\\v3gen3.nsp";
            {
                MakeNpdm(env.NpdmFile, metaFileV3, env.DefaultDescFile);
                MakeFileImpl(accessibleUrlsFilePath, "http://test3.com\n");

                var error = ExecuteProgram(string.Format("creatensp -o {0} --keygeneration 3 --type Application --meta {1} --desc {2} --program {3} {3} --accessible-urls {4} --icon AmericanEnglish {5} Japanese {5}", v3gen3, metaFileV3, env.DefaultDescFile, env.TestCodeDir, accessibleUrlsDir, iconPath));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v3gen3, null);
                CheckNcaKeyGeneration(v3gen3, new List<byte>() { 3, 3, 3, 3 }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption -o {0} --no-nspu --no-check --keyconfig {1} {2}", env.OutputDir, env.DefaultKeyConfigFile, v3gen3));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v3gen3.Replace(".nsp", "_prod.nsp"), new List<byte>() { 3, 3, 3, 3 });
            }

            // v4 (generation latest) //
            var v4genLatest = env.OutputDir + "\\v4genLatest.nsp";
            {
                MakeNpdm(env.NpdmFile, metaFileV4, env.DefaultDescFile);
                MakeFileImpl(accessibleUrlsFilePath, "http://test4.com\n");

                var error = ExecuteProgram(string.Format("creatensp -o {0} --keygeneration {6} --type Application --meta {1} --desc {2} --program {3} {3} --accessible-urls {4} --icon AmericanEnglish {5} Japanese {5}", v4genLatest, metaFileV4, env.DefaultDescFile, env.TestCodeDir, accessibleUrlsDir, iconPath, (int)latestKeyGen));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v4genLatest, null);
                CheckNcaKeyGeneration(v4genLatest, new List<byte>() { latestKeyGen, latestKeyGen, latestKeyGen, latestKeyGen }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption -o {0} --no-nspu --no-check --keyconfig {1} {2}", env.OutputDir, env.DefaultKeyConfigFile, v4genLatest));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v4genLatest.Replace(".nsp", "_prod.nsp"), new List<byte>() { latestKeyGen, latestKeyGen, latestKeyGen, latestKeyGen });

                // desired-safe-keygen の方が低い場合、更新されない
                error = ExecuteProgram(string.Format("prodencryption -o {0} --desired-safe-keygen 3 --no-nspu --no-check --keyconfig {1} {2}", env.OutputDir, env.DefaultKeyConfigFile, v4genLatest));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v4genLatest.Replace(".nsp", "_prod.nsp"), new List<byte>() { latestKeyGen, latestKeyGen, latestKeyGen, latestKeyGen });
            }

            // patch v1
            var v1patch = env.OutputDir + "\\v1patch.nsp";
            {
                var error = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3}", v1patch, env.DefaultDescFile, v0gen0, v1gen0));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v1patch, null);
                CheckNcaKeyGeneration(v1patch, new List<byte>() { 0, 0, 0, 0 }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption-patch -o {0} --no-nspu --no-check --keyconfig {1} --original {2} --original-prod {3} {4}", env.OutputDir, env.DefaultKeyConfigFile, v0gen0, v0gen0.Replace(".nsp", "_prod.nsp"), v1patch));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v1patch.Replace(".nsp", "_prod.nsp"), new List<byte>() { 0, 0, 0, 0 });
            }

            // patch v1 no html
            var v1patchNoHtml = env.OutputDir + "\\v1patchNoHtml.nsp";
            {
                var error = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3}", v1patchNoHtml, env.DefaultDescFile, v0gen0, v1gen0noHtml));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v1patchNoHtml, null);
                CheckNcaKeyGeneration(v1patchNoHtml, new List<byte>() { 0, 0, 0 }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption-patch -o {0} --no-nspu --no-check --keyconfig {1} --original {2} --original-prod {3} {4}", env.OutputDir, env.DefaultKeyConfigFile, v0gen0, v0gen0.Replace(".nsp", "_prod.nsp"), v1patchNoHtml));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v1patchNoHtml.Replace(".nsp", "_prod.nsp"), new List<byte>() { 0, 0, 0 });
            }

            // patch v1 (generation 2)
            var v1patchGen2 = env.OutputDir + "\\v1patchGen2.nsp";
            {
                var error = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3}", v1patchGen2, env.DefaultDescFile, v0gen0, v1gen2));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v1patchGen2, null);
                CheckNcaKeyGeneration(v1patchGen2, new List<byte>() { 2, 2, 2, 2 }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption-patch -o {0} --no-nspu --no-check --keyconfig {1} --original {2} --original-prod {3} {4}", env.OutputDir, env.DefaultKeyConfigFile, v0gen0, v0gen0.Replace(".nsp", "_prod.nsp"), v1patchGen2));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v1patchGen2.Replace(".nsp", "_prod.nsp"), new List<byte>() { 2, 2, 2, 2 });
            }

            // patch v2 (not opt)
            var v2patchNoOpt = env.OutputDir + "\\v2patchNoOpt.nsp";
            {
                var error = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3}", v2patchNoOpt, env.DefaultDescFile, v0gen0, v2gen2));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v2patchNoOpt, null);
                CheckNcaKeyGeneration(v2patchNoOpt, new List<byte>() { 2, 2, 2, 2 }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption-patch -o {0} --no-nspu --no-check --keyconfig {1} --original {2} --original-prod {3} {4}", env.OutputDir, env.DefaultKeyConfigFile, v0gen0, v0gen0.Replace(".nsp", "_prod.nsp"), v2patchNoOpt));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v2patchNoOpt.Replace(".nsp", "_prod.nsp"), new List<byte>() { 2, 2, 2, 2 });
            }

            // patch v2
            var v2patch = env.OutputDir + "\\v2patch.nsp";
            {
                var error = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3} --previous {4}", v2patch, env.DefaultDescFile, v0gen0, v2gen2, v1patch));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v2patch, null);
                CheckNcaKeyGeneration(v2patch, new List<byte>() { 0, 0, 0, 0 }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption-patch -o {0} --no-nspu --no-check --keyconfig {1} --original {2} --original-prod {3} --previous-prod {4} {5}", env.OutputDir, env.DefaultKeyConfigFile, v0gen0, v0gen0.Replace(".nsp", "_prod.nsp"), v1patch.Replace(".nsp", "_prod.nsp"), v2patch));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v2patch.Replace(".nsp", "_prod.nsp"), new List<byte>() { 0, 0, 0, 0, 0, 0, 0, 0 });
            }

            // patch v2 (opt add html)
            var v2patchAddHtml = env.OutputDir + "\\v2patchAddHtml.nsp";
            {
                var error = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3} --previous {4}", v2patchAddHtml, env.DefaultDescFile, v0gen0, v2gen2, v1patchNoHtml));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v2patchAddHtml, null);
                CheckNcaKeyGeneration(v2patchAddHtml, new List<byte>() { 0, 0, 0, 0 }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption-patch -o {0} --no-nspu --no-check --keyconfig {1} --original {2} --original-prod {3} --previous-prod {4} {5}", env.OutputDir, env.DefaultKeyConfigFile, v0gen0, v0gen0.Replace(".nsp", "_prod.nsp"), v1patchNoHtml.Replace(".nsp", "_prod.nsp"), v2patchAddHtml));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v2patchAddHtml.Replace(".nsp", "_prod.nsp"), new List<byte>() { 0, 0, 0, 0, 0, 0, 0, 0 });
            }

            // patch v2 (v1 generation2)
            var v2patchV1Gen2 = env.OutputDir + "\\v2patchV1Gen2.nsp";
            {
                var error = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3} --previous {4}", v2patchV1Gen2, env.DefaultDescFile, v0gen0, v2gen2, v1patchGen2));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v2patchV1Gen2, null);
                CheckNcaKeyGeneration(v2patchV1Gen2, new List<byte>() { 2, 2, 2, 2 }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption-patch -o {0} --no-nspu --no-check --keyconfig {1} --original {2} --original-prod {3} --previous-prod {4} {5}", env.OutputDir, env.DefaultKeyConfigFile, v0gen0, v0gen0.Replace(".nsp", "_prod.nsp"), v1patchGen2.Replace(".nsp", "_prod.nsp"), v2patchV1Gen2));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v2patchV1Gen2.Replace(".nsp", "_prod.nsp"), new List<byte>() { 2, 2, 2, 2, 2, 2, 2, 2 });

                error = ExecuteProgram(string.Format("prodencryption-patch -o {0} --desired-safe-keygen 3 --no-nspu --no-check --keyconfig {1} --original {2} --original-prod {3} --previous-prod {4} {5}", env.OutputDir, env.DefaultKeyConfigFile, v0gen0, v0gen0.Replace(".nsp", "_prod.nsp"), v1patchGen2.Replace(".nsp", "_prod.nsp"), v2patchV1Gen2));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v2patchV1Gen2.Replace(".nsp", "_prod.nsp"), new List<byte>() { 3, 3, 3, 3, 3, 3, 3, 3 });

                // desired-safe-keygen の方が低い場合、更新されない
                error = ExecuteProgram(string.Format("prodencryption-patch -o {0} --desired-safe-keygen 0 --no-nspu --no-check --keyconfig {1} --original {2} --original-prod {3} --previous-prod {4} {5}", env.OutputDir, env.DefaultKeyConfigFile, v0gen0, v0gen0.Replace(".nsp", "_prod.nsp"), v1patchGen2.Replace(".nsp", "_prod.nsp"), v2patchV1Gen2));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v2patchV1Gen2.Replace(".nsp", "_prod.nsp"), new List<byte>() { 2, 2, 2, 2, 2, 2, 2, 2 });
            }

            // patch v2 (keep-gen)
            var v2patchKeepGen = env.OutputDir + "\\v2patchKeepGen.nsp";
            {
                var error = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3} --previous {4}", v2patchKeepGen, env.DefaultDescFile, v0gen0, v2gen2KeepGen, v1patch));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v2patchKeepGen, null);
                CheckNcaKeyGeneration(v2patchKeepGen, new List<byte>() { 0, 0, 0, 0 }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption-patch -o {0} --desired-safe-keygen 3 --no-nspu --no-check --keyconfig {1} --original {2} --original-prod {3} --previous-prod {4} {5}", env.OutputDir, env.DefaultKeyConfigFile, v0gen0, v0gen0.Replace(".nsp", "_prod.nsp"), v1patch.Replace(".nsp", "_prod.nsp"), v2patchKeepGen));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v2patchKeepGen.Replace(".nsp", "_prod.nsp"), new List<byte>() { 0, 0, 0, 0, 0, 0, 0, 0 });
            }

            // patch v3
            var v3patch = env.OutputDir + "\\v3patch.nsp";
            {
                var error = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3} --previous {4}", v3patch, env.DefaultDescFile, v0gen0, v3gen3, v2patch));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v3patch, null);
                CheckNcaKeyGeneration(v3patch, new List<byte>() { 0, 0, 0, 0 }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption-patch -o {0} --no-nspu --no-check --keyconfig {1} --original {2} --original-prod {3} --previous-prod {4} {5}", env.OutputDir, env.DefaultKeyConfigFile, v0gen0, v0gen0.Replace(".nsp", "_prod.nsp"), v2patch.Replace(".nsp", "_prod.nsp"), v3patch));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v3patch.Replace(".nsp", "_prod.nsp"), new List<byte>() { 0, 0, 0, 0, 0, 0, 0, 0 });
            }

            // patch v4
            var v4patch = env.OutputDir + "\\v4patch.nsp";
            {
                var error = ExecuteProgram(string.Format("makepatch -o {0} --desc {1} --original {2} --current {3} --previous {4}", v4patch, env.DefaultDescFile, v0gen0, v4genLatest, v3patch));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v4patch, null);
                CheckNcaKeyGeneration(v4patch, new List<byte>() { 0, 0, 0, 0 }, config.GetKeyConfiguration());
                error = ExecuteProgram(string.Format("prodencryption-patch -o {0} --no-nspu --no-check --keyconfig {1} --original {2} --original-prod {3} --previous-prod {4} {5}", env.OutputDir, env.DefaultKeyConfigFile, v0gen0, v0gen0.Replace(".nsp", "_prod.nsp"), v3patch.Replace(".nsp", "_prod.nsp"), v4patch));
                Assert.IsTrue(error == string.Empty, error);
                CheckNcaKeyGeneration(v4patch.Replace(".nsp", "_prod.nsp"), new List<byte>() { 0, 0, 0, 0, 0, 0, 0, 0 });
            }
        }

        [TestMethod]
        public void TestTicketKeyGeneration()
        {
            TestEnvironment env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);
            var testPath = new TestUtility.TestPath(this.TestContext);
            var testSourceDir = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources";
            var testExtractDir = Path.Combine(env.OutputDir, "extract");
            var config = new AuthoringConfiguration();

            var accessibleUrlsDir = Path.Combine(env.OutputDir, "accessible-urls");
            var accessibleUrlsFilePath = Path.Combine(accessibleUrlsDir, "accessible-urls.txt");

            var metaFile = env.SourceDir + "\\ApplicationMeta\\describe_all.nmeta";
            var metaFileV1 = env.OutputDir + "\\describe_all_v1.nmeta";
            var metaFileV2 = env.OutputDir + "\\describe_all_v2.nmeta";
            var iconPath = testSourceDir + "\\Icon\\describe_all.bmp";

            MakeSpecifiedVersionMetaFile(metaFile, metaFileV1, 1);
            MakeSpecifiedVersionMetaFile(metaFile, metaFileV2, 2);

            var latestKeyGen = NintendoContentFileSystemMetaConstant.SupportedKeyGenerationMax;

            // v0 (generation 0)
            var v0gen0 = env.OutputDir + "\\v0gen0.nsp";
            {
                MakeNpdm(env.NpdmFile, metaFile, env.DefaultDescFile);
                MakeFileImpl(accessibleUrlsFilePath, "http://test0.com\n");

                var error = ExecuteProgram(string.Format("creatensp -o {0} --keygeneration 0 --type Application --meta {1} --desc {2} --program {3} {3} --accessible-urls {4} --ticket --icon AmericanEnglish {5} Japanese {5}", v0gen0, metaFile, env.DefaultDescFile, env.TestCodeDir, accessibleUrlsDir, iconPath));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v0gen0, null);
                CheckNcaKeyGeneration(v0gen0, new List<byte>() { 0, 0, 0, 0 }, config.GetKeyConfiguration());
                CheckTicketCommonKeyId(v0gen0, 0);
            }

            // v0 (generation 2)
            var v0gen2 = env.OutputDir + "\\v0gen2.nsp";
            {
                MakeNpdm(env.NpdmFile, metaFile, env.DefaultDescFile);
                MakeFileImpl(accessibleUrlsFilePath, "http://test0.com\n");

                var error = ExecuteProgram(string.Format("creatensp -o {0} --keygeneration 2 --type Application --meta {1} --desc {2} --program {3} {3} --accessible-urls {4} --ticket --icon AmericanEnglish {5} Japanese {5}", v0gen2, metaFile, env.DefaultDescFile, env.TestCodeDir, accessibleUrlsDir, iconPath));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v0gen2, null);
                CheckNcaKeyGeneration(v0gen2, new List<byte>() { 2, 2, 2, 2 }, config.GetKeyConfiguration());
                CheckTicketCommonKeyId(v0gen2, 2);
            }

            // v0 (generation 3)
            var v0gen3 = env.OutputDir + "\\v0gen3.nsp";
            {
                MakeNpdm(env.NpdmFile, metaFile, env.DefaultDescFile);
                MakeFileImpl(accessibleUrlsFilePath, "http://test0.com\n");

                var error = ExecuteProgram(string.Format("creatensp -o {0} --keygeneration 3 --type Application --meta {1} --desc {2} --program {3} {3} --accessible-urls {4} --ticket --icon AmericanEnglish {5} Japanese {5}", v0gen3, metaFile, env.DefaultDescFile, env.TestCodeDir, accessibleUrlsDir, iconPath));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v0gen3, null);
                CheckNcaKeyGeneration(v0gen3, new List<byte>() { 3, 3, 3, 3 }, config.GetKeyConfiguration());
                CheckTicketCommonKeyId(v0gen3, 3);
            }

            // v0 (generation latest)
            var v0genLatest = env.OutputDir + "\\v0genLatest.nsp";
            {
                MakeNpdm(env.NpdmFile, metaFile, env.DefaultDescFile);
                MakeFileImpl(accessibleUrlsFilePath, "http://test0.com\n");

                var error = ExecuteProgram(string.Format("creatensp -o {0} --keygeneration {6} --type Application --meta {1} --desc {2} --program {3} {3} --accessible-urls {4} --ticket --icon AmericanEnglish {5} Japanese {5}", v0genLatest, metaFile, env.DefaultDescFile, env.TestCodeDir, accessibleUrlsDir, iconPath, (int)latestKeyGen));
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(v0genLatest, null);
                CheckNcaKeyGeneration(v0genLatest, new List<byte>() { latestKeyGen, latestKeyGen, latestKeyGen, latestKeyGen }, config.GetKeyConfiguration());
                CheckTicketCommonKeyId(v0genLatest, latestKeyGen);
            }
        }

        [TestMethod]
        public void TestExecutionProdEncryptionOnCardAoc()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            var uppV268435906 = env.SourceDir + "\\UppTest\\dummyUpp_v268435906.nsp";

            var metaFileV0 = env.SourceDir + "\\UppTest\\describe_all_required_system_version.nmeta";
            var metaFileV1 = env.OutputDir + "\\" + Path.GetFileName(metaFileV0).Replace(".nmeta", "_v1.nmeta");
            MakeSpecifiedVersionMetaFile(metaFileV0, metaFileV1, 65536);

            var aocMetaFile = env.OutputDir + "\\multiple_aoc_required_version.nmeta";
            var othersAocMetaFile = env.OutputDir + "\\others_aoc.nmeta";
            File.Copy(env.SourceDir + "\\UppTest\\multiple_aoc_required_version.nmeta", aocMetaFile);
            File.Copy(env.SourceDir + "\\UppTest\\others_aoc.nmeta", othersAocMetaFile);

            var iconPath = env.SourceDir + "\\Icon\\describe_all.bmp";
            var legalInformationZip = MakeLegalInfoZipfile(env.OutputDir);
            var accessibleUrlsDir = Path.Combine(env.OutputDir, "accessible-urls");
            var accessibleUrlsFilePath = Path.Combine(accessibleUrlsDir, "accessible-urls.txt");
            MakeFileImpl(accessibleUrlsFilePath, "http://test0.com\n");

            Action<string, string, string> createNsp = (nsp, metaFile, addCmd) =>
            {
                var cmd = string.Format("creatensp -o {0} --type Application --meta {1} --desc {2} --program {3} {3} --icon AmericanEnglish {4} Japanese {4} --keygeneration 4", nsp, metaFile, env.DefaultDescFile, env.TestCodeDir, iconPath) + addCmd;
                MakeNpdm(env.NpdmFile, metaFile, env.DefaultDescFile);
                var error = ExecuteProgram(cmd);
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(nsp, null);
            };

            Action<string, string, string> createPatch = (nsp, orig, cur) =>
            {
                var cmd = string.Format("makepatch -o {0} --desc {1} --original {2} --current {3}", nsp, env.DefaultDescFile, orig, cur);
                var error = ExecuteProgram(cmd);
                Assert.IsTrue(error == string.Empty, error);
                VerifyNsp(nsp, null);
            };

            Func<string, string, string, string, string> createAocForCard = (nsp, metaFile, baseNsp, addCmd) =>
            {
                var convertedNsp = nsp.Replace(".nsp", "_forcard.nsp");
                var error = ExecuteProgram(string.Format("creatensp -o {0} --type AddOnContent --meta {1} --keygeneration 4", nsp, metaFile));
                Assert.IsTrue(error == string.Empty, error);
                error = ExecuteProgram(string.Format("convertaoc -o {0} --base {1} {2}", convertedNsp, baseNsp, nsp) + addCmd);
                return error;
            };

            Action<string, string, string, string> checkAocInfo = (aocForCardNsp, aocNsp, origNsp, patchNsp) =>
            {
                var contentMetaList = new List<ContentMetaModel>();
                foreach (var nsp in new List<string>() { aocNsp, origNsp, patchNsp })
                {
                    if (string.IsNullOrEmpty(nsp))
                    {
                        continue;
                    }
                    using (var nspReader = new NintendoSubmissionPackageReader(nsp))
                    {
                        contentMetaList.AddRange(ArchiveReconstructionUtils.ReadContentMetaInNsp(nspReader));
                    }
                }
                using (var nspReader = new NintendoSubmissionPackageReader(aocForCardNsp))
                {
                    var model = nspReader.ListFileInfo().Where(x => x.Item1 == "oncardaoc.xml").Select(x => ArchiveReconstructionUtils.ReadXml<OnCardAddOnContentInfoModel>(nspReader, x.Item1, x.Item2)).First();
                    foreach (var aocContentMeta in model.OriginalAddOnContentMeta.ContentMetaList)
                    {
                        Assert.IsTrue(contentMetaList.Where(x => x.Id == aocContentMeta.Id && x.Version == aocContentMeta.Version && x.Digest == aocContentMeta.Digest).Any(x => (x as AddOnContentContentMetaModel).Tag == aocContentMeta.Tag));
                    }
                    var appAndPatch = model.Application.ContentMetaList;
                    if (model.Patch != null)
                    {
                        appAndPatch.AddRange(model.Patch.ContentMetaList);
                    }
                    foreach (var contentMeta in model.Application.ContentMetaList)
                    {
                        Assert.IsTrue(contentMetaList.Any(x => x.Id == contentMeta.Id && x.Version == contentMeta.Version && x.Digest == contentMeta.Digest));
                    }
                }
            };

            Func<string, string, string, string> createOnCardAoc = (nsp, aocNsp, addCmd) =>
            {
                var error = ExecuteProgram(string.Format("prodencryption --no-check --keyconfig {0} --gamecard --includes-cnmt --no-padding --no-xcie -o {1} --upp {2} --aoc {3} {4}", env.DefaultKeyConfigFile, env.OutputDir, uppV268435906, aocNsp, nsp) + addCmd);
                return error;
            };

            Action<string, string, byte, byte, byte> checkOnCardAoc = (xci, aocForCardNsp, keyGeneration, romSize, launchFlags) =>
            {
                // ResultXml に aoc のコンテンツが含まれていること、鍵世代が上がっていることを確認
                List<ContentMetaModel> contentMetaList;
                using (var nspReader = new NintendoSubmissionPackageReader(aocForCardNsp))
                {
                    contentMetaList = ArchiveReconstructionUtils.ReadContentMetaInNsp(nspReader);
                }
                var xml = xci.Replace(".xci", ".xci.result.xml");
                CheckXmlEncoding(xml);
                using (var fs = new FileStream(xml, FileMode.Open, FileAccess.Read))
                {
                    var serializer = new XmlSerializer(typeof(ResultModel));
                    var model = (ResultModel)serializer.Deserialize(fs);
                    var resultContentMetaList = model.ContentMetaList.Where(x => x.Type == NintendoContentMetaConstant.ContentMetaTypeAddOnContent);
                    foreach (var resultContentMeta in resultContentMetaList)
                    {
                        Assert.IsTrue(!resultContentMeta.ContentList.Any(x => x.KeyGeneration != keyGeneration));
                    }
                    foreach (var contentMeta in contentMetaList)
                    {
                        Assert.IsTrue(resultContentMetaList.Any(x => x.Id == contentMeta.Id && x.Digest == contentMeta.Digest));
                    }
                    Assert.IsTrue(model.CardHeader.RomSize == XciUtils.GetBytesString(romSize));
                    Assert.IsTrue(model.CardHeader.Flags == XciUtils.GetBytesString(launchFlags));
                }
            };

            // v0
            var v0 = env.OutputDir + "\\v0.nsp";
            createNsp(v0, metaFileV0, string.Empty);
            var v0gen2 = env.OutputDir + "\\v0gen2.nsp";
            createNsp(v0gen2, metaFileV0, " --keygeneration 2");

            // v1 & v1 patch
            var v1 = env.OutputDir + "\\v1.nsp";
            createNsp(v1, metaFileV1, string.Empty);
            var v1gen2 = env.OutputDir + "\\v1gen2.nsp";
            createNsp(v1gen2, metaFileV1, " --keygeneration 2");
            var v1patch = env.OutputDir + "\\v1patch.nsp";
            createPatch(v1patch, v0, v1);
            var v1patchGen2 = env.OutputDir + "\\v1patchGen2.nsp";
            createPatch(v1patchGen2, v0gen2, v1gen2);

            // RequiredApplicationVersion を満たさない場合エラー
            {
                var error = createAocForCard(env.OutputDir + "\\aoc_err.nsp", aocMetaFile, v0, string.Empty);
                Assert.IsTrue(error.IndexOf("The verision of base Application or Patch is lower than RequiredApplicationVersion of AddOnContent") >= 0, error);
            }

            // base となるアプリ・パッチがない場合エラー
            {
                var error = createAocForCard(env.OutputDir + "\\aoc_err.nsp", othersAocMetaFile, v0, string.Empty);
                Assert.IsTrue(error.IndexOf("There is no Application or Patch that corresponds to AddOnContent") >= 0, error);
            }

            // ID が重複している場合エラー
            {
                var error = ExecuteProgram(string.Format("creatensp -o {0} --type AddOnContent --meta {1}", env.OutputDir + "\\aoc_err.nsp", aocMetaFile));
                Assert.IsTrue(error == string.Empty, error);
                error = ExecuteProgram(string.Format("convertaoc -o {0} --base {1} {2} {2}", env.OutputDir + "\\aoc_err_c.nsp", v1, env.OutputDir + "\\aoc_err.nsp"));
                Assert.IsTrue(error.IndexOf("AddOnContent IDs in on-card-aoc cannot be duplicated.") >= 0, error);
            }

            var aoc = env.OutputDir + "\\aoc.nsp";
            var aocForCard = env.OutputDir + "\\aoc_forcard.nsp";
            {
                // base = アプリ
                var error = createAocForCard(aoc, aocMetaFile, v1, string.Empty);
                Assert.IsTrue(error == string.Empty, error);
                checkAocInfo(aocForCard, aoc, v1, null);
                error = createOnCardAoc(v1, aocForCard, string.Empty);
                Assert.IsTrue(error == string.Empty, error);
                checkOnCardAoc(v1.Replace(".nsp", "_prod.xci"), aocForCard, 4, XciInfo.RomSize2GB, 0);
                // base としたアプリが与えられなかった場合（Digest 不一致）製品化時エラー
                error = createOnCardAoc(v1gen2, aocForCard, string.Empty);
                Assert.IsTrue(error.IndexOf("is not input, which was used to convert AddOnContent.") >= 0, error);

                // base = パッチ
                // OnCardAoc で指定したカードサイズで作成されることの確認
                error = createAocForCard(aoc, aocMetaFile, v1patch, " --cardsize 16 --cardlaunchflags 3");
                Assert.IsTrue(error == string.Empty, error);
                checkAocInfo(aocForCard, aoc, v0, v1patch);
                error = createOnCardAoc(v0, aocForCard, string.Format(" --patch {0}", v1patch));
                Assert.IsTrue(error == string.Empty, error);
                checkOnCardAoc(v0.Replace(".nsp", "_prod.xci"), aocForCard, 4, XciInfo.RomSize16GB, 3);
                // base としたパッチが与えられなかった場合（Digest 不一致）製品化時エラー
                error = createOnCardAoc(v0, aocForCard, string.Format(" --patch {0}", v1patchGen2));
                Assert.IsTrue(error.IndexOf("is not input, which was used to convert AddOnContent.") >= 0, error);

                // AocForCard は内部鍵暗号化（= チケットが含まれないが list 成功）
                {
                    using (var nspReader = new NintendoSubmissionPackageReader(aocForCard))
                    {
                        Assert.IsTrue(!nspReader.ListFileInfo().Any(x => x.Item1.EndsWith(".tik") || x.Item1.EndsWith(".cert")));
                    }
                    error = ExecuteProgram(string.Format("list {0}", aocForCard));
                    Assert.IsTrue(error == string.Empty, error);
                }

                // AocForCard は replace 不可
                error = ExecuteProgram(string.Format("replace {0} test {1}", aocForCard, metaFileV1)); // target は適当なファイル
                Assert.IsTrue(error.IndexOf("To replace AddOnContent converted for card is not supported.") >= 0, error);

                // AocForCard は DL 版製品化不可
                error = ExecuteProgram(string.Format("prodencryption --no-check --keyconfig {0} --no-nspu -o {1} {2}", env.DefaultKeyConfigFile, env.OutputDir, aocForCard));
                Assert.IsTrue(error.IndexOf("This nsp cannot be prod-encrypted.") >= 0, error);

                // Convert されていない AddOnContent はカード製品化不可
                error = createOnCardAoc(v1, aoc, string.Empty);
                Assert.IsTrue(error.IndexOf("This AddOnContent is not converted for card.") >= 0, error);
            }
        }
    }
}
