﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Collections.Generic;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Security.Cryptography;
using System.Text;
using System.Text.RegularExpressions;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using TestUtility;

namespace AuthoringToolsTest
{
    [TestClass]
    public class ExcecCompatibilityTest : ExcecutionTestBase
    {
        private class TestDataResourcePath
        {
            public string AuthoringTool { get; set; }
            public string MakeMeta { get; set; }
            public string Meta { get; set; }
            public string Icon { get; set; }
            public string Desc { get; set; }

            public TestDataResourcePath(string rootPath, bool isPackageDir)
            {
                if(isPackageDir)
                {
                    AuthoringTool = Path.Combine(rootPath, "NintendoSDK\\Tools\\CommandLineTools\\AuthoringTool\\AuthoringTool.exe");
                    MakeMeta = Path.Combine(rootPath, "NintendoSDK\\Tools\\CommandLineTools\\MakeMeta\\MakeMeta.exe");
                    Meta = Path.Combine(rootPath, "NintendoSDK\\Resources\\SpecFiles\\Application.aarch64.lp64.nmeta");
                    Icon = Path.Combine(rootPath, "NintendoSDK\\Resources\\SpecFiles\\NintendoSDK_Application.bmp");
                    Desc = Path.Combine(rootPath, "NintendoSDK\\Resources\\SpecFiles\\Application.desc");
                }
                else
                {
                    AuthoringTool = Path.Combine(rootPath, "Tools\\CommandLineTools\\AuthoringTool\\AuthoringTool.exe");
                    MakeMeta = Path.Combine(rootPath, "Tools\\CommandLineTools\\MakeMeta\\MakeMeta.exe");
                    Meta = Path.Combine(rootPath, "Programs\\Iris\\Resources\\SpecFiles\\Application.aarch64.lp64.nmeta");
                    Icon = Path.Combine(rootPath, "Programs\\Iris\\Resources\\SpecFiles\\NintendoSDK_Application.bmp");
                    Desc = Path.Combine(rootPath, "Programs\\Iris\\Resources\\SpecFiles\\Application.desc");
                }
            }
        }

        [TestInitialize]
        public void ForceGC()
        {
            ForceGCImpl();
        }

        private RNGCryptoServiceProvider m_Rng = new RNGCryptoServiceProvider();
        private const ulong m_ProgramId = 0x01004b9000490000;

        private string ExecuteProgram(string exePath, string args)
        {
            Console.WriteLine(exePath + " " + args);
            var process = new Process();
            process.StartInfo.FileName = exePath;
            process.StartInfo.Arguments = args;
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardOutput = true;
            process.StartInfo.RedirectStandardError = true;
            process.Start();

            var errorMsg = process.StandardError.ReadToEnd();
            process.WaitForExit();
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);

            return process.StandardOutput.ReadToEnd();
        }

        private void MakeNpdm(string exePath, string npdmFile, string metaFile, string descFile)
        {
            Process process = new Process();
            process.StartInfo.FileName = exePath;
            process.StartInfo.Arguments = String.Format("-o {0} --meta {1} --desc {2} --no_check_programid", npdmFile, metaFile, descFile);
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardError = true;
            process.Start();

            string errorMsg = process.StandardError.ReadToEnd();
            process.WaitForExit();
            Assert.IsTrue(errorMsg == string.Empty, errorMsg);
        }

        private string GetTestResourceDirectoryPath()
        {
            var testPath = new TestPath(this.TestContext);
            var testResourceDir = testPath.GetSigloRoot() + "\\Externals\\TestBinaries\\Tools\\AuthoringToolsExecCompatibilityTest";
            return testResourceDir;
        }

        private string GetFileNameSuffix(uint version, bool useHtml)
        {
            return "_v" + version.ToString("d2") + (useHtml ? "_Html" : "_noHtml") + ".nsp";
        }

        private string GetApplicationPath(string toolVersion, uint version, bool useHtml)
        {
            var filename = "app" + GetFileNameSuffix(version, useHtml);
            return Path.Combine(toolVersion, filename);
        }

        private string GetPatchPath(string toolVersion, uint version, bool useHtml)
        {
            var filename = "patch" + GetFileNameSuffix(version, useHtml);
            return Path.Combine(toolVersion, filename);
        }

        private string GetAddOnContentPath(string toolVersion)
        {
            return Path.Combine(toolVersion, "aoc.nsp");
        }

        private string CreateDataDir(string outputDirPath, string dirName, List<Tuple<string, int, byte[]>> dataPatterns)
        {
            var dirPath = Path.Combine(outputDirPath, dirName);
            Directory.CreateDirectory(dirPath);
            foreach (var dataPattern in dataPatterns)
            {
                var filePath = Path.Combine(dirPath, dataPattern.Item1);
                Directory.CreateDirectory(Path.GetDirectoryName(filePath));
                using (var fs = File.Create(filePath))
                {
                    var bytes = dataPattern.Item3;
                    if (bytes == null)
                    {
                        bytes = new byte[dataPattern.Item2];
                        m_Rng.GetBytes(bytes);
                    }
                    fs.Write(bytes, 0, bytes.Length);
                }
            }
            return dirPath;
        }

        private byte[] GetIncrementBytes(uint offset, int size)
        {
            var bytes = new byte[size];
            int wrote = 0;
            int i = 0;
            uint current = offset;
            while (wrote < size)
            {
                var data = BitConverter.GetBytes(current);
                Buffer.BlockCopy(data, 0, bytes, i * data.Length, data.Length);
                wrote += data.Length;
            }
            return bytes;
        }

        private void CreateTestApplication(TestEnvironment env, TestDataResourcePath resourcePath, string outputPath, int version,
            List<Tuple<string, int, byte[]>> dataPattern, List<Tuple<string, int, byte[]>> dataPatternHtml)
        {
            var codeSource = Path.Combine(env.SourceDir, "Code");
            var codeDir = Path.Combine(env.OutputDir, "code");
            SafeDeleteDirectory(codeDir);
            Directory.CreateDirectory(codeDir);
            foreach (var codeFile in Directory.GetFiles(codeSource, "*.*", SearchOption.AllDirectories))
            {
                File.Copy(codeFile, Path.Combine(codeDir, Path.GetFileName(codeFile)));
            }

            var dataDir = CreateDataDir(env.OutputDir, "data", dataPattern);
            var legalInformationZip = MakeLegalInfoZipfile(env.OutputDir);
            var meta = Path.Combine(env.OutputDir, "app.nmeta");

            var versionValue = version << 16;

            MakeNpdm(resourcePath.MakeMeta, Path.Combine(codeDir, "main.npdm"), resourcePath.Meta, resourcePath.Desc);
            MakeSpecifiedVersionMetaFile(resourcePath.Meta, meta, versionValue, m_ProgramId);
            File.Copy(resourcePath.Icon, Path.Combine(env.OutputDir, Path.GetFileName(resourcePath.Icon)), true);

            if (dataPatternHtml != null)
            {
                var dataHtmlDir = CreateDataDir(env.OutputDir, "data_html", dataPatternHtml);
                ExecuteProgram(resourcePath.AuthoringTool, string.Format($"creatensp -o {outputPath} --type Application --meta {meta} --desc {resourcePath.Desc} --program {codeDir} {dataDir} --legal-information {legalInformationZip} --html-document {dataHtmlDir + "\\docs"} --accessible-urls {dataHtmlDir + "\\accessibles"}"));
            }
            else
            {
                ExecuteProgram(resourcePath.AuthoringTool, string.Format($"creatensp -o {outputPath} --type Application --meta {meta} --desc {resourcePath.Desc} --program {codeDir} {dataDir} --legal-information {legalInformationZip}"));
            }
        }

        private void CreateTestPatch(TestEnvironment env, TestDataResourcePath resourcePath, string outputPath, string originalPath, string currentPath)
        {
            ExecuteProgram(resourcePath.AuthoringTool, string.Format($"makepatch -o {outputPath} --desc {resourcePath.Desc} --original {originalPath} --current {currentPath}"));
        }

        private void CreateTestAoc(TestEnvironment env, TestDataResourcePath resourcePath, string outputPath, List<Tuple<string, int, byte[]>> dataPatterns)
        {
            var dataAocDir = CreateDataDir(env.OutputDir, "data_aoc", dataPatterns);

            var aocMeta = Path.Combine(env.OutputDir, "aoc.nmeta");
            MakeAocMetaFile(resourcePath.Meta, aocMeta, 0, 1, "data_aoc", m_ProgramId);
            File.Copy(resourcePath.Icon, Path.Combine(env.OutputDir, Path.GetFileName(resourcePath.Icon)), true);

            ExecuteProgram(resourcePath.AuthoringTool, string.Format($"creatensp -o {outputPath} --type AddOnContent --meta {aocMeta} --data {dataAocDir}"));
        }

        private void PrepareTestDataImpl(TestEnvironment env, TestDataResourcePath resourcePath, string outputPath, string version)
        {
            SafeDeleteDirectory(outputPath);
            Directory.CreateDirectory(outputPath);

            var dataPattern0 = new List<Tuple<string, int, byte[]>>()
            {
                Tuple.Create<string, int, byte[]>("data0\\0.dat", 256 * 1024, GetIncrementBytes(0, 256 * 1024)),
                Tuple.Create<string, int, byte[]>("data0\\1.dat", 0, new byte[256 * 1024]),
                Tuple.Create<string, int, byte[]>("data0\\2.dat", 256 * 1024, GetIncrementBytes(256 * 1024, 256 * 1024)),
                Tuple.Create<string, int, byte[]>("data0\\3.dat", 256 * 1024, GetIncrementBytes(512 * 1024, 256 * 1024)),
            };

            var dataPattern1 = new List<Tuple<string, int, byte[]>>()
            {
                Tuple.Create<string, int, byte[]>("data0\\0.dat", 256 * 1024, GetIncrementBytes(0, 256 * 1024)),
                Tuple.Create<string, int, byte[]>("data0\\1.dat", 256 * 1024, null),
                Tuple.Create<string, int, byte[]>("data0\\2.dat", 256 * 1024, GetIncrementBytes(256 * 1024, 256 * 1024)),
                Tuple.Create<string, int, byte[]>("data0\\3.dat", 256 * 1024, GetIncrementBytes(512 * 1024, 256 * 1024)),
                Tuple.Create<string, int, byte[]>("data1\\0.dat", 512 * 1024, null),
            };

            var dataPatternHtml0 = new List<Tuple<string, int, byte[]>>()
            {
                Tuple.Create<string, int, byte[]>("docs\\index.html", 1014, null),
                Tuple.Create<string, int, byte[]>("docs\\style.css", 3436, null),
                Tuple.Create<string, int, byte[]>("accessibles\\accessible-urls.txt", 0, Encoding.ASCII.GetBytes(@"https://www.nintendo.co.jp
https://www.nintendo.co.jp/index.html&hoge=soge")),
            };

            var dataPatternHtml1 = new List<Tuple<string, int, byte[]>>()
            {
                Tuple.Create<string, int, byte[]>("docs\\index.html", 1014, null),
                Tuple.Create<string, int, byte[]>("docs\\style.css", 3436, null),
                Tuple.Create<string, int, byte[]>("accessibles\\accessible-urls.txt", 0, Encoding.ASCII.GetBytes(@"https://www.nintendo.co.jp
https://www.nintendo.co.jp/index.html&foo=var")),
            };

            var dataPatternAoc = new List<Tuple<string, int, byte[]>>()
            {
                Tuple.Create<string, int, byte[]>("aoc\\0.dat", 256 * 1024, null),
            };

            foreach (var useHtml in new bool[] { true, false })
            {
                // アプリケーションを作成
                CreateTestApplication(env, resourcePath, Path.Combine(outputPath, GetApplicationPath(version, 0, useHtml)), 0, dataPattern0, useHtml ? dataPatternHtml0 : null);
                CreateTestApplication(env, resourcePath, Path.Combine(outputPath, GetApplicationPath(version, 1, useHtml)), 1, dataPattern1, useHtml ? dataPatternHtml1 : null);
                // パッチを作成
                CreateTestPatch(env, resourcePath,
                    Path.Combine(outputPath, GetPatchPath(version, 1, useHtml)),
                    Path.Combine(outputPath, GetApplicationPath(version, 0, useHtml)),
                    Path.Combine(outputPath, GetApplicationPath(version, 1, useHtml)));
            }
            // AOC を作成
            CreateTestAoc(env, resourcePath, Path.Combine(outputPath, GetAddOnContentPath(version)), dataPatternAoc);

            // テスト用に残さないデータを削除
            File.Delete(Path.Combine(outputPath, GetApplicationPath(version, 1, false)));
            File.Delete(Path.Combine(outputPath, GetApplicationPath(version, 1, true)));
        }

        private void TestCommand(string cmd, TestEnvironment env)
        {
            var stdout = ExecuteProgram(env.ToolPath, cmd);
            // 失敗しなければ OK
            Assert.IsTrue(stdout.IndexOf("Failed") < 0, stdout);
        }

        private void TestListCompatibility(TestEnvironment env, string version)
        {
            var resourcePath = GetTestResourceDirectoryPath();
            foreach (var useHtml in new bool[] { true, false })
            {
                var app = Path.Combine(resourcePath, GetApplicationPath(version, 0, useHtml));
                var patch = Path.Combine(resourcePath, GetPatchPath(version, 1, useHtml));

                TestCommand(string.Format($"list {app}"), env);
                TestCommand(string.Format($"list {patch} --original {app}"), env);
            }
            var aoc = Path.Combine(resourcePath, GetAddOnContentPath(version));
            TestCommand(string.Format($"list {aoc}"), env);
        }

        private void TestExtractCompatibility(TestEnvironment env, string version)
        {
            var resourcePath = GetTestResourceDirectoryPath();
            foreach (var useHtml in new bool[] { true, false })
            {
                var app = Path.Combine(resourcePath, GetApplicationPath(version, 0, useHtml));
                var patch = Path.Combine(resourcePath, GetPatchPath(version, 1, useHtml));
                TestCommand(string.Format($"extract -o {env.OutputDir} {app}"), env);
                TestCommand(string.Format($"extract -o {env.OutputDir} --original {app} {patch}"), env);
            }
            var aoc = Path.Combine(resourcePath, GetAddOnContentPath(version));
            TestCommand(string.Format($"extract -o {env.OutputDir} {aoc}"), env);
        }

        private void TestGetPropertyCompatibility(TestEnvironment env, string version)
        {
            var resourcePath = GetTestResourceDirectoryPath();
            foreach (var useHtml in new bool[] { true, false })
            {
                var app = Path.Combine(resourcePath, GetApplicationPath(version, 0, useHtml));
                var patch = Path.Combine(resourcePath, GetPatchPath(version, 1, useHtml));

                TestCommand(string.Format($"getproperty {app}"), env);
                TestCommand(string.Format($"getproperty {patch}"), env);
            }
            var aoc = Path.Combine(resourcePath, GetAddOnContentPath(version));
            TestCommand(string.Format($"getproperty {aoc}"), env);
        }

        private void TestConvertAocCompatibility(TestEnvironment env, string version)
        {
            var resourcePath = GetTestResourceDirectoryPath();
            foreach (var useHtml in new bool [] {true, false})
            {
                var app = Path.Combine(resourcePath, GetApplicationPath(version, 0, useHtml));
                var aoc = Path.Combine(resourcePath, GetAddOnContentPath(version));

                TestCommand(string.Format($"convertaoc -o {env.OutputDir} --base {app} {aoc}"), env);
            }
        }

        private void TestMakePatchCompatibility(TestEnvironment env, string version)
        {
            var testPath = new TestPath(this.TestContext);
            var testResourcePath = new TestDataResourcePath(testPath.GetSigloRoot(), false);

            // 現バージョンのツールを利用してアプリを作成
            var dataPattern2 = new List<Tuple<string, int, byte[]>>()
            {
                Tuple.Create<string, int, byte[]>("data0\\0.dat", 256 * 1024, GetIncrementBytes(0, 256 * 1024)),
                Tuple.Create<string, int, byte[]>("data0\\1.dat", 256 * 1024, null),
                Tuple.Create<string, int, byte[]>("data0\\3.dat", 256 * 1024, GetIncrementBytes(512 * 1024, 256 * 1024)),
                Tuple.Create<string, int, byte[]>("data0\\4.dat", 256 * 1024, GetIncrementBytes(256 * 1024, 256 * 1024)),
            };

            var dataPatternHtml2 = new List<Tuple<string, int, byte[]>>()
            {
                Tuple.Create<string, int, byte[]>("docs\\index.html", 1014, null),
                Tuple.Create<string, int, byte[]>("docs\\style.css", 3436, null),
                Tuple.Create<string, int, byte[]>("accessibles\\accessible-urls.txt", 0, Encoding.ASCII.GetBytes(@"https://www.nintendo.co.jp
https://www.nintendo.co.jp/index.html&fuga=fuga")),
            };

            var currentVersion = "current";
            var patch = Path.Combine(env.OutputDir, "current_patch.nsp");
            var resourcePath = GetTestResourceDirectoryPath();

            foreach (var useHtmlCurrent in new bool[] { true, false })
            {
                var current = Path.Combine(env.OutputDir, GetApplicationPath(currentVersion, 2, useHtmlCurrent));
                CreateTestApplication(env, testResourcePath, current, 2, dataPattern2, useHtmlCurrent ? dataPatternHtml2 : null);

                foreach (var useHtml in new bool[] { true, false })
                {
                    var original = Path.Combine(resourcePath, GetApplicationPath(version, 0, useHtml));
                    var previous = Path.Combine(resourcePath, GetPatchPath(version, 1, useHtml));

                    // パッチを生成・展開
                    TestCommand(string.Format($"makepatch -o {patch} --original {original} --current {current} --previous {previous} --desc {testResourcePath.Desc}"), env);
                    TestCommand(string.Format($"extract -o {env.OutputDir}  --original {original} {patch}"), env);
                    TestCommand(string.Format($"makepatch -o {patch} --original {original} --current {current} --desc {testResourcePath.Desc}"), env);
                    TestCommand(string.Format($"extract -o {env.OutputDir}  --original {original} {patch}"), env);
                }
            }
        }

        // [TestMethod]
        public void PrepareTestData()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            const string PackageRootPath = "";
            const string OutputPath = "";
            var versions = new string[]
            {
            };

            foreach (var version in versions)
            {
                string rootPath = Path.Combine(PackageRootPath, version);
                PrepareTestDataImpl(env, new TestDataResourcePath(rootPath, true), OutputPath, version);
            }

            SafeDeleteDirectory(env.OutputDir);
        }

        [TestMethod]
        public void TestExecutionCompatibility()
        {
            var env = new TestEnvironment(new TestPath(this.TestContext), MethodBase.GetCurrentMethod().Name);

            var testResourceDir = GetTestResourceDirectoryPath();
            if (!Directory.Exists(testResourceDir))
            {
                Console.WriteLine(string.Format($"{testResourceDir} is not found. Skipped {MethodBase.GetCurrentMethod().Name}."));
                return;
            }

            // バージョンごとにテストを実施
            var subdires = Directory.GetDirectories(testResourceDir);
            foreach (var subdir in subdires.Where(s => !s.EndsWith(".svn")))
            {
                var version = Path.GetFileName(subdir);

                Console.WriteLine("Version: " + version);

                // list
                TestListCompatibility(env, version);
                // extract
                TestExtractCompatibility(env, version);
                // getproperty
                TestGetPropertyCompatibility(env, version);
                // convertaoc
                TestConvertAocCompatibility(env, version);
                // makepatch
                TestMakePatchCompatibility(env, version);
            }

            SafeDeleteDirectory(env.OutputDir);
        }
    }
}
