﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Text.RegularExpressions;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.ApplicationControlProperty;
using Nintendo.Authoring.AuthoringLibrary;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace AuthoringToolsTest
{
    using UnpublishableErrorExpectType = Tuple<string, string, string[]>;

    // 下記のチェック項目は単体テストは実施しない
    // SbmChk-41b(テストデータの動的作成が困難なので TestExecutionErrorUnpublishableLegalinfoHashCheck にて実データを使ってテストする)
    // SbmChk-1_9b(テストデータの動的作成が困難なので TestExecutionErrorUnpublishableIconCheck にて実データを使ってテストする)
    // SbmChk-1_7(テストデータの動的作成が困難なので TestExecutionErrorUnpublishableIconCheck にて実データを使ってテストする)

    // １．UnpublishableErrorCheckのテスト
    [TestClass]
    public class UnpublishableErrorCheckTest
    {
        private void CheckUnpublishableErrorContent(string expectApplicationId, string expectApplicationType, int expectErrorCount, List<string> expectErrorTypeList, List<string> expectErrorIdList, List<string[]> expectResourceStringList, UnpublishableErrorContentModel actualErrorModel)
        {
            if (expectErrorCount == 0)
            {
                // エラーが発生していない場合のチェック
                // アプリケーション情報のチェック
                Assert.AreEqual(expectApplicationId, actualErrorModel.Id);
                Assert.AreEqual(expectApplicationType, actualErrorModel.Type);

                // エラー情報のチェック
                // エラーではないエントリが1つ登録されている
                Assert.AreEqual(1, actualErrorModel.ErrorList.Count());
                UnpublishableErrorDetailModel error = actualErrorModel.ErrorList[0];
                Assert.AreEqual(error.Type, null);
                Assert.AreEqual(error.Message, null);

                return;
            }

            Assert.AreEqual(expectApplicationId, actualErrorModel.Id);
            Assert.AreEqual(expectApplicationType, actualErrorModel.Type);

            Regex replaceRegex = new Regex(@"([a-zA-Z0-9]+) ([0-9]+)-([0-9]+)");

            // エラー情報のチェック
            Assert.AreEqual(expectErrorCount, actualErrorModel.ErrorList.Count());
            for (int i = 0; i < actualErrorModel.ErrorList.Count(); i++)
            {
                var resourceId = replaceRegex.Replace(expectErrorIdList[i], @"$1_$2_$3");
                var error = actualErrorModel.ErrorList[i];
                Assert.AreEqual(expectErrorTypeList[i], error.Type);
                Assert.AreEqual(expectErrorIdList[i], error.Message.Id);
                Assert.AreEqual(
                    string.Format(
                        AuthoringTool.Resources.ResourceManager.GetString(
                            resourceId + "_Title", CultureInfo.GetCultureInfo("ja")), expectResourceStringList[i]),
                                error.Message.Title.Japanese);
                Assert.AreEqual(
                    string.Format(
                        AuthoringTool.Resources.ResourceManager.GetString(
                            resourceId + "_Description", CultureInfo.GetCultureInfo("ja")), expectResourceStringList[i]),
                                error.Message.Description.Japanese);
                Assert.AreEqual(
                    string.Format(
                        AuthoringTool.Resources.ResourceManager.GetString(
                            resourceId + "_Title", CultureInfo.GetCultureInfo("en")), expectResourceStringList[i]),
                                error.Message.Title.English);
                Assert.AreEqual(
                    string.Format(
                        AuthoringTool.Resources.ResourceManager.GetString(
                            resourceId + "_Description", CultureInfo.GetCultureInfo("en")), expectResourceStringList[i]),
                                error.Message.Description.English);
            }
        }

        private void CheckUnpublishableErrorContent(string expectApplicationId, string expectApplicationType, int expectErrorCount, List<string> expectErrorTypeList, List<string> expectErrorIdList, List<string> expectResourceStringList, UnpublishableErrorContentModel actualErrorModel)
        {
            var resourceStringList = expectResourceStringList.Select(entry => { return new string[] { entry }; }).ToList();
            CheckUnpublishableErrorContent(expectApplicationId, expectApplicationType, expectErrorCount, expectErrorTypeList, expectErrorIdList, resourceStringList, actualErrorModel);
        }
        private void CheckUnpublishableErrorCommon<T>(UnpublishableErrorCheckData checkData,
                                                      List<Tuple<bool, T>> testPatternList,
                                                      Action<UnpublishableErrorCheckData, Tuple<bool, T>> testSettingFunction,
                                                      Func<UnpublishableErrorCheckData, Tuple<bool, T>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator)
        {
            CheckUnpublishableErrorCommon(checkData, testPatternList, testSettingFunction, expectUnpublishableErrorModelGenerator, null);
        }
        private void CheckUnpublishableErrorCommon<T>(UnpublishableErrorCheckData checkData,
                                                      List<Tuple<bool, T>> testPatternList,
                                                      Action<UnpublishableErrorCheckData, Tuple<bool, T>> testSettingFunction,
                                                      Func<UnpublishableErrorCheckData, Tuple<bool, T>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator,
                                                      List<string> ignoreIdList)
        {
            // テスト実施
            foreach (var testPattern in testPatternList)
            {
                testSettingFunction(checkData, testPattern);

                // UnpublishableErrorエラーの解析
                var unpublishableErrorModel = UnpublishableError.GetList(checkData);
                if (ignoreIdList != null)
                {
                    // 無視するエラーが指定されている場合は、エラーを削除する
                    for (int i = 0; i < unpublishableErrorModel.Contents.Count(); i++)
                    {
                        unpublishableErrorModel.Contents[i].ErrorList.RemoveAll(entry => entry.Message != null && ignoreIdList.Contains(entry.Message.Id));
                        if (unpublishableErrorModel.Contents[i].ErrorList.Count() == 0)
                        {
                            // エラーが存在しなし場合は、エラー要因無しのモデルが一つ追加されることを期待しているので
                            // 無視するエラー削除した結果エラーが0の場合は、エラー要因無しのモデルを追加しておく。
                            unpublishableErrorModel.Contents[i].ErrorList.Add(UnpublishableErrorDetailModel.Create(null, null));
                        }
                    }
                }

                // UnpublishableErrorエラーの発生数を取得する
                int errorCount = 0;
                int warningCount = 0;
                UnpublishableError.GetUnpublishableErrorCount(out errorCount, out warningCount, unpublishableErrorModel);

                // テスト結果判定
                List<UnpublishableErrorExpectType> expectList;
                if (testPattern.Item1 == true)
                {
                    // 成功
                    expectList = new List<UnpublishableErrorExpectType>();
                }
                else
                {
                    // 失敗
                    expectList = expectUnpublishableErrorModelGenerator(checkData, testPattern);
                }

                // エラー数のチェック
                Assert.AreEqual(expectList.FindAll(entry => entry.Item1 == "Error").Count(), errorCount);
                Assert.AreEqual(expectList.FindAll(entry => entry.Item1 == "Warning").Count(), warningCount);

                var id = string.Empty;
                var type = string.Empty;
                if (checkData.Nsp.ContentMeta != null)
                {
                    // チェック対象がnsp
                    id = checkData.Nsp.ContentMeta.Id;
                    type = checkData.Nsp.ContentMeta.Type;
                }
                else if (checkData.Nmeta.AddOnContent != null)
                {
                    // チェック対象がnmeta(Aoc)
                    id = checkData.Nmeta.AddOnContent.Id;
                    type = NintendoContentMetaConstant.ContentMetaTypeAddOnContent;
                }
                else
                {
                    id = string.Empty;
                    type = checkData.CheckContentType;
                }

                // xmlモデルのチェック
                Assert.AreEqual(1, unpublishableErrorModel.Contents.Count());
                CheckUnpublishableErrorContent(id, type, errorCount + warningCount,
                    expectList.Select(entry => entry.Item1).ToList(),
                        expectList.Select(entry => entry.Item2).ToList(),
                            expectList.Select(entry => entry.Item3).ToList(), unpublishableErrorModel.Contents.Single());
            }
        }

        private UnpublishableErrorCheckData CreateSimpleSuccessData()
        {
            var checkData = new UnpublishableErrorCheckData();
            checkData.CheckTarget = UnpublishableErrorCheckData.CheckTargetType.Nmeta;
            checkData.CheckContentType = NintendoContentMetaConstant.ContentMetaTypeApplication;

            checkData.Nsp.ContentMeta = new ContentMetaModel();
            System.Random rand = new System.Random();
            checkData.Nsp.ContentMeta.Id = string.Format("0x{0}", rand.Next());
            checkData.Nsp.ContentMeta.Type = NintendoContentMetaConstant.ContentMetaTypeApplication;
            var contentModelList = new List<ContentModel>();
            // ContentMetaにProgramが登録されている
            contentModelList.Add(new ContentModel() { Type = NintendoContentMetaConstant.ContentTypeProgram });
            // ContentMetaにLegalInformationが登録されている
            contentModelList.Add(new ContentModel() { Type = NintendoContentMetaConstant.ContentTypeLegalInformation });
            checkData.Nsp.ContentMeta.ContentList = contentModelList;

            checkData.Nsp.ProgramInfo = new ProgramInfoModel();
            // SdkVersionとToolVersionが同じ
            checkData.Nsp.ProgramInfo.SdkVersion = "1_2_3";
            checkData.Nsp.ProgramInfo.ToolVersion = checkData.Nsp.ProgramInfo.SdkVersion + ":4_5678";

            // ビルドタイプがRelease
            checkData.Nsp.ProgramInfo.BuildType = "Release";

            // 適当なDescを設定
            checkData.Nsp.ProgramInfo.Desc = "XXXXXXXXXX";
            checkData.Nsp.ProgramInfo.DescFlags = new DescFlags() { Production = true, UnqualifiedApproval = true };

            // DebugApiが空
            checkData.Nsp.ProgramInfo.DebugApiListData = DebugApiList.Create();

            // PrivateApiが空
            checkData.Nsp.ProgramInfo.PrivateApiListData = PrivateApiList.Create();

            // Middlewareが空
            checkData.Nsp.ProgramInfo.MiddlewareListData = MiddlewareList.Create();
            checkData.Nsp.AllMiddlewareListInNsp = checkData.Nsp.ProgramInfo.MiddlewareListData.GetModuleNameList();

            // UnresolvedApiが空
            checkData.Nsp.ProgramInfo.UnresolvedApiListData = UnresolvedApiList.Create();

            // FsAccessControlData
            List<SaveDataOwnerIdsModel> saveDataOwnerIdsList = new List<SaveDataOwnerIdsModel>();
            checkData.Nsp.ProgramInfo.FsAccessControlData = new FsAccessControlData() { Entries = saveDataOwnerIdsList };

            // LegalInformationとContentMetaのアプリケーションIDが同じ
            checkData.Nsp.LegalInformation = new LegalInformationModel();
            // LegalInformationのフォーマットバージョンが2.x.xより古い
            checkData.Nsp.LegalInformation.FormatVersion = "1.2.3";
            checkData.Nsp.LegalInformation.ApplicationId = checkData.Nsp.ContentMeta.Id;
            // LegalInformationのTechnologyが空
            var declarationRequiredNotationsTechnologyNameList = new List<string>() { };
            checkData.Nsp.LegalInformation.DeclarationRequiredNotations = new LegalInformationDeclarationRequiredNotationsModel();
            checkData.Nsp.LegalInformation.DeclarationRequiredNotations.TechnologyName = declarationRequiredNotationsTechnologyNameList;
            var copyrightNotationsTechnologyNameList = new List<string>();
            checkData.Nsp.LegalInformation.CopyrightNotations = new LegalInformationCopyrightNotationsModel();
            checkData.Nsp.LegalInformation.CopyrightNotations.TechnologyName = copyrightNotationsTechnologyNameList;

            checkData.ApplicationControlProperty = new ApplicationControlPropertyModel();
            checkData.ApplicationControlProperty.Title = new List<Title>
            {
                new Title { Language = "AmericanEnglish", Name = "Name", Publisher = "Publisher" }
            };
            SetSuccessIconData(checkData);
            checkData.ApplicationControlProperty.SupportedLanguage = new List<string>
            {
                "AmericanEnglish",
            };
            checkData.ApplicationControlProperty.DisplayVersion = "1.0.0";
            checkData.ApplicationControlProperty.UserAccountSaveDataSize = "0x0000000000400000";
            checkData.ApplicationControlProperty.UserAccountSaveDataSizeMax = "0x0000000000000000";
            checkData.ApplicationControlProperty.UserAccountSaveDataJournalSize = "0x0000000000100000";
            checkData.ApplicationControlProperty.UserAccountSaveDataJournalSizeMax = "0x0000000000000000";
            checkData.ApplicationControlProperty.DeviceSaveDataSize = "0x0000000000000000";
            checkData.ApplicationControlProperty.DeviceSaveDataSizeMax = "0x0000000000000000";
            checkData.ApplicationControlProperty.DeviceSaveDataJournalSize = "0x0000000000000000";
            checkData.ApplicationControlProperty.DeviceSaveDataJournalSizeMax = "0x0000000000000000";
            checkData.ApplicationControlProperty.TemporaryStorageSize = "0x0000000000000000";
            checkData.ApplicationControlProperty.CacheStorageSize = "0x0000000000000000";
            checkData.ApplicationControlProperty.CacheStorageJournalSize = "0x0000000000000000";

            var properties = new NintendoContentMetaProperty[] { };
            checkData.Nsp.ContentMetaPropertyList = new NintendoContentMetaPropertyList() { Properties = properties };
            return checkData;
        }

        private void SetSuccessIconData(UnpublishableErrorCheckData checkData)
        {
            // Titleと同じ言語のIconデータを登録
            var icon = checkData.ApplicationControlProperty.Title.Select(entry => new Icon() { Language = entry.Language, IconPath = new DataPath() { Path = "icon.bmp" } }).ToList();
            checkData.ApplicationControlProperty.Icon = icon;
        }

        // エラーが発生しない
        [TestMethod]
        public void UnpublishableErrorSuccessTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            // ビルドタイプがRelease
            checkData.Nsp.ProgramInfo.BuildType = "Release";
            // DebugApiが空
            checkData.Nsp.ProgramInfo.DebugApiListData = DebugApiList.Create();
            // PrivateApiに情報が登録されている
            ModuleInfo[] privateApiInfos = new ModuleInfo[]
            {
                new ModuleInfo(ModuleInfoType.PrivateApiInfo, "testModule", "testVendor", "testFile"),
            };
            checkData.Nsp.ProgramInfo.PrivateApiListData = PrivateApiList.Create(privateApiInfos);
            // UnresolvedApiに使用禁止ではないAPIが登録されている。
            UndefSymbolInfo[] unresolvedApiModuleInfos = new UndefSymbolInfo[1]
            {
                new UndefSymbolInfo(UnpublishableApiList.UnpublishableDebugApiTable.First().Key + ".NG", "test.nso"),
            };
            checkData.Nsp.ProgramInfo.UnresolvedApiListData = UnresolvedApiList.Create(unresolvedApiModuleInfos);
            // Middlewareに情報が登録されている
            ModuleInfo[] middlewareInfos = new ModuleInfo[]
            {
                // LegalInfoXMLのCopyrightNotations・DeclarationRequiredNotationsに存在しない
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "testMiddlewareInfosOnly_0_0", "testVendor", "testFile"),
                // LegalInfoXMLのCopyrightNotations・DeclarationRequiredNotationsに存在する
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "testAllowTechnology_0_0", "testVendor", "testFile"),
            };
            checkData.Nsp.ProgramInfo.MiddlewareListData = MiddlewareList.Create(middlewareInfos);
            checkData.Nsp.AllMiddlewareListInNsp = checkData.Nsp.ProgramInfo.MiddlewareListData.GetModuleNameList();

            // LegalInfoXMLのDeclarationRequiredNotationsの文字列が含まれるMiddlewareが存在する
            var declarationRequiredNotationsTechnologyNameList = new List<string>()
            {
                "testAllowTechnology",
                // LegalInfoXMLのDeclarationRequiredNotationsにのみ存在する
                "testDeclarationRequiredNotationsOnly",
            };
            checkData.Nsp.LegalInformation.DeclarationRequiredNotations.TechnologyName = declarationRequiredNotationsTechnologyNameList;

            // LegalInfoXMLのCopyrightNotationsにDeclarationRequiredNotationsと同じ情報が存在する
            var copyrightNotationsTechnologyNameList = new List<string>()
            {
                "testAllowTechnology",
                // LegalInfoXMLのCopyrightNotationsにのみ存在する
                "testCopyrightNotationsNameOnly",
            };
            checkData.Nsp.LegalInformation.CopyrightNotations.TechnologyName = copyrightNotationsTechnologyNameList;

            // テスト実施
            // UnpublishableErrorエラーの解析
            var unpublishableErrorModel = UnpublishableError.GetList(checkData);
            // UnpublishableErrorエラーの発生数を取得する
            int errorCount = 0;
            int warningCount = 0;
            UnpublishableError.GetUnpublishableErrorCount(out errorCount, out warningCount, unpublishableErrorModel);

            // UnpublishableErrorエラー情報を表示する (例外が発生しなければ問題なしとする)
            UnpublishableError.ShowUnpublishableError(unpublishableErrorModel, true);
            UnpublishableError.ShowUnpublishableError(unpublishableErrorModel, false);

            // テスト結果判定
            // エラー数のチェック
            Assert.AreEqual(0, errorCount);
            Assert.AreEqual(0, warningCount);
            // xmlモデルのチェック
            Assert.AreEqual(1, unpublishableErrorModel.Contents.Count());
            CheckUnpublishableErrorContent(checkData.Nsp.ContentMeta.Id, checkData.Nsp.ContentMeta.Type,
                0, null, null, (List<string[]>)null, unpublishableErrorModel.Contents.Single());
        }

        // SbmChk-23bでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk23bTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            // DebugApiが複数登録されている
            System.Random rand = new System.Random();
            ModuleInfo[] moduleInfos = new ModuleInfo[]
            {
                new ModuleInfo( ModuleInfoType.DebugApiInfo,
                    string.Format("module{0}", rand.Next()),
                    string.Format("vendor{0}", rand.Next()),
                    string.Format("file{0}", rand.Next())),
                new ModuleInfo( ModuleInfoType.DebugApiInfo,
                    string.Format("module{0}", rand.Next()),
                    string.Format("vendor{0}", rand.Next()),
                    string.Format("file{0}", rand.Next())),
                new ModuleInfo( ModuleInfoType.DebugApiInfo,
                    string.Format("NEX-DebugLoginDirect"),
                    string.Format("vendor{0}", rand.Next()),
                    string.Format("file{0}", rand.Next())),
            };
            checkData.Nsp.ProgramInfo.DebugApiListData = DebugApiList.Create(moduleInfos);

            // テスト実施
            // UnpublishableErrorエラーの解析
            var unpublishableErrorModel = UnpublishableError.GetList(checkData);
            // UnpublishableErrorエラーの発生数を取得する
            int errorCount = 0;
            int warningCount = 0;
            UnpublishableError.GetUnpublishableErrorCount(out errorCount, out warningCount, unpublishableErrorModel);

            // UnpublishableErrorエラー情報を表示する (例外が発生しなければ問題なしとする)
            UnpublishableError.ShowUnpublishableError(unpublishableErrorModel, true);
            UnpublishableError.ShowUnpublishableError(unpublishableErrorModel, false);

            // テスト結果判定
            // エラーとなるexpectApiName（期待値）
            List<string> expectApiNameList = moduleInfos.Select(entry =>
            {
                if (entry.ModuleName == "NEX-DebugLoginDirect")
                {
                    // 名前が"NEX-DebugLoginDirect"の場合は"nn::nex::NgsFacade::DebugLoginDirect"が表示される
                    return "nn::nex::NgsFacade::DebugLoginDirect";
                }
                else
                {
                    return entry.ModuleName;
                }
            }).ToList();

            // 表示するAPI名を改行区切りで結合する
            var expectApiName = new List<string>() { string.Join(Environment.NewLine, expectApiNameList) };

            // エラー数のチェック
            Assert.AreEqual(expectApiName.Count(), errorCount);
            Assert.AreEqual(0, warningCount);
            // xmlモデルのチェック
            Assert.AreEqual(1, unpublishableErrorModel.Contents.Count());
            CheckUnpublishableErrorContent(checkData.Nsp.ContentMeta.Id, checkData.Nsp.ContentMeta.Type, errorCount,
                Enumerable.Repeat<string>("Error", errorCount).ToList(), Enumerable.Repeat<string>("Issue 11-004", errorCount).ToList(),
                    expectApiName, unpublishableErrorModel.Contents.Single());
        }

        // ３．SbmChk-23cでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk23cTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            // UnresolvedApiに使用禁止APIと使用可能APIが交互に登録されている
            System.Random rand = new System.Random();
            UndefSymbolInfo[] unresolvedApiModuleInfos = new UndefSymbolInfo[]
            {
                new UndefSymbolInfo( UnpublishableApiList.UnpublishableDebugApiTable.First().Key + ".NG",
                    string.Format("{0}.nso", rand.Next())),
                new UndefSymbolInfo( UnpublishableApiList.UnpublishableDebugApiTable.First().Key,
                    string.Format("{0}.nso", rand.Next())),
                new UndefSymbolInfo( UnpublishableApiList.UnpublishableDebugApiTable.First().Key + ".NG",
                    string.Format("{0}.nso", rand.Next())),
                new UndefSymbolInfo( UnpublishableApiList.UnpublishableDebugApiTable.Last().Key,
                    string.Format("{0}.nso", rand.Next())),
                new UndefSymbolInfo( UnpublishableApiList.UnpublishableDebugApiTable.Last().Key + ".NG",
                    string.Format("{0}.nso", rand.Next())),
            };
            checkData.Nsp.ProgramInfo.UnresolvedApiListData = UnresolvedApiList.Create(unresolvedApiModuleInfos);

            // テスト実施
            // UnpublishableErrorエラーの解析
            var unpublishableErrorModel = UnpublishableError.GetList(checkData);
            // UnpublishableErrorエラーの発生数を取得する
            int errorCount = 0;
            int warningCount = 0;
            UnpublishableError.GetUnpublishableErrorCount(out errorCount, out warningCount, unpublishableErrorModel);

            // UnpublishableErrorエラー情報を表示する (例外が発生しなければ問題なしとする)
            UnpublishableError.ShowUnpublishableError(unpublishableErrorModel, true);
            UnpublishableError.ShowUnpublishableError(unpublishableErrorModel, false);

            // テスト結果判定
            // マングリング前のAPI名（期待値）
            // 表示するAPI名を改行区切りで結合する
            string[] expectApiName =
            {
                UnpublishableApiList.UnpublishableDebugApiTable.First().Value
                + Environment.NewLine
                + UnpublishableApiList.UnpublishableDebugApiTable.Last().Value
            };
            // エラー数のチェック
            Assert.AreEqual(expectApiName.Count(), errorCount);
            Assert.AreEqual(0, warningCount);
            // xmlモデルのチェック
            Assert.AreEqual(1, unpublishableErrorModel.Contents.Count());
            CheckUnpublishableErrorContent(checkData.Nsp.ContentMeta.Id, checkData.Nsp.ContentMeta.Type, errorCount,
                Enumerable.Repeat<string>("Error", errorCount).ToList(), Enumerable.Repeat<string>("Issue 11-006", errorCount).ToList(),
                    expectApiName.ToList(), unpublishableErrorModel.Contents.Single());
        }

        // 以下の項目でエラーが発生するケースが混在する
        // SbmChk-23b
        // SbmChk-23c
        // SbmChk-26a
        // SbmChk-26b
        // SbmChk-41a
        // SbmChk-21
        // SbmChk-42
        // SbmChk-33d
        // SbmChk-63
        // SbmChk-11
        // SbmChk-12
        // SbmChk-12-1
        // SbmChk-12-6
        // SbmChk-12-7
        // SbmChk-13
        // SbmChk-15
        // SbmChk-1-8
        // SbmChk-52b
        // SbmChk-52c
        // SbmChk-54
        // SbmChk-22
        // SbmChk-8b～k
        // SbmChk-16
        // SbmChk-17a
        // SbmChk-55
        // SbmChk-56
        // SbmChk-57
        // SbmChk-58
        // SbmChk-39-1
        // 下記のチェック項目は混在しない
        // SbmChk-28 (SbmChk-26a,SbmChk-26b,SbmChk-41aと同時にNGにすることはできない)
        // SbmChk-9c,SbmChk-12-2,12-3,12-4,12-5 (SbmChk-11,12,15エラー用にTitleとIconを登録しないため同時にNGにすることはできない)
        // SbmChk-9a (SbmChk-11,12,15エラー用にTitleを登録しないため同時にNGにすることはできない)
        // SbmChk-52d (SbmChk-52b,SbmChk-52cと同時にNGにすることはできない)
        // SbmChk-27a,SbmChk-50b,SbmChk-38a,SbmChk-38b (他のテストで複合テストを実施しているのでここには含めない)
        // SbmChk-29b,SbmChk-50a,SbmChk-43b (同じ条件でもパッチ、アプリによってエラー内容が変わるためここには含めない)
        // SbmChk-1-10 (programinfoがないとその他のテストが実施できない)
        // SbmChk-39-2,SbmChk-39-3,SbmChk-61(SbmChk-39-1と同時にNGにすることはできない)
        [TestMethod]
        public void UnpublishableErrorMixTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            // ビルドタイプがReleaseではない(SbmChk-21エラー)
            checkData.Nsp.ProgramInfo.BuildType = "Debug";

            // SDKバージョンとToolバージョンのメジャーバージョン一致しない(SbmChk-42エラー)
            checkData.Nsp.ProgramInfo.SdkVersion = "1_2_3";
            // Toolバージョンの最後が99(SbmChk-33dエラー)
            checkData.Nsp.ProgramInfo.ToolVersion = "2_2_99:1_2_3";

            // ContentMetaとLegalInformationのアプリケーションIDが異なる(SbmChk-41aエラー)
            checkData.Nsp.ContentMeta.Id = "0x0100000000001000";
            checkData.Nsp.LegalInformation.ApplicationId = "0x0100000000000000";

            // DebugApiが登録されている
            System.Random rand = new System.Random();
            ModuleInfo[] debugApiInfos = new ModuleInfo[]
            {
                new ModuleInfo( ModuleInfoType.DebugApiInfo,
                    string.Format("module{0}", rand.Next()),
                    string.Format("vendor{0}", rand.Next()),
                    string.Format("file{0}", rand.Next())),
            };
            checkData.Nsp.ProgramInfo.DebugApiListData = DebugApiList.Create(debugApiInfos);
            // UnresolvedApiに使用禁止APIが登録されている (SbmChk-23c,SbmChk-54,SbmChk-58エラー)
            UndefSymbolInfo[] unresolvedApiModuleInfos = new UndefSymbolInfo[]
            {
                new UndefSymbolInfo( UnpublishableApiList.UnpublishableDebugApiTable.First().Key,
                    string.Format("{0}.nso", rand.Next())),
                new UndefSymbolInfo( UnpublishableApiList.UnpublishableXpadApiTable.First().Key,
                    string.Format("{0}.nso", rand.Next())),
                new UndefSymbolInfo( "_ZN2nn8irsensor20RunPointingProcessorERKNS0_14IrCameraHandleE",
                    string.Format("{0}.nso", rand.Next())),
            };
            checkData.Nsp.ProgramInfo.UnresolvedApiListData = UnresolvedApiList.Create(unresolvedApiModuleInfos);
            // マングリング前のAPI名を期待値とする
            string expectUnpublishableDebugApiName = UnpublishableApiList.UnpublishableDebugApiTable.First().Value;
            string expectUnpublishableXpadApiName = UnpublishableApiList.UnpublishableXpadApiTable.First().Value;
            // Middlewareに情報が登録されている
            ModuleInfo[] middlewareInfos = new ModuleInfo[]
            {
                // LegalInfoXMLのDeclarationRequiredNotationsに前方一致するものが存在するがCopyrightNotationsに存在しない(SbmChk-26aエラー)
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "testNothingTechnologyName_0_0_0", "testVendor", "testFile"),
                // モジュール名に"Debug"が含まれている(SbmChk-35エラー)
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "testDebugModule", "testVendor", "testFile"),
                // モジュール名に"PiaChat"が含まれている(SbmChk-55エラー)
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "testPiaChatModule_1_2_3", "testVendor", "testFile"),
                // モジュール名に"NEX"の4_3_0以下が含まれている(SbmChk-63エラー)
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "NEX-4_3_0-", "testVendor", "testFile"),
            };
            checkData.Nsp.ProgramInfo.MiddlewareListData = MiddlewareList.Create(middlewareInfos);
            checkData.Nsp.AllMiddlewareListInNsp = checkData.Nsp.ProgramInfo.MiddlewareListData.GetModuleNameList();
            var declarationRequiredNotationsTechnologyNameList = new List<string>()
            {
                "testNothingTechnologyName",
                // LegalInfoXMLのCopyrightNotationsとDeclarationRequiredNotationsに存在するがMiddlewareに存在しない(SbmChk-26bエラー)
                "testNothingMiddleware",
            };
            checkData.Nsp.LegalInformation.DeclarationRequiredNotations.TechnologyName = declarationRequiredNotationsTechnologyNameList;
            var copyrightNotationsTechnologyNameList = new List<string>()
            {
                "testNothingMiddleware",
            };
            checkData.Nsp.LegalInformation.CopyrightNotations.TechnologyName = copyrightNotationsTechnologyNameList;
            var unresolvedApiEntry = new UnresolvedApiModel();
            unresolvedApiEntry.ApiName = "_ZN2nn3hid21InitializeTouchScreenEv";
            unresolvedApiEntry.NsoName = string.Format("{0}.nso", rand.Next());
            checkData.Nsp.ProgramInfo.UnresolvedApiListData.Entries.Add(unresolvedApiEntry);
            // ApplicationControlPropertyXMLのタイトルエントリがnull(SbmChk-11,SbmChk-12エラー)
            checkData.ApplicationControlProperty.Title = null;
            // ApplicationControlPropertyXMLのIconエントリがnull(SbmChk-1-8エラー)
            checkData.ApplicationControlProperty.Icon = null;
            // ApplicationControlPropertyXMLの表示バージョンが空(SbmChk-13エラー)
            checkData.ApplicationControlProperty.DisplayVersion = string.Empty;
            // ApplicationControlPropertyXMLのSupportedLanguageが存在しない(SbmChk-15エラー)
            checkData.ApplicationControlProperty.SupportedLanguage = null;
            // ApplicationControlPropertyXMLのHDCPがNoneではない(SbmChk-56エラー)
            checkData.ApplicationControlProperty.Hdcp = "Required";

            // BcatPassphraseが設定されていて且つ、SDKのメジャーバージョンが0か1(SbmChk-52bエラー)
            checkData.ApplicationControlProperty.BcatPassphrase = "test";
            // BcatDeliveryCacheStorageSizeが64MiBより大きい(SbmChk-52cエラー)
            checkData.ApplicationControlProperty.BcatDeliveryCacheStorageSize = "0x4000001";

            // SDKのメジャーバージョンが1の場合に使用禁止のPrivateAPIが登録されている
            var privateApiEntry = new UnresolvedApiModel();
            unresolvedApiEntry.ApiName = "_ZN2nn2oe18ReportUserIsActiveEv";
            unresolvedApiEntry.NsoName = string.Format("{0}.nso", rand.Next());
            checkData.Nsp.ProgramInfo.UnresolvedApiListData.Entries.Add(unresolvedApiEntry);

            // Isbnにascii以外の文字が設定されている(SbmChk-12-6エラー)
            checkData.ApplicationControlProperty.Isbn = "１２３";

            // ApplicationErrorCodeCategoryの文字数が制限(7文字)を超える(SbmChk-12-7エラー)
            checkData.ApplicationControlProperty.ApplicationErrorCodeCategory = "12345678";

            // 0が設定されるべきタグに0以外が設定されている(SbmChk-8b～kエラー)
            checkData.ApplicationControlProperty.UserAccountSaveDataSizeMax = "0x0000000000000001";
            checkData.ApplicationControlProperty.UserAccountSaveDataJournalSizeMax = "0x0000000000000001";
            checkData.ApplicationControlProperty.DeviceSaveDataSize = "0x0000000000000001";
            checkData.ApplicationControlProperty.DeviceSaveDataSizeMax = "0x0000000000000001";
            checkData.ApplicationControlProperty.DeviceSaveDataJournalSize = "0x0000000000000001";
            checkData.ApplicationControlProperty.DeviceSaveDataJournalSizeMax = "0x1000000000000000";
            checkData.ApplicationControlProperty.TemporaryStorageSize = "0x1000000000000000";
            checkData.ApplicationControlProperty.CacheStorageSize = "0x1000000000000000";
            checkData.ApplicationControlProperty.CacheStorageJournalSize = "0xFFFFFFFFFFFFFFFF";
            checkData.ApplicationControlProperty.CacheStorageDataAndJournalSizeMax = "0x0000000003F00000";
            checkData.ApplicationControlProperty.CacheStorageIndexMax = "0x100";

            // PresenceGroupIdとApplicationIdが異なる(SbmChk-16エラー)
            checkData.ApplicationControlProperty.PresenceGroupId = "0x0100000000000000";
            // ローカル通信識別子とApplicationIdが異なる(SbmChk-17a)
            checkData.ApplicationControlProperty.LocalCommunicationId = new List<string>() { "0x0100000000000001", "0x0100000000000002" };

            // CardSpec/LaunchFlagsが0ではない(SbmChk-57エラー)
            checkData.CardSpec = new CardSpecModel();
            checkData.CardSpec.LaunchFlags = 1;

            // テストするタイプ
            string[] testType =
            {
                NintendoContentMetaConstant.ContentMetaTypeApplication,
                NintendoContentMetaConstant.ContentMetaTypePatch,
                NintendoContentMetaConstant.ContentMetaTypeAddOnContent
            };

            foreach (var type in testType)
            {
                checkData.CheckContentType = type;
                string applicationId = checkData.Nsp.ContentMeta.Id;
                if (type == NintendoContentMetaConstant.ContentMetaTypePatch)
                {
                    var patchMeta = new PatchContentMetaModel();
                    patchMeta.Id = string.Format("0x{0}", rand.Next());
                    patchMeta.ApplicationId = applicationId;
                    var contentModelList = new List<ContentModel>();
                    // ContentMetaにProgramが登録されている
                    contentModelList.Add(new ContentModel() { Type = NintendoContentMetaConstant.ContentTypeProgram });
                    // ContentMetaにLegalInformationが登録されている
                    contentModelList.Add(new ContentModel() { Type = NintendoContentMetaConstant.ContentTypeLegalInformation });
                    patchMeta.ContentList = contentModelList;
                    checkData.Nsp.ContentMeta = patchMeta;
                }
                else if (type == NintendoContentMetaConstant.ContentMetaTypeAddOnContent)
                {
                    var aocMeta = new AddOnContentContentMetaModel();
                    aocMeta.Id = string.Format("0x{0}", rand.Next());
                    aocMeta.ApplicationId = applicationId;
                    aocMeta.ContentList = new List<ContentModel>() { new ContentModel() };
                    aocMeta.Type = type;
                    checkData.Nsp.ContentMeta = aocMeta;
                }

                checkData.Nsp.ContentMeta.Type = type;

                // テスト実施
                // UnpublishableErrorエラーの解析
                var unpublishableErrorModel = UnpublishableError.GetList(checkData);
                // UnpublishableErrorエラーの発生数を取得する
                int errorCount = 0;
                int warningCount = 0;
                UnpublishableError.GetUnpublishableErrorCount(out errorCount, out warningCount, unpublishableErrorModel);

                // UnpublishableErrorエラー情報を表示する (例外が発生しなければ問題なしとする)
                UnpublishableError.ShowUnpublishableError(unpublishableErrorModel, true);
                UnpublishableError.ShowUnpublishableError(unpublishableErrorModel, false);

                // テスト結果判定
                // エラーが発生するIDとモジュールor関数名（期待値）
                List<Tuple<string, string, string[]>> expectInfo = new List<Tuple<string, string, string[]>>();
                if (type == NintendoContentMetaConstant.ContentMetaTypeApplication ||
                    type == NintendoContentMetaConstant.ContentMetaTypePatch)
                {
                    expectInfo.AddRange(
                        new List<Tuple<string, string, string[]>>
                        {
                            Tuple.Create("Error", "Issue 11-004", new string[] { debugApiInfos[0].ModuleName }),
                            Tuple.Create("Error", "Issue 11-006", new string[] { expectUnpublishableDebugApiName }),
                            Tuple.Create("Error", "Issue 12-001", new string[] { "testNothingTechnologyName" }),
                            Tuple.Create("Error", "Issue 12-002", new string[] { "testNothingMiddleware" }),
                            Tuple.Create("Error", "Issue 12-004", new string[] { applicationId, checkData.Nsp.LegalInformation.ApplicationId }),
                            Tuple.Create("Error", "Issue 11-008", new string[] { "testDebugModule" }),
                            Tuple.Create("Error", "Issue 11-002", new string[] { "Debug" }),
                            Tuple.Create("Error", "Issue 11-009", new string[] { "1.2.3", "2.2.99:1.2.3" }),
                            Tuple.Create("Error", "Issue 02-007", new string[] { "2.2.99:1.2.3" }),
                            Tuple.Create("Error", "Issue 11-016", new string[] { "dummy" }),
                            Tuple.Create("Error", "Issue 10-010", new string[] { "dummy" }),
                            Tuple.Create("Error", "Issue 10-011", new string[] { "dummy" }),
                            Tuple.Create("Error", "Issue 10-801", new string[] { "dummy" }),
                            Tuple.Create("Error", "Issue 10-809", new string[] { "Isbn", "１２３" }),
                            Tuple.Create("Error", "Issue 10-810", new string[] { "ApplicationErrorCodeCategory", "12345678", "7" }),
                            Tuple.Create("Error", "Issue 10-012", new string[] { "dummy" }),
                            Tuple.Create("Error", "Issue 10-014", new string[] { "dummy" }),
                            Tuple.Create("Error", "Issue 10-607", new string[] { "dummy" }),
                            Tuple.Create("Error", "Issue 10-029", new string[] { "dummy" }),
                            Tuple.Create("Warning", "Issue 10-030", new string[] { "dummy" }),
                            Tuple.Create("Error", "Issue 11-010", new string[] { expectUnpublishableXpadApiName }),
                            Tuple.Create("Warning", "Issue 11-003", new string[] { "nn::oe::ReportUserIsActive()", "Bundle_OeReportUserIsActive_NX" }),
                            Tuple.Create("Warning", "Issue 10-033", new string[] { "dummy" }),
                            Tuple.Create("Warning", "Issue 10-034", new string[] { "dummy" }),
                            Tuple.Create("Warning", "Issue 10-035", new string[] { "dummy" }),
                            Tuple.Create("Warning", "Issue 10-036", new string[] { "dummy" }),
                            Tuple.Create("Warning", "Issue 10-037", new string[] { "dummy" }),
                            Tuple.Create("Warning", "Issue 10-038", new string[] { "dummy" }),
                            Tuple.Create("Warning", "Issue 10-039", new string[] { "dummy" }),
                            Tuple.Create("Warning", "Issue 10-040", new string[] { "dummy" }),
                            Tuple.Create("Warning", "Issue 10-041", new string[] { "dummy" }),
                            Tuple.Create("Warning", "Issue 10-049", new string[] { "dummy" }),
                            Tuple.Create("Warning", "Issue 10-015", new string[] { "0x0100000000000000" }),
                            Tuple.Create("Warning", "Issue 10-016", new string[] { "0x0100000000000001, 0x0100000000000002" }),
                            Tuple.Create("Warning", "Issue 11-013", new string[] { "dummy" }),
                            Tuple.Create("Warning", "Issue 10-045", new string[] { "Required" }),
                            Tuple.Create("Warning", "Issue 11-014", new string[] { "1" }),
                            Tuple.Create("Warning", "Issue 11-015", new string[] { "nn::irsensor::RunPointingProcessor(nn::irsensor::IrCameraHandle const&)" }),
                        });
                }
                else if (type == NintendoContentMetaConstant.ContentMetaTypeAddOnContent)
                {
                    expectInfo.AddRange(
                        new List<Tuple<string, string, string[]>>
                        {
                            Tuple.Create("Error", "Issue 10-611", new string[] { "dummy" }),
                        });
                }

                // エラー数のチェック
                Assert.AreEqual(expectInfo.FindAll(entry => entry.Item1 == "Error").Count(), errorCount);
                Assert.AreEqual(expectInfo.FindAll(entry => entry.Item1 == "Warning").Count(), warningCount);
                // xmlモデルのチェック
                Assert.AreEqual(1, unpublishableErrorModel.Contents.Count());
                CheckUnpublishableErrorContent(checkData.Nsp.ContentMeta.Id, checkData.Nsp.ContentMeta.Type, errorCount + warningCount,
                    expectInfo.Select(entry => { return entry.Item1; }).ToList(), expectInfo.Select(entry => { return entry.Item2; }).ToList(),
                        expectInfo.Select(entry => { return entry.Item3; }).ToList(), unpublishableErrorModel.Contents.Single());
            }
        }

        // programInfo.xmlにAPIが含まれない
        [TestMethod]
        public void UnpublishableErrorProgramInfoNoApiTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();
            // 作成したデータにAPIがないことを確認しておく
            Assert.AreEqual(0, checkData.Nsp.ProgramInfo.DebugApiListData.GetApiNameList().Count());
            Assert.AreEqual(0, checkData.Nsp.ProgramInfo.PrivateApiListData.GetApiNameList().Count());
            Assert.AreEqual(0, checkData.Nsp.ProgramInfo.UnresolvedApiListData.GetApiNameList().Count());

            // テスト実施
            // UnpublishableErrorエラーの解析
            var unpublishableErrorModel = UnpublishableError.GetList(checkData);
            // UnpublishableErrorエラーの発生数を取得する
            int errorCount = 0;
            int warningCount = 0;
            UnpublishableError.GetUnpublishableErrorCount(out errorCount, out warningCount, unpublishableErrorModel);

            // UnpublishableErrorエラー情報を表示する (例外が発生しなければ問題なしとする)
            UnpublishableError.ShowUnpublishableError(unpublishableErrorModel, true);
            UnpublishableError.ShowUnpublishableError(unpublishableErrorModel, false);

            // テスト結果判定
            // エラー数のチェック
            Assert.AreEqual(0, errorCount);
            Assert.AreEqual(0, warningCount);
            // xmlモデルのチェック
            Assert.AreEqual(1, unpublishableErrorModel.Contents.Count());
            CheckUnpublishableErrorContent(checkData.Nsp.ContentMeta.Id, checkData.Nsp.ContentMeta.Type,
                0, null, null, (List<string[]>)null, unpublishableErrorModel.Contents.Single());
        }

        // SbmChk-26aでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk26aTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            // Middlewareに情報が登録されている
            ModuleInfo[] middlewareInfos = new ModuleInfo[]
            {
                // LegalInfoXMLのDeclarationRequiredNotationsに完全一致するものが存在するがCopyrightNotationsに存在しない(エラーになる)
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "testFailure1", "testVendor", "testFile"),
                // LegalInfoXMLのDeclarationRequiredNotationsに前方一致するものが存在するがCopyrightNotationsに存在しない(エラーになる)
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "testFailure2_1", "testVendor", "testFile"),
                // LegalInfoXMLのDeclarationRequiredNotationsに後方一致するものが存在するがCopyrightNotationsに存在しない(エラーにならない)
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "_testSuccess1", "testVendor", "testFile"),
            };
            checkData.Nsp.ProgramInfo.MiddlewareListData = MiddlewareList.Create(middlewareInfos);
            var declarationRequiredNotationsTechnologyNameList = new List<string>()
            {
                // LegalInfoXMLのCopyrightNotationsに完全一致するものが存在しない
                "testSuccess1",
                "testFailure1",
                // LegalInfoXMLのCopyrightNotationsに完全一致するものが存在しないが、前方一致するものは存在する
                "testFailure2",
            };
            checkData.Nsp.LegalInformation.DeclarationRequiredNotations.TechnologyName = declarationRequiredNotationsTechnologyNameList;
            var copyrightNotationsTechnologyNameList = new List<string>()
            {
                "testFailure2_1",
            };
            checkData.Nsp.LegalInformation.CopyrightNotations.TechnologyName = copyrightNotationsTechnologyNameList;

            // テスト実施
            // UnpublishableErrorエラーの解析
            var unpublishableErrorModel = UnpublishableError.GetList(checkData);
            // UnpublishableErrorエラーの発生数を取得する
            int errorCount = 0;
            int warningCount = 0;
            UnpublishableError.GetUnpublishableErrorCount(out errorCount, out warningCount, unpublishableErrorModel);

            // テスト結果判定
            // エラーとなるDeclarationRequiredNotations（期待値）
            string[] expectErrorDeclarationRequiredNotationsName =
            {
                "testFailure1"
                + Environment.NewLine
                + "testFailure2"
            };
            // エラー数のチェック
            Assert.AreEqual(expectErrorDeclarationRequiredNotationsName.Count(), errorCount);
            Assert.AreEqual(0, warningCount);
            // xmlモデルのチェック
            Assert.AreEqual(1, unpublishableErrorModel.Contents.Count());
            CheckUnpublishableErrorContent(checkData.Nsp.ContentMeta.Id, checkData.Nsp.ContentMeta.Type, errorCount,
                Enumerable.Repeat<string>("Error", errorCount).ToList(), Enumerable.Repeat<string>("Issue 12-001", errorCount).ToList(),
                    expectErrorDeclarationRequiredNotationsName.ToList(), unpublishableErrorModel.Contents.Single());
        }

        // SbmChk-26aでエラーが発生する(LogoTypeによるチェック処理の違いのテスト)
        [TestMethod]
        public void UnpublishableErrorSbmChk26aLogoTypeTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            // Middlewareに情報が登録されている
            ModuleInfo[] middlewareInfos = new ModuleInfo[]
            {
                // LegalInfoXMLのDeclarationRequiredNotationsに完全一致するものが存在するがCopyrightNotationsに存在しない(エラーになる)
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "Havok", "testVendor", "testFile"),
                // LegalInfoXMLのDeclarationRequiredNotationsに前方一致するものが存在するがCopyrightNotationsに存在しない(エラーになる)
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "UnrealEngine_1", "testVendor", "testFile"),
            };
            checkData.Nsp.ProgramInfo.MiddlewareListData = MiddlewareList.Create(middlewareInfos);
            var declarationRequiredNotationsTechnologyNameList = new List<string>()
            {
                // LegalInfoXMLのCopyrightNotationsに完全一致するものが存在しない
                "Havok",
                "UnrealEngine",
            };
            checkData.Nsp.LegalInformation.DeclarationRequiredNotations.TechnologyName = declarationRequiredNotationsTechnologyNameList;

            // Application/LogoTypeの値
            List<Tuple<bool, string>> testPatternList = new List<Tuple<bool, string>>
            {
                Tuple.Create(false, "Nintendo"),              // NG
                Tuple.Create(true,  "LicensedByNintendo"),    // OK
                Tuple.Create(false, "licensedbynintendo"),    // NG(通常はありえない)
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, string>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> input)
            {
                // LogoTypeを設定
                data.ApplicationControlProperty.LogoType = input.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, string>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> pattern)
            {
                // エラーとなるDeclarationRequiredNotations（期待値）
                var name = string.Join(Environment.NewLine, checkData.Nsp.LegalInformation.DeclarationRequiredNotations.TechnologyName);
                return new List<UnpublishableErrorExpectType> { new Tuple<string, string, string[]>("Error", "Issue 12-001", new string[] { name }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-26bでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk26bTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            // Middlewareに情報が登録されている
            ModuleInfo[] middlewareInfos = new ModuleInfo[]
            {
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "testSuccess1", "testVendor", "testFile"),
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "testSuccess2_1", "testVendor", "testFile"),
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "_testFailure2", "testVendor", "testFile"),
            };

            // SbmChk-26bはコンテンツ毎のProgramInfoに含まれるMiddlewareListではチェックせずに、nsp全体のMiddlewareListでチェックを実施する
            checkData.Nsp.ProgramInfo.MiddlewareListData = MiddlewareList.Create();
            checkData.Nsp.AllMiddlewareListInNsp = MiddlewareList.Create(middlewareInfos).GetModuleNameList();

            var declarationRequiredNotationsTechnologyNameList = new List<string>()
            {
                // LegalInfoXMLのCopyrightNotationsとMiddlewareに完全一致するものが存在する(エラーにならない)
                "testSuccess1",
                // LegalInfoXMLのCopyrightNotationsに完全一致するものが存在し、Middlewareに前方一致するものがある(エラーにならない)
                "testSuccess2",
                // LegalInfoXMLのCopyrightNotationsに完全一致するものが存在するが、Middlewareに一致するものがない(エラーになる)
                "testFailure1",
                // LegalInfoXMLのCopyrightNotationsに完全一致するものが存在し、Middlewareに後方一致するものがある(エラーになる)
                "testFailure2",
            };
            checkData.Nsp.LegalInformation.DeclarationRequiredNotations.TechnologyName = declarationRequiredNotationsTechnologyNameList;
            var copyrightNotationsTechnologyNameList = new List<string>()
            {
                "testSuccess1",
                "testSuccess2",
                "testFailure1",
                "testFailure2",
            };
            checkData.Nsp.LegalInformation.CopyrightNotations.TechnologyName = copyrightNotationsTechnologyNameList;

            // テスト実施
            // UnpublishableErrorエラーの解析
            var unpublishableErrorModel = UnpublishableError.GetList(checkData);
            // UnpublishableErrorエラーの発生数を取得する
            int errorCount = 0;
            int warningCount = 0;
            UnpublishableError.GetUnpublishableErrorCount(out errorCount, out warningCount, unpublishableErrorModel);

            // テスト結果判定
            // エラーとなるDeclarationRequiredNotations（期待値）
            string[] expectErrorDeclarationRequiredNotationsName =
            {
                "testFailure1"
                + Environment.NewLine
                + "testFailure2"
            };
            // エラー数のチェック
            Assert.AreEqual(expectErrorDeclarationRequiredNotationsName.Count(), errorCount);
            Assert.AreEqual(0, warningCount);
            // xmlモデルのチェック
            Assert.AreEqual(1, unpublishableErrorModel.Contents.Count());
            CheckUnpublishableErrorContent(checkData.Nsp.ContentMeta.Id, checkData.Nsp.ContentMeta.Type, errorCount,
                Enumerable.Repeat<string>("Error", errorCount).ToList(), Enumerable.Repeat<string>("Issue 12-002", errorCount).ToList(), expectErrorDeclarationRequiredNotationsName.ToList(),
                    unpublishableErrorModel.Contents.Single());
        }

        // SbmChk-26bでエラーが発生する(LogoTypeによるチェック処理の違いのテスト)
        [TestMethod]
        public void UnpublishableErrorSbmChk26bLogoTypeTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            // Middlewareに情報がないTechnologyがDeclarationRequiredNotationsとCopyrightNotationsに存在する
            var technologyNameList = new List<string>()
            {
                "Havok",
                "UnrealEngine",
            };
            checkData.Nsp.LegalInformation.CopyrightNotations.TechnologyName = technologyNameList;
            checkData.Nsp.LegalInformation.DeclarationRequiredNotations.TechnologyName = technologyNameList;

            // Application/LogoTypeの値
            List<Tuple<bool, string>> testPatternList = new List<Tuple<bool, string>>
            {
                Tuple.Create(false, "Nintendo"),              // NG
                Tuple.Create(true,  "LicensedByNintendo"),    // OK
                Tuple.Create(false, "licensedbynintendo"),    // NG(通常はありえない)
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, string>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> input)
            {
                // LogoTypeを設定
                data.ApplicationControlProperty.LogoType = input.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, string>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> pattern)
            {
                // エラーとなるDeclarationRequiredNotations（期待値）
                var name = string.Join(Environment.NewLine, checkData.Nsp.LegalInformation.DeclarationRequiredNotations.TechnologyName);
                return new List<UnpublishableErrorExpectType> { new Tuple<string, string, string[]>("Error", "Issue 12-002", new string[] { name }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-28でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk28Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            // ContentMetaにLegalInformationが登録されていない
            string[] testTypeList =
            {
                NintendoContentMetaConstant.ContentMetaTypeAddOnContent,
                NintendoContentMetaConstant.ContentMetaTypeApplication,
                NintendoContentMetaConstant.ContentMetaTypeBootImagePackage,
                NintendoContentMetaConstant.ContentMetaTypeBootImagePackageSafe,
                NintendoContentMetaConstant.ContentMetaTypeDelta,
                NintendoContentMetaConstant.ContentMetaTypePatch,
                NintendoContentMetaConstant.ContentMetaTypeSystemData,
                NintendoContentMetaConstant.ContentMetaTypeSystemProgram,
                NintendoContentMetaConstant.ContentMetaTypeSystemUpdate,
                NintendoContentMetaConstant.ContentTypeControl,
                NintendoContentMetaConstant.ContentTypeData,
                NintendoContentMetaConstant.ContentTypeDeltaFragment,
                NintendoContentMetaConstant.ContentTypeHtmlDocument,
                NintendoContentMetaConstant.ContentTypeMeta,
                NintendoContentMetaConstant.ContentTypeProgram,
                NintendoContentMetaConstant.ContentTypePublicData,
                "LEGALINFORMATION", // 通常はありえない
                "LegalInformatio",  // 通常はありえない
            };
            var contentModelList = new List<ContentModel>();
            foreach (var type in testTypeList)
            {
                ContentModel contentModel = new ContentModel();
                contentModel.Type = type;
                contentModelList.Add(contentModel);
            }
            checkData.Nsp.ContentMeta.ContentList = contentModelList;

            // UnpublishableErrorエラーの解析
            var unpublishableErrorModel = UnpublishableError.GetList(checkData);
            // UnpublishableErrorエラーの発生数を取得する
            int errorCount = 0;
            int warningCount = 0;
            UnpublishableError.GetUnpublishableErrorCount(out errorCount, out warningCount, unpublishableErrorModel);

            // テスト結果判定
            // エラー数のチェック
            Assert.AreEqual(1, errorCount);
            Assert.AreEqual(0, warningCount);
            // xmlモデルのチェック
            Assert.AreEqual(1, unpublishableErrorModel.Contents.Count());
            CheckUnpublishableErrorContent(checkData.Nsp.ContentMeta.Id, checkData.Nsp.ContentMeta.Type, errorCount,
                Enumerable.Repeat<string>("Error", errorCount).ToList(), Enumerable.Repeat<string>("Issue 12-003", errorCount).ToList(),
                    new List<string[]> { new string[] { "dummy" } }, unpublishableErrorModel.Contents.Single());
        }

        // SbmChk-41aでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk41aTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, string[]>> testPatternList = new List<Tuple<bool, string[]>>
            {
                Tuple.Create(false, new string[] { "0x0000000000000000", "0x8000000000000000" }), // NG
                Tuple.Create(false, new string[] { "0xFFFFFFFFFFFFFFFF", "0xFFFFFFFFFFFFFFFE" }), // NG
                Tuple.Create(false, new string[] { "0x5a5a5a5a5a5a5a5a", "0xa5a5a5a5a5a5a5a5" }), // NG
                Tuple.Create(true,  new string[] { "0x0000000000000000", "0x0000000000000000" }), // OK
                Tuple.Create(true,  new string[] { "0xFFFFFFFFFFFFFFFF", "0xffffffffffffffff" }), // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, string[]>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, string[]> input)
            {
                // Idを設定
                data.Nsp.ContentMeta.Id = input.Item2[0];
                data.Nsp.LegalInformation.ApplicationId = input.Item2[1];
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, string[]>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, string[]> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new Tuple<string, string, string[]>("Error", "Issue 12-004", pattern.Item2) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-35でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk35Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            // Middlewareに"Debug"・"Develop"を含む情報が登録されている
            ModuleInfo[] middlewareInfos = new ModuleInfo[]
            {
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "Debug", "testVendor", "testFile"),           // NG(Debugと完全一致)
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "Develop", "testVendor", "testFile"),         // NG(Developと完全一致)
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "_Debug_", "testVendor", "testFile"),         // NG(Debugが部分一致)
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "testDeveloptest", "testVendor", "testFile"), // NG(Developが部分一致)
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "debug", "testVendor", "testFile"),           // OK(全て小文字)
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "DEVELOP", "testVendor", "testFile"),         // OK(全て大文字)
                new ModuleInfo(ModuleInfoType.MiddlewareInfo, "Debuevelop", "Debug", "Develop"),            // OK(NG文字列の一部が欠けている)
            };
            checkData.Nsp.ProgramInfo.MiddlewareListData = MiddlewareList.Create(middlewareInfos);

            // テスト実施
            // UnpublishableErrorエラーの解析
            var unpublishableErrorModel = UnpublishableError.GetList(checkData);
            // UnpublishableErrorエラーの発生数を取得する
            int errorCount = 0;
            int warningCount = 0;
            UnpublishableError.GetUnpublishableErrorCount(out errorCount, out warningCount, unpublishableErrorModel);

            // テスト結果判定
            // エラーとなるモジュール名（期待値）
            // 入力したミドルウェア情報の先頭の4つでNGが発生する
            var name = string.Join(Environment.NewLine, middlewareInfos.Take(4).Select(entry => { return entry.ModuleName; }).ToList());
            List<string> expectMiddlewareName = new List<string>() { name };
            // エラー数のチェック
            Assert.AreEqual(expectMiddlewareName.Count(), errorCount);
            Assert.AreEqual(0, warningCount);
            // xmlモデルのチェック
            Assert.AreEqual(1, unpublishableErrorModel.Contents.Count());
            CheckUnpublishableErrorContent(checkData.Nsp.ContentMeta.Id, checkData.Nsp.ContentMeta.Type, errorCount,
                Enumerable.Repeat<string>("Error", errorCount).ToList(), Enumerable.Repeat<string>("Issue 11-008", errorCount).ToList(),
                    expectMiddlewareName.ToList(), unpublishableErrorModel.Contents.Single());
        }

        // SbmChk-21でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk21Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            // ProgramInfo/BuildTypeが"Release"以外
            List<Tuple<bool, string>> testPatternList = new List<Tuple<bool, string>>
            {
                Tuple.Create(false, "Debug"),   // NG
                Tuple.Create(false, "Develop"), // NG
                Tuple.Create(false, "Releas"),  // 実際にはありえない
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, string>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> input)
            {
                // ビルドタイプを設定
                data.Nsp.ProgramInfo.BuildType = input.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, string>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> pattern)
            {
                // エラーとなるビルドタイプ（期待値）
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 11-002", new string[] { pattern.Item2 }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-42でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk42Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, string[]>> testPatternList = new List<Tuple<bool, string[]>>
            {
                Tuple.Create(false, new string[] { "1_2_3", "2_2_3" }),       // NG
                Tuple.Create(true,  new string[] { "1_2_3", "1_2_3" }),       // OK
                Tuple.Create(false, new string[] { "1_1_1", "2_1_1" }),       // NG
                Tuple.Create(false, new string[] { "1_2_3", "01_2_3:abc" }),  // NG
                Tuple.Create(false, new string[] { "2_2_3", "20_2_3" }),      // NG
                Tuple.Create(true,  new string[] { "10_2_3", "10_0_0:abc" }), // OK
                Tuple.Create(true,  new string[] { "09_2_3", "09_3_11" }),    // OK
                Tuple.Create(false, new string[] { null, "2_2_3" }),          // NG (テスト用 通常はありえない)
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, string[]>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, string[]> input)
            {
                // バージョンを設定
                data.Nsp.ProgramInfo.SdkVersion = input.Item2[0];
                data.Nsp.ProgramInfo.ToolVersion = input.Item2[1];
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, string[]>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, string[]> pattern)
            {
                var expectVersion = pattern.Item2.Select(entry => { return (entry == null) ? "null" : entry.Replace('_', '.'); }).ToArray();
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 11-009", expectVersion) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-20cでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk20cTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, string>> testPatternList = new List<Tuple<bool, string>>
            {
                Tuple.Create(false, "1_2_99"),      // NG
                Tuple.Create(false, "1_2_3_99"),    // NG
                Tuple.Create(true, "1_99_1"),       // OK
                Tuple.Create(true, "1_2_999"),      // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, string>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> input)
            {
                // バージョンを設定
                data.Nsp.ProgramInfo.SdkVersion = input.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, string>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 11-011", new string[] { pattern.Item2.Replace('_', '.') }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-33dでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk33dTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, string>> testPatternList = new List<Tuple<bool, string>>
            {
                Tuple.Create(false, "1_2_99:4_22"),  // NG
                Tuple.Create(true, "99_99_1:99_99"), // OK
                Tuple.Create(true, "1_2_999:0_1"),   // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, string>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> input)
            {
                // ツールバージョンを設定
                data.Nsp.ProgramInfo.ToolVersion = input.Item2;
                // SbmChk-42エラーの抑制の為に、SDKバージョンとツールバージョンのメジャーバージョンを同じに設定
                data.Nsp.ProgramInfo.SdkVersion = input.Item2.Split('_').First() + Regex.Replace(data.Nsp.ProgramInfo.SdkVersion, "[0-9]+(_.+)", @"$1");
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, string>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 02-007", new string[] { pattern.Item2.Replace('_', '.') }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-24aでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk24aTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();
            string descFileName = "Test.Desc";
            checkData.Nsp.ProgramInfo.DescFileName = descFileName;

            List<Tuple<bool, Tuple <DescFlags, string>>> testPatternList = new List<Tuple<bool, Tuple<DescFlags, string>>>
            {
                Tuple.Create(false, Tuple.Create(new DescFlags() { Production = false, UnqualifiedApproval = true }, "rJWdlhkpc5IJ8WocWnan4CD3aQaarfe6DfIfL0YJMMUy+nxqnm7q46208nIdtElX/b0SdB4kQV7rmJDkN5bawI2ESZgLA89FngWxfWswPBXyF7805W6JX3/FbM+DmxaDk2/2GXuuOoChjFa0lNGZcmwewxEeOR+Dzy2p9iRVdy5D3P7D/Rg6i+7k+l5Um9hLYI1957dWfxJWUsAkEsWdsV4SJtiPGH7DJ7OGpVFMOOPW0MB5AGD1HhlUUflLAJcgp1qcaAM2sI1jnZVDZ0cFTVX/E40cPStftx4AAc5SB3HE8NVU4LohLOGva7l1ThmUhuwEvAEWF3KGVY76t1P0heDlRSQ5nSXASENNWdreAwzAkg/rbLCdqeedAy1mXK9DOp/ICuJx3vnOGwBNwkGpYcMhG6CC0my0SDF97N5qM12KLNpQtILgODC5uw3l0fYQjCS/j5phF+N3Fp+IruBtzAyDcK4fGNttGdNtIMJadrBe3vfFwKMorA0TlC6/NZkc70VAJpcP6MFQcqdileF2gYNnzY4t5/oWoJmZS/S4/TxY55G3G/W78hMvlQvgTpHnURDX1qCB7eDdFJtGgkPd1YhZgDkh+LzVkBwBiXqG9pUfKFloO1yzt7NG04x85RrSuHC9/a8VGDCrHrDpTCNiY5vEHe7OqpLdQl1RxHHKn0tBQ0lEpAIAAAAAAAAAAAAAAAABAAAAAAH/////////AUACAAAsAAAAcAIAAAgBAACAAwAAJAAAAAAAAAAAAAAAAQAAAAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABWFjYzp1MARhb2M6dQJhcG0EYXBtOnAHYXBwbGV0T0UGYXVkaW46dQdhdWRvdXQ6dQdhdWRyZW46dQRic2Q6dQVic2RjZmcEY3NybmcCZmdtB2ZyaWVuZDp1BmZzcC1zcnYCaGlkAmh0YwdodGM6dGVudgNodGNzBWh3b3B1cwJpcnMEbGRuOnUFbGRyOnJvAWxtBG1paTp1A21tOnUHbmZjOm1mOnUHbmZjOnVzZXIHbmZwOnVzZXIFbmlmbTp1BG5zZDp1Am50YwRudmRydgJwY20DcGN0bANwbDp1BnByZXBvOnUCc2V0B3NmZG5zcmVzAnNzbAV0aW1lOnUDdHNwbQN2aTp1AAAAAAAAAAC3cwACz///H+//fzCPAQBADwAAcP9fAAD/PxgA/38AAv//AgA=")),   // NG
                Tuple.Create(false, Tuple.Create(new DescFlags() { Production = false, UnqualifiedApproval = true }, "XHPzuv3LYKRsmsvegp2i7ulPuXDKYh9+f65jJZRXMyr49RJ/oV4mA/Kei1pWJrL86INPZ3JVmqHYKOs6oqqwx3fj7ISNKGMNGM0UJSyqQV+v9eJSZNE/CSb5A+VoZnf5Dn2Uw6Tvq/nNa0mziUAQBMWjScHusb/vR4jZD1XrHw7DussSCVTVgR+4OMw4cHBJcZQlQUUTFQP5oAw9rI7KYZWkOZbs6VBSRERdR7BVcWfVOepjE0u0XJMayRRqxbhEtvzwCyVsgUu4WXu77gvJHj8ov2P822/hch2Nd6otXmAIz1+Bdd+DIAr1gSc8EQobOY+BdiOG1ojQyImYDa5xwN0GL5RoaSYsRVWw40GET33k1Y0KgJckQQSbl29Wjq1cCTFNDIplVu+qI+Gql8RlOwl0OJC74OvK/z4SnN6l2kgGL4L/+tXP5Gvu7Ym48xYoxgpNlFE/JDFMeBBxxykDfyOfsWwnjpk6oMh3gBdYfRwprASEISEm9ziEpquJmVNp4s1Gt5yYqqsdWizMSJbjmh5Unp/i2bWtflgik65QeYCGhIdeoI+4XaLOSxZpvkiU4K5hqKyVq8EAETm7NghqJsbeD6Wr51M4BMleMl1jy+P5FVxx9HyAjgwFvWWJybGPM93jM2bnIRibyCmfAaydKrWE5pWRIPc7shoHlsF91QFBQ0lEpAIAAAAAAAAAAAAAAAABAAAAAAH/////////AUACAAAsAAAAcAIAAAoBAACAAwAAJAAAAAAAAAAAAAAAAQAAAAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABWFjYzp1MARhb2M6dQJhcG0EYXBtOnAHYXBwbGV0T0UGYXVkaW46dQdhdWRvdXQ6dQdhdWRyZW46dQViY2F0OnUEYnNkOnUFYnNkY2ZnBGNzcm5nAmZnbQdmcmllbmQ6dQZmc3Atc3J2AmhpZAJodGMHaHRjOnRlbnYDaHRjcwVod29wdXMCaXJzBGxkbjp1BWxkcjpybwFsbQRtaWk6dQNtbTp1B25mYzptZjp1B25mYzp1c2VyB25mcDp1c2VyBW5pZm06dQRuc2Q6dQJudGMEbnZkcnYCcGNtA3BjdGwDcGw6dQZwcmVwbzp1AnNldAdzZmRuc3JlcwJzc2wFdGltZTp1A3ZpOnUAAAAAAAC3cwACz///H+//fzCPAQBADwAAcP9fAAD/PzAA/38AAv//AgA=")),   // NG
                Tuple.Create(false, Tuple.Create(new DescFlags() { Production = false, UnqualifiedApproval = true }, "QM+XUg5nWnSLjsG02YmG4wI7ln0s+zTaOZ5qH4s12PIoHFMwLdJuKX88DVMXpn/1C+IaOFDRDNm60t3f8JDrO+PRowOV+GZAnV5sYpGgIRD2/XWnG4YAwu4rZhECHGS52M5P7q4vPxclhK7QH0m/3pDW8GleW3OqYVTy/+zPCgRU3DDsPrhlDVHSUXipg9JS66ygvDp7+Xc82aU5pqcAKiZheIc4dxCl9dsH5wqX2p2gW/Motn4zAllwa22q9wLmEWNxothpT0guksdWFVQpi4YeDHEMax4CT7UCieTfqWNi3im7Qx0Uowrlsqb2qxkV7DEt5DUF4JHyg3ZSK0HsRfU43+GWYTbfjJL6eCfraQb50iAUR27gS84AOY7RLWQ7wPavJ9oHvKGK97eB6kLeeyohIZdbENSTBK9KNVvvJ7lJmS5PIQtYY9akh3vE7G9Xrs05fWOyzkgmJToKEGZ7F1mtbyqP4L8s6GqdLdRWw/jT9h3g++TaaoVBkLachPdTNoxnXqeUoAZl2/s8w//Fj1j5lcHGJXc+CC60WcG2NL9juJL3DLOcjIrLoKVkLDvOhH5e6eHVlL5xC2bxKW8uN4QU7WQb4lHoXJkM4850r9C2OQdpKtcwR153Q0SlMpu7qeORAVIbv4A/s7jP+InCDEs0Vd8qtlXK8eklPl8RurVBQ0lEtAIAAAAAAAAAAAAAAMAJAHA8AAEAwAkAcDwAAUACAAAsAAAAcAIAABUBAACQAwAAJAAAAAAAAAAAAAAAAQAAAAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABWFjYzp1MARhb2M6dQJhcG0EYXBtOnAHYXBwbGV0T0UGYXVkaW46dQdhdWRvdXQ6dQdhdWRyZW46dQViY2F0OnUEYnNkOnUFYnNkY2ZnBWNhcHM6YQZjYXBzOnNzBGNzcm5nB2ZyaWVuZDp1BmZzcC1zcnYCaGlkAmh0YwdodGM6dGVudgNodGNzBWh3b3B1cwJpcnMEbGRuOnUFbGRyOnJvAWxtBG1paTp1A21tOnUHbmZjOm1mOnUHbmZjOnVzZXIHbmZwOnVzZXIFbmlmbTp1BG5zZDp1Am50YwRudmRydgJwY20DcGN0bANwbDp1BnByZXBvOnUCc2V0B3NmZG5zcmVzAnNzbAV0aW1lOnUDdmk6dQAAAAAAAAAAAAAAt3MAAs///x/v/38wjwEAQA8AAHD/XwAA/z8wAP9/AAL//wIA")),   // NG
                Tuple.Create(false, Tuple.Create(new DescFlags() { Production = false, UnqualifiedApproval = true }, "LjhN33KJNWWC+auJMXBf//SfYEuJqKPoLGguJ5i4QtIkT2Y72VNKVX1ZvqMFVXM3RAPHscHRLtgWh/PI7L91X9Cz1R8Adbl7obiAUesQchkPERVjFfAccOCAqwKjtgCgCjfbecygQbZTsMJ2CpltMnE2ZTINmEkGu+nS4Chg0xiRjBq1tAoEKi/fbN+AgLqZ0658umYLuQJgFifTkdAlyxJtc/QcYO4nI5zC78J6yWX+OqNQPPzZEdhVlZCM1IjLFWqGezHRvyKxZJlYstKGErMxTBO9zWRuWC4Ce55N4OciZTBM/CtuC+YHSuqPiduYeWajLuTf/hcl1d7xKzP+P8VKKD3cL2/sz6M9G82GBPSJPi09b8S9cizWGc0GKQIQjYQrhcLnr8yhEChTSDXoGg7CGU3LXgSN8O6pB97lvhjIUCBliwCPy8EhXBRogHm7OCt8FBUqjbHZC959NUcps6dLvqJxxLOzH33MPCtozQqKGJHPhTVx8yR3200HRh07e2lNPKHwzvl3ippMzdnz5DS438PgGbMKdF/7M/q/lAyxsQU3KZotb6Z8aBt+USs31UWT4Kcu4ORdjGvIGX4mRAM0wMItVUycd3agLbwZrODXIr9qnMaw7GMxPvzYZrxfbuvn+os3i+8eZSHt3FGnrGBKTwp7rrUhMcqY14XWeUVBQ0lEtAIAAAAAAAAAAAAAAAAKAMA7AAEAAAoAwDsAAUACAAAsAAAAcAIAABUBAACQAwAAJAAAAAAAAAAAAAAAAQAAAAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABWFjYzp1MARhb2M6dQJhcG0EYXBtOnAHYXBwbGV0T0UGYXVkaW46dQdhdWRvdXQ6dQdhdWRyZW46dQViY2F0OnUEYnNkOnUFYnNkY2ZnBWNhcHM6YQZjYXBzOnNzBGNzcm5nB2ZyaWVuZDp1BmZzcC1zcnYCaGlkAmh0YwdodGM6dGVudgNodGNzBWh3b3B1cwJpcnMEbGRuOnUFbGRyOnJvAWxtBG1paTp1A21tOnUHbmZjOm1mOnUHbmZjOnVzZXIHbmZwOnVzZXIFbmlmbTp1BG5zZDp1Am50YwRudmRydgJwY20DcGN0bANwbDp1BnByZXBvOnUCc2V0B3NmZG5zcmVzAnNzbAV0aW1lOnUDdmk6dQAAAAAAAAAAAAAAt3MAAs///x/v/38wjwEAQA8AAHD/XwAA/z8wAP9/AAL//wIA")),   // NG
                Tuple.Create(false, Tuple.Create(new DescFlags() { Production = false, UnqualifiedApproval = true }, "M5iti02lPVW/MBl0AGd3dtfPwIGOZjoVjX3eIlezC77WfupnOzB+GPyO9pL5/IcWMvsk6UcWul0bFSBgirguyaTXZHmCvNgrFRJ+57a+zYsJeG4tDEb7xj5AT2nC9cpHFgSlZg9B53J3QCqk5x8bQGOZheR/S2S6PCV9aNa40SGVimb8al5H/XK14n2cC1sap2O3kfU9CCuVsFt6puio7s941mikYGUGClfoXI6V0kILwFvZHM2Yr+hgnRJ5M2nMmv8pkath6NFcV9Jc3jSKAfCkOYBGR7LqJfIjwJOOr8Fsbzo3OE8Kx73HYNrvUTwo3e4qwhbo4c9O7t8U3ziMTqvrkusrayaVFZTKrpFY71Bz4Zfaij8A/B7EWbk5jmCp1l4i3hpHH/DPnYcToXIGRJGoDKFLiBJMAkMmJet5V/B6MMh2e7FsSbrZfL3lG6uR+aVnguv0fF/0SNZGJgSFBdGoNWlpHt6PD7dpm2Phbi8O8B68RZCkydeuPVd/92BVXl+2e/uzfL4Vn2bouHIG24FwXrVd/4ieKP30X8D3j5EmSNn35nIXgxaYEwOS6RvdQj7eL3v3X+HWAM+EHkwpxCs1cjodz4Tz3grZi57JIGPVbUeogJNdB5BzZTpFFdE30cCXNzRkSyxQOabgU/9K+YR0ad3swgw/IB1Qg72Tu89BQ0lEtAIAAAAAAAAAAAAAACAKAPD4AAEAIAoA8PgAAUACAAAsAAAAcAIAABUBAACQAwAAJAAAAAAAAAAAAAAAAQAAAAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABWFjYzp1MARhb2M6dQJhcG0EYXBtOnAHYXBwbGV0T0UGYXVkaW46dQdhdWRvdXQ6dQdhdWRyZW46dQViY2F0OnUEYnNkOnUFYnNkY2ZnBWNhcHM6YQZjYXBzOnNzBGNzcm5nB2ZyaWVuZDp1BmZzcC1zcnYCaGlkAmh0YwdodGM6dGVudgNodGNzBWh3b3B1cwJpcnMEbGRuOnUFbGRyOnJvAWxtBG1paTp1A21tOnUHbmZjOm1mOnUHbmZjOnVzZXIHbmZwOnVzZXIFbmlmbTp1BG5zZDp1Am50YwRudmRydgJwY20DcGN0bANwbDp1BnByZXBvOnUCc2V0B3NmZG5zcmVzAnNzbAV0aW1lOnUDdmk6dQAAAAAAAAAAAAAAt3MAAs///x/v/38wjwEAQA8AAHD/XwAA/z8wAP9/AAL//wIA")),   // NG
                Tuple.Create(true, Tuple.Create(new DescFlags() { Production = false, UnqualifiedApproval = true }, "M5iti02lPVW/MBl0AGd3dtfPwIGOZjoVjX3eIlezC77WfupnOzB+GPyO9pL5/IcWMvsk6UcWul0bFSBgirguyaTXZHmCvNgrFRJ+57a+zYsJeG4tDEb7xj5AT2nC9cpHFgSlZg9B53J3QCqk5x8bQGOZheR/S2S6PCV9aNa40SGVimb8al5H/XK14n2cC1sap2O3kfU9CCuVsFt6puio7s941mikYGUGClfoXI6V0kILwFvZHM2Yr+hgnRJ5M2nMmv8pkath6NFcV9Jc3jSKAfCkOYBGR7LqJfIjwJOOr8Fsbzo3OE8Kx73HYNrvUTwo3e4qwhbo4c9O7t8U3ziMTqvrkusrayaVFZTKrpFY71Bz4Zfaij8A/B7EWbk5jmCp1l4i3hpHH/DPnYcToXIGRJGoDKFLiBJMAkMmJet5V/B6MMh2e7FsSbrZfL3lG6uR+aVnguv0fF/0SNZGJgSFBdGoNWlpHt6PD7dpm2Phbi8O8B68RZCkydeuPVd/92BVXl+2e/uzfL4Vn2bouHIG24FwXrVd/4ieKP30X8D3j5EmSNn35nIXgxaYEwOS6RvdQj7eL3v3X+HWAM+EHkwpxCs1cjodz4Tz3grZi57JIGPVbUeogJNdB5BzZTpFFdE30cCXNzRkSyxQOabgU/9K+YR0ad3swgw/IB1Qg72Tu89BQ0lEtAIAAAAAAAAAAAAAACAKAPD4AAEAIAoA8PgAAUACAAAsAAAAcAIAABUBAACQAwAAJAAAAAAAAAAAAAAAAQAAAAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABWFjYzp1MARhb2M6dQJhcG0EYXBtOnAHYXBwbGV0T0UGYXVkaW46dQdhdWRvdXQ6dQdhdWRyZW46dQViY2F0OnUEYnNkOnUFYnNkY2ZnBWNhcHM6YQZjYXBzOnNzBGNzcm5nB2ZyaWVuZDp1BmZzcC1zcnYCaGlkAmh0YwdodGM6dGVudgNodGNzBWh3b3B1cwJpcnMEbGRuOnUFbGRyOnJvAWxtBG1paTp1A21tOnUHbmZjOm1mOnUHbmZjOnVzZXIHbmZwOnVzZXIFbmlmbTp1BG5zZDp1Am50YwRudmRydgJwY20DcGN0bANwbDp1BnByZXBvOnUCc2V0B3NmZG5zcmVzAnNzbAV0aW1lOnUDdmk6dQAAAAAAAAAAAAAAt3MAAs///x/v/38wjwEAQA8AAHD/XwAA/z8wAP9/AAL//wIB")),    // OK
                Tuple.Create(true, Tuple.Create(new DescFlags() { Production = true, UnqualifiedApproval = true }, "M5iti02lPVW/MBl0AGd3dtfPwIGOZjoVjX3eIlezC77WfupnOzB+GPyO9pL5/IcWMvsk6UcWul0bFSBgirguyaTXZHmCvNgrFRJ+57a+zYsJeG4tDEb7xj5AT2nC9cpHFgSlZg9B53J3QCqk5x8bQGOZheR/S2S6PCV9aNa40SGVimb8al5H/XK14n2cC1sap2O3kfU9CCuVsFt6puio7s941mikYGUGClfoXI6V0kILwFvZHM2Yr+hgnRJ5M2nMmv8pkath6NFcV9Jc3jSKAfCkOYBGR7LqJfIjwJOOr8Fsbzo3OE8Kx73HYNrvUTwo3e4qwhbo4c9O7t8U3ziMTqvrkusrayaVFZTKrpFY71Bz4Zfaij8A/B7EWbk5jmCp1l4i3hpHH/DPnYcToXIGRJGoDKFLiBJMAkMmJet5V/B6MMh2e7FsSbrZfL3lG6uR+aVnguv0fF/0SNZGJgSFBdGoNWlpHt6PD7dpm2Phbi8O8B68RZCkydeuPVd/92BVXl+2e/uzfL4Vn2bouHIG24FwXrVd/4ieKP30X8D3j5EmSNn35nIXgxaYEwOS6RvdQj7eL3v3X+HWAM+EHkwpxCs1cjodz4Tz3grZi57JIGPVbUeogJNdB5BzZTpFFdE30cCXNzRkSyxQOabgU/9K+YR0ad3swgw/IB1Qg72Tu89BQ0lEtAIAAAAAAAAAAAAAACAKAPD4AAEAIAoA8PgAAUACAAAsAAAAcAIAABUBAACQAwAAJAAAAAAAAAAAAAAAAQAAAAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABWFjYzp1MARhb2M6dQJhcG0EYXBtOnAHYXBwbGV0T0UGYXVkaW46dQdhdWRvdXQ6dQdhdWRyZW46dQViY2F0OnUEYnNkOnUFYnNkY2ZnBWNhcHM6YQZjYXBzOnNzBGNzcm5nB2ZyaWVuZDp1BmZzcC1zcnYCaGlkAmh0YwdodGM6dGVudgNodGNzBWh3b3B1cwJpcnMEbGRuOnUFbGRyOnJvAWxtBG1paTp1A21tOnUHbmZjOm1mOnUHbmZjOnVzZXIHbmZwOnVzZXIFbmlmbTp1BG5zZDp1Am50YwRudmRydgJwY20DcGN0bANwbDp1BnByZXBvOnUCc2V0B3NmZG5zcmVzAnNzbAV0aW1lOnUDdmk6dQAAAAAAAAAAAAAAt3MAAs///x/v/38wjwEAQA8AAHD/XwAA/z8wAP9/AAL//wIA")),   // OK
                Tuple.Create(true, Tuple.Create((DescFlags)null, "M5iti02lPVW/MBl0AGd3dtfPwIGOZjoVjX3eIlezC77WfupnOzB+GPyO9pL5/IcWMvsk6UcWul0bFSBgirguyaTXZHmCvNgrFRJ+57a+zYsJeG4tDEb7xj5AT2nC9cpHFgSlZg9B53J3QCqk5x8bQGOZheR/S2S6PCV9aNa40SGVimb8al5H/XK14n2cC1sap2O3kfU9CCuVsFt6puio7s941mikYGUGClfoXI6V0kILwFvZHM2Yr+hgnRJ5M2nMmv8pkath6NFcV9Jc3jSKAfCkOYBGR7LqJfIjwJOOr8Fsbzo3OE8Kx73HYNrvUTwo3e4qwhbo4c9O7t8U3ziMTqvrkusrayaVFZTKrpFY71Bz4Zfaij8A/B7EWbk5jmCp1l4i3hpHH/DPnYcToXIGRJGoDKFLiBJMAkMmJet5V/B6MMh2e7FsSbrZfL3lG6uR+aVnguv0fF/0SNZGJgSFBdGoNWlpHt6PD7dpm2Phbi8O8B68RZCkydeuPVd/92BVXl+2e/uzfL4Vn2bouHIG24FwXrVd/4ieKP30X8D3j5EmSNn35nIXgxaYEwOS6RvdQj7eL3v3X+HWAM+EHkwpxCs1cjodz4Tz3grZi57JIGPVbUeogJNdB5BzZTpFFdE30cCXNzRkSyxQOabgU/9K+YR0ad3swgw/IB1Qg72Tu89BQ0lEtAIAAAAAAAAAAAAAACAKAPD4AAEAIAoA8PgAAUACAAAsAAAAcAIAABUBAACQAwAAJAAAAAAAAAAAAAAAAQAAAAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABWFjYzp1MARhb2M6dQJhcG0EYXBtOnAHYXBwbGV0T0UGYXVkaW46dQdhdWRvdXQ6dQdhdWRyZW46dQViY2F0OnUEYnNkOnUFYnNkY2ZnBWNhcHM6YQZjYXBzOnNzBGNzcm5nB2ZyaWVuZDp1BmZzcC1zcnYCaGlkAmh0YwdodGM6dGVudgNodGNzBWh3b3B1cwJpcnMEbGRuOnUFbGRyOnJvAWxtBG1paTp1A21tOnUHbmZjOm1mOnUHbmZjOnVzZXIHbmZwOnVzZXIFbmlmbTp1BG5zZDp1Am50YwRudmRydgJwY20DcGN0bANwbDp1BnByZXBvOnUCc2V0B3NmZG5zcmVzAnNzbAV0aW1lOnUDdmk6dQAAAAAAAAAAAAAAt3MAAs///x/v/38wjwEAQA8AAHD/XwAA/z8wAP9/AAL//wIA")),   // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<DescFlags, string>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<DescFlags, string>> input)
            {
                // Descを設定

                data.Nsp.ProgramInfo.DescFlags = input.Item2.Item1;
                data.Nsp.ProgramInfo.Desc = input.Item2.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<DescFlags, string>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<DescFlags, string>> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 11-007", new string[] { descFileName }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-24bでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk24bTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();
            string descFileName = "Test.Desc";
            checkData.Nsp.ProgramInfo.DescFileName = descFileName;

            List<Tuple<bool, bool>> testPatternList = new List<Tuple<bool, bool>>
            {
                Tuple.Create(false, false),   // NG
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, bool>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, bool> input)
            {
                // Descを設定

                data.Nsp.ProgramInfo.DescFlags.UnqualifiedApproval = input.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, bool>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, bool> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Warning", "Issue 11-012", new string[] { descFileName }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-48でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk48Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, Tuple<string, string, uint>>> testPatternList = new List<Tuple<bool, Tuple<string, string, uint>>>
            {
                Tuple.Create(false, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication, "0x00000000", (uint)0x00000001)),  // NG
                Tuple.Create(false, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication, "0x00000001", (uint)0x00000000)),  // NG
                Tuple.Create(false, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication, "0x00000001", (uint)0x00000001)),  // NG
                Tuple.Create(false, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypePatch, "0x00000001", (uint)0xFFFF0000)),        // NG
                Tuple.Create(false, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeAddOnContent, "0xFFFF0000", (uint)0x00000001)), // NG
                Tuple.Create(true, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication, "0xFFFF0000", (uint)0xFFFF0000)),   // OK
                Tuple.Create(true, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypePatch, "0xFFFF0000", (uint)0xFFFF0000)),         // OK
                Tuple.Create(true, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeAddOnContent, "0xFFFF0000", (uint)0xFFFF0000)),  // OK
                Tuple.Create(true, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication, (string)null, (uint)0xFFFF0000)),   // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, uint>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, uint>> input)
            {
                var type = input.Item2.Item1;
                string applicationId = data.Nsp.ContentMeta.Id;

                if (type == NintendoContentMetaConstant.ContentMetaTypePatch)
                {
                    data = CreateSimpleSuccessData();
                    System.Random rand = new System.Random();
                    var patchMeta = new PatchContentMetaModel();
                    patchMeta.Id = string.Format("0x{0}", rand.Next());
                    patchMeta.ApplicationId = applicationId;
                    // ContentMetaにLegalInformationが登録されている
                    var contentModelList = new List<ContentModel> { new ContentModel() };
                    contentModelList.Single().Type = NintendoContentMetaConstant.ContentTypeLegalInformation;
                    patchMeta.ContentList = contentModelList;
                    patchMeta.Type = type;
                    data.Nsp.ContentMeta = patchMeta;
                }
                else if (type == NintendoContentMetaConstant.ContentMetaTypeAddOnContent)
                {
                    data = CreateSimpleSuccessData();
                    System.Random rand = new System.Random();
                    var aocMeta = new AddOnContentContentMetaModel();
                    aocMeta.Id = string.Format("0x{0}", rand.Next());
                    aocMeta.ApplicationId = applicationId;
                    aocMeta.ContentList = new List<ContentModel>() { new ContentModel() };
                    aocMeta.Type = type;
                    data.Nsp.ContentMeta = aocMeta;
                }

                data.CheckContentType = type;
                data.ApplicationControlProperty.Version = input.Item2.Item2;
                data.Nsp.ContentMeta.Version = input.Item2.Item3;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, uint>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, uint>> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 10-031", new string[] { "dummy" }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-63でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk63Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, string>> testPatternList = new List<Tuple<bool, string>>
            {
                Tuple.Create(false, "NEX-4_0_0-"),       // NG
                Tuple.Create(false, "NEX-4_0_1-"),       // NG
                Tuple.Create(false, "NEX-4_0_2-"),       // NG
                Tuple.Create(false, "NEX-4_0_3-"),       // NG
                Tuple.Create(false, "NEX-4_1_1-"),       // NG
                Tuple.Create(false, "NEX-4_1_2-"),       // NG
                Tuple.Create(false, "NEX-4_2_0-"),       // NG
                Tuple.Create(false, "NEX-4_2_1-"),       // NG
                Tuple.Create(false, "NEX-4_3_0-"),       // NG

                Tuple.Create(true,  "NEX-4_0_10-"),      // OK(エラーとするモジュール名と完全一致しない場合はエラーとしない)
                Tuple.Create(true,  "Nex-4_0_0-"),       // OK
                Tuple.Create(true,  "NEX-4_0_0"),        // OK
                Tuple.Create(true,  "-NEX-4_0_0"),       // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, string>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> input)
            {
                // MiddlewareListを設定
                var middlewareList = MiddlewareList.Create();
                middlewareList.Entries.Add(new MiddlewareModel() { ModuleName = input.Item2, NsoName = "dummy.nso", VenderName = "dummy" });
                checkData.Nsp.ProgramInfo.MiddlewareListData = middlewareList;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, string>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 11-016", new string[] { "dummy" }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-11でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk11Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, List<Title>>> testPatternList = new List<Tuple<bool, List<Title>>>
            {
                Tuple.Create(false,     // NG
                    new List<Title>
                    {
                        new Title { Name = null, Language = "AmericanEnglish", Publisher = "Publisher" },
                    }),
                Tuple.Create(false,     // NG
                    new List<Title>
                    {
                        new Title { Name = string.Empty, Language = "AmericanEnglish", Publisher = "Publisher" },
                        new Title { Name = "Name", Language = "AmericanEnglish", Publisher = "Publisher" }
                    }),
                Tuple.Create(true,     // OK
                     new List<Title>
                     {
                        new Title { Name = "Name1", Language = "AmericanEnglish", Publisher = "Publisher" },
                        new Title { Name = "Name2", Language = "AmericanEnglish", Publisher = "Publisher" }
                    }),
                // タイトルエントリーが存在しない場合はSbmChk-11とSbmChk-12でエラーになるのでUnpublishableErrorMixTestにてテストを実施する
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, List<Title>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, List<Title>> input)
            {
                // タイトル情報を設定
                data.ApplicationControlProperty.Title = input.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, List<Title>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, List<Title>> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 10-010", new string[] { "dummy" }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-11でエラーが発生する(タイトルとIconの言語設定異常)
        [TestMethod]
        public void UnpublishableErrorSbmChk11TestLanguage()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, Tuple<List<Title>, List<Icon>>>> testPatternList = new List<Tuple<bool, Tuple<List<Title>, List<Icon>>>>
            {
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Title>
                        {
                            new Title { Name = null, Language = "AmericanEnglish", Publisher = "Publisher" },
                            new Title { Name = "Name", Language = "Japanese", Publisher = "Publisher" },
                        },
                        new List<Icon>
                        {
                            new Icon { Language = "AmericanEnglish", IconPath = new DataPath() { Path = "icon.bmp" } },
                            new Icon { Language = "Japanese", IconPath = new DataPath() { Path = "icon.bmp" } },
                        })),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Title>
                        {
                            new Title { Name = "Name", Language = "AmericanEnglish", Publisher = "Publisher" },
                            new Title { Name = string.Empty, Language = "Japanese", Publisher = "Publisher" }
                        },
                        new List<Icon>
                        {
                            new Icon { Language = "AmericanEnglish", IconPath = new DataPath() { Path = "icon.bmp" } },
                            new Icon { Language = "Japanese", IconPath = new DataPath() { Path = "icon.bmp" } },
                        })),
                Tuple.Create(true,     // OK
                    Tuple.Create(
                        new List<Title>
                        {
                            new Title { Name = "Name", Language = "AmericanEnglish", Publisher = "Publisher" },
                            new Title { Name = "Name", Language = "Japanese", Publisher = "Publisher" }
                        },
                        new List<Icon>
                        {
                            new Icon { Language = "AmericanEnglish", IconPath = new DataPath() { Path = "icon.bmp" } },
                            new Icon { Language = "Japanese", IconPath = new DataPath() { Path = "icon.bmp" } },
                        })),
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<Title>, List<Icon>>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<Title>, List<Icon>>> input)
            {
                // タイトル情報を設定
                data.ApplicationControlProperty.Title = input.Item2.Item1;
                data.ApplicationControlProperty.Icon = input.Item2.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<Title>, List<Icon>>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<Title>, List<Icon>>> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 10-010", new string[] { "dummy" }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-12でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk12Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, List<Title>>> testPatternList = new List<Tuple<bool, List<Title>>>
            {
                Tuple.Create(false,     // NG
                    new List<Title>
                    {
                        new Title { Name = "Name", Language = "AmericanEnglish", Publisher = null },
                    }),
                Tuple.Create(false,     // NG
                    new List<Title>
                    {
                        new Title { Name = "Name", Language = "AmericanEnglish", Publisher = string.Empty },
                        new Title { Name = "Name", Language = "AmericanEnglish", Publisher = "Publisher" }
                    }),
                Tuple.Create(true,     // OK
                     new List<Title>
                     {
                        new Title { Name = "Name1", Language = "AmericanEnglish", Publisher = "Publisher" },
                        new Title { Name = "Name2", Language = "AmericanEnglish", Publisher = "Publisher" }
                    }),
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, List<Title>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, List<Title>> input)
            {
                // タイトル情報を設定
                data.ApplicationControlProperty.Title = input.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, List<Title>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, List<Title>> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 10-011", new string[] { "dummy" }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-12-1でエラーが発生する(SbmChk-12-3のテストも兼用)
        [TestMethod]
        public void UnpublishableErrorSbmChk12_1Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, List<Title>>> testPatternList = new List<Tuple<bool, List<Title>>>
            {
                Tuple.Create(false,     // NG
                    new List<Title>
                    {
                        new Title { Name = "Name", Language = null, Publisher = "Publisher" },
                    }),
                Tuple.Create(false,     // NG
                    new List<Title>
                    {
                        new Title { Name = "Name", Language = string.Empty, Publisher = "Publisher" },
                        new Title { Name = "Name", Language = "AmericanEnglish", Publisher = "Publisher" }
                    }),
                Tuple.Create(true,     // OK
                     new List<Title>
                     {
                        new Title { Name = "Name1", Language = "AmericanEnglish", Publisher = "Publisher" },
                        new Title { Name = "Name2", Language = "AmericanEnglish", Publisher = "Publisher" }
                    }),
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, List<Title>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, List<Title>> input)
            {
                // タイトル情報を設定
                data.ApplicationControlProperty.Title = input.Item2;
                SetSuccessIconData(data);
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, List<Title>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, List<Title>> pattern)
            {
                return new List<UnpublishableErrorExpectType>
                {
                    new UnpublishableErrorExpectType("Error", "Issue 10-801", new string[] { "dummy" }),
                    new UnpublishableErrorExpectType("Error", "Issue 10-806", new string[] { "dummy" }),
                    new UnpublishableErrorExpectType("Error", "Issue 10-607", new string[] { "dummy" }) // Iconの言語が空になるのでこのエラーも発生する
                };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-12_2でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk12_2Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>>> testPatternList = new List<Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>>>
            {
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Title>
                        {
                            new Title { Name = "Name", Language = "AmericanEnglish", Publisher = "Publisher" },
                        },
                        new List<Icon>
                        {
                            new Icon { Language = "AmericanEnglish", IconPath = new DataPath() { Path = "icon.bmp" } },
                            new Icon { Language = "Japanese", IconPath = new DataPath() { Path = "icon.bmp" } },
                            new Icon { Language = "German", IconPath = new DataPath() { Path = "icon.bmp" } },
                        },
                        new string[] { "Japanese, German" })),
               Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Title>
                        {
                            new Title { Name = "Name", Language = "AmericanEnglish", Publisher = "Publisher" },
                            new Title { Name = "Name", Language = "Japanese", Publisher = "Publisher" },
                            new Title { Name = "Name", Language = "Japanese", Publisher = "Publisher" },
                        },
                        new List<Icon>
                        {
                            new Icon { Language = "AmericanEnglish", IconPath = new DataPath() { Path = "icon.bmp" } },
                            new Icon { Language = "Japanese", IconPath = new DataPath() { Path = "icon.bmp" } },
                            new Icon { Language = "German", IconPath = new DataPath() { Path = "icon.bmp" } },
                        },
                        new string[] { "German" })),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        (List<Title>)null,
                        new List<Icon>
                        {
                            new Icon { Language = "AmericanEnglish", IconPath = new DataPath() { Path = "icon.bmp" } },
                        },
                        new string[] { "AmericanEnglish" })),
                Tuple.Create(true,     // OK
                    Tuple.Create(
                        new List<Title>
                        {
                            new Title { Name = "Name", Language = "AmericanEnglish", Publisher = "Publisher" },
                            new Title { Name = "Name", Language = "Japanese", Publisher = "Publisher" },
                            new Title { Name = "Name", Language = "German", Publisher = "Publisher" },
                        },
                        new List<Icon>
                        {
                            new Icon { Language = "AmericanEnglish", IconPath = new DataPath() { Path = "icon.bmp" } },
                            new Icon { Language = "Japanese", IconPath = new DataPath() { Path = "icon.bmp" } },
                            new Icon { Language = "German", IconPath = new DataPath() { Path = "icon.bmp" } },
                        },
                        new string[] { string.Empty })),
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>> input)
            {
                // タイトル情報を設定
                data.ApplicationControlProperty.Title = input.Item2.Item1;
                data.ApplicationControlProperty.Icon = input.Item2.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>> pattern)
            {
                if (pattern.Item2.Item1 == null)
                {
                    // Titleがnullの場合、他のエラーも発生する
                    return new List<UnpublishableErrorExpectType>
                    {
                        new UnpublishableErrorExpectType("Error", "Issue 10-010", pattern.Item2.Item3),
                        new UnpublishableErrorExpectType("Error", "Issue 10-011", pattern.Item2.Item3),
                        new UnpublishableErrorExpectType("Error", "Issue 10-801", pattern.Item2.Item3),
                        new UnpublishableErrorExpectType("Error", "Issue 10-802", pattern.Item2.Item3),
                    };
                }
                else
                {
                    return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 10-802", pattern.Item2.Item3) };
                }
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-12-4でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk12_4Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>>> testPatternList = new List<Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>>>
            {
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Title>
                        {
                            new Title { Name = "Name", Language = "Invalid", Publisher = "Publisher" },
                        },
                        new List<Icon>
                        {
                            new Icon { Language = "Invalid", IconPath = new DataPath() { Path = "icon.bmp" } },
                        },
                        new string[] { "Invalid" })),
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>> input)
            {
                // タイトル情報を設定
                data.ApplicationControlProperty.Title = input.Item2.Item1;
                data.ApplicationControlProperty.Icon = input.Item2.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 10-807", pattern.Item2.Item3) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-12-5でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk12_5Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, Tuple<List<Tuple<string, string>>, List<string[]>>>> testPatternList = new List<Tuple<bool, Tuple<List<Tuple<string, string>>, List<string[]>>>>
            {
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Tuple<string, string>>
                        {
                            Tuple.Create("Name", "Name\u0180")
                        },
                        new List<string[]> { new string[] { "Name", "Name\u0180" } })),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Tuple<string, string>>
                        {
                            Tuple.Create("Publisher", "\u0180")
                        },
                        new List<string[]> { new string[] { "Publisher", "\u0180" } })),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Tuple<string, string>>
                        {
                            Tuple.Create("Tag", "\uD867\uDE3D"),
                        },
                        new List<string[]> { new string[] { "Tag", "\uD867\uDE3D" } })),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Tuple<string, string>>
                        {
                            Tuple.Create("Name", "\u0181"),
                            Tuple.Create("Publisher", "Publisher\u0182"),
                            Tuple.Create("Tag", "Tag\u0184")
                        },
                        new List<string[]> { new string[] { "Name", "\u0181" }, new string[] { "Publisher", "Publisher\u0182" }, new string[] { "Tag", "Tag\u0184" } })),
                Tuple.Create(true,     // OK
                    Tuple.Create(
                        new List<Tuple<string, string>>
                        {
                            Tuple.Create("Name", "NameTest"),
                            Tuple.Create("Publisher", "PublisherTest"),
                            Tuple.Create("Tag", "TagTest")
                        },
                        (List<string[]>)null)),
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<Tuple<string, string>>, List<string[]>>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<Tuple<string, string>>, List<string[]>>> input)
            {
                var title = new Title() { Name = "Name", Language = "AmericanEnglish", Publisher = "Publisher" };
                var aocNmetaModel = new AddOnContentModel() { Id = checkData.Nsp.ContentMeta.Id + 1 };

                foreach (var entry in input.Item2.Item1)
                {
                    switch (entry.Item1)
                    {
                        case "Name":
                            title.Name = entry.Item2;
                            break;
                        case "Publisher":
                            title.Publisher = entry.Item2;
                            break;
                        case "Tag":
                            aocNmetaModel.Tag = entry.Item2;
                            break;
                        default:
                            break;
                    }
                }

                // タイトル情報を設定
                data.ApplicationControlProperty.Title = new List<Title> { title };
                data.ApplicationControlProperty.Icon = new List<Icon> { new Icon { Language = "AmericanEnglish", IconPath = new DataPath() { Path = "icon.bmp" } } };
                // Aocタグ名を設定
                checkData.Nmeta.AddOnContent = aocNmetaModel;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<Tuple<string, string>>, List<string[]>>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<Tuple<string, string>>, List<string[]>>> pattern)
            {
                return pattern.Item2.Item2.Select(entry => new UnpublishableErrorExpectType("Error", "Issue 10-808", entry)).ToList();
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-12-6でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk12_6Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            // Bcatの設定をする為に必要な設定を行う
            checkData.Nsp.ProgramInfo.SdkVersion = "3";
            checkData.Nsp.ProgramInfo.ToolVersion = checkData.Nsp.ProgramInfo.SdkVersion;
            checkData.ApplicationControlProperty.BcatDeliveryCacheStorageSize = "0x400000";

            List<Tuple<bool, Tuple<string, string, string, string, List<string[]>>>> testPatternList = new List<Tuple<bool, Tuple<string, string, string, string, List<string[]>>>>
            {
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        "test１", // Isbn
                        "test2", // DisplayVersion
                        "test3", // ApplicationErrorCodeCategory
                        "1234", // BcatPassphrase
                        new List<string[]> { new string[] { "Isbn", "test１" } })),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        "test1", // Isbn
                        "テスト２", // DisplayVersion
                        "test3", // ApplicationErrorCodeCategory
                        "1234", // BcatPassphrase
                        new List<string[]> { new string[] { "DisplayVersion", "テスト２" } })),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        "test1", // Isbn
                        "test2", // DisplayVersion
                        "てすと３", // ApplicationErrorCodeCategory
                        "1234", // BcatPassphrase
                        new List<string[]> { new string[] { "ApplicationErrorCodeCategory", "てすと３" } })),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        "test1", // Isbn
                        "test2", // DisplayVersion
                        "test3", // ApplicationErrorCodeCategory
                        "１２３", // BcatPassphrase
                        new List<string[]> { new string[] { "BcatPassphrase", "１２３" } })),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        "テスト一", // Isbn
                        "テスト二", // DisplayVersion
                        "テスト三", // ApplicationErrorCodeCategory
                        "テスト四", // BcatPassphrase
                        new List<string[]>
                        {
                            new string[] { "Isbn", "テスト一" },
                            new string[] { "DisplayVersion", "テスト二" },
                            new string[] { "ApplicationErrorCodeCategory", "テスト三" },
                            new string[] { "BcatPassphrase", "テスト四" },
                        })),
                Tuple.Create(true,     // OK
                    Tuple.Create(
                        "test1", // Isbn
                        "test2", // DisplayVersion
                        "test3", // ApplicationErrorCodeCategory
                        "test4", // BcatPassphrase
                        new List<string[]> { new string[] { string.Empty } })),
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, string, string, List<string[]>>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, string, string, List<string[]>>> input)
            {
                // 各種情報を設定
                data.ApplicationControlProperty.Isbn = input.Item2.Item1;
                data.ApplicationControlProperty.DisplayVersion = input.Item2.Item2;
                data.ApplicationControlProperty.ApplicationErrorCodeCategory = input.Item2.Item3;
                data.ApplicationControlProperty.BcatPassphrase = input.Item2.Item4;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, string, string, List<string[]>>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, string, string, List<string[]>>> pattern)
            {
                return pattern.Item2.Item5.Select(entry => new UnpublishableErrorExpectType("Error", "Issue 10-809", entry)).ToList();
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-12-7でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk12_7Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            // Bcatの設定をする為に必要な設定を行う
            checkData.Nsp.ProgramInfo.SdkVersion = "3";
            checkData.Nsp.ProgramInfo.ToolVersion = checkData.Nsp.ProgramInfo.SdkVersion;
            checkData.ApplicationControlProperty.BcatDeliveryCacheStorageSize = "0x400000";

            List<Tuple<bool, Tuple<string, string, string, string, List<string[]>>>> testPatternList = new List<Tuple<bool, Tuple<string, string, string, string, List<string[]>>>>
            {
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        "1234567890123456789012345678901234567", // Isbn(最大36文字)
                        "123456789012345", // DisplayVersion(最大15文字)
                        "1234567", // ApplicationErrorCodeCategory(最大7文字)
                        "1234567890123456789012345678901234567890123456789012345678901234", // BcatPassphrase(最大64文字)
                        new List<string[]> { new string[] { "Isbn", "1234567890123456789012345678901234567", "36" } })),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        "123456789012345678901234567890123456", // Isbn(最大36文字)
                        "1234567890123456", // DisplayVersion(最大15文字)
                        "1234567", // ApplicationErrorCodeCategory(最大7文字)
                        "1234567890123456789012345678901234567890123456789012345678901234", // BcatPassphrase(最大64文字)
                        new List<string[]> { new string[] { "DisplayVersion", "1234567890123456", "15" } })),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        "123456789012345678901234567890123456", // Isbn(最大36文字)
                        "123456789012345", // DisplayVersion(最大15文字)
                        "12345678", // ApplicationErrorCodeCategory(最大7文字)
                        "1234567890123456789012345678901234567890123456789012345678901234", // BcatPassphrase(最大64文字)
                        new List<string[]> { new string[] { "ApplicationErrorCodeCategory", "12345678", "7" } })),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        "123456789012345678901234567890123456", // Isbn(最大36文字)
                        "123456789012345", // DisplayVersion(最大15文字)
                        "1234567", // ApplicationErrorCodeCategory(最大7文字)
                        "12345678901234567890123456789012345678901234567890123456789012345", // BcatPassphrase(最大64文字)
                        new List<string[]> { new string[] { "BcatPassphrase", "12345678901234567890123456789012345678901234567890123456789012345", "64" } })),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        "1234567890123456789012345678901234567", // Isbn(最大36文字)
                        "1234567890123456", // DisplayVersion(最大15文字)
                        "12345678", // ApplicationErrorCodeCategory(最大7文字)
                        "12345678901234567890123456789012345678901234567890123456789012345", // BcatPassphrase(最大64文字)
                        new List<string[]>
                        {
                            new string[] { "Isbn", "1234567890123456789012345678901234567", "36" },
                            new string[] { "DisplayVersion", "1234567890123456", "15" },
                            new string[] { "ApplicationErrorCodeCategory", "12345678", "7" },
                            new string[] { "BcatPassphrase", "12345678901234567890123456789012345678901234567890123456789012345", "64" },
                        })),
                Tuple.Create(true,     // OK
                    Tuple.Create(
                        "123456789012345678901234567890123456", // Isbn(最大36文字)
                        "123456789012345", // DisplayVersion(最大15文字)
                        "1234567", // ApplicationErrorCodeCategory(最大7文字)
                        "1234567890123456789012345678901234567890123456789012345678901234", // BcatPassphrase(最大64文字)
                        new List<string[]> { new string[] { string.Empty } })),
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, string, string, List<string[]>>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, string, string, List<string[]>>> input)
            {
                // 各種情報を設定
                data.ApplicationControlProperty.Isbn = input.Item2.Item1;
                data.ApplicationControlProperty.DisplayVersion = input.Item2.Item2;
                data.ApplicationControlProperty.ApplicationErrorCodeCategory = input.Item2.Item3;
                data.ApplicationControlProperty.BcatPassphrase = input.Item2.Item4;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, string, string, List<string[]>>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, string, string, List<string[]>>> pattern)
            {
                return pattern.Item2.Item5.Select(entry => new UnpublishableErrorExpectType("Error", "Issue 10-810", entry)).ToList();
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-13でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk13Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, string>> testPatternList = new List<Tuple<bool, string>>
            {
                Tuple.Create(false, (string)null),  // NG
                Tuple.Create(false, string.Empty),  // NG
                Tuple.Create(true, "1.0.0"),        // OK
                Tuple.Create(true, "abcde"),        // OK(空ではないので正常扱い)
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, string>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> input)
            {
                // 表示バージョンを設定
                data.ApplicationControlProperty.DisplayVersion = input.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, string>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 10-012", new string[] { "dummy" }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-15でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk15Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, List<string>>> testPatternList = new List<Tuple<bool, List<string>>>
            {
                Tuple.Create(false, (List<string>)null),                    // NG
                Tuple.Create(false, new List<string> { string.Empty, "test" }),        // NG
                Tuple.Create(true,  new List<string> { "test1", "test2" }),  // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, List<string>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, List<string>> input)
            {
                // 対応言語を設定
                data.ApplicationControlProperty.SupportedLanguage = input.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, List<string>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, List<string>> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 10-014", new string[] { "dummy" }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-1-8でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk1_8Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, Tuple<List<Title>, List<Icon>>>> testPatternList = new List<Tuple<bool, Tuple<List<Title>, List<Icon>>>>
            {
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Title>
                        {
                            new Title { Name = "Name", Language = "AmericanEnglish", Publisher = "Publisher" },
                        },
                        (List<Icon>)null)),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Title>
                        {
                            new Title { Name = "Name", Language = "AmericanEnglish", Publisher = "Publisher" },
                        },
                        new List<Icon> { })),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Title>
                        {
                            new Title { Name = "Name", Language = "AmericanEnglish", Publisher = "Publisher" },
                        },
                        new List<Icon>
                        {
                            new Icon { Language = null, IconPath = new DataPath() { Path = "icon.bmp" } },
                        })),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Title>
                        {
                            new Title { Name = "Name", Language = "AmericanEnglish", Publisher = "Publisher" },
                        },
                        new List<Icon>
                        {
                            new Icon { Language = string.Empty, IconPath = new DataPath() { Path = string.Empty } },
                        })),
                Tuple.Create(true,     // OK
                    Tuple.Create(
                        new List<Title>
                        {
                            new Title { Name = "Name", Language = "AmericanEnglish", Publisher = "Publisher" },
                        },
                        new List<Icon>
                        {
                            new Icon { Language = "AmericanEnglish", IconPath = new DataPath() { Path = "icon.bmp" } },
                        })),
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<Title>, List<Icon>>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<Title>, List<Icon>>> input)
            {
                // タイトル情報を設定
                data.ApplicationControlProperty.Title = input.Item2.Item1;
                data.ApplicationControlProperty.Icon = input.Item2.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<Title>, List<Icon>>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<Title>, List<Icon>>> pattern)
            {
                return new List<UnpublishableErrorExpectType>
                {
                    new UnpublishableErrorExpectType("Error", "Issue 10-607", new string[] { "dummy" }),
                    new UnpublishableErrorExpectType("Error", "Issue 10-608", new string[] { "AmericanEnglish" }) // Iconの言語が空になるのでこのエラーも発生する
                };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-9aでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk9aTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>>> testPatternList = new List<Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>>>
            {
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Title>
                        {
                            new Title { Name = "Name", Language = "AmericanEnglish", Publisher = "Publisher" },
                            new Title { Name = "Name", Language = "Japanese", Publisher = "Publisher" },
                        },
                        new List<Icon>
                        {
                            new Icon { Language = "Japanese", IconPath = new DataPath() { Path = "icon.bmp" } },
                        },
                        new string[] { "AmericanEnglish" })),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Title>
                        {
                            new Title { Name = "Name", Language = "AmericanEnglish", Publisher = "Publisher" },
                            new Title { Name = "Name", Language = "Japanese", Publisher = "Publisher" },
                            new Title { Name = "Name", Language = "French", Publisher = "Publisher" }
                        },
                        new List<Icon>
                        {
                            new Icon { Language = "French", IconPath = new DataPath() { Path = "icon.bmp" } },
                        },
                        new string[] { "AmericanEnglish, Japanese" })),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Title>
                        {
                            new Title { Name = "Name", Language = "AmericanEnglish", Publisher = "Publisher" },
                        },
                        (List<Icon>)null,
                        new string[] { "AmericanEnglish" })),
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>> input)
            {
                // タイトル情報を設定
                data.ApplicationControlProperty.Title = input.Item2.Item1;
                data.ApplicationControlProperty.Icon = input.Item2.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<Title>, List<Icon>, string[]>> pattern)
            {
                if (pattern.Item2.Item2 == null)
                {
                    // Iconがnullの場合、他のエラーも発生する
                    return new List<UnpublishableErrorExpectType>
                    {
                        new UnpublishableErrorExpectType("Error", "Issue 10-607", pattern.Item2.Item3),
                        new UnpublishableErrorExpectType("Error", "Issue 10-608", pattern.Item2.Item3)
                    };
                }
                else
                {
                    return new List<UnpublishableErrorExpectType>
                    {
                        new UnpublishableErrorExpectType("Error", "Issue 10-608", pattern.Item2.Item3)
                    };
                }
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-9cでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk9cTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            string DefaultRawIconHash = "402d395f3ee2eec3ac4d8adfec446a52";
            string DefaultNxIconHash = "8866647b2b59decfc0198d05e1e5cb95";
            string SuccessIconHash = "8866647b2b59decfc0198d05e1e5cb94";

            List<Tuple<bool, Tuple<List<Icon>, string[]>>> testPatternList = new List<Tuple<bool, Tuple<List<Icon>, string[]>>>
            {
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Icon>
                        {
                            new Icon
                            {
                                Language = "Japanese",
                                IconPath = new DataPath() { Path = "icon.bmp" },
                                NxIconHash = DefaultNxIconHash,
                                RawIconHash = DefaultRawIconHash,
                            },
                        },
                        new string[] { "Japanese" })), // エラーメッセージに表示される言語情報
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Icon>
                        {
                            new Icon
                            {
                                Language = "AmericanEnglish",
                                IconPath = new DataPath() { Path = "icon.bmp" },
                                NxIconHash = SuccessIconHash,
                                RawIconHash = DefaultRawIconHash,
                            },
                            new Icon
                            {
                                Language = "Japanese",
                                IconPath = new DataPath() { Path = "icon.bmp" },
                                NxIconHash = DefaultNxIconHash,
                                RawIconHash = SuccessIconHash,
                            },
                        },
                        new string[] { "AmericanEnglish, Japanese" })), // エラーメッセージに表示される言語情報
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        new List<Icon>
                        {
                            new Icon
                            {
                                Language = "AmericanEnglish",
                                IconPath = new DataPath() { Path = "icon.bmp" },
                                NxIconHash = SuccessIconHash,
                                RawIconHash = DefaultRawIconHash,
                            },
                            new Icon
                            {
                                Language = "Japanese",
                                IconPath = new DataPath() { Path = "icon.bmp" },
                                NxIconHash = SuccessIconHash,
                                RawIconHash = SuccessIconHash,
                            },
                        },
                        new string[] { "AmericanEnglish" })), // エラーメッセージに表示される言語情報
                Tuple.Create(true,     // OK
                    Tuple.Create(
                        new List<Icon>
                        {
                            new Icon
                            {
                                Language = "AmericanEnglish",
                                IconPath = new DataPath() { Path = "icon.bmp" },
                                NxIconHash = SuccessIconHash,
                                RawIconHash = SuccessIconHash,
                            },
                            new Icon
                            {
                                Language = "Japanese",
                                IconPath = new DataPath() { Path = "icon.bmp" },
                                NxIconHash = SuccessIconHash,
                                RawIconHash = SuccessIconHash,
                            },
                        },
                        (string[])null)),
                Tuple.Create(true,     // OK
                    Tuple.Create(
                        new List<Icon>
                        {
                            new Icon
                            {
                                Language = "AmericanEnglish",
                                IconPath = new DataPath() { Path = "icon.bmp" },
                                NxIconHash = null,
                                RawIconHash = null,
                            },
                            new Icon
                            {
                                Language = "Japanese",
                                IconPath = new DataPath() { Path = "icon.bmp" },
                                NxIconHash = null,
                                RawIconHash = null,
                            },
                        },
                        (string[])null))
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<Icon>, string[]>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<Icon>, string[]>> input)
            {
                // アイコン情報を設定
                data.ApplicationControlProperty.Icon = input.Item2.Item1;

                // 別エラーを回避する為にタイトル情報を作成
                var title = data.ApplicationControlProperty.Icon.Select(entry => new Title() { Name = "dummy", Publisher = "dummy", Language = entry.Language }).ToList();
                data.ApplicationControlProperty.Title = title;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<Icon>, string[]>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<Icon>, string[]>> pattern)
            {
                return new List<UnpublishableErrorExpectType>
                {
                    new UnpublishableErrorExpectType("Error", "Issue 10-047", pattern.Item2.Item2),
                };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-52bでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk52bTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, Tuple<string, string, string>>> testPatternList = new List<Tuple<bool, Tuple<string, string, string>>>
            {
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        "0_1_2",
                        "0x4000000",
                        "BcatPassphrase")),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        "0_1_2",
                        (string)null,
                        "BcatPassphrase")),
                Tuple.Create(false,     // NG
                    Tuple.Create(
                        "1_1_2",
                        "0x4000000",
                        (string)null)),
                Tuple.Create(true,     // OK
                    Tuple.Create(
                        "3_1_2",
                        "0x4000000",
                        "BcatPassphrase")),
                Tuple.Create(true,     // OK
                    Tuple.Create(
                        "0_1_2",
                        (string)null,
                        (string)null)),
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, string>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, string>> input)
            {
                // SDKバージョンを設定
                data.Nsp.ProgramInfo.SdkVersion = input.Item2.Item1;
                data.Nsp.ProgramInfo.ToolVersion = data.Nsp.ProgramInfo.SdkVersion;
                // BcatDeliveryCacheStorageSize,BcatPassphraseを設定
                data.ApplicationControlProperty.BcatDeliveryCacheStorageSize = input.Item2.Item2;
                data.ApplicationControlProperty.BcatPassphrase = input.Item2.Item3;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, string>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, string>> pattern)
            {
                if ((pattern.Item2.Item2 != pattern.Item2.Item3) && ((pattern.Item2.Item2 == null) || (pattern.Item2.Item3 == null)))
                {
                    // BcatPassphraseとBcatDeliveryCacheStorageSizeの一方しか設定していない時は52dのエラーも発生するのでエラーを追加
                    string[] param;
                    if (pattern.Item2.Item2 == null)
                    {
                        param = new string[] { "BcatPassphrase", "BcatDeliveryCacheStorageSize" };
                    }
                    else
                    {
                        param = new string[] { "BcatDeliveryCacheStorageSize", "BcatPassphrase" };
                    }
                    return new List<UnpublishableErrorExpectType>
                    {
                        new UnpublishableErrorExpectType("Error", "Issue 10-029", new string[] { "dummy" }),
                        new UnpublishableErrorExpectType("Error", "Issue 10-803", param)
                    };
                }
                else
                {
                    return new List<UnpublishableErrorExpectType>
                    {
                        new UnpublishableErrorExpectType("Error", "Issue 10-029", new string[] { "dummy" })
                    };
                }
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-52cで警告が発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk52cTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, string>> testPatternList = new List<Tuple<bool, string>>
            {
                Tuple.Create(false, "0x4000001"),                     // NG
                Tuple.Create(false, long.MaxValue.ToString("X")),     // NG (longの最大値)
                Tuple.Create(true,  "0x4000000"),                     // OK
                Tuple.Create(true,  "0x1"),                           // OK
            };

            // SDKバージョンのメジャー番号を0と1以外に設定する
            checkData.Nsp.ProgramInfo.SdkVersion = "2_0_0";
            checkData.Nsp.ProgramInfo.ToolVersion = checkData.Nsp.ProgramInfo.SdkVersion;

            Action<UnpublishableErrorCheckData, Tuple<bool, string>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> input)
            {
                // BcatDeliveryCacheStorageSizeを設定
                data.ApplicationControlProperty.BcatDeliveryCacheStorageSize = input.Item2;
                data.ApplicationControlProperty.BcatPassphrase = "test";
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, string>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> pattern)
            {
                return new List<UnpublishableErrorExpectType>
                {
                    new UnpublishableErrorExpectType("Warning", "Issue 10-030", new string[] { "dummy" })
                };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-52dで警告が発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk52dTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();
            // 53bを抑制するためSDKバージョンを2以降に設定
            checkData.Nsp.ProgramInfo.SdkVersion = "2_0_0";
            checkData.Nsp.ProgramInfo.ToolVersion = checkData.Nsp.ProgramInfo.SdkVersion;

            List<Tuple<bool, Tuple<string, string, string[]>>> testPatternList = new List<Tuple<bool, Tuple<string, string, string[]>>>
            {
                Tuple.Create(false, Tuple.Create("0x4000000", string.Empty, new string[] { "BcatDeliveryCacheStorageSize", "BcatPassphrase" })), // NG
                Tuple.Create(false, Tuple.Create(string.Empty, "test", new string[] { "BcatPassphrase", "BcatDeliveryCacheStorageSize" })), // NG
                Tuple.Create(true,  Tuple.Create(string.Empty, string.Empty, new string[] { "dummy" })), // OK
                Tuple.Create(true,  Tuple.Create("0x4000000", "test", new string[] { "dummy" })), // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, string[]>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, string[]>> input)
            {
                // BcatDeliveryCacheStorageSizeとBcatPassphraseを設定
                data.ApplicationControlProperty.BcatDeliveryCacheStorageSize = input.Item2.Item1;
                data.ApplicationControlProperty.BcatPassphrase = input.Item2.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, string[]>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, string[]>> pattern)
            {
                return new List<UnpublishableErrorExpectType>
                {
                    new UnpublishableErrorExpectType("Error", "Issue 10-803", pattern.Item2.Item3)
                };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-43bでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk43bTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, Tuple<string, string, string[], string, string[]>>> testPatternList = new List<Tuple<bool, Tuple<string, string, string[], string, string[]>>>
            {
                // Coreのチェック
                Tuple.Create(false, // NG
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 "0x01004b9000490000", // Core/ApplicationId
                                 new string[] { "0x01004b9000490000" }, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null, // ContentMeta/Id
                                 (string[])null)), // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                Tuple.Create(false, // NG
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 "0x01004b9000490000", // Core/ApplicationId
                                 new string[] { "0x01004b9000490000", "0" }, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null, // ContentMeta/Id
                                 (string[])null)), // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                Tuple.Create(true, // OK
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 "0x01004b9000490000", // Core/ApplicationId
                                 new string[] { "0" }, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null, // ContentMeta/Id
                                 (string[])null)), // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                Tuple.Create(true, // OK
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 (string)null, // Core/ApplicationId
                                 new string[] { "0" }, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null, // ContentMeta/Id
                                 (string[])null)), // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                Tuple.Create(true, // OK
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 "0x01004b9000490000", // Core/ApplicationId
                                 (string[])null, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null, // ContentMeta/Id
                                 (string[])null)), // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id

                // ContentMetaのチェック
                Tuple.Create(false, // NG
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 (string)null, // Core/ApplicationId
                                 (string[])null, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 "0x01004b9000490000", // ContentMeta/Id
                                 new string[] { "0x01004b9000490000" })), // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                Tuple.Create(false, // NG
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 (string)null, // Core/ApplicationId
                                 (string[])null, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 "0x01004b9000490000", // ContentMeta/Id
                                 new string[] { "0x01004b9000490000", "0" })), // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                Tuple.Create(true, // OK
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 (string)null, // Core/ApplicationId
                                 (string[])null, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 "0x01004b9000490000", // ContentMeta/Id
                                 new string[] { "0" })), // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                Tuple.Create(true, // OK
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 (string)null, // Core/ApplicationId
                                 (string[])null, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null, // ContentMeta/Id
                                 new string[] { "0" })), // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                Tuple.Create(true, // OK
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 (string)null, // Core/ApplicationId
                                 (string[])null, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 "0x01004b9000490000", // ContentMeta/Id
                                 (string[])null)), // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id

                // パッチのチェック
                Tuple.Create(false, // NG
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypePatch,
                                 "0x01004b9000490000", // Core/ApplicationId
                                 new string[] { "0" }, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 "0x01004b9000490000", // ContentMeta/Id
                                 new string[] { "0x01004b9000490000" })), // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                Tuple.Create(true, // OK
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypePatch,
                                 "0x01004b9000490000", // Core/ApplicationId
                                 new string[] { "0" }, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 "0x01004b9000490000", // ContentMeta/Id
                                 new string[] { "0" })), // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, string[], string, string[]>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, string[], string, string[]>> input)
            {
                // Coreの設定
                data.Nmeta.Core = new CoreModel();
                data.Nmeta.Core.ApplicationId = input.Item2.Item2;
                data.Nmeta.CoreFsAccessControlData = new FsAccessControlData() { Entries = null };
                if (input.Item2.Item3 != null)
                {
                    data.Nmeta.CoreFsAccessControlData.Entries = input.Item2.Item3.Select(entry => new SaveDataOwnerIdsModel() { Id = entry }).ToList();
                }

                // ContentMetaの設定
                if (input.Item2.Item1 == NintendoContentMetaConstant.ContentMetaTypePatch)
                {
                    var patchMeta = new PatchContentMetaModel();
                    if (input.Item2.Item4 == input.Item2.Item5[0])
                    {
                        // SaveDataOwnerIds/Idの中にContentMeta/ApplicationIdと同じものがある場合はContentMeta/Idに適当な値を設定する
                        patchMeta.Id = "0x0000000000000000";
                    }
                    else
                    {
                        // SaveDataOwnerIds/Idの中にContentMeta/ApplicationIdと同じものが無い場合はContentMeta/IdにSaveDataOwnerIds/Idの値を設定する
                        patchMeta.Id = input.Item2.Item5[0];
                    }

                    patchMeta.ApplicationId = input.Item2.Item4;
                    var contentModelList = new List<ContentModel>();
                    // ContentMetaにProgramが登録されている
                    contentModelList.Add(new ContentModel() { Type = NintendoContentMetaConstant.ContentTypeProgram });
                    // ContentMetaにLegalInformationが登録されている
                    contentModelList.Add(new ContentModel() { Type = NintendoContentMetaConstant.ContentTypeLegalInformation });
                    patchMeta.ContentList = contentModelList;
                    checkData.Nsp.ContentMeta = patchMeta;
                }
                else
                {
                    data.Nsp.ContentMeta.Id = input.Item2.Item4;
                }
                checkData.Nsp.ContentMeta.Type = input.Item2.Item1;

                // SbmChk-41aのエラーを抑制するために SoftwareLegalInformation/ApplicationId を設定ContentMeta/Id と同じにする
                data.Nsp.LegalInformation.ApplicationId = input.Item2.Item4;

                // ProgramInfoの設定
                if (input.Item2.Item5 != null)
                {
                    data.Nsp.ProgramInfo.FsAccessControlData.Entries = input.Item2.Item5.Select(entry => new SaveDataOwnerIdsModel() { Id = entry }).ToList();
                }
                else
                {
                    data.Nsp.ProgramInfo.FsAccessControlData.Entries = null;
                }
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, string[], string, string[]>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, string[], string, string[]>> pattern)
            {
                return new List<UnpublishableErrorExpectType>
                {
                    new UnpublishableErrorExpectType("Error", "Issue 10-811", new string[] { "dummy" })
                };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-18cでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk18cTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, Tuple<string, string>>> testPatternList = new List<Tuple<bool, Tuple<string, string>>>
            {
                Tuple.Create(false,
                    Tuple.Create(string.Empty, "_ZN2nn3err20ShowApplicationErrorERKNS0_19ApplicationErrorArgE")),
                Tuple.Create(false,
                    Tuple.Create((string)null, "_ZN2nn3err20ShowApplicationErrorERKNS0_19ApplicationErrorArgE")),
                Tuple.Create(true,
                    Tuple.Create("ABC12", "_ZN2nn3err20ShowApplicationErrorERKNS0_19ApplicationErrorArgE")),
                Tuple.Create(true,
                    Tuple.Create(string.Empty, "ZN2nn3err20ShowApplicationErrorERKNS0_19ApplicationErrorArgE")),
                Tuple.Create(true,
                    Tuple.Create((string)null, "__ZN2nn3err20ShowApplicationErrorERKNS0_19ApplicationErrorArgE")),
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string>> input)
            {
                // ApplicationErrorCodeCategoryを設定
                data.ApplicationControlProperty.ApplicationErrorCodeCategory = input.Item2.Item1;
                // UnresolvedApiにAPIを登録
                var apiList = UnresolvedApiList.Create();
                apiList.Entries.Add(new UnresolvedApiModel() { ApiName = input.Item2.Item2, NsoName = "test.nso" });
                checkData.Nsp.ProgramInfo.UnresolvedApiListData = apiList;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string>> pattern)
            {
                return new List<UnpublishableErrorExpectType>() { new UnpublishableErrorExpectType("Error", "Issue 10-046", new string[] { "dummy" }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-54でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk54Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, string[]>> testPatternList = new List<Tuple<bool, string[]>>
            {
                Tuple.Create(false, UnpublishableApiList.UnpublishableXpadApiTable.Select(entry => entry.Key).ToArray()), // NG(使用禁止の全てのXpadAPIのマングル名)
                Tuple.Create(true,  UnpublishableApiList.UnpublishableXpadApiTable.Select(entry => entry.Key + "_").ToArray()), // OK(使用禁止の全てのXpadAPIのマングル名の最後に"_"を追加)
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, string[]>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, string[]> input)
            {
                // UnresolvedApiにAPIを登録
                var apiList = UnresolvedApiList.Create();
                apiList.Entries.AddRange(input.Item2.Select(entry => new UnresolvedApiModel() { ApiName = entry, NsoName = "test.nso" }).ToList());
                checkData.Nsp.ProgramInfo.UnresolvedApiListData = apiList;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, string[]>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, string[]> pattern)
            {
                var demanglingApiList =  pattern.Item2.Where(entry => UnpublishableApiList.IsUnpublishableXpadApi(entry)).Select(entry => UnpublishableApiList.GetDemanglingApiNameForXpadApi(entry)).ToArray();

                return new List<UnpublishableErrorExpectType>() { new UnpublishableErrorExpectType("Error", "Issue 11-010", new string[] { string.Join(Environment.NewLine, demanglingApiList) }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-22でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk22Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, Tuple<string, string[], List<string[]>>>> testPatternList = new List<Tuple<bool, Tuple<string, string[], List<string[]>>>>
            {
                // NGとならない項目
                Tuple.Create(true, Tuple.Create("1_0", new string[] { "_ZN2nn3nsd9ResolveExEPNS0_4FqdnERKS1_" }, new List<string[]>() { new string[] { "dummy" } })), // OK
                Tuple.Create(true, Tuple.Create("3_0", new string[] { "_ZN2nn3nsd9ResolveExEPNS0_4FqdnERKS1_" }, new List<string[]>() { new string[] { "dummy" } })), // OK
                Tuple.Create(true, Tuple.Create("4_0", new string[] { "_ZN2nn3nsd9ResolveExEPNS0_4FqdnERKS1_" }, new List<string[]>() { new string[] { "dummy" } })), // OK
                Tuple.Create(true, Tuple.Create("1_0", new string[] { "nnsocketCancel" }, new List<string[]>() { new string[] { "dummy" } })), // OK
                Tuple.Create(true, Tuple.Create("3_0", new string[] { "nnsocketGetHostByNameCancel" }, new List<string[]>() { new string[] { "dummy" } })), // OK
                Tuple.Create(true, Tuple.Create("4_0", new string[] { "nnsocketRequestCancelHandle" }, new List<string[]>() { new string[] { "dummy" } })), // OK

                // SDKバージョンが0でのみNGとなる項目
                Tuple.Create(false, Tuple.Create("0_12", new string[] { "_ZN2nn2oe23BeginBlockingHomeButtonENS_8TimeSpanE" }, new List<string[]>() { new string[] { "nn::oe::BeginBlockingHomeButton(nn::TimeSpan)", "Bundle_BlockingHomeButton_NX" } })), // NG
                Tuple.Create(true,  Tuple.Create("1_12", new string[] { "_ZN2nn2oe23BeginBlockingHomeButtonENS_8TimeSpanE" }, new List<string[]>() { new string[] { "dummy" } })), // OK
                Tuple.Create(true,  Tuple.Create("3_0", new string[] { "_ZN2nn2oe23BeginBlockingHomeButtonENS_8TimeSpanE" }, new List<string[]>() { new string[] { "dummy" } })),  // OK
                // SDKバージョンが1,3,4でNGとなる項目
                Tuple.Create(false, Tuple.Create("1_7", new string[] { "_ZN2nn2oe18ReportUserIsActiveEv" }, new List<string[]>() { new string[] { "nn::oe::ReportUserIsActive()", "Bundle_OeReportUserIsActive_NX" } })), // NG
                Tuple.Create(false, Tuple.Create("3_0", new string[] { "_ZN2nn2pl21GetCurrentIlluminanceEv" }, new List<string[]>() { new string[] { "nn::pl::GetCurrentIlluminance()", "Bundle_Private_Illuminance_NX" } })), // NG
                Tuple.Create(false, Tuple.Create("4_0", new string[] { "_ZN2nn2os13SuspendThreadEPNS0_10ThreadTypeE" }, new List<string[]>() { new string[] { "nn::os::SuspendThread(nn::os::ThreadType*)", "Bundle_Private_Unity_Generic" } })), // NG
                // SDKバージョンが存在しないバージョン    4_BundleRules_NX_Private_RetailInteractiveDisplay_RetailInteractiveDisplay
                Tuple.Create(true,  Tuple.Create("2_0", new string[] { "_ZN2nn2oe23BeginBlockingHomeButtonENS_8TimeSpanE" }, new List<string[]>() { new string[] { "dummy" } })), // OK

                // 複数の禁止APIが登録されている + 一つのAPIに複数のBundle名がある場合、Bundle名毎にエラーが作成される
                Tuple.Create(false, Tuple.Create("1",
                    new string[] { "_ZN2nn2oe18ReportUserIsActiveEv", "_ZN2nn7friends17SendFriendRequestEPNS0_12AsyncContextERKNS_7account3UidENS3_23NetworkServiceAccountIdERKNS0_15InAppScreenNameESA_" },
                        new List<string[]>() {
                            new string[] { "nn::oe::ReportUserIsActive()", "Bundle_OeReportUserIsActive_NX" },
                            new string[] { "nn::friends::SendFriendRequest(nn::friends::AsyncContext*, nn::account::Uid const&, nn::account::NetworkServiceAccountId, nn::friends::InAppScreenName const&, nn::friends::InAppScreenName const&)", "Bundle_FriendRequestKit_Generic" },
                            new string[] { "nn::friends::SendFriendRequest(nn::friends::AsyncContext*, nn::account::Uid const&, nn::account::NetworkServiceAccountId, nn::friends::InAppScreenName const&, nn::friends::InAppScreenName const&)", "Bundle_FriendRequestKit_NX" },
                        })), //NG

                // 複数の禁止APIが登録されている + 同じBundleに複数のAPIがある場合、Bundle名毎に複数のAPIが記載されたエラーが作成される
                Tuple.Create(false, Tuple.Create("0",
                    new string[] { "eglBindAPI", "_ZN2nv22SetGraphicsServiceNameEPKc", "eglChooseConfig", "_ZN2nn2oe20GetDefaultThemeColorEv", "_ZN2nn2oe21GetExpectedThemeColorEv" },
                        new List<string[]>() {
                            new string[] { "eglBindAPI" + Environment.NewLine
                                            + "nv::SetGraphicsServiceName(char const*)" + Environment.NewLine
                                            + "eglChooseConfig",
                                            "Bundle_GLPrivate_NX" },
                            new string[] { "nv::SetGraphicsServiceName(char const*)",
                                            "Bundle_NvnPrivate_NX" },
                            new string[] { "nn::oe::GetDefaultThemeColor()" + Environment.NewLine
                                            + "nn::oe::GetExpectedThemeColor()",
                                            "Bundle_TurboSKit_NX" },
                        })), //NG

                // Bundle名変換のチェック
                // 0_12_BundleRules_PlayReportKit_Generic_Files_play-report-kit → Bundle_PlayReportKit_Generic
                Tuple.Create(false, Tuple.Create("0", new string[] { "_ZN2nn5prepo10InitializeEv" }, new List<string[]>() { new string[] { "nn::prepo::Initialize()", "Bundle_PlayReportKit_Generic" } })),             // NG
                // 0_12_BundleRules_NintendoSDK_NX_Files_consolesixaxissensor → Bundle_NintendoSDK_NX
                Tuple.Create(false, Tuple.Create("0", new string[] { "_ZN3nnd7lsm6ds310InitializeEv" }, new List<string[]>() { new string[] { "nnd::lsm6ds3::Initialize()", "Bundle_NintendoSDK_NX" } })),              // NG
                // 1_BundleDefinitions_NX_Ldn_Private → Bundle_Ldn_Private_NX
                Tuple.Create(false, Tuple.Create("1", new string[] { "_ZN2nn3ldnL11PriorityMinE" }, new List<string[]>() { new string[] { "nn::ldn::PriorityMin", "Bundle_Ldn_Private_NX" } })),                        // NG
                // 1_BundleDefinitions_NX_OeReportUserIsActive → Bundle_OeReportUserIsActive_NX
                Tuple.Create(false, Tuple.Create("1", new string[] { "_ZN2nn2oe18ReportUserIsActiveEv" }, new List<string[]>() { new string[] { "nn::oe::ReportUserIsActive()", "Bundle_OeReportUserIsActive_NX" } })), // NG
                // 3_BundleRules_Generic_Private_PlayReport_play-report → Bundle_Private_PlayReport_Generic
                Tuple.Create(false, Tuple.Create("3", new string[] { "_ZN2nn5prepo10InitializeEv" }, new List<string[]>() {        // NG
                    new string[] { "nn::prepo::Initialize()", "Bundle_Private_PlayReport_Generic" } })),

                // APIリストに存在しないsdkバージョン
                Tuple.Create(false, Tuple.Create("99_0", new string[] { "_ZN2nn2os13SuspendThreadEPNS0_10ThreadTypeE" }, new List<string[]>() { new string[] { "nn::os::SuspendThread(nn::os::ThreadType*)", "Bundle_Private_Unity_Generic" } })), // NG
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string[], List<string[]>>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string[], List<string[]>>> input)
            {
                // SDKバージョンを設定
                checkData.Nsp.ProgramInfo.SdkVersion = input.Item2.Item1;
                checkData.Nsp.ProgramInfo.ToolVersion = checkData.Nsp.ProgramInfo.SdkVersion;

                // UnresolvedApiを設定
                var apiList = UnresolvedApiList.Create();
                apiList.Entries.AddRange(input.Item2.Item2.Select(entry => new UnresolvedApiModel() { ApiName = entry, NsoName = "test.nso" }).ToList());
                checkData.Nsp.ProgramInfo.UnresolvedApiListData = apiList;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string[], List<string[]>>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string[], List<string[]>>> pattern)
            {
                return pattern.Item2.Item3.Select(entry => new UnpublishableErrorExpectType("Warning", "Issue 11-003", entry)).ToList();
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-22b,SbmChk-22cでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk22SpecificRulesTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();
            checkData.Nsp.ProgramInfo.SdkVersion = "0_12";
            checkData.Nsp.ProgramInfo.ToolVersion = checkData.Nsp.ProgramInfo.SdkVersion;

            List<Tuple<bool, Tuple<string[], string[], List<string[]>>>> testPatternList = new List<Tuple<bool, Tuple<string[], string[], List<string[]>>>>
            {
                // SbmChk-22c-BundleRules_NetworkMiddlewareKit_[NX|Generic]_Files_NetworkMiddleware
                Tuple.Create(false, Tuple.Create(new string[] { "Unity" }, new string[] { "_ZN2nn5codec11OpusEncoderD1Ev" }, new List<string[]>() { new string[] { "nn::codec::OpusEncoder::~OpusEncoder()", "Bundle_NetworkMiddlewareKit_Generic" } })),       // NG
                Tuple.Create(false, Tuple.Create(new string[] { "pia", "nex" }, new string[] { "_ZN2nn5codec11OpusEncoderD1Ev" }, new List<string[]>() { new string[] { "nn::codec::OpusEncoder::~OpusEncoder()", "Bundle_NetworkMiddlewareKit_Generic" } })),  // NG
                Tuple.Create(true, Tuple.Create(new string[] { "Pia", "Unity" }, new string[] { "_ZN2nn5codec11OpusEncoderD1Ev" }, new List<string[]>() { new string[] { "dummy" } })), // OK
                Tuple.Create(true, Tuple.Create(new string[] { "Unity", "NEX" }, new string[] { "_ZN2nn5codec11OpusEncoderD1Ev" }, new List<string[]>() { new string[] { "dummy" } })), // OK
                Tuple.Create(true,  Tuple.Create(new string[] { "Pia", "NEX" }, new string[] { "_ZN2nn5codec11OpusEncoderD1Ev" }, new List<string[]>() { new string[] { "dummy" } })),                 // OK
                Tuple.Create(true,  Tuple.Create(new string[] { "HohePia-0_0", "NEXHoge-0_0" }, new string[] { "_ZN2nn5codec11OpusEncoderD1Ev" }, new List<string[]>() { new string[] { "dummy" } })), // OK

                // SbmChk-22c-BundleRules_LdnPrivate_NX_Ldn_Private
                Tuple.Create(false, Tuple.Create(new string[] { "Unity" }, new string[] { "_ZN2nn3ldn11ScanPrivateEPNS0_11NetworkInfoEPiiRKNS0_10ScanFilterEi" }, new List<string[]>() { new string[] { "nn::ldn::ScanPrivate(nn::ldn::NetworkInfo*, int*, int, nn::ldn::ScanFilter const&, int)", "Bundle_LdnPrivate_NX" } })),       // NG
                Tuple.Create(false,  Tuple.Create(new string[] { "pia", "nex" }, new string[] { "_ZN2nn3ldn11ScanPrivateEPNS0_11NetworkInfoEPiiRKNS0_10ScanFilterEi" }, new List<string[]>() { new string[] { "nn::ldn::ScanPrivate(nn::ldn::NetworkInfo*, int*, int, nn::ldn::ScanFilter const&, int)", "Bundle_LdnPrivate_NX" } })), // NG
                Tuple.Create(true, Tuple.Create(new string[] { "Pia", "Unity" }, new string[] { "_ZN2nn3ldn11ScanPrivateEPNS0_11NetworkInfoEPiiRKNS0_10ScanFilterEi" }, new List<string[]>() { new string[] { "dummy" } })), // OK
                Tuple.Create(true, Tuple.Create(new string[] { "Unity", "NEX" }, new string[] { "_ZN2nn3ldn11ScanPrivateEPNS0_11NetworkInfoEPiiRKNS0_10ScanFilterEi" }, new List<string[]>() { new string[] { "dummy" } })), // OK
                Tuple.Create(true,  Tuple.Create(new string[] { "Pia", "NEX" }, new string[] { "_ZN2nn3ldn11ScanPrivateEPNS0_11NetworkInfoEPiiRKNS0_10ScanFilterEi" }, new List<string[]>() { new string[] { "dummy" } })),                 // OK
                Tuple.Create(true,  Tuple.Create(new string[] { "PiaHoge-0_0", "HogeNEX-0_0" }, new string[] { "_ZN2nn3ldn11ScanPrivateEPNS0_11NetworkInfoEPiiRKNS0_10ScanFilterEi" }, new List<string[]>() { new string[] { "dummy" } })), // OK
                Tuple.Create(false, Tuple.Create(new string[] { "Pia", "NEX" }, new string[] { "_ZN2nn3ldn14FinalizeSystemEv" }, new List<string[]>() { new string[] { "nn::ldn::FinalizeSystem()", "Bundle_LdnSystem_NX" } })),            // NG(LdnSystem,Ldn_SystemはNEX,Piaが登録されていてもエラーになる)
                Tuple.Create(false, Tuple.Create(new string[] { "Pia", "NEX" }, new string[] { "_ZN2nn3ldn18GetStateForMonitorEv" }, new List<string[]>() { new string[] { "nn::ldn::GetStateForMonitor()", "Bundle_LdnMonitor_NX" } })),   // NG(LdnMonitor,Ldn_MonitorはNEX,Piaが登録されなくてもエラーになる)

                // SbmChk-22b-BundleRules_UnityKit_Generic_Files
                Tuple.Create(false, Tuple.Create(new string[] { "Pia", "NEX" }, new string[] { "_ZN2nn2os12ResumeThreadEPNS0_10ThreadTypeE" }, new List<string[]>() { new string[] { "nn::os::ResumeThread(nn::os::ThreadType*)", "Bundle_UnityKit_Generic" } })),      // NG
                Tuple.Create(false, Tuple.Create(new string[] { "unity" }, new string[] { "_ZN2nn2os12ResumeThreadEPNS0_10ThreadTypeE" }, new List<string[]>() { new string[] { "nn::os::ResumeThread(nn::os::ThreadType*)", "Bundle_UnityKit_Generic" } })),           // NG
                Tuple.Create(true,  Tuple.Create(new string[] { "Unity" }, new string[] { "_ZN2nn2os12ResumeThreadEPNS0_10ThreadTypeE" }, new List<string[]>() { new string[] { "dummy" } })),           // OK
                Tuple.Create(true,  Tuple.Create(new string[] { "UnityHoge-0_0" }, new string[] { "_ZN2nn2os12ResumeThreadEPNS0_10ThreadTypeE" }, new List<string[]>() { new string[] { "dummy" } })),   // OK
                Tuple.Create(true,  Tuple.Create(new string[] { "HogeUnity-0_0" }, new string[] { "_ZN2nn2os12ResumeThreadEPNS0_10ThreadTypeE" }, new List<string[]>() { new string[] { "dummy" } })),   // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string[], string[], List<string[]>>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string[], string[], List<string[]>>> input)
            {
                // MiddlewareListを設定
                var middlewareList = MiddlewareList.Create();
                middlewareList.Entries.AddRange(input.Item2.Item1.Select(entry => new MiddlewareModel() { ModuleName = entry, NsoName = "dummy.nso", VenderName = "dummy" }).ToList());
                checkData.Nsp.ProgramInfo.MiddlewareListData = middlewareList;

                // UnresolvedApiを設定
                var apiList = UnresolvedApiList.Create();
                apiList.Entries.AddRange(input.Item2.Item2.Select(entry => new UnresolvedApiModel() { ApiName = entry, NsoName = "test.nso" }).ToList());
                checkData.Nsp.ProgramInfo.UnresolvedApiListData = apiList;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string[], string[], List<string[]>>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string[], string[], List<string[]>>> pattern)
            {
                return pattern.Item2.Item3.Select(entry => new UnpublishableErrorExpectType("Warning", "Issue 11-003", entry)).ToList();
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-8aでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk8aTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            const ulong LimitSize = 67108864;

            List<Tuple<bool, Tuple<ulong, ulong>>> testPatternList = new List<Tuple<bool, Tuple<ulong, ulong>>>
            {
                Tuple.Create(false, Tuple.Create(LimitSize, (ulong)1)),  // NG
                Tuple.Create(false, Tuple.Create((ulong)1, LimitSize)),  // NG
                Tuple.Create(false, Tuple.Create(LimitSize + 1, (ulong)0)),  // NG
                Tuple.Create(false, Tuple.Create((ulong)0, LimitSize + 1)),  // NG
                Tuple.Create(true, Tuple.Create(LimitSize, (ulong)0)),  // OK
                Tuple.Create(true, Tuple.Create((ulong)0, LimitSize)),  // OK
                Tuple.Create(true, Tuple.Create(LimitSize / 2, LimitSize / 2)),  // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<ulong, ulong>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<ulong, ulong>> input)
            {
                // バージョンを設定
                data.ApplicationControlProperty.UserAccountSaveDataSize = "0x" + input.Item2.Item1.ToString("X16");
                data.ApplicationControlProperty.UserAccountSaveDataJournalSize = "0x" + input.Item2.Item2.ToString("X16");
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<ulong, ulong>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<ulong, ulong>> pattern)
            {
                var size = (pattern.Item2.Item1 + pattern.Item2.Item2) / (1024 * 1024);

                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Warning", "Issue 10-032", new string[] { size.ToString() }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-8kでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk8kTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, Tuple<string, string>>> testPatternList = new List<Tuple<bool, Tuple<string, string>>>
            {
                Tuple.Create(true,  Tuple.Create((string)null, (string)null)),     // OK
                Tuple.Create(true,  Tuple.Create("0x0", "0x0")),                   // OK
                Tuple.Create(false, Tuple.Create("0x0000000003F00000", "0x100")),  // NG
                Tuple.Create(false, Tuple.Create("0x0000000003F00000", "0x0")),    // NG
                Tuple.Create(false, Tuple.Create("0x0", "0x100")),                 // NG
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string>> input)
            {
                // バージョンを設定
                data.ApplicationControlProperty.CacheStorageDataAndJournalSizeMax = input.Item2.Item1;
                data.ApplicationControlProperty.CacheStorageIndexMax = input.Item2.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string>> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Warning", "Issue 10-049", new string[] { "dummy" }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-16でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk16Test()
        {
            // nmetaのテスト
            {
                // テストデータ作成
                var checkData = CreateSimpleSuccessData();

                var id1 = "0x0100000000001000";
                var id2 = "0x0100000000001002";
                List<Tuple<bool, Tuple<string, string>>> testPatternList = new List<Tuple<bool, Tuple<string, string>>>
                {
                    Tuple.Create(false, Tuple.Create(id1, id2)), // NG
                    Tuple.Create(true, Tuple.Create(id1, id1)), // OK
                };

                Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string>> input)
                {
                    data.Nmeta.Core = new CoreModel();
                    data.Nmeta.Core.ApplicationId = input.Item2.Item1;
                    data.ApplicationControlProperty.PresenceGroupId = input.Item2.Item2;

                    // 他のエラー抑制の為、各情報のIDを設定する
                    data.Nsp.ContentMeta.Id = data.Nmeta.Core.ApplicationId;
                    data.Nsp.LegalInformation.ApplicationId = data.Nmeta.Core.ApplicationId;
                };

                Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string>> pattern)
                {
                    return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Warning", "Issue 10-015", new string[] { pattern.Item2.Item2 }) };
                };

                // テスト実施
                CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
            }

            // ApplicationControlPropertyXMLのテスト
            {
                // テストデータ作成
                var checkData = CreateSimpleSuccessData();

                var id1 = "0x0100000000001000";
                var id2 = "0x0100000000001002";
                List<Tuple<bool, Tuple<string, string, string>>> testPatternList = new List<Tuple<bool, Tuple<string, string, string>>>
                {
                    Tuple.Create(false, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication, id1, id2)), // NG
                    Tuple.Create(true, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication, id1, id1)), // OK
                    Tuple.Create(false, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypePatch, id1, id2)), // NG
                    Tuple.Create(true, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypePatch, id2, id2))  // OK
                };

                Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, string>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, string>> input)
                {
                    var type = input.Item2.Item1;
                    data.CheckContentType = type;

                    string applicationId = input.Item2.Item2;
                    data.Nsp.ContentMeta.Id = applicationId;
                    data.Nsp.LegalInformation.ApplicationId = applicationId;
                    data.ApplicationControlProperty.PresenceGroupId = input.Item2.Item3;
                    if (type == NintendoContentMetaConstant.ContentMetaTypePatch)
                    {
                        System.Random rand = new System.Random();
                        var patchMeta = new PatchContentMetaModel();
                        patchMeta.Id = string.Format("0x{0}", rand.Next());
                        patchMeta.ApplicationId = applicationId;
                        var contentModelList = new List<ContentModel>();
                        // ContentMetaにProgramが登録されている
                        contentModelList.Add(new ContentModel() { Type = NintendoContentMetaConstant.ContentTypeProgram });
                        // ContentMetaにLegalInformationが登録されている
                        contentModelList.Add(new ContentModel() { Type = NintendoContentMetaConstant.ContentTypeLegalInformation });
                        patchMeta.ContentList = contentModelList;
                        patchMeta.Type = type;
                        data.Nsp.ContentMeta = patchMeta;
                    }
                };

                Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, string>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, string>> pattern)
                {
                    return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Warning", "Issue 10-015", new string[] { pattern.Item2.Item3 }) };
                };

                // テスト実施
                CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
            }
        }

        // SbmChk-17aでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk17aTest()
        {
            // nmetaのテスト
            {
                // テストデータ作成
                var checkData = CreateSimpleSuccessData();
                var id1 = "0x0100000000001000";
                var id2 = "0x0100000000001002";
                var id3 = "0x0100000000001003";
                List<Tuple<bool, Tuple<string, List<string>, string>>> testPatternList = new List<Tuple<bool, Tuple<string, List<string>, string>>>
                {
                    Tuple.Create(false, Tuple.Create(id1, new List<string> { id2 }, id2)), // NG
                    Tuple.Create(false, Tuple.Create(id1, new List<string> { id2, id3 }, id2 + ", " + id3)), // NG
                    Tuple.Create(true, Tuple.Create(id1, new List<string> { id1 }, (string)null)), // OK
                };

                Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, List<string>, string>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, List<string>, string>> input)
                {
                    data.Nmeta.Core = new CoreModel();
                    data.Nmeta.Core.ApplicationId = input.Item2.Item1;
                    data.ApplicationControlProperty.LocalCommunicationId = input.Item2.Item2;

                    // 他のエラー抑制の為、各情報のIDを設定する
                    data.Nsp.ContentMeta.Id = data.Nmeta.Core.ApplicationId;
                    data.Nsp.LegalInformation.ApplicationId = data.Nmeta.Core.ApplicationId;
                };

                Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, List<string>, string>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, List<string>, string>> pattern)
                {
                    return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Warning", "Issue 10-016", new string[] { pattern.Item2.Item3 }) };
                };

                // テスト実施
                CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
            }

            // ApplicationControlPropertyXMLのテスト
            {
                // テストデータ作成
                var checkData = CreateSimpleSuccessData();
                var id1 = "0x0100000000001000";
                var id2 = "0x0100000000001002";
                var id3 = "0x0100000000001003";
                List<Tuple<bool, Tuple<string, string, List<string>, string>>> testPatternList = new List<Tuple<bool, Tuple<string, string, List<string>, string>>>
                {
                    Tuple.Create(false, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication, id1, new List<string> { id2 }, id2)), // NG
                    Tuple.Create(true, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication, id1, new List<string> { id1 }, (string)null)), // OK
                    Tuple.Create(false, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypePatch, id1, new List<string> { id2 }, id2)), // NG
                    Tuple.Create(false, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypePatch, id1, new List<string> { id2, id3 }, id2 + ", " + id3)), // NG
                    Tuple.Create(true, Tuple.Create(NintendoContentMetaConstant.ContentMetaTypePatch, id2, new List<string> { id2 }, (string)null)) // OK
                };

                Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, List<string>, string>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, List<string>, string>> input)
                {
                    var type = input.Item2.Item1;
                    data.CheckContentType = type;

                    string applicationId = input.Item2.Item2;
                    data.Nsp.ContentMeta.Id = applicationId;
                    data.Nsp.LegalInformation.ApplicationId = applicationId;
                    data.ApplicationControlProperty.LocalCommunicationId = input.Item2.Item3;
                    if (type == NintendoContentMetaConstant.ContentMetaTypePatch)
                    {
                        System.Random rand = new System.Random();
                        var patchMeta = new PatchContentMetaModel();
                        patchMeta.Id = string.Format("0x{0}", rand.Next());
                        patchMeta.ApplicationId = applicationId;
                        var contentModelList = new List<ContentModel>();
                        // ContentMetaにProgramが登録されている
                        contentModelList.Add(new ContentModel() { Type = NintendoContentMetaConstant.ContentTypeProgram });
                        // ContentMetaにLegalInformationが登録されている
                        contentModelList.Add(new ContentModel() { Type = NintendoContentMetaConstant.ContentTypeLegalInformation });
                        patchMeta.ContentList = contentModelList;
                        patchMeta.Type = type;
                        data.Nsp.ContentMeta = patchMeta;
                    }
                };

                Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, List<string>, string>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, List<string>, string>> pattern)
                {
                    return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Warning", "Issue 10-016", new string[] { pattern.Item2.Item4 }) };
                };

                // テスト実施
                CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
            }
        }

        // SbmChk-43でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk43Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, Tuple<string, string, string[], string, string[], string>>> testPatternList = new List<Tuple<bool, Tuple<string, string, string[], string, string[], string>>>
            {
                // Coreのチェック
                Tuple.Create(false, // NG
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 "0x01004b9000490000", // Core/ApplicationId
                                 new string[] { "0x01004b9000490001" }, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null, // ContentMeta/Id
                                 (string[])null, // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                                 "0x01004b9000490001")),
                Tuple.Create(false, // NG
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 "0x01004b9000490000", // Core/ApplicationId
                                 new string[] { "0x01004b9000490001", "0", "0x01004b9000490002" }, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null, // ContentMeta/Id
                                 (string[])null, // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                                 "0x01004b9000490001, 0x01004b9000490002")),
                Tuple.Create(true, // OK
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 "0x01004b9000490000", // Core/ApplicationId
                                 new string[] { "0" }, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null, // ContentMeta/Id
                                 (string[])null, // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null)),
                Tuple.Create(true, // OK
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 (string)null, // Core/ApplicationId
                                 new string[] { "0x01004b9000490001" }, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null, // ContentMeta/Id
                                 (string[])null, // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null)),
                Tuple.Create(true, // OK
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 "0x01004b9000490000", // Core/ApplicationId
                                 (string[])null, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null, // ContentMeta/Id
                                 (string[])null, // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null)),

                // SaveDataOwnerIds/Id が ApplicationIdと同じ場合は、SbmChk-43エラーにはならないが、SbmChk-43bエラーとなる
                // SbmChk-43bの試験内でSbmChk-43エラーが発生しないことを確認しているので、
                // 本試験内ではSaveDataOwnerIds/Id と ApplicationIdが同じ場合にSbmChk-43エラーが発生しないことの試験は実施しない。

                // ContentMetaのチェック
                Tuple.Create(false, // NG
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 (string)null, // Core/ApplicationId
                                 (string[])null, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 "0x01004b9000490000", // ContentMeta/Id
                                 new string[] { "0x01004b9000490001" }, // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                                 "0x01004b9000490001")),
                Tuple.Create(false, // NG
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 (string)null, // Core/ApplicationId
                                 (string[])null, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 "0x01004b9000490000", // ContentMeta/Id
                                 new string[] { "0x01004b9000490001", "0", "0x01004b9000490002" }, // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                                 "0x01004b9000490001, 0x01004b9000490002")),
                Tuple.Create(true, // OK
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 (string)null, // Core/ApplicationId
                                 (string[])null, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 "0x01004b9000490000", // ContentMeta/Id
                                 new string[] { "0" }, // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null)),
                Tuple.Create(true, // OK
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 (string)null, // Core/ApplicationId
                                 (string[])null, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null, // ContentMeta/Id
                                 new string[] { "0x01004b9000490001" }, // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null)),
                Tuple.Create(true, // OK
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypeApplication,
                                 (string)null, // Core/ApplicationId
                                 (string[])null, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 "0x01004b9000490000", // ContentMeta/Id
                                 (string[])null, // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null)),

                // パッチのチェック
                Tuple.Create(false, // NG
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypePatch,
                                 "0x01004b9000490000", // Core/ApplicationId
                                 new string[] { "0x01004b9000490001", "0", "0x01004b9000490002" }, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 "0x01004b9000490000", // ContentMeta/Id
                                 new string[] { "0" }, // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                                 "0x01004b9000490001, 0x01004b9000490002")),
                Tuple.Create(true, // OK
                    Tuple.Create(NintendoContentMetaConstant.ContentMetaTypePatch,
                                 "0x01004b9000490000", // Core/ApplicationId
                                 new string[] { "0" }, // Core/FsAccessControlData/SaveDataOwnerIds/Id
                                 "0x01004b9000490000", // ContentMeta/Id
                                 new string[] { "0" }, // ProgramInfo/FsAccessControlData/SaveDataOwnerIds/Id
                                 (string)null)),
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, string[], string, string[], string>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, string[], string, string[], string>> input)
            {
                // Coreの設定
                data.Nmeta.Core = new CoreModel();
                data.Nmeta.Core.ApplicationId = input.Item2.Item2;
                data.Nmeta.CoreFsAccessControlData = new FsAccessControlData() { Entries = null };
                if (input.Item2.Item3 != null)
                {
                    data.Nmeta.CoreFsAccessControlData.Entries = input.Item2.Item3.Select(entry => new SaveDataOwnerIdsModel() { Id = entry }).ToList();
                }

                // ContentMetaの設定
                if (input.Item2.Item1 == NintendoContentMetaConstant.ContentMetaTypePatch)
                {
                    var patchMeta = new PatchContentMetaModel();
                    if (input.Item2.Item4 == input.Item2.Item5[0])
                    {
                        // SaveDataOwnerIds/Idの中にContentMeta/ApplicationIdと同じものがある場合はContentMeta/Idに適当な値を設定する
                        patchMeta.Id = "0x0000000000000000";
                    }
                    else
                    {
                        // SaveDataOwnerIds/Idの中にContentMeta/ApplicationIdと同じものが無い場合はContentMeta/IdにSaveDataOwnerIds/Idの値を設定する
                        patchMeta.Id = input.Item2.Item5[0];
                    }

                    patchMeta.ApplicationId = input.Item2.Item4;
                    var contentModelList = new List<ContentModel>();
                    // ContentMetaにProgramが登録されている
                    contentModelList.Add(new ContentModel() { Type = NintendoContentMetaConstant.ContentTypeProgram });
                    // ContentMetaにLegalInformationが登録されている
                    contentModelList.Add(new ContentModel() { Type = NintendoContentMetaConstant.ContentTypeLegalInformation });
                    patchMeta.ContentList = contentModelList;
                    checkData.Nsp.ContentMeta = patchMeta;
                }
                else
                {
                    data.Nsp.ContentMeta.Id = input.Item2.Item4;
                }
                checkData.Nsp.ContentMeta.Type = input.Item2.Item1;

                // SbmChk-41aのエラーを抑制するために SoftwareLegalInformation/ApplicationId を設定ContentMeta/Id と同じにする
                data.Nsp.LegalInformation.ApplicationId = input.Item2.Item4;

                // ProgramInfoの設定
                if (input.Item2.Item5 != null)
                {
                    data.Nsp.ProgramInfo.FsAccessControlData.Entries = input.Item2.Item5.Select(entry => new SaveDataOwnerIdsModel() { Id = entry }).ToList();
                }
                else
                {
                    data.Nsp.ProgramInfo.FsAccessControlData.Entries = null;
                }
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string, string[], string, string[], string>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string, string[], string, string[], string>> pattern)
            {
                return new List<UnpublishableErrorExpectType>
                {
                    new UnpublishableErrorExpectType("Warning", "Issue 10-024", new string[] { pattern.Item2.Item6 })
                };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-27a,SbmChk-50bでエラーが発生する(SbmChk-50bでエラーが発生する場合は必ずSbmChk-27aでも失敗するので同時にテストを実行する)
        [TestMethod]
        public void UnpublishableErrorSbmChk27a50bTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();
            var patchMeta = new PatchContentMetaModel();
            patchMeta.Id = checkData.Nsp.ContentMeta.Id;
            patchMeta.ApplicationId = patchMeta.Id;
            var contentModelList = new List<ContentModel>();
            // ContentMetaにProgramが登録されている
            contentModelList.Add(new ContentModel() { Type = NintendoContentMetaConstant.ContentTypeProgram });
            // ContentMetaにLegalInformationが登録されている
            contentModelList.Add(new ContentModel() { Type = NintendoContentMetaConstant.ContentTypeLegalInformation });
            patchMeta.ContentList = contentModelList;
            patchMeta.Type = NintendoContentMetaConstant.ContentMetaTypePatch;
            checkData.Nsp.ContentMeta = patchMeta;
            checkData.CheckContentType = patchMeta.Type;

            const long LimitSize27a = 536870912;
            const long LimitSize50b = 68719476736;
            List<Tuple<bool, List<long>>> testPatternList = new List<Tuple<bool, List<long>>>
            {
                Tuple.Create(false, new List<long> { LimitSize27a + 1 }),                      // NG
                Tuple.Create(false, new List<long> { 0, LimitSize27a + 1, LimitSize27a + 2 }), // NG
                Tuple.Create(false, new List<long> { LimitSize50b }),                           // NG
                Tuple.Create(false, new List<long> { LimitSize50b + 1, LimitSize27a + 1 }),    // NG
                Tuple.Create(true, new List<long> { LimitSize27a, LimitSize27a }),             // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, List<long>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, List<long>> input)
            {
                // ContentMetaPropertyListを設定
                var properties = input.Item2.Select(entry => new NintendoContentMetaProperty() { Size = new NintendoContentMetaProperty.SizeInfo() { DownLoad = entry } }).ToArray();
                data.Nsp.ContentMetaPropertyList = new NintendoContentMetaPropertyList() { Properties = properties };
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, List<long>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, List<long>> pattern)
            {
                List<UnpublishableErrorExpectType> expect = new List<UnpublishableErrorExpectType>();
                var size = pattern.Item2.Find(entry => entry > LimitSize27a) / (1024 * 1024);
                expect.Add(new UnpublishableErrorExpectType("Warning", "Issue 20-001", new string[] { size.ToString() }));

                if (pattern.Item2.FindAll(entry => entry > LimitSize50b).Count() > 0)
                {
                    size = pattern.Item2.Find(entry => entry > LimitSize50b) / (1024 * 1024 * 1024);
                    expect.Add(new UnpublishableErrorExpectType("Warning", "Issue 10-042", new string[] { size.ToString() }));
                }
                return expect;
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-29bでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk29bTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            const long LimitSize = 1073741824;
            List<Tuple<bool, Tuple<List<string>, List<long>>>> testPatternList = new List<Tuple<bool, Tuple<List<string>, List<long>>>>
            {
                Tuple.Create(false, Tuple.Create(new List<string> { "Demo" }, new List<long> { LimitSize + 1 })), // NG
                Tuple.Create(false, Tuple.Create(new List<string> { "Test", "Demo" }, new List<long> { 0, LimitSize + 1, LimitSize + 2 })), // NG
                Tuple.Create(true, Tuple.Create(new List<string> { "Demo" }, new List<long> { LimitSize, LimitSize - 1 })), // OK
                Tuple.Create(true, Tuple.Create(new List<string> { "Dem" }, new List<long> { LimitSize + 1 })), // OK
                Tuple.Create(true, Tuple.Create(new List<string> { "Demo2" }, new List<long> { LimitSize + 1 })), // OK
                Tuple.Create(true, Tuple.Create((List<string>)null, new List<long> { LimitSize + 1 })) // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<string>, List<long>>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<string>, List<long>>> input)
            {
                // ContentMetaPropertyListを設定
                var properties = input.Item2.Item2.Select(entry => new NintendoContentMetaProperty() { Size = new NintendoContentMetaProperty.SizeInfo() { DownLoad = entry } }).ToArray();
                data.Nsp.ContentMetaPropertyList = new NintendoContentMetaPropertyList() { Properties = properties };

                // Attributeを設定
                data.ApplicationControlProperty.Attribute = input.Item2.Item1;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<string>, List<long>>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<string>, List<long>>> pattern)
            {
                List<UnpublishableErrorExpectType> expect = new List<UnpublishableErrorExpectType>();
                var size = pattern.Item2.Item2.Find(entry => entry > LimitSize) / (1024 * 1024);
                expect.Add(new UnpublishableErrorExpectType("Warning", "Issue 10-021", new string[] { size.ToString() }));

                return expect;
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-50aでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk50aTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            const long LimitSize = 68719476736;
            List<Tuple<bool, List<long>>> testPatternList = new List<Tuple<bool, List<long>>>
            {
                Tuple.Create(false, new List<long> { LimitSize + 1 }), // NG
                Tuple.Create(false, new List<long> { 0, LimitSize + 1, LimitSize + 2 }), // NG
                Tuple.Create(true, new List<long> { LimitSize, LimitSize - 1 }), // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, List<long>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, List<long>> input)
            {
                // ContentMetaPropertyListを設定
                var properties = input.Item2.Select(entry => new NintendoContentMetaProperty() { Size = new NintendoContentMetaProperty.SizeInfo() { DownLoad = entry } }).ToArray();
                data.Nsp.ContentMetaPropertyList = new NintendoContentMetaPropertyList() { Properties = properties };
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, List<long>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, List<long>> pattern)
            {
                var size = pattern.Item2.Find(entry => entry > LimitSize) / (1024 * 1024 * 1024);
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Warning", "Issue 10-027", new string[] { size.ToString() }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-55でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk55Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, string>> testPatternList = new List<Tuple<bool, string>>
            {
                Tuple.Create(false, "PiaChat"),       // NG
                Tuple.Create(false, "_PiaChat_"),     // NG
                Tuple.Create(false, "PiaChat_1.1"),   // NG
                Tuple.Create(false, "piachat"),       // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, string>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> input)
            {
                // バージョンを設定
                data.Nsp.ProgramInfo.MiddlewareListData.Entries.Add(new MiddlewareModel() { ModuleName = input.Item2, NsoName = "test.nso", VenderName = "vender" });
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, string>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Warning", "Issue 11-013", new string[] { "dummy" }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-38(a-b)でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk38Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();
            var aocMeta = new AddOnContentContentMetaModel();
            aocMeta.Id = checkData.Nsp.ContentMeta.Id;
            aocMeta.Tag = "name";
            aocMeta.ApplicationId = aocMeta.Id;
            aocMeta.Type = NintendoContentMetaConstant.ContentMetaTypeAddOnContent;

            checkData.Nsp.ContentMeta = aocMeta;
            checkData.CheckContentType = aocMeta.Type;

            const long LimitSize38a = 10737418240;
            const long LimitSize38b = 17179869184;

            List<Tuple<bool, List<long>>> testPatternList = new List<Tuple<bool, List<long>>>
            {
                Tuple.Create(false, new List<long> { LimitSize38a + 1 }),                      // NG
                Tuple.Create(false, new List<long> { 0, LimitSize38a + 1, LimitSize38a + 2 }), // NG
                Tuple.Create(false, new List<long> { LimitSize38b }),                          // NG
                Tuple.Create(false, new List<long> { LimitSize38b + 1, LimitSize38a + 1 }),    // NG
                Tuple.Create(true, new List<long> { LimitSize38a, LimitSize38a }),             // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, List<long>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, List<long>> input)
            {
                // contentModelListに複数のコンテンツメタXMLを追加
                var contentList = input.Item2.Select(entry => new ContentModel() { Size = entry }).ToList();
                data.Nsp.ContentMeta.ContentList = contentList;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, List<long>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, List<long>> pattern)
            {
                List<UnpublishableErrorExpectType> expect = new List<UnpublishableErrorExpectType>();
                var size = pattern.Item2.Find(entry => entry > LimitSize38a) / (1024 * 1024 * 1024);
                expect.Add(new UnpublishableErrorExpectType("Warning", "Issue 10-043", new string[] { size.ToString() }));

                if (pattern.Item2.FindAll(entry => entry > LimitSize38b).Count() > 0)
                {
                    size = pattern.Item2.Find(entry => entry > LimitSize38b) / (1024 * 1024 * 1024);
                    expect.Add(new UnpublishableErrorExpectType("Warning", "Issue 10-044", new string[] { size.ToString() }));
                }
                return expect;
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-56でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk56Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, string>> testPatternList = new List<Tuple<bool, string>>
            {
                Tuple.Create(false, "Required"),   // NG
                Tuple.Create(false, string.Empty), // NG
                Tuple.Create(true,  "None"),       // OK
                Tuple.Create(true,  (string)null), // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, string>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> input)
            {
                // ApplicationControlProperty/Hdcpを設定
                data.ApplicationControlProperty.Hdcp = input.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, string>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, string> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Warning", "Issue 10-045", new string[] { pattern.Item2 }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-57でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk57Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, CardSpecModel>> testPatternList = new List<Tuple<bool, CardSpecModel>>
            {
                Tuple.Create(false, new CardSpecModel() { LaunchFlags = 1 }),    // NG
                Tuple.Create(false, new CardSpecModel() { LaunchFlags = 2 }),    // NG
                Tuple.Create(true,  new CardSpecModel() { LaunchFlags = 0 }),    // OK
                Tuple.Create(true,  new CardSpecModel() { LaunchFlags = null }), // OK
                Tuple.Create(true,  (CardSpecModel)null), // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, CardSpecModel>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, CardSpecModel> input)
            {
                // カードスペックを設定
                data.CardSpec = input.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, CardSpecModel>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, CardSpecModel> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Warning", "Issue 11-014", new string[] { pattern.Item2.LaunchFlags.ToString() }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-58でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk58Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, Tuple<string[], string>>> testPatternList = new List<Tuple<bool, Tuple<string[], string>>>
            {
                // NG項目
                // NGとなる各APIが1つ存在する
                Tuple.Create(false, Tuple.Create(
                    new string[] { "_ZN2nn8irsensor20RunPointingProcessorERKNS0_14IrCameraHandleE" },
                        "nn::irsensor::RunPointingProcessor(nn::irsensor::IrCameraHandle const&)")),
                Tuple.Create(false, Tuple.Create(
                    new string[] { "_ZN2nn8irsensor26GetPointingProcessorStatesEPNS0_22PointingProcessorStateEPiiRKNS0_14IrCameraHandleE" },
                        "nn::irsensor::GetPointingProcessorStates(nn::irsensor::PointingProcessorState*, int*, int, nn::irsensor::IrCameraHandle const&)")),
                Tuple.Create(false, Tuple.Create(
                    new string[] { "_ZN2nn8irsensor32GetPointingProcessorMarkerStatesEPNS0_28PointingProcessorMarkerStateEPiiRKNS0_14IrCameraHandleE" },
                        "nn::irsensor::GetPointingProcessorMarkerStates(nn::irsensor::PointingProcessorMarkerState*, int*, int, nn::irsensor::IrCameraHandle const&)")),

                // NGとなる各APIが複数存在する(API名が改行で接続されて表示される)
                Tuple.Create(false, Tuple.Create(
                    new string[] { "_ZN2nn8irsensor20RunPointingProcessorERKNS0_14IrCameraHandleE",
                                   "_ZN2nn8irsensor26GetPointingProcessorStatesEPNS0_22PointingProcessorStateEPiiRKNS0_14IrCameraHandleE",
                                   "_ZN2nn8irsensor32GetPointingProcessorMarkerStatesEPNS0_28PointingProcessorMarkerStateEPiiRKNS0_14IrCameraHandleE" },
                        "nn::irsensor::RunPointingProcessor(nn::irsensor::IrCameraHandle const&)"
                        + Environment.NewLine + "nn::irsensor::GetPointingProcessorStates(nn::irsensor::PointingProcessorState*, int*, int, nn::irsensor::IrCameraHandle const&)"
                        + Environment.NewLine + "nn::irsensor::GetPointingProcessorMarkerStates(nn::irsensor::PointingProcessorMarkerState*, int*, int, nn::irsensor::IrCameraHandle const&)")),

                // OK項目(NGとなるAPI名を 一文字削除 or 一文字追加 or 大文字→小文字)
                Tuple.Create(true, Tuple.Create(
                    new string[] { "ZN2nn8irsensor20RunPointingProcessorERKNS0_14IrCameraHandleE3" }, "dummy")),
                Tuple.Create(true, Tuple.Create(
                    new string[] { "_ZN2nn8irsensor26GetPointingProcessorStatesEPNS0_22PointingProcessorStateEPiiRKNS0_14IrCameraHandleE_" }, "dummy")),
                Tuple.Create(true, Tuple.Create(
                    new string[] { "_zN2nn8irsensor32GetPointingProcessorMarkerStatesEPNS0_28PointingProcessorMarkerStateEPiiRKNS0_14IrCameraHandleE" }, "dummy")),
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string[], string>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string[], string>> input)
            {
                // UnresolvedApiを設定
                var apiList = UnresolvedApiList.Create();
                apiList.Entries.AddRange(input.Item2.Item1.Select(entry => new UnresolvedApiModel() { ApiName = entry, NsoName = "test.nso" }).ToList());
                checkData.Nsp.ProgramInfo.UnresolvedApiListData = apiList;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string[], string>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string[], string>> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Warning", "Issue 11-015", new string[] { pattern.Item2.Item2 }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-7cでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk7cTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            List<Tuple<bool, CardSpecModel>> testPatternList = new List<Tuple<bool, CardSpecModel>>
            {
                Tuple.Create(false, new CardSpecModel() { Size = "1" }),  // NG
                Tuple.Create(true, new CardSpecModel() { Size = "2" }),  // OK
                Tuple.Create(true,  new CardSpecModel() { Size = "32" }), // OK
                Tuple.Create(true,  new CardSpecModel() { Size = "0" }),  // OK (実際に設定されることはない)
                Tuple.Create(true,  new CardSpecModel() { Size = null }), // OK
                Tuple.Create(true,  (CardSpecModel)null), // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, CardSpecModel>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, CardSpecModel> input)
            {
                // カードスペックを設定
                data.CardSpec = input.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, CardSpecModel>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, CardSpecModel> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 13-801", new string[] { pattern.Item2.LaunchFlags.ToString() }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-1-10でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk1_10Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            var programinfo = checkData.Nsp.ProgramInfo;
            List<Tuple<bool, Tuple<ProgramInfoModel, string[]>>> testPatternList = new List<Tuple<bool, Tuple<ProgramInfoModel, string[]>>>
            {
                Tuple.Create(false, Tuple.Create((ProgramInfoModel)null,    // NG
                    new string[] {
                        NintendoContentMetaConstant.ContentTypeProgram,
                        NintendoContentMetaConstant.ContentTypeLegalInformation })),
                Tuple.Create(false, Tuple.Create(programinfo,               // NG
                    new string[] {
                        NintendoContentMetaConstant.ContentMetaTypeAddOnContent,
                        NintendoContentMetaConstant.ContentMetaTypeApplication,
                        NintendoContentMetaConstant.ContentMetaTypeBootImagePackage,
                        NintendoContentMetaConstant.ContentMetaTypeBootImagePackageSafe,
                        NintendoContentMetaConstant.ContentMetaTypeDelta,
                        NintendoContentMetaConstant.ContentMetaTypePatch,
                        NintendoContentMetaConstant.ContentMetaTypeSystemData,
                        NintendoContentMetaConstant.ContentMetaTypeSystemProgram,
                        NintendoContentMetaConstant.ContentMetaTypeSystemUpdate,
                        NintendoContentMetaConstant.ContentTypeControl,
                        NintendoContentMetaConstant.ContentTypeData,
                        NintendoContentMetaConstant.ContentTypeDeltaFragment,
                        NintendoContentMetaConstant.ContentTypeHtmlDocument,
                        NintendoContentMetaConstant.ContentTypeLegalInformation,
                        NintendoContentMetaConstant.ContentTypeMeta,
                        NintendoContentMetaConstant.ContentTypePublicData })),
                Tuple.Create(false, Tuple.Create(programinfo, (string[])null)), // NG
                Tuple.Create(true, Tuple.Create(programinfo, new string[] { NintendoContentMetaConstant.ContentTypeProgram, NintendoContentMetaConstant.ContentTypeLegalInformation }))  // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<ProgramInfoModel, string[]>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<ProgramInfoModel, string[]>> input)
            {
                // ProgramInfo を設定
                checkData.Nsp.ProgramInfo = input.Item2.Item1;
                // ContentMetaを設定
                checkData.Nsp.ContentMeta.ContentList = null;
                if (input.Item2.Item2 != null)
                {
                    checkData.Nsp.ContentMeta.ContentList = input.Item2.Item2.Select(entry => new ContentModel() { Type = entry }).ToList();
                }
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<ProgramInfoModel, string[]>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<ProgramInfoModel, string[]>> pattern)
            {
                var list = new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 10-813", new string[] { "dummy" }) };
                if (pattern.Item2.Item2 == null)
                {
                    // ContentListがない場合は、SbmChk-28（ソフトリーガル情報がない）エラーも発生するのでエラーを追加
                    list.Insert(0, new UnpublishableErrorExpectType("Error", "Issue 12-003", new string[] { "dummy" }));
                }
                return list;
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-39-1でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk39_1Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();
            checkData.CheckContentType = NintendoContentMetaConstant.ContentMetaTypeAddOnContent;
            var id = checkData.Nsp.ContentMeta.Id;

            List<Tuple<bool, Tuple<string, string>>> testPatternList = new List<Tuple<bool, Tuple<string, string>>>
            {
                Tuple.Create(false, Tuple.Create("nmeta", (string)null)),   // NG
                Tuple.Create(false, Tuple.Create("nmeta", string.Empty)),   // NG
                Tuple.Create(true,  Tuple.Create("nmeta", "AocName1")),     // NG
                Tuple.Create(false, Tuple.Create("nsp", string.Empty)),     // NG
                Tuple.Create(true,  Tuple.Create("nsp", "AocName1")),       // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string>> input)
            {
                if (input.Item2.Item1 == "nmeta")
                {
                    var aocNmetaModel = new AddOnContentModel();
                    aocNmetaModel.Id = id;
                    aocNmetaModel.Tag = input.Item2.Item2;
                    data.Nmeta.AddOnContent = aocNmetaModel;
                    data.Nsp.ContentMeta = null;
                }
                else
                {
                    data.Nmeta.AddOnContent = null;
                    var aocMeta = new AddOnContentContentMetaModel();
                    aocMeta.Id = id;
                    aocMeta.ApplicationId = id;
                    aocMeta.ContentList = new List<ContentModel>() { new ContentModel() };
                    aocMeta.Type = NintendoContentMetaConstant.ContentMetaTypeAddOnContent;
                    aocMeta.Tag = input.Item2.Item2;
                    data.Nsp.ContentMeta = aocMeta;
                }
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string>> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 10-611", new string[] { "dummy" }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-39-2でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk39_2Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();
            checkData.CheckContentType = NintendoContentMetaConstant.ContentMetaTypeAddOnContent;
            var id = checkData.Nsp.ContentMeta.Id;

            List<Tuple<bool, Tuple<string, string[]>>> testPatternList = new List<Tuple<bool, Tuple<string, string[]>>>
            {
                Tuple.Create(false, Tuple.Create("nmeta", new string[] { "Duplicate", "Duplicate" })),  // NG
                Tuple.Create(true,  Tuple.Create("nmeta", new string[] { "AocName1", "AocName2" })),  // OK
                Tuple.Create(false, Tuple.Create("nsp", new string[] { "Duplicate", "Duplicate" })),    // NG
                Tuple.Create(true,  Tuple.Create("nsp", new string[] { "AocName1", "AocName2" })),    // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string[]>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string[]>> input)
            {
                if (input.Item2.Item1 == "nmeta")
                {
                    List<AddOnContentModel> aocContentModelList = new List<AddOnContentModel>();
                    foreach (var tag in input.Item2.Item2)
                    {
                        var aocMetaModel = new AddOnContentModel();
                        aocMetaModel.Id = id;
                        aocMetaModel.ApplicationId = id;
                        aocMetaModel.Tag = tag;
                        aocContentModelList.Add(aocMetaModel);
                    }
                    data.Nmeta.AddOnContent = aocContentModelList.First();
                    data.Nmeta.AddOnContentList = aocContentModelList;
                    data.Nsp.ContentMeta = null;
                    data.Nsp.AddonContentMetaList = null;
                }
                else
                {
                    List<AddOnContentContentMetaModel> aocContentMetaList = new List<AddOnContentContentMetaModel>();
                    foreach (var tag in input.Item2.Item2)
                    {
                        var aocMeta = new AddOnContentContentMetaModel();
                        aocMeta.Id = id;
                        aocMeta.ApplicationId = id;
                        aocMeta.Tag = tag;
                        aocMeta.ContentList = new List<ContentModel>() { new ContentModel() };
                        aocMeta.Type = NintendoContentMetaConstant.ContentMetaTypeAddOnContent;
                        aocContentMetaList.Add(aocMeta);
                    }
                    data.Nsp.ContentMeta = aocContentMetaList.First();
                    data.Nsp.AddonContentMetaList = aocContentMetaList;
                }
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string[]>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string[]>> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 10-612", new string[] { "Duplicate" }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-61でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk61Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();
            checkData.CheckContentType = NintendoContentMetaConstant.ContentMetaTypeAddOnContent;
            var aocMeta = new AddOnContentContentMetaModel();
            aocMeta.Id = checkData.Nsp.ContentMeta.Id + 1;
            aocMeta.ApplicationId = checkData.Nsp.ContentMeta.Id;
            aocMeta.ContentList = new List<ContentModel>() { new ContentModel() };
            aocMeta.Type = NintendoContentMetaConstant.ContentMetaTypeAddOnContent;
            checkData.Nsp.ContentMeta = aocMeta;
            var aocNmetaModel = new AddOnContentModel() { Id = aocMeta.Id };
            checkData.Nmeta.AddOnContent = aocNmetaModel;

            // 127文字の文字列を用意する
            string name127 = "1234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567";
            List<Tuple<bool, Tuple<string, string>>> testPatternList = new List<Tuple<bool, Tuple<string, string>>>
            {
                // 127文字の文字列に1文字or2文字を結合して検証する
                Tuple.Create(false, Tuple.Create("nmeta", "89")),          // NG
                Tuple.Create(false, Tuple.Create("nmeta", "８９")),        // NG
                Tuple.Create(true, Tuple.Create("nmeta", "8")),            // OK
                Tuple.Create(true, Tuple.Create("nmeta", "あ")),           // Ok
                Tuple.Create(false, Tuple.Create("nsp", "89")),          // NG
                Tuple.Create(false, Tuple.Create("nsp", "８９")),        // NG
                Tuple.Create(true, Tuple.Create("nsp", "8")),            // OK
                Tuple.Create(true, Tuple.Create("nsp", "あ")),           // Ok
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string>> input)
            {
                if (input.Item2.Item1 == "nmeta")
                {
                    (data.Nsp.ContentMeta as AddOnContentContentMetaModel).Tag = name127;
                    data.Nmeta.AddOnContent.Tag = name127 + input.Item2.Item2;
                }
                else
                {
                    (data.Nsp.ContentMeta as AddOnContentContentMetaModel).Tag = name127 + input.Item2.Item2;
                    data.Nmeta.AddOnContent.Tag = name127;
                }
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, string>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, string>> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 10-613", new string[] { "dummy" }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-39-3でエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk39_3Test()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();
            checkData.CheckContentType = NintendoContentMetaConstant.ContentMetaTypeAddOnContent;
            checkData.Nsp.ContentMeta = null;

            List<Tuple<bool, Tuple<string, short?>>> testPatternList = new List<Tuple<bool, Tuple<string, short?>>>
            {
                Tuple.Create(false, Tuple.Create(string.Empty, (short?)null)), // NG
                Tuple.Create(false, Tuple.Create((string)null, (short?)null)), // NG
                Tuple.Create(true,  Tuple.Create(string.Empty, (short?)1)),    // OK
                Tuple.Create(true,  Tuple.Create((string)null, (short?)1)),    // OK
                Tuple.Create(true,  Tuple.Create("0x12345678", (short?)null)), // OK
                Tuple.Create(true,  Tuple.Create("0x12345678", (short?)1)),    // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, short?>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, short?>> input)
            {
                var aocNmetaModel = new AddOnContentModel() { Id = input.Item2.Item1, Index = input.Item2.Item2, Tag = "name" };
                checkData.Nmeta.AddOnContent = aocNmetaModel;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<string, short?>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<string, short?>> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 10-814", new string[] { "dummy" }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }

        // SbmChk-47aでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk47aTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            // nmetaのテストはファイル読み込みが発生するのでここではHtmlDocumentXmlModelのテストのみ実施する
            // nmetaを使用したテストはTestExecutionErrorUnpublishableHtmlDocumentCheck()で実施する
            List<Tuple<bool, List<string>>> testPatternList = new List<Tuple<bool, List<string>>>
            {
                // http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-73358 に記載されているテスト
                Tuple.Create(true,  new List<string> { @"^https?://([0-9A-Za-z\-]+\.)*nintendo\.com(/|$)" }), // OK
                Tuple.Create(false, new List<string> { @".*" }), // NG
                Tuple.Create(false, new List<string> { @"^http" }), // NG
                Tuple.Create(true,  new List<string> { @"http://localhost((" }), // OK
                Tuple.Create(false, new List<string> { @"http://localhost" }), // NG
                Tuple.Create(false, new List<string> { @"https://localhost" }), // NG
                Tuple.Create(false, new List<string> { @"http://example.com" }), // NG
                Tuple.Create(false, new List<string> { @"https://www.example.com" }), // NG
                Tuple.Create(false, new List<string> { @"http://example.org" }), // NG
                Tuple.Create(false, new List<string> { @"https://www.example.org" }), // NG
                Tuple.Create(false, new List<string> { @"http://127.0.0.1" }), // NG
                Tuple.Create(false, new List<string> { @"https://127.0.0.1" }), // NG
                Tuple.Create(false, new List<string> { @"http://192.168.0.11" }), // NG
                Tuple.Create(false, new List<string> { @"https://192.168.0.11" }), // NG
                Tuple.Create(false, new List<string> { @"http://localhost" }), // NG
                Tuple.Create(true,  new List<string> { @"http://localhost",
                                                       @"---- http://localhost" }), // OK

                // ブラックリスト・ホワイトリストが複数含まれる場合のテスト
                Tuple.Create(false, new List<string> { @"http://localhost",
                                                       @"http://example.com",
                                                       @"---- http://localhost" }), // NG
                Tuple.Create(true,  new List<string> { @"http://localhost",
                                                       @"---- http://localhost",
                                                       @"http://example.com",
                                                       @"---- http://example.com" }), // OK

                // ブラックリストで正規表現のマッチングが実行されることのテスト
                Tuple.Create(false,  new List<string> { @"http://localhost",
                                                       @"---- http[aclo:/]+ost" }), // NG
                Tuple.Create(true,  new List<string> { @"http://localhost",
                                                       @"---- http[aclo:/]+host" }), // OK

                // 空の場合に例外が発生しないことのテスト
                Tuple.Create(true,  new List<string> { string.Empty,
                                                       @"---- " }), // OK
                Tuple.Create(true,  new List<string>()), // OK
                Tuple.Create(true,  (List<string>)null), // OK
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, List<string>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, List<string>> input)
            {
                data.Nsp.HtmlDocumentXml = new HtmlDocumentXmlModel() { AccessibleUrls = new AccessibleUrlsModel() };
                data.Nsp.HtmlDocumentXml.AccessibleUrls.Url = input.Item2;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, List<string>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, List<string>> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 10-609", new string[] { string.Join(Environment.NewLine, pattern.Item2) }) };
            };

            // テスト実施(同じようなチェック項目である"Issue 10-048"のエラーが発生するので、"Issue 10-048"を無視する)
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator, new List<string> { "Issue 10-048" });
        }

        // SbmChk-47bでエラーが発生する
        [TestMethod]
        public void UnpublishableErrorSbmChk47bTest()
        {
            // テストデータ作成
            var checkData = CreateSimpleSuccessData();

            // nmetaのテストはファイル読み込みが発生するのでここではHtmlDocumentXmlModelのテストのみ実施する
            // nmetaを使用したテストはTestExecutionErrorUnpublishableHtmlDocumentCheck()で実施する
            List<Tuple<bool, Tuple<List<string>, string>>> testPatternList = new List<Tuple<bool, Tuple<List<string>, string>>>
            {
                Tuple.Create(false,
                    Tuple.Create(new List<string> { @"https://www.nintendo.co.jp/" },     // NG
                                                    @"https://www.nintendo.co.jp/")),
                Tuple.Create(false,
                    Tuple.Create(new List<string> { @"^http://www.nintendo.co.jp/" },     // NG
                                                       @"^http://www.nintendo.co.jp/")),
                Tuple.Create(false,
                    Tuple.Create(new List<string> { @"^http:/www.nintendo.co.jp/" },      // NG
                                                       @"^http:/www.nintendo.co.jp/")),
                Tuple.Create(false,
                    Tuple.Create(new List<string> { @"----https://www.nintendo.co.jp/" }, // NG
                                                       @"----https://www.nintendo.co.jp/")),
                Tuple.Create(true,
                    Tuple.Create(new List<string> { @"^https://www.nintendo.co.jp/" }, (string)null)),    // OK
                Tuple.Create(true,
                    Tuple.Create(new List<string> { @"---- http://www.nintendo.co.jp/" }, (string)null)), // OK

                // OKパターンとNGパターンが含まれる時にNGパターンだけ表示される
                Tuple.Create(false,
                    Tuple.Create(new List<string> { @"^https://www.nintendo.co.jp/",
                                                    @"^http://www.nintendo.co.jp/",
                                                    @"----https://www.nintendo.co.jp/",
                                                    @"---- https://www.nintendo.co.jp/" }, // NG
                                                    @"^http://www.nintendo.co.jp/" + "\r\n" + @"----https://www.nintendo.co.jp/")),
            };

            Action<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<string>, string>>> settingFunction = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<string>, string>> input)
            {
                data.Nsp.HtmlDocumentXml = new HtmlDocumentXmlModel() { AccessibleUrls = new AccessibleUrlsModel() };
                data.Nsp.HtmlDocumentXml.AccessibleUrls.Url = input.Item2.Item1;
            };

            Func<UnpublishableErrorCheckData, Tuple<bool, Tuple<List<string>, string>>, List<UnpublishableErrorExpectType>> expectUnpublishableErrorModelGenerator = delegate (UnpublishableErrorCheckData data, Tuple<bool, Tuple<List<string>, string>> pattern)
            {
                return new List<UnpublishableErrorExpectType> { new UnpublishableErrorExpectType("Error", "Issue 10-048", new string[] { string.Join(Environment.NewLine, pattern.Item2.Item2) }) };
            };

            // テスト実施
            CheckUnpublishableErrorCommon(checkData, testPatternList, settingFunction, expectUnpublishableErrorModelGenerator);
        }
    }
}
