﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Collections.Generic;
using System.Security.Cryptography;
using System.Xml.Serialization;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.Authoring.AuthoringLibrary;
using TestUtility;

namespace AuthoringToolsTest
{
    [TestClass]
    public class SymbolExtractTest
    {
        // １．DynSymテーブルデータがNULLの場合
        [TestMethod]
        public void GetUndefSymbolDynSymDataEmptyTest()
        {
            // 入力データ
            string nsoFileName = "TEST_ERROR02.nso";    // nsoファイル名
            byte[] metaDataBinary = new byte[] {        // main.npdmテストデータ
                    // 0    1    2    3    4    5    6   7     8    9    a    b       metaFlag
                    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00
                };
            byte[] dynSymBinary = null;                 // ELF32 DynSymテーブルテストデータ
            byte[] dynStrBinary = new byte[0];          // DynStrテーブルテストデータ

            // 期待値
            UndefSymbolInfo[] expData = new UndefSymbolInfo[0];

            // 未定義シンボル取得
            bool elf32Mode = SymbolExtract.IsElf32(metaDataBinary);
            Assert.AreEqual(elf32Mode, true);
            List<UndefSymbolInfo> resultData = SymbolExtract.GetUndefSymbolInfoList(dynSymBinary, dynStrBinary, nsoFileName, elf32Mode);

            // 結果判定
            Assert.AreEqual(expData.Length, resultData.Count);
        }

        // ２．ELF32未定義シンボル取得
        [TestMethod]
        public void GetUndefSymbolElf32DataTest()
        {
            // 入力データ
            string nsoFileName = "TEST_ELF32.nso";// nsoファイル名
            byte[] metaDataBinary = new byte[] {  // main.npdmテストデータ
                    // 0    1    2    3    4    5    6   7     8    9    a    b       metaFlag
                    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00
                };
            byte[] dynSymBinary = new byte[] {    // ELF32 DynSymテーブルテストデータ
                    //      m_Name                                                m_Info        m_Shndx
                    0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x12, 0x00, 0x00,0x00, // STT_FUNC && UNDEF
                    0x10,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x22, 0x00, 0x01,0x00, // STT_FUNC && !UNDEF
                    0x20,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x30, 0x00, 0x00,0x00, // STT_NOTYPE && UNDEF
                    0x30,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x40, 0x00, 0x01,0x00, // STT_NOTYPE && !UNDEF
                    0x40,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x53, 0x00, 0x00,0x00, // (!STT_FUNC && !STT_NOTYPE) && UNDEF
                    0x50,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x30, 0x00, 0x00,0x00, // STT_NOTYPE && UNDEF
                };
            byte[] dynStrBinary = new byte[] {    // DynStrテーブルテストデータ
                    // 0    1    2    3    4    5    6   7     8    9    a    b    c    d    e    f
                    0x5F,0x45,0x4C,0x46,0x33,0x32,0x5F,0x46,0x55,0x4E,0x43,0x30,0x30,0x30,0x31,0x00, // "_ELF32_FUNC0001"
                    0x5F,0x45,0x4C,0x46,0x33,0x32,0x5F,0x46,0x55,0x4E,0x43,0x30,0x30,0x30,0x32,0x00, // "_ELF32_FUNC0002"
                    0x5F,0x45,0x4C,0x46,0x33,0x32,0x5F,0x46,0x55,0x4E,0x43,0x30,0x30,0x30,0x33,0x00, // "_ELF32_FUNC0003"
                    0x5F,0x45,0x4C,0x46,0x33,0x32,0x5F,0x46,0x55,0x4E,0x43,0x30,0x30,0x30,0x34,0x00, // "_ELF32_FUNC0004"
                    0x5F,0x45,0x4C,0x46,0x33,0x32,0x5F,0x46,0x55,0x4E,0x43,0x30,0x30,0x30,0x35,0x00, // "_ELF32_FUNC0005"
                    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // シンボル名が空
                };

            // 期待値
            UndefSymbolInfo[] expData = new UndefSymbolInfo[] {
                    new UndefSymbolInfo("_ELF32_FUNC0001", nsoFileName),
                    new UndefSymbolInfo("_ELF32_FUNC0003", nsoFileName),
                };

            // 未定義シンボル取得
            bool elf32Mode = SymbolExtract.IsElf32(metaDataBinary);
            Assert.AreEqual(elf32Mode, true);
            List<UndefSymbolInfo> resultData = SymbolExtract.GetUndefSymbolInfoList(dynSymBinary, dynStrBinary, nsoFileName, elf32Mode);

            // 結果判定
            Assert.AreEqual(expData.Length, resultData.Count);
            for (int i = 0; i < expData.Length; ++i)
            {
                Assert.AreEqual(expData[i].ApiName, resultData[i].ApiName);
                Assert.AreEqual(expData[i].NsoName, resultData[i].NsoName);
            }
        }

        // ３．ELF64未定義シンボル取得
        [TestMethod]
        public void GetUndefSymbolElf64DataTest()
        {
            // 入力データ
            string nsoFileName = "TEST_ELF64.nso";// nsoファイル名
            byte[] metaDataBinary = new byte[] {  // main.npdmテストデータ
                    // 0    1    2    3    4    5    6   7     8    9    a    b       metaFlag
                    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, 0x01,0x00,0x00,0x00
                };
            byte[] dynSymBinary = new byte[] {    // ELF64 DynSymテーブルテストデータ
                    //      m_Name      m_Info        m_Shndx
                    0x00,0x00,0x00,0x00, 0x12, 0x00, 0x00,0x00, 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // STT_FUNC && UNDEF
                    0x10,0x00,0x00,0x00, 0x22, 0x00, 0x01,0x00, 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // STT_FUNC && !UNDEF
                    0x20,0x00,0x00,0x00, 0x30, 0x00, 0x00,0x00, 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // STT_NOTYPE && UNDEF
                    0x30,0x00,0x00,0x00, 0x40, 0x00, 0x01,0x00, 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // STT_NOTYPE && !UNDEF
                    0x40,0x00,0x00,0x00, 0x51, 0x00, 0x00,0x00, 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // (!STT_FUNC && !STT_NOTYPE) && UNDEF
                    0x50,0x00,0x00,0x00, 0x30, 0x00, 0x00,0x00, 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // STT_NOTYPE && UNDEF
                };
            byte[] dynStrBinary = new byte[] {    // DynStrテーブルテストデータ
                    // 0    1    2    3    4    5    6   7     8    9    a    b    c    d    e    f
                    0x5F,0x45,0x4C,0x46,0x36,0x34,0x5F,0x46,0x55,0x4E,0x43,0x30,0x30,0x30,0x31,0x00, // "_ELF64_FUNC0001"
                    0x5F,0x45,0x4C,0x46,0x36,0x34,0x5F,0x46,0x55,0x4E,0x43,0x30,0x30,0x30,0x32,0x00, // "_ELF64_FUNC0002"
                    0x5F,0x45,0x4C,0x46,0x36,0x34,0x5F,0x46,0x55,0x4E,0x43,0x30,0x30,0x30,0x33,0x00, // "_ELF64_FUNC0003"
                    0x5F,0x45,0x4C,0x46,0x36,0x34,0x5F,0x46,0x55,0x4E,0x43,0x30,0x30,0x30,0x34,0x00, // "_ELF64_FUNC0004"
                    0x5F,0x45,0x4C,0x46,0x36,0x34,0x5F,0x46,0x55,0x4E,0x43,0x30,0x30,0x30,0x35,0x00, // "_ELF64_FUNC0005"
                    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // シンボル名が空
                };

            // 期待値
            UndefSymbolInfo[] expData = new UndefSymbolInfo[] {
                    new UndefSymbolInfo("_ELF64_FUNC0001", nsoFileName),
                    new UndefSymbolInfo("_ELF64_FUNC0003", nsoFileName),
                };

            // 未定義シンボル取得
            bool elf32Mode = SymbolExtract.IsElf32(metaDataBinary);
            Assert.AreEqual(elf32Mode, false);
            List<UndefSymbolInfo> resultData = SymbolExtract.GetUndefSymbolInfoList(dynSymBinary, dynStrBinary, nsoFileName, elf32Mode);

            // 結果判定
            Assert.AreEqual(expData.Length, resultData.Count);
            for (int i = 0; i < expData.Length; ++i)
            {
                Assert.AreEqual(expData[i].ApiName, resultData[i].ApiName);
                Assert.AreEqual(expData[i].NsoName, resultData[i].NsoName);
            }
        }
    }
}
