﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.Security.Cryptography;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.Authoring.AuthoringLibrary;
using Nintendo.Authoring.FileSystemMetaLibrary;
using TestUtility;

namespace AuthoringToolsTest
{
    [TestClass]
    public class SubmissionPackageTest
    {
        public TestContext TestContext { get; set; }

        private static void CreateAnyNsp(TestPath testPath, string metaPath, string outputAdfPath, string outputPath)
        {
            var nspInfo = new NintendoSubmissionPackageContentInfo();
            nspInfo.MetaFilePath = metaPath;
            nspInfo.MetaType = "SystemUpdate";

            var nspInfoList = new List<NintendoSubmissionPackageContentInfo>();
            nspInfoList.Add(nspInfo);
            var writer = new NintendoSubmissionPackageAdfWriter(outputAdfPath);
            writer.Write(nspInfoList);

            var config = new AuthoringConfiguration();
            config.KeyConfigFilePath = testPath.GetSigloRoot() + "\\Programs\\Chris\\Sources\\Tools\\AuthoringTools\\AuthoringTool\\AuthoringTool.repository.keyconfig.xml";

            using (var fs = new FileStream(outputPath, FileMode.Create,
                       FileAccess.ReadWrite, FileShare.None, 4096, FileOptions.RandomAccess))
            {
                ContentArchiveLibraryInterface.CreateArchiveFromAdf(fs, null, outputAdfPath, config);
            }
        }

        [TestMethod]
        public void TestSystemUpdateMeta()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var metaDirectoryPath = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources\\SystemUpdateMeta";
            var outputAdfPath = Directory.GetCurrentDirectory() + "/test.nsp.adf";
            var outputPath = Directory.GetCurrentDirectory() + "/test.nsp";
            var extractedDirectory = Directory.GetCurrentDirectory() + "/extracted";

            var config = new AuthoringConfiguration();
            config.KeyConfigFilePath = testPath.GetSigloRoot() + "\\Programs\\Chris\\Sources\\Tools\\AuthoringTools\\AuthoringTool\\AuthoringTool.repository.keyconfig.xml";

            foreach (var file in Directory.EnumerateFiles(metaDirectoryPath))
            {
                var nspInfo = new NintendoSubmissionPackageContentInfo();
                nspInfo.MetaFilePath = file;
                nspInfo.MetaType = "SystemUpdate";

                var nspInfoList = new List<NintendoSubmissionPackageContentInfo>();
                nspInfoList.Add(nspInfo);
                var writer = new NintendoSubmissionPackageAdfWriter(outputAdfPath);
                writer.Write(nspInfoList);

                using (FileStream fs = new FileStream(outputPath, FileMode.Create,
                           FileAccess.ReadWrite, FileShare.None, 4096, FileOptions.RandomAccess))
                {
                    ContentArchiveLibraryInterface.CreateArchiveFromAdf(fs, null, outputAdfPath, config);
                }

                Assert.IsTrue(File.Exists(outputAdfPath));
                Assert.IsTrue(File.Exists(outputPath));

                Directory.CreateDirectory(extractedDirectory);
                using (var readerStream = new FileStream(outputPath, FileMode.Open, FileAccess.Read))
                {
                    var reader = new PartitionFileSystemArchiveReader(readerStream);
                    foreach (var info in reader.ListFileInfo())
                    {
                        var fileName = info.Item1;
                        var fileSize = info.Item2;

                        var extractedPath = extractedDirectory + "/" + fileName;
                        Console.WriteLine("Extracting {0}.", extractedPath);

                        using (var writerStream = new FileStream(extractedPath, FileMode.Create,
                                   FileAccess.ReadWrite, FileShare.None, 4096, FileOptions.RandomAccess))
                        {
                            long readSize = 0;
                            while (readSize < fileSize)
                            {
                                uint bufferSize = (uint)Math.Min(1024 * 1024, fileSize - readSize);
                                var bytes = reader.ReadFile(fileName, readSize, bufferSize);
                                writerStream.Write(bytes, 0, bytes.Length);
                                readSize += bytes.Length;
                            }
                        }

                        if (Path.GetExtension(fileName) == ".xml")
                        {
                            using (var streamReader = new StreamReader(extractedPath))
                            {
                                Console.WriteLine(streamReader.ReadToEnd());
                            }
                        }
                    }
                }

                Directory.Delete(extractedDirectory, true);

                File.Delete(outputAdfPath);
                File.Delete(outputPath);
            }
        }

        [TestMethod]
        [SuppressMessage("Microsoft.Usage", "CA2202", Justification = "NintendoSubmissionPackageReader は Stream を Dispose しない")]
        public void TestNintendoSubmissionPackageReaderDispose()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var metaPath = testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\AuthoringToolsTest\\TestResources\\SystemUpdateMeta\\basic.nmeta";
            var testDirectory = Directory.GetCurrentDirectory() + "/TestNintendoSubmissionPackageReaderDispose";
            var outputAdfPath = testDirectory + "/test.nsp.adf";
            var outputPath = testDirectory + "/test.nsp";

            Utils.DeleteDirectoryIfExisted(testDirectory);
            Directory.CreateDirectory(testDirectory);

            CreateAnyNsp(testPath, metaPath, outputAdfPath, outputPath);

            var disposed = new DisposeCheckStream.DisposedValue();
            using (var stream = new DisposeCheckStream(outputPath, disposed))
            {
                using (var nsp = new NintendoSubmissionPackageReader(stream))
                {
                }
                Assert.IsFalse(disposed.Value);
            }
        }
    }
}
