﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.Authoring.AuthoringLibrary;

namespace AuthoringToolsTest
{
    [TestClass]
    public class StreamTest
    {
        [TestMethod]
        public void SubStreamTest()
        {
            byte[] incBuffer = new byte[256];
            byte[] zeroBuffer = new byte[256];
            for (int i = 0; i < incBuffer.Length; i++)
            {
                incBuffer[i] = (byte)i;
            }

            // サポートされていないメソッド
            {
                using (Stream baseStream = new MemoryStream(256))
                {
                    baseStream.Write(incBuffer, 0, incBuffer.Length);
                    using (Stream ss = new SubStream(baseStream, 0, baseStream.Length))
                    {
                        Utils.CheckReturnException(new NotSupportedException(), () =>
                                {
                                    ss.SetLength(10);
                                    return true;
                                });
                    }
                }
            }
            // オフセット設定違反
            {
                using (Stream baseStream = new MemoryStream(256))
                {
                    baseStream.Write(incBuffer, 0, incBuffer.Length);
                    Utils.CheckReturnException(new IOException("ストリームの先頭より前に位置を移動しようとしました。"), () =>
                            {
                                Stream ss = new SubStream(baseStream, -1, 1);
                                return true;
                            });
                }
            }
            // プロパティチェック
            {
                using (Stream baseStream = new MemoryStream(256))
                {
                    baseStream.Write(incBuffer, 0, incBuffer.Length);
                    using (Stream ss = new SubStream(baseStream, 16, baseStream.Length - 32))
                    {
                        Assert.IsTrue(ss.CanRead);
                        Assert.IsTrue(ss.CanSeek);
                        Assert.IsTrue(ss.CanWrite);
                        Assert.IsTrue(ss.Length == baseStream.Length - 32);
                        Assert.IsTrue(ss.Position == 0);
                        ss.Seek(128, SeekOrigin.Begin);
                        Assert.IsTrue(ss.Position == 128);
                        ss.Seek(-1, SeekOrigin.Current);
                        Assert.IsTrue(ss.Position == 127);
                        ss.Seek(1, SeekOrigin.Current);
                        Assert.IsTrue(ss.Position == 128);
                        ss.Seek(-1, SeekOrigin.End);
                        Assert.IsTrue(ss.Position == ss.Length - 1);
                        ss.Seek(0, SeekOrigin.End);
                        Assert.IsTrue(ss.Position == ss.Length);
                    }
                }
            }
            // サイズを超える読出し
            {
                using (Stream baseStream = new MemoryStream(256))
                {
                    baseStream.Write(incBuffer, 0, incBuffer.Length);
                    using (Stream ss = new SubStream(baseStream, 16, baseStream.Length - 32))
                    {
                        byte[] buffer = new byte[ss.Length + 16];
                        int read = ss.Read(buffer, 0, buffer.Length);
                        Assert.IsTrue(read == ss.Length);
                        for (int i = 0; i < buffer.Length; i++)
                        {
                            if (i < ss.Length)
                            {
                                Assert.IsTrue(buffer[i] == 16 + i);
                            }
                            else
                            {
                                Assert.IsTrue(buffer[i] == 0);
                            }
                        }
                        Assert.IsTrue(ss.Position == ss.Length);
                        read = ss.Read(buffer, 0, 16);
                        Assert.IsTrue(read == 0);
                    }
                }
            }
            // サイズを超える書込み
            {
                using (Stream baseStream = new MemoryStream(256))
                {
                    baseStream.Seek(0, SeekOrigin.Begin);
                    baseStream.Write(incBuffer, 0, incBuffer.Length);
                    using (Stream ss = new SubStream(baseStream, 16, baseStream.Length - 32))
                    {
                        byte[] buffer = new byte[ss.Length + 32];
                        Utils.CheckReturnException(new ArgumentOutOfRangeException(), () =>
                                {
                                    ss.Write(buffer, 0, buffer.Length);
                                    return true;
                                });
                        Assert.IsTrue(ss.Position == 0);

                        byte[] baseBuffer = new byte[256];
                        baseStream.Seek(0, SeekOrigin.Begin);
                        baseStream.Read(baseBuffer, 0, 256);
                        for (int i = 0; i < buffer.Length; i++)
                        {
                            Assert.IsTrue(baseBuffer[i] == i);
                        }
                    }
                }
            }
            // 範囲外へのシーク
            {
                using (Stream baseStream = new MemoryStream(256))
                {
                    baseStream.Write(incBuffer, 0, incBuffer.Length);
                    using (Stream ss = new SubStream(baseStream, 16, baseStream.Length - 32))
                    {
                        Utils.CheckReturnException(new ArgumentOutOfRangeException(), () =>
                                {
                                    ss.Seek(-1, SeekOrigin.Begin);
                                    return true;
                                });
                        Utils.CheckReturnException(new ArgumentOutOfRangeException(), () =>
                                {
                                    ss.Seek(ss.Length + 1, SeekOrigin.Begin);
                                    return true;
                                });
                        Utils.CheckReturnException(new ArgumentOutOfRangeException(), () =>
                                {
                                    ss.Seek(1, SeekOrigin.End);
                                    return true;
                                });
                        ss.Seek(128, SeekOrigin.Begin);
                        Utils.CheckReturnException(new ArgumentOutOfRangeException(), () =>
                                {
                                    ss.Seek(ss.Length + 1 - 128, SeekOrigin.Current);
                                    return true;
                                });
                    }
                }
            }
            // オフセット読出し・シーク後終端まで読出し
            {
                using (Stream baseStream = new MemoryStream(256))
                {
                    baseStream.Seek(0, SeekOrigin.Begin);
                    baseStream.Write(incBuffer, 0, incBuffer.Length);
                    using (Stream ss = new SubStream(baseStream, 16, baseStream.Length - 32))
                    {
                        byte[] buffer = new byte[ss.Length];
                        int read = ss.Read(buffer, 0, buffer.Length - 128);
                        Assert.IsTrue(read == buffer.Length - 128);
                        Assert.IsTrue(ss.Position == buffer.Length - 128);
                        read = ss.Read(buffer, read, 128);
                        Assert.IsTrue(read == 128);
                        Assert.IsTrue(ss.Position == ss.Length);
                        for (int i = 0; i < buffer.Length; i++)
                        {
                            Assert.IsTrue(buffer[i] == 16 + i);
                        }
                    }
                }
            }
            // シーク後サイズを超える読出し
            {
                using (Stream baseStream = new MemoryStream(256))
                {
                    baseStream.Seek(0, SeekOrigin.Begin);
                    baseStream.Write(incBuffer, 0, incBuffer.Length);
                    using (Stream ss = new SubStream(baseStream, 16, baseStream.Length - 32))
                    {
                        byte[] buffer = new byte[ss.Length];
                        ss.Seek(128, SeekOrigin.Begin);
                        int read = ss.Read(buffer, 0, 128 - 16);
                        Assert.IsTrue(read == ss.Length - 128);
                        Assert.IsTrue(ss.Position == ss.Length);
                        for (int i = 0; i < buffer.Length; i++)
                        {
                            if (i < ss.Length - 128)
                            {
                                Assert.IsTrue(buffer[i] == 16 + 128 + i);
                            }
                            else
                            {
                                Assert.IsTrue(buffer[i] == 0);
                            }
                        }
                    }
                }
            }
            // オフセット書込み・シーク後終端まで書込み
            {
                using (Stream baseStream = new MemoryStream(256))
                {
                    baseStream.Seek(0, SeekOrigin.Begin);
                    baseStream.Write(zeroBuffer, 0, incBuffer.Length);
                    using (Stream ss = new SubStream(baseStream, 16, baseStream.Length - 32))
                    {
                        byte[] buffer = new byte[ss.Length];
                        for (int i = 0; i < buffer.Length; i++)
                        {
                            buffer[i] = (byte)i;
                        }
                        ss.Write(buffer, 0, buffer.Length - 128);
                        Assert.IsTrue(ss.Position == buffer.Length - 128);
                        ss.Write(buffer, buffer.Length - 128, 128);
                        Assert.IsTrue(ss.Position == ss.Length);

                        byte[] baseBuffer = new byte[256];
                        baseStream.Seek(0, SeekOrigin.Begin);
                        baseStream.Read(baseBuffer, 0, 256);
                        for (int i = 0; i < baseBuffer.Length; i++)
                        {
                            if (16 <= i && i < 16 + ss.Length)
                            {
                                Assert.IsTrue(baseBuffer[i] == i - 16);
                            }
                            else
                            {
                                Assert.IsTrue(baseBuffer[i] == 0);
                            }
                        }
                    }
                }
            }
            // シーク後サイズを超える書込み
            {
                using (Stream baseStream = new MemoryStream(256))
                {
                    baseStream.Seek(0, SeekOrigin.Begin);
                    baseStream.Write(zeroBuffer, 0, incBuffer.Length);
                    using (Stream ss = new SubStream(baseStream, 16, baseStream.Length - 16))
                    {
                        byte[] buffer = new byte[ss.Length];
                        for (int i = 0; i < buffer.Length; i++)
                        {
                            buffer[i] = (byte)i;
                        }
                        ss.Seek(128, SeekOrigin.Begin);
                        Assert.IsTrue(ss.Position == 128);

                        Utils.CheckReturnException(new ArgumentOutOfRangeException(), () =>
                                {
                                    ss.Write(buffer, 0, buffer.Length);
                                    return true;
                                });
                        Assert.IsTrue(ss.Position == 128);

                        byte[] baseBuffer = new byte[256];
                        baseStream.Seek(0, SeekOrigin.Begin);
                        baseStream.Read(baseBuffer, 0, 256);
                        for (int i = 0; i < baseBuffer.Length; i++)
                        {
                            Assert.IsTrue(baseBuffer[i] == 0);
                        }
                    }
                }
            }
            // base より長い範囲を切り出し、シークによるサイズ拡張
            {
                using (Stream baseStream = new MemoryStream(256))
                {
                    baseStream.Seek(0, SeekOrigin.Begin);
                    baseStream.Write(zeroBuffer, 0, incBuffer.Length);
                    using (Stream ss = new SubStream(baseStream, 16, 256))
                    {
                        Assert.IsTrue(baseStream.Length == 256);
                        Assert.IsTrue(ss.Length == 256);
                        ss.Seek(256, SeekOrigin.Begin);
                        Assert.IsTrue(ss.Length == 256);
                        Assert.IsTrue(baseStream.Length == 256 + 16);
                    }
                }
            }
            // base より長い範囲を切り出し、書込みによるサイズ拡張
            {
                using (Stream baseStream = new MemoryStream(256))
                {
                    baseStream.Seek(0, SeekOrigin.Begin);
                    baseStream.Write(zeroBuffer, 0, incBuffer.Length);
                    using (Stream ss = new SubStream(baseStream, 16, 256))
                    {
                        Assert.IsTrue(baseStream.Length == 256);
                        Assert.IsTrue(ss.Length == 256);
                        byte[] buffer = new byte[ss.Length];
                        for (int i = 0; i < buffer.Length; i++)
                        {
                            buffer[i] = (byte)i;
                        }
                        ss.Write(buffer, 0, buffer.Length);
                        Assert.IsTrue(ss.Length == 256);
                        Assert.IsTrue(baseStream.Length == 256 + 16);
                    }
                }
            }
            // base より長い範囲を切り出し、サイズを超える読出し
            {
                using (Stream baseStream = new MemoryStream(256))
                {
                    baseStream.Write(incBuffer, 0, incBuffer.Length);
                    using (Stream ss = new SubStream(baseStream, 16, 256))
                    {
                        byte[] buffer = new byte[ss.Length];
                        Utils.CheckReturnException(new ArgumentOutOfRangeException(), () =>
                                {
                                    ss.Read(buffer, 0, buffer.Length);
                                    return true;
                                });

                        // サイズ拡充
                        ss.Seek(256 - 16, SeekOrigin.Begin);
                        ss.Write(incBuffer, 0, 16);
                        ss.Seek(0, SeekOrigin.Begin);

                        int read = ss.Read(buffer, 0, buffer.Length);
                        Assert.IsTrue(read == ss.Length);
                        for (int i = 0; i < buffer.Length; i++)
                        {
                            if (i < ss.Length - 16)
                            {
                                Assert.IsTrue(buffer[i] == 16 + i);
                            }
                            else if (i < ss.Length)
                            {
                                Assert.IsTrue(buffer[i] == i - (256 - 16));
                            }
                            else
                            {
                                Assert.IsTrue(buffer[i] == 0);
                            }
                        }
                        Assert.IsTrue(ss.Position == ss.Length);
                        read = ss.Read(buffer, 0, 16);
                        Assert.IsTrue(read == 0);
                    }
                }
            }
            // base より長い範囲を切り出し、サイズを超える書き込み
            {
                using (Stream baseStream = new MemoryStream(256))
                {
                    baseStream.Seek(0, SeekOrigin.Begin);
                    baseStream.Write(incBuffer, 0, incBuffer.Length);
                    using (Stream ss = new SubStream(baseStream, 16, 256))
                    {
                        byte[] buffer = new byte[ss.Length + 16];
                        Utils.CheckReturnException(new ArgumentOutOfRangeException(), () =>
                                {
                                    ss.Write(buffer, 0, buffer.Length);
                                    return true;
                                });
                        Assert.IsTrue(ss.Position == 0);

                        byte[] baseBuffer = new byte[256];
                        baseStream.Seek(0, SeekOrigin.Begin);
                        baseStream.Read(baseBuffer, 0, 256);
                        for (int i = 0; i < 256; i++)
                        {
                            Assert.IsTrue(baseBuffer[i] == i);
                        }
                    }
                }
            }
            // base より長い範囲を切り出し、サイズを超えるシーク
            {
                using (Stream baseStream = new MemoryStream(256))
                {
                    baseStream.Write(incBuffer, 0, incBuffer.Length);
                    using (Stream ss = new SubStream(baseStream, 16, 256))
                    {
                        Utils.CheckReturnException(new ArgumentOutOfRangeException(), () =>
                                {
                                    ss.Seek(256 + 1, SeekOrigin.Begin);
                                    return true;
                                });
                    }
                }
            }
            // オフセット書込み・シーク後終端まで書込み
            {
                using (Stream baseStream = new MemoryStream(256))
                {
                    using (Stream ss = new SubStream(baseStream, 16, 256))
                    {
                        ss.Flush();
                    }
                }
            }
        }

        [TestMethod]
        public void SourceBasedStreamTest()
        {
            byte[] incBuffer = new byte[256];
            for (int i = 0; i < incBuffer.Length; i++)
            {
                incBuffer[i] = (byte)i;
            }
            ISource source = new MemorySource(incBuffer, 0, incBuffer.Length);

            // サポートされていないメソッド
            {
                using (Stream sbs = new SourceBasedStream(source))
                {
                    Utils.CheckReturnException(new NotSupportedException(), () =>
                            {
                                byte[] buffer = new byte[10];
                                sbs.Write(buffer, 0, 10);
                                return true;
                            });
                    Utils.CheckReturnException(new NotSupportedException(), () =>
                            {
                                sbs.SetLength(10);
                                return true;
                            });
                }
            }
            // プロパティ
            {
                using (Stream sbs = new SourceBasedStream(source))
                {
                    Assert.IsTrue(sbs.CanRead);
                    Assert.IsTrue(sbs.CanSeek);
                    Assert.IsFalse(sbs.CanWrite);
                    Assert.IsTrue(sbs.Length == incBuffer.Length);
                    Assert.IsTrue(sbs.Position == 0);
                    sbs.Seek(128, SeekOrigin.Begin);
                    Assert.IsTrue(sbs.Position == 128);
                    sbs.Seek(-1, SeekOrigin.Current);
                    Assert.IsTrue(sbs.Position == 127);
                    sbs.Seek(1, SeekOrigin.Current);
                    Assert.IsTrue(sbs.Position == 128);
                    sbs.Seek(-1, SeekOrigin.End);
                    Assert.IsTrue(sbs.Position == sbs.Length - 1);
                    sbs.Seek(0, SeekOrigin.End);
                    Assert.IsTrue(sbs.Position == sbs.Length);
                }
            }
            // 読み出し
            {
                using (Stream sbs = new SourceBasedStream(source))
                {
                    byte[] buffer = new byte[sbs.Length];
                    int read = sbs.Read(buffer, 0, 128);
                    Assert.IsTrue(read == 128);
                    read = sbs.Read(buffer, 128, 128);
                    Assert.IsTrue(read == 128);
                    for (int i = 0; i < buffer.Length; i++)
                    {
                        Assert.IsTrue(buffer[i] == incBuffer[i]);
                    }
                }
                using (Stream sbs = new SourceBasedStream(source))
                {
                    byte[] buffer = new byte[sbs.Length];
                    int read = sbs.Read(buffer, 0, 128);
                    Assert.IsTrue(read == 128);
                    Assert.IsTrue(sbs.Position == 128);
                    read = sbs.Read(buffer, 128, 128);
                    Assert.IsTrue(read == 128);
                    Assert.IsTrue(sbs.Position == 256);
                    for (int i = 0; i < buffer.Length; i++)
                    {
                        Assert.IsTrue(buffer[i] == incBuffer[i]);
                    }
                }
            }
            // サイズを超えた読み出し
            {
                using (Stream sbs = new SourceBasedStream(source))
                {
                    byte[] buffer = new byte[sbs.Length + 16];
                    int read = sbs.Read(buffer, 0, (int)sbs.Length + 16);
                    Assert.IsTrue(read == 256);
                    Assert.IsTrue(sbs.Position == 256);
                    for (int i = 0; i < read; i++)
                    {
                        Assert.IsTrue(buffer[i] == incBuffer[i]);
                    }
                    read = sbs.Read(buffer, 0, 1);
                    Assert.IsTrue(read == 0);
                    Assert.IsTrue(sbs.Position == 256);
                }
                using (Stream sbs = new SourceBasedStream(source))
                {
                    byte[] buffer = new byte[sbs.Length + 16];
                    sbs.Seek(16, SeekOrigin.Begin);
                    int read = sbs.Read(buffer, 0, (int)sbs.Length);
                    Assert.IsTrue(read == 256 - 16);
                    Assert.IsTrue(sbs.Position == 256);
                    for (int i = 0; i < read; i++)
                    {
                        Assert.IsTrue(buffer[i] == incBuffer[i + 16]);
                    }
                    read = sbs.Read(buffer, 0, 1);
                    Assert.IsTrue(read == 0);
                    Assert.IsTrue(sbs.Position == 256);
                }
            }
            // 範囲外へのシーク
            {
                using (Stream sbs = new SourceBasedStream(source))
                {
                    Utils.CheckReturnException(new ArgumentOutOfRangeException(), () =>
                            {
                                sbs.Seek(-1, SeekOrigin.Begin);
                                return true;
                            });
                    Utils.CheckReturnException(new ArgumentOutOfRangeException(), () =>
                            {
                                sbs.Seek(257, SeekOrigin.Begin);
                                return true;
                            });
                    Utils.CheckReturnException(new ArgumentOutOfRangeException(), () =>
                            {
                                sbs.Seek(128, SeekOrigin.Begin);
                                sbs.Seek(-129, SeekOrigin.Current);
                                return true;
                            });
                    Utils.CheckReturnException(new ArgumentOutOfRangeException(), () =>
                            {
                                sbs.Seek(128, SeekOrigin.Begin);
                                sbs.Seek(129, SeekOrigin.Current);
                                return true;
                            });
                    Utils.CheckReturnException(new ArgumentOutOfRangeException(), () =>
                            {
                                sbs.Seek(-257, SeekOrigin.End);
                                return true;
                            });
                    Utils.CheckReturnException(new ArgumentOutOfRangeException(), () =>
                            {
                                sbs.Seek(1, SeekOrigin.End);
                                return true;
                            });
                }
            }
            // Flush
            {
                using (Stream sbs = new SourceBasedStream(source))
                {
                    sbs.Flush();
                }
            }
        }
    }
}
