﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Reflection;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.Authoring.AuthoringLibrary;

namespace AuthoringToolsTest
{
    [TestClass]
    public class SinkTest
    {
        [TestMethod]
        public void FileSinkTest()
        {
            var outputDir = Path.Combine(Directory.GetCurrentDirectory().Replace("\\" + Assembly.GetExecutingAssembly().GetName().Name, string.Empty), MethodBase.GetCurrentMethod().Name);
            Utils.DeleteDirectoryIfExisted(outputDir);
            Directory.CreateDirectory(outputDir);
            string filePath = Path.Combine(outputDir, "test.dat");
            int fileSize = 1024;
            using (Stream stream = new FileStream(filePath, FileMode.Create, FileAccess.Write, FileShare.None))
            {
            }
            FileSink sink = new FileSink(filePath, 0, fileSize);

            // 通常書込み
            byte[] data = Utils.GetTestBuffer(fileSize);
            Assert.IsTrue(sink.PushData(new ByteData(new ArraySegment<byte>(data)), 0) == fileSize);
            using (Stream stream = new FileStream(filePath, FileMode.Open, FileAccess.Read, FileShare.None))
            {
                byte[] ans = new byte[fileSize];
                stream.Read(ans, 0, ans.Length);
                Utils.CheckBufferEquality(data, 0, ans, 0, fileSize);
            }
            // offset 書込み
            Assert.IsTrue(sink.PushData(new ByteData(new ArraySegment<byte>(data)), 512) == fileSize - 512);
            using (Stream stream = new FileStream(filePath, FileMode.Open, FileAccess.Read, FileShare.None))
            {
                byte[] ans = new byte[fileSize];
                stream.Read(ans, 0, ans.Length);
                Utils.CheckBufferEquality(data, 0, ans, 0, 512);
                Utils.CheckBufferEquality(data, 0, ans, 512, 512);
            }
            // offset data 書込み
            Assert.IsTrue(sink.PushData(new ByteData(new ArraySegment<byte>(data, 512, 512)), 512) == 512);
            using (Stream stream = new FileStream(filePath, FileMode.Open, FileAccess.Read, FileShare.None))
            {
                byte[] ans = new byte[fileSize];
                stream.Read(ans, 0, ans.Length);
                Utils.CheckBufferEquality(data, 0, ans, 0, fileSize);
            }
            File.Delete(filePath);
        }

        [TestMethod]
        public void MemorySinkTest()
        {
        }

        [TestMethod]
        public void SourceThroughSinkTest()
        {
        }

        [TestMethod]
        public void StreamSinkTest()
        {
            int bufferSize = 1024;
            byte[] buffer = new byte[bufferSize];
            StreamSink sink = new StreamSink(new MemoryStream(buffer));
            sink.SetSize(bufferSize);
            // 通常書込み
            byte[] data = Utils.GetTestBuffer(bufferSize);
            Assert.IsTrue(sink.PushData(new ByteData(new ArraySegment<byte>(data)), 0) == bufferSize);
            Utils.CheckBufferEquality(data, 0, buffer, 0, bufferSize);
            // offset 書込み
            Assert.IsTrue(sink.PushData(new ByteData(new ArraySegment<byte>(data)), 512) == bufferSize - 512);
            Utils.CheckBufferEquality(data, 0, buffer, 0, 512);
            Utils.CheckBufferEquality(data, 0, buffer, 512, 512);
            // offset data 書込み
            Assert.IsTrue(sink.PushData(new ByteData(new ArraySegment<byte>(data, 512, 512)), 512) == 512);
            Utils.CheckBufferEquality(data, 0, buffer, 0, bufferSize);
        }

        [TestMethod]
        public void SubSinkTest()
        {
            byte[] buffer = Utils.GetTestBuffer(1024);
            {
                MemorySink sink = new MemorySink(1024);
                SubSink ss = new SubSink(sink, 128, 512);
                ByteData data = new ByteData(new ArraySegment<byte>(buffer, 0, 1024));
                int pushed = ss.PushData(data, 0);
                Assert.IsTrue(pushed == 512);
                ByteData dst = sink.ToSource().PullData(0, 1024);
                for (int i = 0; i < dst.Buffer.Count; i++)
                {
                    if (i < 128 || i >= 128 + 512)
                    {
                        Assert.IsTrue(dst.Buffer.Array[i + dst.Buffer.Offset] == 0);
                    }
                    else
                    {
                        Assert.IsTrue(dst.Buffer.Array[i + dst.Buffer.Offset] == data.Buffer.Array[i - 128]);
                    }
                }
            }
            {
                MemorySink sink = new MemorySink(1024);
                SubSink ss = new SubSink(sink, 512, 1024);
                ByteData data = new ByteData(new ArraySegment<byte>(buffer, 0, 1024));
                int pushed = ss.PushData(data, 0);
                Assert.IsTrue(pushed == 512);
                ByteData dst = sink.ToSource().PullData(0, 1024);
                for (int i = 0; i < dst.Buffer.Count; i++)
                {
                    if (i < 512)
                    {
                        Assert.IsTrue(dst.Buffer.Array[i + dst.Buffer.Offset] == 0);
                    }
                    else
                    {
                        Assert.IsTrue(dst.Buffer.Array[i + dst.Buffer.Offset] == data.Buffer.Array[i - 512]);
                    }
                }
            }
        }
    }
}
