﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Collections.Generic;
using System.Reflection;
using System.Security.Cryptography;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.Authoring.AuthoringLibrary;
using Nintendo.Authoring.FileSystemMetaLibrary;
using TestUtility;

namespace AuthoringToolsTest
{
    using EntryFilterRule = Pair<FilterType, string>;
    using EntryFilterRuleRegex = Pair<FilterType, System.Text.RegularExpressions.Regex>;
    using DirectoryConnector = Pair<string, string>;

    [TestClass]
    public class FilterTest
    {
        public TestContext TestContext { get; set; }

        private delegate void AdfWrite(string adfPath, string dirPath, List<EntryFilterRule> filter);
        private delegate int AdfRead(string adfPath);
        private void CompareFilterringResults(FilterTestUtil.GenerateSampleFiles sampleFiles,
            List<EntryFilterRule> filter, List<EntryFilterRule> filterRemoveOnly,
            bool isFilterValid, AdfWrite writeDelegate, AdfRead readDelegate)
        {
            // adf を作成
            var adfPath = Directory.GetCurrentDirectory() + "/default.adf";
            writeDelegate(adfPath, sampleFiles.testDirPath, null);
            int defaultCount = readDelegate(adfPath);

            adfPath = Directory.GetCurrentDirectory() + "/filter.adf";
            writeDelegate(adfPath, sampleFiles.testDirPath, filter);
            int filteredCount = readDelegate(adfPath);

            adfPath = Directory.GetCurrentDirectory() + "/filter.remove_only.adf";
            writeDelegate(adfPath, sampleFiles.testDirPath, filterRemoveOnly);
            int filteredRemoveOnlyCount = readDelegate(adfPath);

            if (isFilterValid)
            {
                Assert.IsTrue(defaultCount == sampleFiles.fileCount);
                Assert.IsTrue(filteredCount == (defaultCount - sampleFiles.fdfTest0FileRemoveCount + sampleFiles.fdfTest0FileRemoveExceptionCount));
                Assert.IsTrue(filteredRemoveOnlyCount == (sampleFiles.fileCount - sampleFiles.fdfTest0FileRemoveCount));
            }
            else
            {
                Assert.IsTrue(defaultCount == sampleFiles.fileCount);
                Assert.IsTrue(filteredCount == sampleFiles.fileCount);
                Assert.IsTrue(filteredRemoveOnlyCount == sampleFiles.fileCount);
            }
        }

        [TestMethod]
        public void TestFilterBasic()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var outputDir = Path.Combine(Directory.GetCurrentDirectory().Replace("\\" + Assembly.GetExecutingAssembly().GetName().Name, string.Empty), MethodBase.GetCurrentMethod().Name);
            var sampleFiles = new FilterTestUtil.GenerateSampleFiles();
            sampleFiles.Generate(outputDir);

            List<EntryFilterRule>[] filterList = new List<EntryFilterRule>[sampleFiles.fdfPathList.Length];
            for (int i = 0; i < sampleFiles.validfdfPathListNum; i++)
            {
                filterList[i] = FilterDescription.ParseFdf(sampleFiles.fdfPathList[i]);
            }
            //!! フィルターのテスト追加時はここを追加
            var filter0 = filterList[0];
            var filter1 = filterList[1];
            var filter2 = filterList[2];
            var filter3 = filterList[3];
            var filter4 = filterList[4];
            var filter5 = filterList[5];
            var filter6 = filterList[6];
            var filter7 = filterList[7];
            var filter8 = filterList[8];

            Assert.IsTrue(filter0.Count > filter1.Count);
            // 環境変数は消えているはず
            Assert.IsTrue(filter0[0].second.ToString().Contains("WINDIR") == false);

            // filter0 は filter1 を内包しているはず
            foreach (var flt1 in filter1)
            {
                bool found = false;
                foreach (var flt0 in filter0)
                {
                    if (flt0.first == flt1.first && flt0.second.ToString() == flt1.second.ToString())
                    {
                        found = true;
                        break;
                    }
                }
                Assert.IsTrue(found);
            }

            // filter0, filter2（filter0 + filter1 の削除のみ）を用いたフィルタリング結果のチェック
            CompareFilterringResults(sampleFiles, filter0, filter2, false,
                (adfPath, dirPath, filter) => new PartitionFsAdfWriter(adfPath).Write(dirPath),
                (adfPath) => new PartitionFsAdfReader(adfPath).GetFileSystemInfo().entries.Count);
            CompareFilterringResults(sampleFiles, filter0, filter2, true,
                (adfPath, dirPath, filter) => new RomFsAdfWriter(adfPath).Write(dirPath, filter),
                (adfPath) => new RomFsAdfReader(adfPath).GetFileSystemInfo().entries.Count);

            // filter3 ですべて除外した場合のチェック
            {
                int count = 0;
                var filter3Regex = FilterDescription.ConvertFilterRuleStringToRegex(filter3);
                foreach (var file in sampleFiles.fileTree)
                {
                    if (FilterDescription.IsEntryInFilterList(file, "", filter3Regex) == false)
                    {
                        count++;
                    }
                }
                Assert.AreEqual(count, sampleFiles.fdfTest3FileRemainCount);

                // 一応 adf を作ってもチェック
                var adfPath = Directory.GetCurrentDirectory() + "/filter3.adf";
                new RomFsAdfWriter(adfPath).Write(sampleFiles.testDirPath, filter3);
                count = new RomFsAdfReader(adfPath).GetFileSystemInfo().entries.Count;
                Assert.AreEqual(count, sampleFiles.fdfTest3FileRemainCount);
            }

            // filter4, 5 で相対パス + 循環参照のチェック
            {
                // filter4 と filter5 はどっちから呼んでもフィルタルール数が 2 で終わるはずで、
                // 相対パスを正しく処理できないなどで fdf が見つからなければ例外が発生しているはず
                Assert.AreEqual(filter4.Count, filter5.Count);
                Assert.AreEqual(filter4.Count, sampleFiles.fdfTest4FilterRuleCount);
            }

            // filter6 で固定パス絶対追加
            {
                // folder1/folder2 以下のファイルがすべて入っていること(3 つ)
                // folder1/folderB/fuga.bin が入っていること(1つ) -> 計4 ファイルであること
                var adfPath = Directory.GetCurrentDirectory() + "/filter6.adf";
                new RomFsAdfWriter(adfPath).Write(sampleFiles.testDirPath, filter6);
                var count = new RomFsAdfReader(adfPath).GetFileSystemInfo().entries.Count;
                Assert.AreEqual(count, sampleFiles.fdfTest6FileRemainCount);
            }

            // filter7, 8 で空フィルタの動作をチェック
            {
                // filter7 で空のファイルだとフィルタリストは null になって動作する
                Assert.AreEqual(filter7, null);

                // 空のフィルタを include する確認
                // (filter8 は空の filter7 を include しているだけなので null になる)
                Assert.AreEqual(filter8, null);
            }

            // filter9,10 で*+ が途中で入ってる場合に弾くテスト
            {
                Utils.CheckReturnException(new ArgumentException("You can NOT use '+' or '-' before '*+'."), () =>
                {
                    FilterDescription.ParseFdf(sampleFiles.fdfPathList[9]);
                    return true;
                });
                Utils.CheckReturnException(new ArgumentException("You can NOT use '+' or '-' before '*+'."), () =>
                {
                    FilterDescription.ParseFdf(sampleFiles.fdfPathList[10]);
                    return true;
                });
            }
        }
    }
}
