﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.Authoring.AuthoringLibrary;
using Nintendo.Authoring.ETicketLibrary;

namespace AuthoringToolsTest.ContentArchiveLibraryTest
{
    [TestClass]
    public class ETicketTest
    {
        [TestMethod]
        public void RightsIdTest()
        {
            UInt64 applicationId = 0x0005000c10000000;

            var keyGenerationPattern = new Tuple<byte, string>[]
            {
                Tuple.Create((byte)0, "0000000000000000"),
                Tuple.Create((byte)2, "0000000000000000"),
                Tuple.Create((byte)3, "0000000000000003"),
                Tuple.Create((byte)4, "0000000000000004"),
            };

            foreach (var pattern in keyGenerationPattern)
            {
                Byte[] rightsId = TicketUtility.CreateRightsId(applicationId, pattern.Item1);
                string rightsIdText = TicketUtility.CreateRightsIdText(applicationId, pattern.Item1);

                string applicationIdText = applicationId.ToString("x16");

                Assert.AreEqual(rightsIdText.Length, 32);
                Assert.AreEqual(applicationId, TicketUtility.GetContentMetaIdFromRightsId(rightsId));
                Assert.AreEqual(rightsIdText, applicationIdText + pattern.Item2);
            }
        }

        [TestMethod]
        public void PublishTicketTest()
        {
            UInt64 applicationId = 0x0005000c10000000;
            UInt32 ticketBufferLength = 1024;
            Byte[] titleKey = new byte[16];
            UInt64 deviceId = 0;
            UInt64 ticketId = 0;
            Byte[] rightsId = TicketUtility.CreateRightsId(applicationId, (byte)2);
            Byte commonKeyId = 2;

            Byte[] ticketBuffer = new Byte[ticketBufferLength];
            TicketPublication.PublishTicket(ticketBuffer, ticketBufferLength, titleKey, deviceId, ticketId, rightsId, commonKeyId, System.Text.Encoding.ASCII.GetBytes("Root-CA00000004-XS00000020"));

            // Byte[] の offset から 8bit 読み取り UInt64 に変換する
            Func<Byte[], int, UInt64> ByteToUInt64 = (Byte[] bytes, int offset) =>
            {
                UInt64 result = 0;

                result |= (UInt64)bytes[offset + 7] << 56;
                result |= (UInt64)bytes[offset + 6] << 48;
                result |= (UInt64)bytes[offset + 5] << 40;
                result |= (UInt64)bytes[offset + 4] << 32;
                result |= (UInt64)bytes[offset + 3] << 24;
                result |= (UInt64)bytes[offset + 2] << 16;
                result |= (UInt64)bytes[offset + 1] << 8;
                result |= (UInt64)bytes[offset + 0] << 0;

                return result;
            };

            int titleKeyOffset = 0x180;
            int commonKeyIdOffset = 0x285;
            int ticketIdOffset = 0x290;
            int deviceIdOffset = 0x298;
            int rightIdOffset = 0x2a0;

            // Ticket に titleKey が正しく埋め込まれているかのチェック
            for (int i = 0; i < titleKey.Length; i++)
            {
                Assert.AreEqual(titleKey[i], ticketBuffer[titleKeyOffset + i]);
            }
            // Ticket に commonKeyId が正しく埋め込まれているかのチェック
            Assert.AreEqual(commonKeyId, ticketBuffer[commonKeyIdOffset]);
            // Ticket に ticketId が正しく埋め込まれているかのチェック
            Assert.AreEqual(ticketId, ByteToUInt64(ticketBuffer, ticketIdOffset));
            // Ticket に deviceId が正しく埋め込まれているかのチェック
            Assert.AreEqual(deviceId, ByteToUInt64(ticketBuffer, deviceIdOffset));
            // Ticket に rightsId が正しく埋め込まれているかのチェック
            for (int i = 0; i < rightsId.Length; i++)
            {
                Assert.AreEqual(rightsId[i], ticketBuffer[rightIdOffset + i]);
            }
        }
    }
}
