﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Xml.Linq;
using System.Xml.XPath;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params;
using Reactive.Bindings;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Test.MainWindow.ProjectEditPanel.Pages
{
    public class TitlePageVmTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Fact]
        public void DefaultCtor()
        {
            using (var vm = _context.DiContainer.GetInstance<TitlePageVm>())
            {
                Assert.NotEmpty(vm.Params);
                Assert.NotNull(vm.TitleParam);
                Assert.False(vm.CanAddOrRemoveTitle.Value);
            }
        }

        [Theory]
        [InlineData(AppModeType.Project, true, false)]
        [InlineData(AppModeType.AocMeta, false, false)]
        [InlineData(AppModeType.AocNsp, false, false)]
        [InlineData(AppModeType.ApplicationMeta, true, false)]
        [InlineData(AppModeType.PatchNsp, false, false)]
        public void CanAddOrRemoveTitle(AppModeType t, bool canAddOrRemove, bool isReadOnly)
        {
            _context.DiContainer.GetInstance<Project>().Meta.IsReadOnly = isReadOnly;

            var profile = _context.DiContainer.GetInstance<AppProfile>();
            profile.AppMode = t;

            using (var vm = _context.DiContainer.GetInstance<TitlePageVm>())
            {
                Assert.NotEmpty(vm.Params);
                Assert.NotNull(vm.TitleParam);
                Assert.Equal(canAddOrRemove, vm.TitleParam.AddCommand.CanExecute());
                Assert.Equal(canAddOrRemove, vm.CanAddOrRemoveTitle.Value);
            }
        }

        [Theory]
        [InlineData(true)]
        [InlineData(false)]
        public void CanAddOrRemoveTitleOnAppNsp(bool isReadOnly)
        {
            CanAddOrRemoveTitle(
                AppModeType.ApplicationNsp,
                isReadOnly == false && _context.DiContainer.GetInstance<ApplicationCapability>().IsSupportNspTitleAddOrRemove,
                isReadOnly);
        }

        [Fact]
        public void AddTitle()
        {
            using (var vm = _context.DiContainer.GetInstance<TitlePageVm>())
            {
                var project = _context.DiContainer.GetInstance<Project>();
                project.DiContainer = _context.DiContainer;

                Assert.Equal(0, vm.TitleParam.Titles.Count);
                Assert.True(vm.HasErrors.Value);

                vm.TitleParam.AddCommand.Execute();
                Assert.Equal(1, vm.TitleParam.Titles.Count);
                Assert.True(vm.HasErrors.Value);

                var titleVm = vm.TitleParam.Titles[0];
                Assert.NotNull(titleVm);
                Assert.True(vm.HasErrors.Value);
                Assert.True(titleVm.HasErrors.Value);
                Assert.True(titleVm.IconFilePath.IsUse.Value);
                Assert.False(titleVm.IconFilePath.OpenByExplorerCommand.CanExecute());
            }
        }

        [Fact]
        public void IconFilePath()
        {
            var project = _context.DiContainer.GetInstance<Project>();
            project.DiContainer = _context.DiContainer;

            using (var vm = _context.DiContainer.GetInstance<TitlePageVm>())
            {
                vm.TitleParam.AddCommand.Execute();
                var titleVm = vm.TitleParam.Titles[0];

                ((ReactiveProperty<string>)titleVm.Name.Property).Value = "test";
                ((ReactiveProperty<string>)titleVm.Publisher.Property).Value = "test";

                titleVm.Model.IsReplaceIcon = false;
                Assert.False(titleVm.HasErrors.Value);
                Assert.False(titleVm.IconFilePath.OpenByExplorerCommand.CanExecute());

                titleVm.Model.IsReplaceIcon = true;
                Assert.True(titleVm.HasErrors.Value);
                Assert.False(titleVm.IconFilePath.OpenByExplorerCommand.CanExecute());

                var rp = (ReactiveProperty<string>)titleVm.IconFilePath.Property;

                rp.Value = "";
                Assert.True(rp.HasErrors);

                rp.Value = "not_found_file_path";
                Assert.True(rp.HasErrors);
                Assert.True(titleVm.HasErrors.Value);
                Assert.True(vm.HasErrors.Value);
                Assert.False(titleVm.IconFilePath.OpenByExplorerCommand.CanExecute());

                var iconFilePath = Path.Combine(TestContext.TestDataDirPath, @"icon_testdata\1024x1024x24.bmp");
                rp.Value = iconFilePath;
                Assert.False(rp.HasErrors);
                Assert.False(titleVm.HasErrors.Value);
                Assert.False(vm.HasErrors.Value);
                Assert.True(titleVm.IconFilePath.OpenByExplorerCommand.CanExecute());

                // アイコンを指定しない場合 XML に含まれない
                titleVm.Model.IsReplaceIcon = false;
                Assert.False(rp.HasErrors);
                Assert.False(titleVm.HasErrors.Value);
                Assert.Equal(titleVm.Model, project.Meta.Application.Titles[0]);
                {
                    var xmlText = project.MakeAppMetaXmlForAuthoringTool();
                    using (var reader = new StringReader(xmlText))
                    {
                        var xml = XDocument.Load(reader);
                        Assert.Empty(xml.Root.XPathSelectElements($"Application/Icon/Language")
                            .Where(x => x.Value == titleVm.Model.Language.ToString()));
                    }
                }
                titleVm.Model.IsReplaceIcon = true;

                project.ProjectDirectory = Path.Combine(TestContext.TestDataDirPath, "icon_testdata");

                rp.Value = Path.GetFileName(iconFilePath);
                Assert.False(rp.HasErrors);
                Assert.False(titleVm.HasErrors.Value);
                Assert.False(vm.HasErrors.Value);
                Assert.True(titleVm.IconFilePath.OpenByExplorerCommand.CanExecute());

                rp.Value = Path.GetFileName(iconFilePath) + ".notfound";
                Assert.True(rp.HasErrors);
                Assert.True(titleVm.HasErrors.Value);
                Assert.True(vm.HasErrors.Value);
                Assert.False(titleVm.IconFilePath.OpenByExplorerCommand.CanExecute());

                // 環境変数を使用する場合はエラーチェックがスキップされる
                if (project.AppCapability.IsSupportEnvironmentVariableToPath)
                {
                    var expandEnvVarVm = (ParamVm)(titleVm.IconFilePath.Additional as CompositeParamVm)?.Params.FirstOrDefault();
                    var isExpandEnvVarsRp = (ReactiveProperty<bool>)expandEnvVarVm.Property;
                    isExpandEnvVarsRp.Value = true;
                    {
                        Assert.False(rp.HasErrors);
                        Assert.False(titleVm.HasErrors.Value);
                        Assert.False(vm.HasErrors.Value);
                    }
                    isExpandEnvVarsRp.Value = false;
                }

                var tempIconPath = Path.Combine(_context.TempDirPath, "tmp.bmp");

                rp.Value = tempIconPath;
                Assert.True(rp.HasErrors);
                Assert.False(titleVm.IconFilePath.OpenByExplorerCommand.CanExecute());

                File.Copy(iconFilePath, tempIconPath, true);

                rp.Value = tempIconPath;
                Assert.False(rp.HasErrors);
                Assert.True(titleVm.IconFilePath.OpenByExplorerCommand.CanExecute());
            }
        }

        [Fact]
        public void DefaultMetaTitle()
        {
            var project = Project.Import(_context.DiContainer,
                ImportableFileType.Meta,
                NintendoSdkHelper.ApplicationMetaFilePath);
            _context.DiContainer.GetInstance<App>().Project = project;
            using (var vm = _context.DiContainer.GetInstance<TitlePageVm>())
            {
                if (vm.TitleParam.Titles.Count > 0)
                {
                    var titleVm = vm.TitleParam.Titles[0];
                    Assert.NotNull(titleVm);
                    Assert.False(vm.HasErrors.Value);
                    Assert.False(titleVm.HasErrors.Value);

                    if (string.IsNullOrEmpty(project.Meta.Application.Titles[0].IconFilePath) == false)
                        Assert.True(titleVm.IconFilePath.IsUse.Value);
                }
            }
        }

        [Fact]
        public void IconFilePathOnReadOnly()
        {
            var appProfile = _context.DiContainer.GetInstance<AppProfile>();
            appProfile.AppMode = AppModeType.ApplicationNsp;

            var project = _context.DiContainer.GetInstance<Project>();
            project.DiContainer = _context.DiContainer;
            project.Meta.IsReadOnly = true;
            project.Meta.Application.Titles.Add(new Title()
            {
                DiContainer = _context.DiContainer,
                Name = "Name",
                Publisher = "Publisher",
                Language = LanguageType.AmericanEnglish,
                OriginalIconFilePath = Path.Combine(TestContext.TestDataDirPath, "icon_testdata/1024x1024x24.bmp")
            });

            using (var vm = _context.DiContainer.GetInstance<TitlePageVm>())
            {
                Assert.False(vm.TitleParam.AddCommand.CanExecute());

                var titleVm = vm.TitleParam.Titles.Last();
                Assert.False(titleVm.RemoveCommand.CanExecute());
                Assert.True(titleVm.IconFilePath.IsReadOnly.Value);
            }
        }

        [Fact]
        public void SelectableLanguages()
        {
            var appCapability = _context.DiContainer.GetInstance<ApplicationCapability>();

            using (var vm = _context.DiContainer.GetInstance<TitlePageVm>())
            {
                vm.TitleParam.AddCommand.Execute();

                var languageTypes = vm.TitleParam.Titles.First().Languages
                    .Where(x => !(x is AreaCaption))
                    .Cast<LanguageType>();

                if (appCapability.IsSupportEditChineseAndKoreanLanguages)
                {
                    Assert.Contains(LanguageType.TraditionalChinese, languageTypes);
                    Assert.Contains(LanguageType.SimplifiedChinese, languageTypes);
                    Assert.Contains(LanguageType.Korean, languageTypes);
                }
                else
                {
                    Assert.DoesNotContain(LanguageType.TraditionalChinese, languageTypes);
                    Assert.DoesNotContain(LanguageType.SimplifiedChinese, languageTypes);
                    Assert.DoesNotContain(LanguageType.Korean, languageTypes);
                }
            }
        }

        [Fact]
        public void ImageUnspecifiedNmetaWritable()
        {
            using (var tempDir = new DisposableDirectory())
            {
                var tempNmetaFile = tempDir.CreateFile("test.nmeta");

                using (var project = new Project
                {
                    DiContainer = _context.DiContainer,
                    Meta = new ApplicationMeta { DiContainer = _context.DiContainer }
                })
                using (var vm = new TitlePageVm(_context.DiContainer, project, _context.DiContainer.GetInstance<AppProfile>()))
                {
                    project.Meta.Application.DisplayVersion = "1.0.0";
                    project.Meta.Application.SupportedLanguages.First().IsSupported = true;

                    vm.TitleParam.AddCommand.Execute();

                    var titleVm = vm.TitleParam.Titles.Last();
                    Assert.True(titleVm.HasErrors.Value);
                    titleVm.Name.Property.Value = "test";
                    titleVm.Publisher.Property.Value = "publisher";
                    titleVm.IconFilePath.IsUse.Value = false;
                    Assert.False(titleVm.HasErrors.Value);
                    Assert.False(project.Meta.HasErrors);

                    project.OutputAppMetaXmlFileForAuthoringTool(tempNmetaFile);
                }

                using (var project = Project.Import(_context.DiContainer, ImportableFileType.Meta, tempNmetaFile))
                using (var vm = new TitlePageVm(_context.DiContainer, project, _context.DiContainer.GetInstance<AppProfile>()))
                {
                    var titleVm = vm.TitleParam.Titles.Last();
                    Assert.False(titleVm.HasErrors.Value);
                    Assert.False(project.Meta.HasErrors);
                }
            }
        }
    }
}
