﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Threading.Tasks;
using System.Xml.Linq;
using System.Xml.XPath;
using BezelEditor.Mvvm;
using Nintendo.Authoring.AuthoringEditor.Core;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Pages;
using Nintendo.Authoring.AuthoringEditor.MainWindow.ProjectEditPanel.Params;
using Reactive.Bindings;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Test.MainWindow.ProjectEditPanel.Pages
{
    public class BasicPageVmTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Fact]
        public void DefaultCtor()
        {
            using (var vm = _context.DiContainer.GetInstance<BasicPageVm>())
            {
                Assert.NotEmpty(vm.Params);
            }
        }

        [Fact]
        public void DisplayVersion()
        {
            using (var vm = _context.DiContainer.GetInstance<BasicPageVm>())
            {
                Assert.True(vm.HasErrors.Value);
                var rp = (ReactiveProperty<string>)vm.DisplayVersion.Property;
                Assert.True(rp.HasErrors);
                Assert.Empty(rp.Value);
                rp.Value = "1.0.0";
                Assert.Equal("1.0.0", _context.DiContainer.GetInstance<Project>().Meta.Application.DisplayVersion);
                Assert.False(vm.HasErrors.Value);
                Assert.False(rp.HasErrors);
            }
        }

        [Fact]
        public void ReleaseVersion()
        {
            using (var vm = _context.DiContainer.GetInstance<BasicPageVm>())
            {
                var rp = (ReactiveProperty<ushort>)vm.ReleaseVersion.Property;
                rp.Value = 1024;
                Assert.Equal(1024, _context.DiContainer.GetInstance<Project>().Meta.Application.ReleaseVersion);
                Assert.False(rp.HasErrors);
            }
        }

        public static object[][] PathReferenceTestData = new[]
        {
            new object[]
            {
                nameof(Application.HtmlDocumentPath),
                "html"
            },
            new object[]
            {
                nameof(Application.LegalInformationFilePath),
                "legalinfo.zip"
            },
            new object[]
            {
                nameof(Application.AccessibleUrlsFilePath),
                "urls.txt"
            },
            new object[]
            {
                nameof(Application.FilterDescriptionFilePath),
                "filter.fdf"
            }
        };

        [Theory]
        [MemberData(nameof(PathReferenceTestData))]
        public void ExternalPathReferenceTests(string propertyName, string fileName)
        {
            Func<BasicPageVm, ParamVm> getter =
                x => (ParamVm)x.GetType().GetProperty(propertyName).GetValue(x);

            Func<string> getTestFilePath = () => Path.Combine(_context.TempDirPath, fileName);
            Action createTestFile = () =>
            {
                if (string.IsNullOrEmpty(Path.GetExtension(fileName)))
                    _context.CreateFolder(fileName);
                else
                    _context.CreateFile(fileName);
            };

            Action<bool, ParamVm> assertCanOpenByExplorer = (expected, paramVm) =>
            {
                if (paramVm is FilePathStringParamVm)
                    Assert.Equal(expected, ((FilePathStringParamVm)paramVm).OpenByExplorerCommand.CanExecute());
                else if (paramVm is DirPathStringParamVm)
                    Assert.Equal(expected, ((DirPathStringParamVm)paramVm).OpenByExplorerCommand.CanExecute());
            };

            using (var vm = _context.DiContainer.GetInstance<BasicPageVm>())
            {
                ((ReactiveProperty<string>)vm.DisplayVersion.Property).Value = "1.0.0";
                Assert.False(vm.HasErrors.Value);

                var project = _context.DiContainer.GetInstance<Project>();
                project.DiContainer = _context.DiContainer;

                // パスが空の状態
                var paramVm = getter(vm);

                // フィルタ記述ファイルの指定をサポートしない環境では項目が非表示なのでテストもスキップ
                if (project.AppCapability.IsSupportFilterDescriptionFileInMeta == false &&
                    propertyName == nameof(Application.FilterDescriptionFilePath))
                {
                    Assert.Null(paramVm);
                    return;
                }

                var rp = (ReactiveProperty<string>)paramVm.Property;
                Assert.False(rp.HasErrors);

                // パスが空で使用するフラグは有効
                paramVm.IsUse.Value = true;
                Assert.True(rp.HasErrors);
                assertCanOpenByExplorer(false, paramVm);

                // 存在しないファイルパスを与えてエラー
                rp.Value = "not_found_file_path";
                Assert.True(rp.HasErrors);
                Assert.True(vm.HasErrors.Value);
                assertCanOpenByExplorer(false, paramVm);

                // 存在しない一時フォルダの絶対パスを与えてエラー
                var tempPath = getTestFilePath();
                rp.Value = tempPath;
                Assert.True(rp.HasErrors);
                Assert.True(vm.HasErrors.Value);
                assertCanOpenByExplorer(false, paramVm);

                // 一時フォルダの作成後、絶対パス指定
                createTestFile();
                rp.Value = tempPath; // 同じパスを与えても変更通知が来ることを確認
                Assert.False(rp.HasErrors);
                Assert.False(vm.HasErrors.Value);
                assertCanOpenByExplorer(true, paramVm);

                // 使用するフラグを無効化したときに項目が nmeta の XML に含まれないことを確認
                paramVm.IsUse.Value = false;
                Assert.False(rp.HasErrors);
                {
                    var xmlText = _context.DiContainer.GetInstance<App>().Project.MakeAppMetaXmlForAuthoringTool();
                    using (var reader = new StringReader(xmlText))
                    {
                        var xml = XDocument.Load(reader);
                        Assert.Empty(xml.Root.XPathSelectElements($"/Application/{propertyName}"));
                    }
                }

                paramVm.IsUse.Value = true;
                // プロジェクトフォルダの設定
                project.ProjectDirectory = _context.TempDirPath;

                // プロジェクトフォルダ設定後、相対パス指定
                rp.Value = Path.GetFileName(tempPath);
                Assert.False(rp.HasErrors);
                Assert.False(vm.HasErrors.Value);

                if (File.Exists(tempPath))
                    File.Delete(tempPath);
                else if (Directory.Exists(tempPath))
                    Directory.Delete(tempPath, true);

                // ファイル削除後に相対パス指定でエラー
                rp.Value = Path.GetFileName(tempPath);
                Assert.True(rp.HasErrors);
                Assert.True(vm.HasErrors.Value);

                // 環境変数を使用する、にチェックが入っていればエラーは無視される
                if (project.AppCapability.IsSupportEnvironmentVariableToPath)
                {
                    var expandEnvVarVm =
                        (ParamVm) (paramVm.Additional as CompositeParamVm)?.Params
                            .FirstOrDefault(x => x is CheckboxParamVm);

                    if (expandEnvVarVm != null)
                    {
                        var isExpandEnvVarsRp = (ReactiveProperty<bool>)expandEnvVarVm.Property;
                        isExpandEnvVarsRp.Value = true;
                        {
                            Assert.False(rp.HasErrors);
                            Assert.False(vm.HasErrors.Value);
                        }
                        isExpandEnvVarsRp.Value = false;
                    }
                }

                // 指定するのチェックが外れていれば VM のエラー状態は解除される
                paramVm.IsUse.Value = false;
                Assert.False(vm.HasErrors.Value);
                paramVm.IsUse.Value = true;
                Assert.True(vm.HasErrors.Value);

                // ファイル作成後に相対パス指定
                createTestFile();
                rp.Value = Path.GetFileName(tempPath);
                Assert.False(rp.HasErrors);
                Assert.False(vm.HasErrors.Value);
                assertCanOpenByExplorer(true, paramVm);
            }
        }

        [Fact]
        public void ApplicationId()
        {
            using (var vm = _context.DiContainer.GetInstance<BasicPageVm>())
            {
                vm.DisplayVersion.Property.Value = "1.0.0";

                var rp = (ReactiveProperty<ulong>)vm.ApplicationId.Property;
                rp.Value = ulong.MaxValue;
                Assert.True(rp.HasErrors);
                Assert.True(vm.HasErrors.Value);

                rp.Value = Constants.DefaultMetaProgramId;
                Assert.False(rp.HasErrors);
                Assert.False(vm.HasErrors.Value);
            }
        }

        private class StubResourceImporter : INspExtraResourceImporter
        {
            public Task<bool> ReadHtmlDocumentAsync()
            {
                return Task.FromResult(true);
            }

            public Task<bool> ReadLegalInformationAsync()
            {
                return Task.FromResult(true);
            }
        }

        [Fact]
        public async Task ExternalResourcesAvailable()
        {
            var profile = _context.DiContainer.GetInstance<AppProfile>();
            profile.AppMode = AppModeType.ApplicationNsp;

            var project = _context.DiContainer.GetInstance<Project>();
            var rcImporter = project.GetType().GetProperty(nameof(project.ExtraResourceImporter));
            Assert.NotNull(rcImporter);
            rcImporter.SetValue(project, new StubResourceImporter());

            project.Meta.Application.OriginalAccessibleUrlsFilePath =
                _context.CreateFile("original-accessible-urls.txt");
            project.Meta.Application.OriginalHtmlDocumentPath =
                _context.CreateFolder("html-document");
            project.Meta.Application.OriginalLegalInformationPath =
                _context.CreateFolder("invalid-legalinfo");

            using (var vm = _context.DiContainer.GetInstance<BasicPageVm>())
            {
                await AssertAdditinoalVmContent<FileOpenerVm>(project, vm.HtmlDocumentPath.Additional);
                await AssertAdditinoalVmContent<FileOpenerVm>(project, vm.AccessibleUrlsFilePath.Additional);
                await AssertAdditinoalVmContent<LegalInformationAddtionalVm>(project, vm.LegalInformationFilePath.Additional);
            }
        }

        private static async Task AssertAdditinoalVmContent<TAdditionalParamVm>(Project project, ViewModelBase additionalVm)
        {
            var paramsVm = additionalVm as CompositeParamVm;
            Assert.NotNull(paramsVm);
            var isSupportEnvVars = project.AppCapability.IsSupportEnvironmentVariableToPath;
            // 環境変数のサポートが入っている場合は AdditionalVm の要素 + Env 用の CheckboxVm で 2 つの VM になる
            var paramsCount = isSupportEnvVars ? 2 : 1;
            // ParamVm に規定の要素数が溜まるまで待機
            while (paramsVm.Params.Count != paramsCount)
            {
                await Task.Delay(TimeSpan.FromMilliseconds(100));
            }
            // 環境変数が利用できる場合、環境変数の使用の有無を設定するチェックボックスが存在する
            if (isSupportEnvVars)
            {
                Assert.NotNull(paramsVm.Params.FirstOrDefault(x => x is CheckboxParamVm));
            }
            Assert.NotNull(paramsVm.Params.FirstOrDefault(x => x is TAdditionalParamVm));
        }

        [Theory]
        [InlineData(AppModeType.ApplicationNsp, true)]
        [InlineData(AppModeType.ApplicationMeta, false)]
        public void FilterDescriptionFilePath(AppModeType mode, bool isNull)
        {
            var profile = _context.DiContainer.GetInstance<AppProfile>();
            profile.AppMode = mode;

            var project = _context.DiContainer.GetInstance<Project>();

            using (var vm = _context.DiContainer.GetInstance<BasicPageVm>())
            {
                // フィルタ記述ファイルの指定をサポートしない環境なら常に null
                if (project.AppCapability.IsSupportFilterDescriptionFileInMeta == false)
                {
                    Assert.Null(vm.FilterDescriptionFilePath);
                    return;
                }
                Assert.Equal(isNull, vm.FilterDescriptionFilePath == null);
            }
        }
    }
}
