﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using BezelEditor.Foundation;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Foundation.Test
{
    public class YamlHelperTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        public class Data
        {
            public int A { get; set; }
            public int B{ get; set; }
            public string C{ get; set; }
        }

        [Theory]
        [InlineData("data.yaml")]
        [InlineData("dir/data.yaml")]
        [InlineData("dir/dir/data.yaml")]
        public void Save(string filename)
        {
            var s = new Data
            {
                A = 111,
                B = 222,
                C = filename
            };

            var filePath = Path.Combine(_context.TempDirPath, filename).ToPathString();
            var i = YamlHelper.Save(filePath, s);
            Assert.True(i);
            Assert.True(File.Exists(filePath));
        }

        [Fact]
        public void Save_Error()
        {
            var s = new Data
            {
                A = 111,
                B = 222,
                C = "XYZ"
            };

            var filePath = "a:\\dummy".ToPathString();
            var i = YamlHelper.Save(filePath, s);
            Assert.False(i);
        }

        [Fact]
        public void SaveToString()
        {
            var s = new Data
            {
                A = 111,
                B = 222,
                C = "XYZ"
            };

            var i = YamlHelper.SaveToString(s);
            Assert.NotNull(i);
            Assert.NotEqual(string.Empty, i);
        }


        [Theory]
        [InlineData("data.yaml")]
        [InlineData("dir/data.yaml")]
        [InlineData("dir/dir/data.yaml")]
        public void Load(string filename)
        {
            var s = new Data
            {
                A = 111,
                B = 222,
                C = filename
            };

            var filePath = Path.Combine(_context.TempDirPath, filename).ToPathString();
            YamlHelper.Save(filePath, s);

            var d = YamlHelper.Load<Data>(filePath, null);

            Assert.NotNull(d);
            Assert.Equal(s.A, d.A);
            Assert.Equal(s.B, d.B);
            Assert.Equal(s.C, d.C);
        }

        [Fact]
        public void Load_FilePath_Error()
        {
            var d = YamlHelper.Load("XXX".ToPathString(), () => new Data {C = "ERROR"});

            Assert.NotNull(d);
            Assert.Equal("ERROR", d.C);
        }

        [Fact]
        public void Load_FileImage_Error()
        {
            var d = YamlHelper.Load(new byte[123], () => new Data {C = "ERROR"});

            Assert.NotNull(d);
            Assert.Equal("ERROR", d.C);
        }
    }
}
