﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Foundation.Test
{
    public class SerializeExtensionsTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Theory]
        [InlineData("0x0000000000000000", 0ul)]
        [InlineData("0x0000000000000001", 1ul)]
        [InlineData("0x00000000000000ff", 0xFFul)]
        public void UlongToHex(string expected, ulong actual)
        {
            Assert.Equal(expected, actual.ToHex());
        }

        [Theory]
        [InlineData("0x0000000000000000", 0ul)]
        [InlineData("0x0000000000000001", 1ul)]
        [InlineData("0x00000000000000FF", 0xFFul)]
        public void UlongToUpperHex(string expected, ulong actual)
        {
            Assert.Equal(expected, actual.ToUpperHex());
        }

        [Theory]
        [InlineData(0ul, "0x0")]
        [InlineData(1ul, "0x1")]
        [InlineData(0ul, "0x0000000000000000")]
        [InlineData(1ul, "0x0000000000000001")]
        [InlineData(0ul, "0")]
        [InlineData(0ul, "00")]
        [InlineData(0ul, "000")]
        [InlineData(0xABCul, "abc")]
        public void ToUlong(ulong expected, string actual)
        {
            Assert.Equal(expected, actual.ToUlong());
        }

        public static IEnumerable<object> ErrorData
        {
            get
            {
                yield return new object[] {"0x"};
                yield return new object[] {""};
                yield return new object[] {"0x"};
                yield return new object[] {"xyz"};
                yield return new object[] {"0xZZZ"};
            }
        }

        [Theory]
        [MemberData(nameof(ErrorData))]
        public void ToUlong_Error(string data)
        {
            Assert.Throws<InvalidDataException>(() => data.ToUlong());
        }

        [Theory]
        [MemberData(nameof(ErrorData))]
        public void IsUlong_Error(string data)
        {
            Assert.False(data.IsUlong());
        }

        public class Data
        {
            public int A { get; set; }
            public int B { get; set; }
            public string C { get; set; }
        }

        public class ParameterizedData
        {
            public int A { get; set; }

            public ParameterizedData(int a)
            {
                A = a;
            }
        }

        [Fact]
        public void DeepCloneByYamlSerializer_Basic()
        {
            var s = new Data
            {
                A = 111,
                B = 222,
                C = "XYZ"
            };

            var d = s.DeepCloneByYamlSerializer();

            Assert.NotNull(d);
            Assert.Equal(111, d.A);
            Assert.Equal(222, d.B);
            Assert.Equal("XYZ", d.C);
        }


        [Fact]
        public void DeepCloneByYamlSerializer_Null()
        {
            Data s = null;
            // ReSharper disable once ExpressionIsAlwaysNull
            var d = s.DeepCloneByYamlSerializer();

            Assert.Null(d);
        }
    }
}
