﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Threading;
using System.Threading.Tasks;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Foundation.Test
{
    public class AuthoringToolWrapperTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Theory]
        [InlineData("", "Authoring Tool for Nintendo SDK")]
        public void RunBasic(string args, string containsMessage)
        {
            using (var job = AuthoringToolWrapper.Run(args))
            {
                job.WaitForExit();
                Assert.True(job.ExitCode == 1);
                Assert.Contains(containsMessage,
                    string.Join(Environment.NewLine, job.StandardOutput));
            }
        }

        [Theory]
        [InlineData("", "Authoring Tool for Nintendo SDK")]
        public async Task RunAsyncBasic(string args, string containsMessage)
        {
            var cts = new CancellationTokenSource();
            using (var job = AuthoringToolWrapper.Run(args))
            {
                var ok = await job.WaitForExitAsync(x => true, cts.Token).ConfigureAwait(false);
                Assert.Equal(true, ok);
                Assert.Equal(false, cts.IsCancellationRequested);
                Assert.True(job.ExitCode == 1);
                Assert.Contains(containsMessage,
                    string.Join(Environment.NewLine, job.StandardOutput));
            }
        }

        [Fact]
        public async Task ProcessCancellable()
        {
            var cts = new CancellationTokenSource();
            using (var job = AuthoringToolWrapper.Create(""))
            {
                var t = job.WaitForExitAsync(x => true, cts.Token);
                cts.Cancel();
                await Assert.ThrowsAsync<TaskCanceledException>(async () => await t.ConfigureAwait(false)).ConfigureAwait(false);
                Assert.Equal(true, cts.IsCancellationRequested);
            }
        }

        [Theory]
        [InlineData("", -1)]
        [InlineData("showversion", 100)]
        public async Task ProgressCallback(string args, int expectedProgress)
        {
            var progress = -1;
            var cts = new CancellationTokenSource();
            using (var job = AuthoringToolWrapper.Run(args, p => progress = p))
            {
                await job.WaitForExitAsync(x => true, cts.Token).ConfigureAwait(false);
            }
            // プロセスが成功 (ExitCode == 0) でなければ進捗のコールバックは 100% にならない
            Assert.Equal(expectedProgress, progress);
        }

        [Fact]
        public void NxAddonVersion()
        {
            var version = AuthoringToolWrapper.NxAddonVersion();
            Assert.NotEmpty(version);
        }
    }
}
