﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Runtime.Remoting;
using System.Threading.Tasks;
using System.Xml.Linq;
using System.Xml.XPath;
using BezelEditor.Foundation.Extentions;
using Nintendo.Authoring.AuthoringEditor.Core.Test.Fixtures;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test
{
    public class NspPatchTest : IDisposable, IClassFixture<BuildTestAppNspFixture>
    {
        private readonly TestContext _context = new TestContext();

        private bool IsSupportMakingPatch => _context.GetInstance<Project>().AppCapability.IsSupportMakingPatch;

        public void Dispose()
        {
            _context.Dispose();
        }

        private readonly BuildTestAppNspFixture _fixture;

        public NspPatchTest(BuildTestAppNspFixture fixture)
        {
            _fixture = fixture.Initialize();
        }

        [Fact]
        public async Task Create_Basic()
        {
            if (IsSupportMakingPatch == false)
                return;

            var input = new NspBuilder.MakePatchOption
            {
                OutputPath = Path.Combine(_context.TempDirPath, "output.patch.nsp"),
                CurrentNspFilePath = _fixture.RevisedV1NspFilePath,
                OriginalNspFilePath = _fixture.NspFilePath,
            };

            var i = await new NspBuilder().MakePatchNspAsync(input).ConfigureAwait(false);

            Assert.Equal(NspHandleResult.Ok, i.Result);

            // パッチの作成元 Applicationid の情報が AppMeta と同様に Meta.Core.ApplicationId に格納されていることを確認
            using (var tempDir = new DisposableDirectory())
            {
                var defaultMetaXml = XDocument.Load(NintendoSdkHelper.ApplicationMetaFilePath);
                var defaultAppId = defaultMetaXml.Root.XPathSelectElement("Core/ApplicationId").Value;

                var option = new NspImporter.Option
                {
                    DiContainer = _context.DiContainer,
                    InputNspFile = new NspFile(input.OutputPath),
                    TempNspExtractFilePath = tempDir.RootPath
                };

                {
                    var r = await new NspImporter().ToMetaAsync(option).ConfigureAwait(false);
                    Assert.Equal(NspHandleResult.NeedOriginalApplicationNspError, r.Result);
                }
                {
                    option.InputNspFile = new NspFile(input.OutputPath)
                    {
                        OriginalNspFile = new NspFile(_fixture.NspFilePath)
                    };
                    var r = await new NspImporter().ToMetaAsync(option).ConfigureAwait(false);
                    Assert.Equal(NspHandleResult.Ok, r.Result);
                    Assert.Equal(defaultAppId, r.ApplicationMeta.Core.ApplicationIdHex);
                }
            }
        }

        [Fact]
        public async Task Create_FromPrevious()
        {
            if (IsSupportMakingPatch == false)
                return;

            var v1PatchNsp = Path.Combine(_context.TempDirPath, "output.patch.v1.nsp");
            var v2PatchNsp = Path.Combine(_context.TempDirPath, "output.patch.v2.nsp");

            var input = new NspBuilder.MakePatchOption
            {
                OutputPath = v1PatchNsp,
            };

            // パッチ作成
            {
                input.OriginalNspFilePath = _fixture.NspFilePath;
                input.CurrentNspFilePath = _fixture.RevisedV1NspFilePath;
                var i = await new NspBuilder().MakePatchNspAsync(input).ConfigureAwait(false);
                Assert.Equal(NspHandleResult.Ok, i.Result);
            }

            // v0 のアプリにパッチは指定できない
            {
                input.OriginalNspFilePath = _fixture.NspFilePath;
                input.CurrentNspFilePath = input.OutputPath;
                input.OutputPath = v2PatchNsp;
                var i = await new NspBuilder().MakePatchNspAsync(input).ConfigureAwait(false);
                Assert.Equal(NspHandleResult.NspIsNotApplicationError, i.Result);
            }

            // 前バージョンのパッチにアプリは指定できない
            {
                input.OriginalNspFilePath = _fixture.NspFilePath;
                input.CurrentNspFilePath = _fixture.RevisedV1NspFilePath;
                input.PreviousNspFilePath = _fixture.RevisedV1NspFilePath;
                var i = await new NspBuilder().MakePatchNspAsync(input).ConfigureAwait(false);
                Assert.Equal(NspHandleResult.NspIsNotPatchError, i.Result);
            }

            // 前回バージョンのパッチ (v1) から新しいバージョンのパッチ (v2) を作る
            {
                var replacer = _context.DiContainer.GetInstance<NspReplacer>();
                var project = Project.Import(_context.DiContainer, ImportableFileType.Nsp, _fixture.RevisedV1NspFilePath);
                project.Meta.Application.ReleaseVersion = 2;
                var option = new NspReplacer.ReplaceOption
                {
                    Project = project,
                    InputNspPath = _fixture.RevisedV1NspFilePath,
                    OutputNspPath = Path.Combine(_context.TempDirPath, "output.v2.nsp")
                };
                {
                    var i = await replacer.ReplaceNspAsync(option).ConfigureAwait(false);
                    Assert.Equal(NspHandleResult.Ok, i.Result);
                }

                input.OriginalNspFilePath = _fixture.NspFilePath;
                input.CurrentNspFilePath = option.OutputNspPath;
                input.PreviousNspFilePath = v1PatchNsp;
                input.OutputPath = v2PatchNsp;
                {
                    var i = await new NspBuilder().MakePatchNspAsync(input).ConfigureAwait(false);
                    Assert.Equal(NspHandleResult.Ok, i.Result);
                }
            }
        }

        [Fact]
        public async Task ErrorHandling()
        {
            if (IsSupportMakingPatch == false)
                return;

            var input = new NspBuilder.MakePatchOption
            {
                OutputPath = Path.Combine(_context.TempDirPath, "output.patch.error.nsp"),
            };

            {
                var i = await new NspBuilder().MakePatchNspAsync(input).ConfigureAwait(false);
                Assert.Equal(NspHandleResult.NotFoundNspFile, i.Result);
            }

            {
                input.OriginalNspFilePath = _fixture.NspFilePath;
                var i = await new NspBuilder().MakePatchNspAsync(input).ConfigureAwait(false);
                Assert.Equal(NspHandleResult.NotFoundNspFile, i.Result);
            }

            {
                input.OriginalNspFilePath = _fixture.NspFilePath;
                input.CurrentNspFilePath = _fixture.NspFilePath;
                var i = await new NspBuilder().MakePatchNspAsync(input).ConfigureAwait(false);
                Assert.Equal(NspHandleResult.NspIsNotRevisedVersionError, i.Result);
            }

            {
                input.OriginalNspFilePath = _fixture.RevisedV1NspFilePath;
                input.CurrentNspFilePath = _fixture.NspFilePath;
                var i = await new NspBuilder().MakePatchNspAsync(input).ConfigureAwait(false);
                Assert.Equal(NspHandleResult.NspIsNotOriginalVersionError, i.Result);
            }
        }

        [Fact]
        public async Task DiffPatchFiles()
        {
            var v1Nsp = Path.Combine(_context.TempDirPath, "DiffPatchFiles.v1.nsp");
            var v2Nsp = Path.Combine(_context.TempDirPath, "DiffPatchFiles.v2.nsp");
            var project = Project.Import(_context.DiContainer,
                ImportableFileType.Meta,
                TestContext.DefaultMetaFilePath);

            var v1Data = _context.CreateFolder("data-v1");
            {
                Directory.CreateDirectory(Path.Combine(v1Data, "test"));
                File.WriteAllText(Path.Combine(v1Data, "v0.txt"), "version 0");
                File.WriteAllText(Path.Combine(v1Data, "v1.txt"), "version 1");
                File.WriteAllText(Path.Combine(v1Data, "test/add.txt"), "add from version 1");

                project.Meta.Application.ReleaseVersion = 1;

                await CreateAppNsp(v1Nsp, project.Meta, v1Data);
                Assert.True(File.Exists(v1Nsp));
            }

            var v2Data = _context.CreateFolder("data-v2");
            {
                Directory.CreateDirectory(Path.Combine(v2Data, "test"));
                File.WriteAllText(Path.Combine(v2Data, "v1.txt"), "version 1");
                File.WriteAllText(Path.Combine(v2Data, "v2.txt"), "version 2");
                File.WriteAllText(Path.Combine(v2Data, "test/add.txt"), "add from version 1\r\nadd from version 2");

                project.Meta.Application.ReleaseVersion = 2;

                await CreateAppNsp(v2Nsp, project.Meta, v2Data);
                Assert.True(File.Exists(v2Nsp));
            }

            var v1PatchNsp = Path.Combine(_context.TempDirPath, "DiffPatchFiles.patch.v1.nsp");
            {
                var r = await new NspBuilder().MakePatchNspAsync(new NspBuilder.MakePatchOption
                {
                    OutputPath = v1PatchNsp,
                    OriginalNspFilePath = _fixture.NspFilePath,
                    CurrentNspFilePath = v1Nsp,
                });
                Assert.Equal(NspHandleResult.Ok, r.Result);
                Assert.True(File.Exists(v1PatchNsp));
            }

            var v2PatchNsp = Path.Combine(_context.TempDirPath, "DiffPatchFiles.patch.v2.nsp");
            {
                var r = await new NspBuilder().MakePatchNspAsync(new NspBuilder.MakePatchOption
                {
                    OutputPath = v2PatchNsp,
                    OriginalNspFilePath = _fixture.NspFilePath,
                    CurrentNspFilePath = v2Nsp,
                    PreviousNspFilePath = v1PatchNsp
                });
                Assert.Equal(NspHandleResult.Ok, r.Result);
                Assert.True(File.Exists(v2PatchNsp));
            }

            // パッチ間差分によって得られたファイル
            {
                var patchNspFile = new NspFile(v1PatchNsp) { OriginalNspFile = new NspFile(_fixture.NspFilePath) };
                var diffPatchFiles = patchNspFile.Files(new NspFileEnumerateParameter
                {
                    DiffPatchNspFilePath = v2PatchNsp,
                    EnumerationType = NspFileEnumeration.DiffPatch
                });

                var diffItems = diffPatchFiles.Select(x =>
                {
                    // パスの先頭部分 (hoge.nca/) を消す
                    var parts = x.FilePath.Split('/').ToList();
                    parts.RemoveFirst();
                    return new { Path = string.Join("/", parts), Type = x.ModifiedType };
                }).ToArray();
                Assert.Contains(new { Path = "fs1/v0.txt", Type = NspFileModifiedType.Removed }, diffItems);
                Assert.Contains(new { Path = "fs1/v2.txt", Type = NspFileModifiedType.Added }, diffItems);
                Assert.Contains(new { Path = "fs1/test/add.txt", Type = NspFileModifiedType.Changed }, diffItems);
            }
        }

        private async Task CreateAppNsp(string outputPath, ApplicationMeta meta, string dataDir)
        {
            // Core の変更を反映するには MakeMeta で .npdm を変更する必要あり
            meta.Core.FsAccessControlData.SaveDataOwnerIds.Add(new SaveDataOwnerId
            {
                Accessibility = AccessibilityType.Read,
                ApplicationId = meta.Core.ApplicationId + 1ul
            });
            var builder = BuildTestAppNspFixture.AppBuilder;
            var output = await builder.BuildNspAsync(outputPath, builder.CodeDirPath, dataDir, meta).ConfigureAwait(false);
            Assert.Equal(NspHandleResult.Ok, output.Result);
        }
    }
}
