﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Xml.Linq;
using System.Xml.Serialization;
using Nintendo.Authoring.AuthoringEditor.Core.Test.Fixtures;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test
{
    public class NspImporterTest : IDisposable, IClassFixture<BuildTestAppNspFixture>
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        private readonly BuildTestAppNspFixture _fixture;

        public NspImporterTest(BuildTestAppNspFixture fixture)
        {
            _fixture = fixture.Initialize();
        }

        [Fact]
        public void DefaultCtor()
        {
            // ReSharper disable once HeapView.ObjectAllocation.Evident
            // ReSharper disable once ObjectCreationAsStatement
            new NspImporter();

            var option = new NspImporter.Option();
            Assert.Null(option.InputNspFile);
            Assert.Null(option.TempNspExtractFilePath);

            var result = new NspImporter.ResultType();
            Assert.Equal(NspHandleResult.Ok, result.Result);
            Assert.NotNull(result.ErrorMessages);
            Assert.Equal(0, result.ErrorMessages.Length);
            Assert.Null(result.ApplicationMeta);
        }

        [Fact]
        public async Task ToMeta_TemporaryExtractDirIsEmpty()
        {
            var options = new[]
            {
                new NspImporter.Option
                {
                    InputNspFile = new NspFile(_fixture.NspFilePath)
                },
                new NspImporter.Option
                {
                    InputNspFile = new NspFile(_fixture.NspFilePath),
                    TempNspExtractFilePath = "notfound",
                },
            };
            foreach (var option in options)
            {
                var importer = new NspImporter();
                var output = await importer.ToMetaAsync(option).ConfigureAwait(false);
                Assert.Equal(NspHandleResult.NotFoundTemporaryNspExtractDir, output.Result);
            }
        }

        [Fact]
        public async Task ToMeta_Basic()
        {
            var option = new NspImporter.Option
            {
                DiContainer = _context.DiContainer,
                InputNspFile = new NspFile(_fixture.NspFilePath),
                TempNspExtractFilePath = _context.CreateFolder(Path.GetRandomFileName())
            };
            var importer = new NspImporter();
            var output = await importer.ToMetaAsync(option).ConfigureAwait(false);

            Assert.NotNull(output);
            Assert.Equal(NspHandleResult.Ok, output.Result);

            Assert.NotNull(output.ApplicationContentMeta);
            Assert.NotNull(output.ProgramInfo);

            {
                var xmlText = _context.MakeAppMetaXml(output.ApplicationMeta);
                var meta = (ApplicationMeta)new XmlSerializer(typeof(ApplicationMeta), new XmlRootAttribute
                    {
                        ElementName = TypeHelper.DetectRootMetaKindFromXmlText(xmlText).ToString()
                    })
                    .Deserialize(new StringReader(xmlText));
                var expectedMeta = XDocument.Load(TestContext.DefaultMetaFilePath);

                var expectedAppId = expectedMeta.Descendants("ApplicationId").First().Value;
                Assert.Equal(expectedAppId, meta.Core.ApplicationIdHex);

                var expectedName = expectedMeta.Descendants("Name").First().Value;
                Assert.Equal(expectedName, meta.Core.Name);

                Assert.Equal(2, meta.CardSpec.Size);
                Assert.Equal(25, meta.CardSpec.ClockRate);
            }
        }

        [Fact]
        public async Task ToMeta_InputNspFilePath()
        {
            var option = new NspImporter.Option
            {
                DiContainer = _context.DiContainer,
                InputNspFile = new NspFile("abc"),
                TempNspExtractFilePath = _context.CreateFolder(Path.GetRandomFileName())
            };

            var importer = new NspImporter();
            var output = await importer.ToMetaAsync(option).ConfigureAwait(false);
            Assert.NotNull(output);
            Assert.Null(output.ApplicationMeta);

            Assert.Equal(NspHandleResult.NotFoundNspFile, output.Result);
        }

        [Fact]
        public async Task DetectContentMetaTypeAsync_BrokenNsp()
        {
            var brokenNspFile = new NspFile(
                Path.Combine(TestContext.TestDataDirPath, @"project_testdata\broken.nsp"));

            Assert.Equal(ContentMetaType.Unknown, await NspImporter.DetectContentMetaTypeAsync(brokenNspFile).ConfigureAwait(false));
        }

        [Fact]
        public async Task DetectContentMetaTypeAsync_NotFoundNsp()
        {
            var notFoundNspFile = new NspFile("zz:/zzz.yyy");
            Assert.Equal(ContentMetaType.Unknown, await NspImporter.DetectContentMetaTypeAsync(notFoundNspFile).ConfigureAwait(false));
        }

        [Fact]
        public async Task ProgramInfo_FsAccessControlData()
        {
            if (_context.GetInstance<Project>().AppCapability.IsSupportSaveDataOwnerIds == false)
                return;

            var option = new NspImporter.Option
            {
                DiContainer = _context.DiContainer,
                InputNspFile = new NspFile(_fixture.RevisedV1NspFilePath),
                TempNspExtractFilePath = _context.CreateFolder(Path.GetRandomFileName())
            };
            var importer = new NspImporter();
            var output = await importer.ToMetaAsync(option).ConfigureAwait(false);
            Assert.Equal(NspHandleResult.Ok, output.Result);
            Assert.Equal(1, output.ProgramInfo.FsAccessControlData.SaveDataOwnerIds.Count);

            var owner = output.ProgramInfo.FsAccessControlData.SaveDataOwnerIds[0];
            Assert.Equal(output.ApplicationMeta.Core.ApplicationId + 1ul, owner.ApplicationId);
            Assert.Equal(AccessibilityType.Read, owner.Accessibility);
        }

        [Fact]
        public async Task ToMetaFromDirectory_Basic()
        {
            // TODO: AoC/Patch の場合のテストを書く
            using (var temp = new DisposableDirectory())
            {
                await new NspFile(_fixture.NspFilePath).ExtractAllAsync(temp.RootPath, null, CancellationToken.None).ConfigureAwait(false);
                var importer = new NspImporter();
                var r = await importer.ToMetaAsync(new NspImporter.Option
                {
                    DiContainer = _context.DiContainer,
                    InputNspFile = new NspExtractedDirectory(temp.RootPath),
                    TempNspExtractFilePath = _context.CreateFolder(Path.GetRandomFileName())
                }).ConfigureAwait(false);
                Assert.Equal(NspHandleResult.Ok, r.Result);
                Assert.NotEmpty(r.ApplicationMeta.Application.Titles);
            }
        }
    }
}
