﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Authoring.AuthoringEditor.Core.Test.Fixtures;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test
{
    public class NspBuilderTest : IDisposable, IClassFixture<BuildTestAppNspFixture>
    {
        private readonly TestContext _context = new TestContext();

        private static string MetaFilePath =>
            NintendoSdkHelper.ApplicationMetaFilePath;

        private static string DescFilePath =>
            NintendoSdkHelper.ApplicationDescFilePath;

        public void Dispose()
        {
            _context.Dispose();
        }

        private readonly BuildTestAppNspFixture _fixture;

        public NspBuilderTest(BuildTestAppNspFixture fixture)
        {
            _fixture = fixture.Initialize();
        }

        public static IEnumerable<object> AllOutputTypes =>
            Enum.GetValues(typeof(OutputTypes)).Cast<object>().Select(x => new[] {x});

        [Fact]
        public void DefaultCtor()
        {
            // ReSharper disable once ObjectCreationAsStatement
            new NspBuilder();

            var result = new NspHandleResultType();
            Assert.Equal(NspHandleResult.Ok, result.Result);
            Assert.NotNull(result.ErrorMessages);
            Assert.Equal(0, result.ErrorMessages.Length);

            var option = new NspBuilder.ExportOption();
            Assert.Null(option.OutputPath);
            Assert.Null(option.InputMetaFilePath);
            Assert.Null(option.InputProgramCodePath);
            Assert.Null(option.InputProgramDataPath);
            Assert.Equal(NintendoSdkHelper.ApplicationDescFilePath, option.InputDescFilePath);
            Assert.Equal(NspTypes.Application, option.NspType);
            Assert.Null(option.ProgressChanged);
        }

        [Theory]
        [MemberData(nameof(AllOutputTypes))]
        public async Task CreateNsp_Basic(OutputTypes type)
        {
            var progress = -1;

            var input = new NspBuilder.ExportOption
            {
                OutputPath = Path.Combine(_context.TempDirPath, $"test.{type.ToString().ToLower()}"),
                OutputType = type,
                InputMetaFilePath = TestContext.DefaultMetaFilePath,
                InputDescFilePath = TestContext.DefaultDescFilePath,
                InputProgramCodePath = _fixture.CodeDirPath,
                InputProgramDataPath = _fixture.DataDirPath,
                ProgressChanged = p => progress = p
            };

            var i = await new NspBuilder().ExportNspAsync(input).ConfigureAwait(false);

            Assert.Equal(NspHandleResult.Ok, i.Result);
            Assert.Equal(100, progress);

            switch (type)
            {
                case OutputTypes.Nsp:
                    Assert.True(File.Exists(input.OutputPath));
                    break;

                default:
                    throw new ArgumentOutOfRangeException(nameof(type), type, null);
            }
        }

        [Theory]
        [MemberData(nameof(AllOutputTypes))]
        public async Task CreateNsp_Cancel(OutputTypes type)
        {
            var cts = new CancellationTokenSource();
            var input = new NspBuilder.ExportOption
            {
                OutputPath = Path.Combine(_context.TempDirPath, $"test.{type.ToString().ToLower()}"),
                OutputType = type,
                InputMetaFilePath = MetaFilePath,
                InputDescFilePath = DescFilePath,
                InputProgramCodePath = _fixture.CodeDirPath,
                InputProgramDataPath = _fixture.DataDirPath,
                CancellationToken = cts.Token
            };

            var builder = new NspBuilder();
            var exportTask = builder.ExportNspAsync(input);

            cts.Cancel();
            var i = await exportTask.ConfigureAwait(false);

            Assert.Equal(NspHandleResult.Canceled, i.Result);
        }

        [Theory]
        [MemberData(nameof(AllOutputTypes))]
        public async Task CreateNsp_Icon(OutputTypes type)
        {
            var tempMetaPath = Path.Combine(_context.TempDirPath, Path.GetRandomFileName());
            using (var meta = _context.LoadAppMeta(File.ReadAllText(MetaFilePath, Encoding.UTF8)))
            {
                AddOrUpdateTitle(meta, new Title
                {
                    Name = "Test",
                    Language = LanguageType.AmericanEnglish,
                    IconFilePath = Path.Combine(TestContext.TestDataDirPath, "icon_testdata/1024x1024x24.bmp"),
                });
                _context.OutputAppMetaXml(meta, tempMetaPath);
            }

            var input = new NspBuilder.ExportOption
            {
                OutputPath = Path.Combine(_context.TempDirPath, $"test.{type.ToString().ToLower()}"),
                OutputType = type,
                InputMetaFilePath = tempMetaPath,
                InputDescFilePath = DescFilePath,
                InputProgramCodePath = _fixture.CodeDirPath,
                InputProgramDataPath = _fixture.DataDirPath
            };
            var i = await new NspBuilder().ExportNspAsync(input).ConfigureAwait(false);

            Assert.Equal(NspHandleResult.Ok, i.Result);

            switch (type)
            {
                case OutputTypes.Nsp:
                    Assert.True(File.Exists(input.OutputPath));
                    break;

                default:
                    throw new ArgumentOutOfRangeException(nameof(type), type, null);
            }
        }

        [Theory]
        [MemberData(nameof(AllOutputTypes))]
        public async Task CreateNsp_IconAndNxIcon(OutputTypes type)
        {
            var tempMetaPath = Path.Combine(_context.TempDirPath, Path.GetRandomFileName());
            using (var meta = _context.LoadAppMeta(File.ReadAllText(MetaFilePath, Encoding.UTF8)))
            {
                AddOrUpdateTitle(meta, new Title
                {
                    Name = "Test",
                    Language = LanguageType.AmericanEnglish,
                    IconFilePath = Path.Combine(TestContext.TestDataDirPath, "icon_testdata/1024x1024x24.bmp"),
                    NxIconFilePath = Path.Combine(TestContext.TestDataDirPath, "nxicon_testdata/256x256.jpg")
                });
                _context.OutputAppMetaXml(meta, tempMetaPath);
            }

            var input = new NspBuilder.ExportOption
            {
                OutputPath = Path.Combine(_context.TempDirPath, $"test.{type.ToString().ToLower()}"),
                OutputType = type,
                InputMetaFilePath = tempMetaPath,
                InputDescFilePath = DescFilePath,
                InputProgramCodePath = _fixture.CodeDirPath,
                InputProgramDataPath = _fixture.DataDirPath
            };

            var i = await new NspBuilder().ExportNspAsync(input).ConfigureAwait(false);

            Assert.Equal(NspHandleResult.Ok, i.Result);

            switch (type)
            {
                case OutputTypes.Nsp:
                    Assert.True(File.Exists(input.OutputPath));
                    break;

                default:
                    throw new ArgumentOutOfRangeException(nameof(type), type, null);
            }
        }

        [Theory]
        [MemberData(nameof(AllOutputTypes))]
        public async Task CreateNsp_NoInputProgramDataPath(OutputTypes type)
        {
            var input = new NspBuilder.ExportOption
            {
                OutputPath = Path.Combine(_context.TempDirPath, $"test.{type.ToString().ToLower()}"),
                OutputType = type,
                InputMetaFilePath = MetaFilePath,
                InputDescFilePath = DescFilePath,
                InputProgramCodePath = _fixture.CodeDirPath,
                InputProgramDataPath = null
            };

            var i = await new NspBuilder().ExportNspAsync(input).ConfigureAwait(false);

            Assert.Equal(NspHandleResult.Ok, i.Result);

            switch (type)
            {
                case OutputTypes.Nsp:
                    Assert.True(File.Exists(input.OutputPath));
                    break;

                default:
                    throw new ArgumentOutOfRangeException(nameof(type), type, null);
            }
        }

        [Theory]
        [MemberData(nameof(AllOutputTypes))]
        public async Task CreateNsp_Error_InputMetaFilePath(OutputTypes type)
        {
            var input = new NspBuilder.ExportOption
            {
                OutputPath = Path.Combine(_context.TempDirPath, $"test.{type.ToString().ToLower()}"),
                OutputType = type,
                InputDescFilePath = DescFilePath,
                InputMetaFilePath = null,
                InputProgramCodePath = _fixture.CodeDirPath,
                InputProgramDataPath = _fixture.DataDirPath
            };

            var i = await new NspBuilder().ExportNspAsync(input).ConfigureAwait(false);

            Assert.Equal(NspHandleResult.NotFoundMetaFile, i.Result);
        }

        [Theory]
        [MemberData(nameof(AllOutputTypes))]
        public async Task CreateNsp_Error_InputProgramCodePath(OutputTypes type)
        {
            var input = new NspBuilder.ExportOption
            {
                OutputPath = Path.Combine(_context.TempDirPath, $"test.{type.ToString().ToLower()}"),
                OutputType = type,
                InputMetaFilePath = MetaFilePath,
                InputDescFilePath = DescFilePath,
                InputProgramCodePath = null,
                InputProgramDataPath = _fixture.DataDirPath
            };

            var i = await new NspBuilder().ExportNspAsync(input).ConfigureAwait(false);

            Assert.Equal(NspHandleResult.NotFoundProgramCodeFile, i.Result);
        }

        [Theory]
        [MemberData(nameof(AllOutputTypes))]
        public async Task CreateNsp_Error_InputProgramDataPath(OutputTypes type)
        {
            var input = new NspBuilder.ExportOption
            {
                OutputPath = Path.Combine(_context.TempDirPath, $"test.{type.ToString().ToLower()}"),
                OutputType = type,
                InputMetaFilePath = MetaFilePath,
                InputDescFilePath = DescFilePath,
                InputProgramCodePath = _fixture.CodeDirPath,
                InputProgramDataPath = "AAAA"
            };

            var i = await new NspBuilder().ExportNspAsync(input).ConfigureAwait(false);

            Assert.Equal(NspHandleResult.NotFoundProgramDataFile, i.Result);
        }


        [Theory]
        [MemberData(nameof(AllOutputTypes))]
        public async Task CreateNsp_Error_InputDescFilePath(OutputTypes type)
        {
            var input = new NspBuilder.ExportOption
            {
                OutputPath = Path.Combine(_context.TempDirPath, $"test.{type.ToString().ToLower()}"),
                OutputType = type,
                InputMetaFilePath = MetaFilePath,
                InputDescFilePath = null,
                InputProgramCodePath = _fixture.CodeDirPath,
                InputProgramDataPath = _fixture.DataDirPath
            };

            var i = await new NspBuilder().ExportNspAsync(input).ConfigureAwait(false);

            Assert.Equal(NspHandleResult.NotFoundDescFile, i.Result);
        }

        public static object[][] AllOutputTypesWithExternalResource =
            Enum.GetValues(typeof(OutputTypes)).Cast<object>().Select(x => new []
            {
                x, new [] {
                    nameof(Application.HtmlDocumentPath),
                    nameof(Application.LegalInformationFilePath),
                    nameof(Application.AccessibleUrlsFilePath)
                }
            }).ToArray();

        [Theory]
        [MemberData(nameof(AllOutputTypesWithExternalResource))]
        public async Task CreateNsp_Error_ExternalResources(OutputTypes type, string[] propertyNames)
        {
            foreach (var propertyName in propertyNames)
            {
                var tempMetaPath = Path.Combine(_context.TempDirPath, Path.GetRandomFileName());
                using (var meta = _context.LoadAppMeta(File.ReadAllText(MetaFilePath, Encoding.UTF8)))
                {
                    meta.Application.GetType().GetProperty(propertyName).SetValue(meta.Application, new ExpandablePath("AAAA"));
                    meta.Application.GetType().GetProperty($"IsReplace{propertyName}").SetValue(meta.Application, true);
                    _context.OutputAppMetaXml(meta, tempMetaPath);
                }

                var input = new NspBuilder.ExportOption
                {
                    OutputPath = Path.Combine(_context.TempDirPath, $"test.{type.ToString().ToLower()}"),
                    OutputType = type,
                    InputMetaFilePath = tempMetaPath,
                    InputDescFilePath = DescFilePath,
                    InputProgramCodePath = _fixture.CodeDirPath
                };

                var i = await new NspBuilder().ExportNspAsync(input).ConfigureAwait(false);

                Assert.Equal(NspHandleResult.Error, i.Result);
            }
        }

        [Fact]
        public async Task MakeAocNsp()
        {
            using (var tempDir = new DisposableDirectory())
            {
                tempDir.CreateFolder("aoc");
                tempDir.CreateFile(@"aoc\empty.txt");

                var project = _context.GetInstance<Project>();
                project.DiContainer = _context.DiContainer;
                project.ProjectDirectory = tempDir.RootPath;
                project.AocMeta.AddContent();

                var contents = project.AocMeta.Contents[0];
                contents.Tag = "aoc-tag";
                contents.DataPath.Path = "aoc";

                var builder = new NspBuilder();
                var option = new NspBuilder.MakeAocOption
                {
                    Project = project,
                    OutputPath = Path.Combine(_context.TempDirPath, "aoc.nsp")
                };
                var r = await builder.MakeAocNspAsync(option).ConfigureAwait(false);
                Assert.Equal(NspHandleResult.Ok, r.Result);
                Assert.True(File.Exists(option.OutputPath));
            }
        }

        private void AddOrUpdateTitle(ApplicationMeta appMeta, Title title)
        {
            var t = appMeta.Application.Titles.Where(x => x.Language == title.Language).Select((x, i) => i);
            if (t.Any())
            {
                var i = t.First();
                appMeta.Application.Titles[i] = title;
            }
            else
            {
                appMeta.Application.Titles.Add(title);
            }
        }
    }
}

