﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using BezelEditor.Foundation;
using BezelEditor.Foundation.Utilities;
using Nintendo.Authoring.AuthoringEditor.Foundation;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test.Fixtures
{
    public class TestApplicationBuildErrorException : ApplicationException
    {
        public TestApplicationBuildErrorException(string message) : base(message)
        {
        }

        public TestApplicationBuildErrorException(string message, Exception innerException)
            : base(message, innerException)
        {

        }
    }

    public class TestAppBuilder : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public string TempDirPath
        {
            get { return _context.TempDirPath; }
            set { _context.TempDirPath = value; }
        }

        public string CodeDirPath =>
            Path.Combine(TempDirPath, @"TestApp\NX64\Debug\TestApp.nspd\program0.ncd\code");

        public string DataDirPath =>
            Path.Combine(TempDirPath, @"TestApp\Data");

        public string NspFilePath =>
            Path.Combine(TempDirPath, @"test.nsp");
        private string SolutionDirPath =>
            Path.Combine(TempDirPath, @"TestApp\");

        private static string MakeMetaExe => Path.Combine(
            NintendoSdkHelper.SdkRootPath,
            @"Tools\CommandLineTools\MakeMeta\MakeMeta.exe");

        private bool _isBuilt;

        public bool IsTestAppBuilt => _isBuilt;

        public void Dispose()
        {
            _context.Dispose();
        }

        public void CloneTestApp()
        {
            var srcDir = Path.Combine(TestContext.TestDataDirPath, "TestApp/");

            new Microsoft.VisualBasic.Devices.Computer().FileSystem.CopyDirectory(srcDir, SolutionDirPath);
        }

        public void BuildTestApp(Func<string, string> sourceCodeUpdater = null)
        {
            if (sourceCodeUpdater != null)
            {
                var testAppCodePath = Path.Combine(SolutionDirPath, @"TestApp\TestApp.cpp");
                File.WriteAllText(testAppCodePath, sourceCodeUpdater(File.ReadAllText(testAppCodePath)));
            }

            using (new CurrentDirectory(SolutionDirPath))
            {
                // ReSharper disable once PossibleNullReferenceException
                var processInfo = new ProcessStartInfo()
                {
                    FileName = Environment.GetEnvironmentVariable("ComSpec") ?? "cmd.exe",
                    Arguments = "/c build.bat",
                    CreateNoWindow = true,
                    UseShellExecute = false,
                    RedirectStandardError = true,
                    RedirectStandardOutput = true
                };

                // ホスト側の VisualStudio のバージョンが波及して意図しない挙動をしないようにする
                foreach (var k in new [] {"VisualStudioDir" , "VisualStudioEdition", "VisualStudioVersion"})
                    processInfo.Environment.Remove(k);

                var process = new Process();

                var errorMessage = new StringBuilder();
                var outputMessage = new StringBuilder();
                process.StartInfo = processInfo;
                process.ErrorDataReceived += (_, e) =>
                {
                    if (string.IsNullOrEmpty(e.Data) == false)
                        errorMessage.AppendLine(e.Data);
                };
                process.OutputDataReceived += (_, e) =>
                {
                    if (string.IsNullOrEmpty(e.Data) == false)
                        outputMessage.AppendLine(e.Data);
                };
                process.Start();
                process.BeginOutputReadLine();
                process.BeginErrorReadLine();
                process.WaitForExit();

                if (process.ExitCode != 0)
                {
                    throw new TestApplicationBuildErrorException(SolutionDirPath);
                }
            }

            _isBuilt = true;
        }

        public Task<NspHandleResultType> BuildNspAsync(string outputPath, ApplicationMeta meta) =>
            BuildNspAsync(outputPath, CodeDirPath, DataDirPath, meta);


        public async Task<NspHandleResultType> BuildNspAsync(string outputPath, string codeDir, string dataDir, ApplicationMeta meta)
        {
            Debug.Assert(_isBuilt);

            SetAmericanEnglishTitle(meta);
            SetJapaneseTitle(meta);

            var tempMetaFile = Path.Combine(TempDirPath, "temp.nmeta").ToPathString();

            using (var project = new Project { Meta = meta, XmlRootMetaKind = RootMetaKind.Meta })
                project.OutputAppMetaXmlFileForAuthoringTool(tempMetaFile);

            var option = new NspBuilder.ExportOption()
            {
                OutputType = OutputTypes.Nsp,
                InputDescFilePath = TestContext.DefaultDescFilePath,
                InputMetaFilePath = tempMetaFile,
                OutputPath = outputPath,
                InputProgramCodePath = codeDir,
                InputProgramDataPath = dataDir
            };

            // Core の変更を反映できるように .npdm を再作成
            MakeMeta(tempMetaFile);

            return await new NspBuilder().ExportNspAsync(option).ConfigureAwait(false);
        }

        private void MakeMeta(PathString metaFilePath)
        {
            var outputPath = Path.Combine(CodeDirPath, "main.npdm").ToPathString();
            var r = ProcessUtility.Execute(
                MakeMetaExe,
                $" --desc {NintendoSdkHelper.ApplicationDescFilePath.SurroundDoubleQuotes}" +
                $" --meta {metaFilePath.SurroundDoubleQuotes}" +
                $" -o {outputPath.SurroundDoubleQuotes}");
            if (r.ExitCode != 0)
                throw new TestApplicationBuildErrorException($"{r.StandardOutput}{r.StandardError}");
        }

        private static void SetAmericanEnglishTitle(ApplicationMeta appMeta)
        {
            var iconPath = Path.Combine(TestContext.TestDataDirPath, @"icon_testdata\1024x1024x24.bmp");
            var title = appMeta.Application.Titles.FirstOrDefault(x => x.Language == LanguageType.AmericanEnglish);
            if (title == null)
            {
                appMeta.Application.Titles.Add(new Title()
                {
                    Name = "Name",
                    Publisher = "Publisher",
                    Language = LanguageType.AmericanEnglish
                });
                title = appMeta.Application.Titles.Last();
            }
            title.IsReplaceIcon = true;
            title.IconFilePath = iconPath;
        }

        private static void SetJapaneseTitle(ApplicationMeta appMeta)
        {
            var title = appMeta.Application.Titles.FirstOrDefault(x => x.Language == LanguageType.Japanese);
            if (title == null)
            {
                appMeta.Application.Titles.Add(new Title()
                {
                    Name = "日本語",
                    Publisher = "パブリッシャー",
                    Language = LanguageType.Japanese,
                });
                title = appMeta.Application.Titles.Last();
            }
            title.IsReplaceIcon = true;
            title.IconFilePath = NintendoSdkHelper.ApplicationIconPath.ToString();
        }
    }
}
