﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Xml.Linq;
using System.Xml.XPath;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test
{
    public class CoreTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Fact]
        public void DefaultCtor()
        {
            var defaultProgramId = XDocument
                .Load(NintendoSdkHelper.ApplicationMetaFilePath)
                .Root.XPathSelectElement("Core/ApplicationId").Value.ToUlong();

            using (var c = new Core())
            {
                Assert.Equal("Application", c.Name);
                Assert.Equal("True", c.Is64BitInstruction);
                Assert.Equal(defaultProgramId, c.ApplicationId);
                Assert.Equal(0x100000ul, c.MainThreadStackSize);
                Assert.Equal(ProcessAddressSpaceType.AddressSpace64Bit, c.ProcessAddressSpace);
                Assert.Equal(0u, c.SystemResourceSize);
                Assert.True(c.IsProcessAddressSpace64Bit);
                Assert.NotNull(c.FsAccessControlData);
            }
        }

        [Theory]
        [InlineData(Core.MainThreadStackSizeValidationType.Empty, 0)]
        [InlineData(Core.MainThreadStackSizeValidationType.AlignError, 2)]
        [InlineData(Core.MainThreadStackSizeValidationType.AlignError, 1024)]
        [InlineData(Core.MainThreadStackSizeValidationType.Ok, 4096)]
        [InlineData(Core.MainThreadStackSizeValidationType.AlignError, 4097)]
        public void ValidationMainThreadStackSize(Core.MainThreadStackSizeValidationType result, ulong size)
        {
            using (var c = new Core())
            {
                c.MainThreadStackSize = size;
                Assert.Equal(result, c.ValidationMainThreadStackSize);
            }
        }

        [Theory]
        // AddressSpace64Bit
        [InlineData(Core.SystemResourceSizeValidationType.Ok, true, 0)]
        [InlineData(Core.SystemResourceSizeValidationType.Ok, true, 0x200000)]
        [InlineData(Core.SystemResourceSizeValidationType.AlignError, true, 0x200000 - 1)]
        [InlineData(Core.SystemResourceSizeValidationType.Ok, true, 0x200000 * 4)]
        // AddressSpace32Bit
        [InlineData(Core.SystemResourceSizeValidationType.Ok, false, 0x200000 - 1)]
        [InlineData(Core.SystemResourceSizeValidationType.Ok, false, 0x200000 * 4)]
        public void ValidationSystemResourceSize(Core.SystemResourceSizeValidationType expected, bool isProcessSpace64Bit, uint size)
        {
            using (var c = new Core())
            {
                c.ProcessAddressSpace = isProcessSpace64Bit
                    ? ProcessAddressSpaceType.AddressSpace64Bit
                    : ProcessAddressSpaceType.AddressSpace32Bit;
                c.SystemResourceSize = size;
                Assert.Equal(expected, c.ValidationSystemResourceSize);
            }
        }
    }
}
