﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.Linq;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test
{
    public class ApplicationTest : IDisposable
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        [Fact]
        public void DefaultCtor()
        {
            using (var a = new Application {DiContainer = _context.DiContainer})
            {
                Assert.NotNull(a.Titles);
                Assert.Equal(0, a.Titles.Count);
                Assert.Equal(0, a.ReleaseVersion);
                Assert.Equal(string.Empty, a.DisplayVersion);
                Assert.Equal(StartupUserAccountType.None, a.StartupUserAccount);
                Assert.Equal(ParentalControlType.None, a.ParentalControl);
                Assert.NotNull(a.SupportedLanguages);
                Assert.Equal(15, a.SupportedLanguages.Count);
                Assert.Equal(0ul, a.PresenceGroupId);
                Assert.True(a.IsUsePresenceGroupIdAppId);
                Assert.True(a.IsAllowScreenshot);
                Assert.Equal(DataLossConfirmationType.None, a.DataLossConfirmation);
                Assert.False(a.IsUseSaveData);
                Assert.Equal(0L, a.SaveDataSize);
                Assert.False(a.IsSpecifiedSaveDataJournal);
                Assert.Equal(0L, a.SaveDataJournalSize);
                Assert.False(a.IsUseUserAccountSaveDataSizeMax);
                Assert.Equal(0L, a.UserAccountSaveDataSizeMax);
                Assert.False(a.IsUseUserAccountSaveDataJournalSizeMax);
                Assert.Equal(0L, a.UserAccountSaveDataJournalSizeMax);
                Assert.Null(a.ApplicationErrorCodeCategory);
                Assert.False(a.IsUseApplicationErrorCode);
                Assert.Equal(LogoTypeType.LicensedByNintendo, a.LogoType);
                Assert.Equal(LogoHandlingType.Auto, a.LogoHandling);
                Assert.NotNull(a.LocalCommunicationIds);
                Assert.Equal(0, a.LocalCommunicationIds.Count);
                Assert.NotNull(a.Ratings);
                Assert.Equal(Constants.AllRatingData.Count, a.Ratings.Count);
                Assert.False(a.IsDemo);
                Assert.False(a.IsReplaceHtmlDocumentPath);
                Assert.Null(a.HtmlDocumentPath.Path);
                Assert.False(a.IsReplaceAccessibleUrlsFilePath);
                Assert.Null(a.AccessibleUrlsFilePath.Path);
                Assert.False(a.IsReplaceLegalInformationFilePath);
                Assert.Null(a.LegalInformationFilePath.Path);
                Assert.True(a.IsUseSeedForPseudoDeviceAppId);
                Assert.Equal(0ul, a.SeedForPseudoDeviceId);
                Assert.Null(a.SeedForPseudoDeviceIdHex);
                Assert.False(a.IsUseBcat);
                Assert.Equal(0L, a.BcatDeliveryCacheStorageSize);
                Assert.Null(a.BcatPassphrase);
                Assert.Equal(0L, a.DeviceSaveDataSize);
                Assert.Equal(0L, a.DeviceSaveDataSizeMax);
                Assert.Equal(0L, a.DeviceSaveDataJournalSize);
                Assert.Equal(0L, a.DeviceSaveDataJournalSizeMax);
                Assert.False(a.IsUseCacheStorageSize);
                Assert.Equal(0L, a.CacheStorageSize);
                Assert.False(a.IsUseCacheStorageJournalSize);
                Assert.Equal(0L, a.CacheStorageJournalSize);
                Assert.False(a.IsUseTemporaryStorageSize);
                Assert.Equal(0L, a.TemporaryStorageSize);
                Assert.Null(a.VideoCapture);
                Assert.Null(a.RuntimeAddOnContentInstall);
                Assert.Null(a.FilterDescriptionFilePath);
            }
        }

        public static IEnumerable<object> ValidationDisplayVersionTestData => new[]
        {
            new object[] {Application.DisplayVersionValidationType.CharaError, "あいう"},
            new object[]
            {
                Application.DisplayVersionValidationType.MaxLengthError,
                "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
            },
            new object[] {Application.DisplayVersionValidationType.CharaError, "あいう"},
            new object[] {Application.DisplayVersionValidationType.Ok, "ABC"}
        };

        [Theory]
        [MemberData(nameof(ValidationDisplayVersionTestData))]
        public void ValidationDisplayVersion(Application.DisplayVersionValidationType result, string displayVersion)
        {
            using (var a = new Application {DiContainer = _context.DiContainer})
            {
                a.DisplayVersion = displayVersion;
                Assert.Equal(result, a.ValidationDisplayVersion);
            }
        }

        [Theory]
        [MemberData(nameof(ValidationDisplayVersionTestData))]
        public void ValidateDisplayVersion(Application.DisplayVersionValidationType result, string displayVersion)
        {
            Assert.Equal(result, Application.ValidateDisplayVersion(displayVersion));
        }

        public static IEnumerable<object> ValidationApplicationErrorCodeCategoryTestData => new[]
        {
            new object[] {Application.ApplicationErrorCodeCategoryValidationType.Ok, false, ""},
            new object[] {Application.ApplicationErrorCodeCategoryValidationType.EmptyError, true, ""},
            new object[] {Application.ApplicationErrorCodeCategoryValidationType.Ok, true, "12345"},
            new object[] {Application.ApplicationErrorCodeCategoryValidationType.LengthError, true, "123456"},
            new object[] {Application.ApplicationErrorCodeCategoryValidationType.Ok, true, "1!あ %"},
        };

        [Theory]
        [MemberData(nameof(ValidationApplicationErrorCodeCategoryTestData))]
        public void ValidationApplicationErrorCodeCategory(
            Application.ApplicationErrorCodeCategoryValidationType result, bool isUse, string category)
        {
            using (var a = new Application {DiContainer = _context.DiContainer})
            {
                a.IsUseApplicationErrorCode = isUse;
                a.ApplicationErrorCodeCategory = category;
                Assert.Equal(result, a.ValidationApplicationErrorCodeCategory);
            }
        }

        [Theory]
        [MemberData(nameof(ValidationApplicationErrorCodeCategoryTestData))]
        public void ValidateApplicationErrorCodeCategory(Application.ApplicationErrorCodeCategoryValidationType result,
            bool isUse, string category)
        {
            Assert.Equal(result, Application.ValidateApplicationErrorCodeCategory(isUse, category));
        }

        [Theory]
        [InlineData(Application.SaveDataSizeValidationType.Ok, false, 0)]
        [InlineData(Application.SaveDataSizeValidationType.SizeError, true, 0)]
        [InlineData(Application.SaveDataSizeValidationType.Ok, true, 16*1024)]
        [InlineData(Application.SaveDataSizeValidationType.AlignError, true, 16*1024 + 1)]
        public void ValidationSaveDataSize(Application.SaveDataSizeValidationType result, bool isUse, long size)
        {
            using (var a = new Application {DiContainer = _context.DiContainer})
            {
                a.IsUseSaveData = isUse;
                a.SaveDataSize = size;
                Assert.Equal(result, a.ValidationSaveDataSize);
            }
        }

        [Theory]
        [InlineData(Application.SaveDataJournalSizeValidationType.Ok, false, 0)]
        [InlineData(Application.SaveDataJournalSizeValidationType.SizeError, true, 0)]
        [InlineData(Application.SaveDataJournalSizeValidationType.SizeError, true, 1)]
        [InlineData(Application.SaveDataJournalSizeValidationType.SizeError, true, 1000)]
        [InlineData(Application.SaveDataJournalSizeValidationType.AlignError, true, 1024*16 + 1)]
        [InlineData(Application.SaveDataJournalSizeValidationType.Ok, true, 1024 * 16 * 10)]
        public void ValidationSaveDataJournalSize(Application.SaveDataJournalSizeValidationType result, bool isUse,
            long size)
        {
            using (var a = new Application {DiContainer = _context.DiContainer})
            {
                a.IsSpecifiedSaveDataJournal = isUse;
                a.SaveDataJournalSize = size;
                Assert.Equal(result, a.ValidationSaveDataJournalSize);
            }
        }

        [Theory]
        [InlineData(Application.SaveDataSizeValidationType.Ok, 0)]
        [InlineData(Application.SaveDataSizeValidationType.Ok, 1024 * 16)]
        [InlineData(Application.SaveDataSizeValidationType.SizeError, 1)]
        [InlineData(Application.SaveDataSizeValidationType.AlignError, 1024*16 + 1)]
        public void ValidationDeviceSaveDataSize(Application.SaveDataSizeValidationType result, long size)
        {
            using (var a = new Application { DiContainer = _context.DiContainer })
            {
                a.IsSpecifiedDeviceSaveDataSize = size > 0;
                a.DeviceSaveDataSize = size;
                Assert.Equal(result, a.ValidationDeviceSaveDataSize);
            }
        }

        [Theory]
        [InlineData(Application.SaveDataJournalSizeValidationType.Ok, 0)]
        [InlineData(Application.SaveDataJournalSizeValidationType.Ok, 1024 * 16)]
        [InlineData(Application.SaveDataJournalSizeValidationType.SizeError, 1)]
        [InlineData(Application.SaveDataJournalSizeValidationType.AlignError, 1024 * 16 + 1)]
        public void ValidationDeviceSaveDataJournalSize(Application.SaveDataJournalSizeValidationType result, long size)
        {
            using (var a = new Application { DiContainer = _context.DiContainer })
            {
                a.IsSpecifiedDeviceSaveDataJournalSize = size > 0;
                a.DeviceSaveDataJournalSize = size;
                Assert.Equal(result, a.ValidationDeviceSaveDataJournalSize);
            }
        }

        [Fact]
        public void ValidationSupportedLanguages()
        {
            using (var a = new Application {DiContainer = _context.DiContainer})
            {
                Assert.True(a.IsErrorSupportedLanguages);

                a.SupportedLanguages.First(x => x.Language == LanguageType.Japanese).IsSupported = true;

                Assert.False(a.IsErrorSupportedLanguages);
            }
        }

        [Fact]
        public void ValidateSupportedLanguages()
        {
            {
                var langs = new ObservableCollection<SupportedLanguage>();
                Assert.Equal(Application.SupportedLanguagesValidationType.EmptyError,
                    Application.ValidateSupportedLanguages(langs));
            }

            {
                var langs = new ObservableCollection<SupportedLanguage>
                {
                    new SupportedLanguage
                    {
                        IsSupported = true,
                        Language = LanguageType.AmericanEnglish
                    }
                };

                Assert.Equal(Application.SupportedLanguagesValidationType.Ok,
                    Application.ValidateSupportedLanguages(langs));
            }
        }

        [Fact]
        public void ValidationTitles()
        {
            using (var a = new Application {DiContainer = _context.DiContainer})
            {
                // タイトルが 1 つ以上必要
                Assert.True(a.IsErrorTitles);

                // タイトルを追加
                a.Titles.Add(new Title
                {
                    Language = LanguageType.Japanese,
                    IconFilePath = Path.Combine(TestContext.TestDataDirPath, "icon_testdata/1024x1024x24.bmp"),
                    Name = "Name",
                    Publisher = "Publisher"
                });

                // 追加内容にエラーが無ければバリデーションも通過
                Assert.False(a.IsErrorTitles);

                // タイトルの内容を意図的に空にした場合はエラーが発生
                a.Titles[0].Name = string.Empty;
                Assert.True(a.Titles[0].HasErrors);
            }
        }

        public static IEnumerable<object> ValidationTitleNameTestData => new[]
        {
            new object[] {Title.NameValidationType.Ok, "abc"},
            new object[] {Title.NameValidationType.EmptyError, string.Empty},
            new object[] {Title.NameValidationType.Ok, new string('c', 127)},
            new object[] {Title.NameValidationType.LengthError, new string('c', 128)}
        };

        [Theory]
        [MemberData(nameof(ValidationTitleNameTestData))]
        public void ValidationTitleName(Title.NameValidationType result, string name)
        {
            var t = new Title {DiContainer = _context.DiContainer};
            {
                t.Name = name;
                Assert.Equal(result, t.ValidationName);
            }
        }

        public static IEnumerable<object> ValidationPublisherNameTestData => new[]
        {
            new object[] {Title.PublisherValidationType.Ok, "abc"},
            new object[] {Title.PublisherValidationType.EmptyError, string.Empty},
            new object[] {Title.PublisherValidationType.Ok, new string('c', 63)},
            new object[] {Title.PublisherValidationType.LengthError, new string('c', 64)}
        };

        [Theory]
        [MemberData(nameof(ValidationPublisherNameTestData))]
        public void ValidationTitlePublisher(Title.PublisherValidationType result, string name)
        {
            var t = new Title {DiContainer = _context.DiContainer};
            {
                t.Publisher = name;
                Assert.Equal(result, t.ValidationPublisher);
            }
        }

        public static IEnumerable<object> ValidationTitleIconFilePathTestData()
        {
            return new[]
            {
                new object[]
                {
                    Title.IconFilePathValidationType.EmptyError,
                    string.Empty
                },
                new object[]
                {
                    Title.IconFilePathValidationType.FileNotFound,
                    "AAA"
                },
                new object[]
                {
                    Title.IconFilePathValidationType.Ok,
                    Path.Combine(TestContext.TestDataDirPath, "icon_testdata/1024x1024x24.bmp")
                },
                new object[]
                {
                    Title.IconFilePathValidationType.FormatError,
                    Path.Combine(TestContext.TestDataDirPath, "icon_testdata/1024x1024x8.bmp")
                },
                new object[]
                {
                    Title.IconFilePathValidationType.SizeError,
                    Path.Combine(TestContext.TestDataDirPath, "icon_testdata/123x234.bmp")
                },
                new object[]
                {
                    Title.IconFilePathValidationType.FormatError,
                    Path.Combine(TestContext.TestDataDirPath, "icon_testdata/error.bmp")
                },
                new object[]
                {
                    Title.IconFilePathValidationType.FormatError,
                    Path.Combine(TestContext.TestDataDirPath, "icon_testdata/not_bmp.bmp")
                }
            };
        }

        [Theory]
        [MemberData(nameof(ValidationTitleIconFilePathTestData))]
        public void ValidationTitleIconFilePath_Enable(Title.IconFilePathValidationType result, string path)
        {
            var t = new Title {DiContainer = _context.DiContainer};
            {
                t.IconFilePath = path;
                Assert.Equal(result, t.ValidationIconFilePath);
            }
        }

        [Theory]
        [MemberData(nameof(ValidationTitleIconFilePathTestData))]
        public void ValidationTitleIconFilePath_Replace(Title.IconFilePathValidationType result, string path)
        {
            var t = new Title {DiContainer = _context.DiContainer};
            {
                t.IconFilePath = path;
                t.IsReplaceIcon = true;
                Assert.Equal(result, t.ValidationIconFilePath);
            }
        }

        [Theory]
        [MemberData(nameof(ValidationTitleIconFilePathTestData))]
        public void ValidationTitleIconFilePath_NoReplace(Title.IconFilePathValidationType result, string path)
        {
            var t = new Title {DiContainer = _context.DiContainer};
            {
                t.IconFilePath = path;
                t.IsReplaceIcon = false;
                Assert.Equal(Title.IconFilePathValidationType.Ok, t.ValidationIconFilePath);
            }
        }

        public static IEnumerable<object> ValidationTitleNxIconFilePathTestData()
        {
            return new[]
            {
                new object[]
                {
                    Title.NxIconFilePathValidationType.EmptyError,
                    string.Empty
                },
                new object[]
                {
                    Title.NxIconFilePathValidationType.FileNotFound,
                    "AAA"
                },
                new object[]
                {
                    Title.NxIconFilePathValidationType.Ok,
                    Path.Combine(TestContext.TestDataDirPath, "nxicon_testdata/256x256.jpg")
                },
                new object[]
                {
                    Title.NxIconFilePathValidationType.SizeError,
                    Path.Combine(TestContext.TestDataDirPath, "nxicon_testdata/123x234.jpg")
                },
                new object[]
                {
                    Title.NxIconFilePathValidationType.FormatError,
                    Path.Combine(TestContext.TestDataDirPath, "nxicon_testdata/error.jpg")
                },
                new object[]
                {
                    Title.NxIconFilePathValidationType.FileSizeError,
                    Path.Combine(TestContext.TestDataDirPath, "nxicon_testdata/filesize_over.jpg")
                },
                new object[]
                {
                    Title.NxIconFilePathValidationType.FormatError,
                    Path.Combine(TestContext.TestDataDirPath, "nxicon_testdata/not_jpg.jpg")
                },
                new object[]
                {
                    Title.NxIconFilePathValidationType.FormatError,
                    Path.Combine(TestContext.TestDataDirPath, "nxicon_testdata/progressive.jpg")
                }
            };
        }

        [Theory]
        [MemberData(nameof(ValidationTitleNxIconFilePathTestData))]
        public void ValidationTitleNxIconFilePath_Replace(Title.NxIconFilePathValidationType result, string path)
        {
            var t = new Title {DiContainer = _context.DiContainer};
            {
                t.IsReplaceNxIcon = true;
                t.NxIconFilePath = path;
                Assert.Equal(result, t.ValidationNxIconFilePath);
            }
        }

        [Theory]
        [MemberData(nameof(ValidationTitleNxIconFilePathTestData))]
        public void ValidationTitleNxIconFilePath_NoReplace(Title.NxIconFilePathValidationType result, string path)
        {
            var t = new Title {DiContainer = _context.DiContainer};
            {
                t.NxIconFilePath = path;
                t.IsReplaceNxIcon = false;
                Assert.Equal(Title.NxIconFilePathValidationType.Ok, t.ValidationNxIconFilePath);
            }
        }

        // 現状の AuthoringEditor では NxIcon を個別指定することはないので常に Validation は通過する
        [Theory]
        [InlineData(Title.NxIconFilePathValidationType.Ok, ImportableFileType.Project)]
        [InlineData(Title.NxIconFilePathValidationType.Ok, ImportableFileType.Meta)]
        [InlineData(Title.NxIconFilePathValidationType.Ok, ImportableFileType.Nsp)]
        public void ValidationTitleNxIconFilePath_TargetFileType(Title.NxIconFilePathValidationType result,
            ImportableFileType filetype)
        {
            _context.DiContainer.GetInstance<AppProfile>().TargetFileType = filetype;

            var t = new Title {DiContainer = _context.DiContainer};

            Assert.Equal(result, t.ValidationNxIconFilePath);
        }

        public static IEnumerable<object> ValidationTitleNxIconFilePath_HasErrorsTestData
        {
            get
            {
                using (var context = new TestContext())
                {
                    yield return
                        new object[]
                        {
                            false,
                            new Title
                            {
                                DiContainer = context.DiContainer,
                                Language = LanguageType.Japanese,
                                IconFilePath =
                                    Path.Combine(TestContext.TestDataDirPath, "icon_testdata/1024x1024x24.bmp"),
                                Name = "Name",
                                Publisher = "Publisher"
                            }
                        };

                    yield return
                        new object[]
                        {
                            true,
                            new Title
                            {
                                DiContainer = context.DiContainer,
                                Language = LanguageType.Japanese,
                                IconFilePath =
                                    Path.Combine(TestContext.TestDataDirPath, "icon_testdata/1024x1024x24.bmp"),
                                Name = string.Empty,
                                Publisher = "Publisher"
                            }
                        };

                    yield return
                        new object[]
                        {
                            true,
                            new Title
                            {
                                DiContainer = context.DiContainer,
                                Language = LanguageType.Japanese,
                                IconFilePath =
                                    Path.Combine(TestContext.TestDataDirPath, "icon_testdata/1024x1024x24.bmp"),
                                Name = "Name",
                                Publisher = string.Empty
                            }
                        };

                    yield return
                        new object[]
                        {
                            true,
                            new Title
                            {
                                DiContainer = context.DiContainer,
                                Language = LanguageType.Japanese,
                                IconFilePath = string.Empty,
                                Name = "Name",
                                Publisher = "Publisher"
                            }
                        };


                    yield return
                        new object[]
                        {
                            true,
                            new Title
                            {
                                DiContainer = context.DiContainer,
                                Language = LanguageType.Japanese,
                                IconFilePath =
                                    Path.Combine(TestContext.TestDataDirPath, "icon_testdata/1024x1024x24.bmp"),
                                NxIconFilePath = string.Empty,
                                IsReplaceNxIcon = true,
                                Name = "Name",
                                Publisher = "Publisher"
                            }
                        };
                }
            }
        }

        [Theory]
        [MemberData(nameof(ValidationTitleNxIconFilePath_HasErrorsTestData))]
        public void ValidationTitleNxIconFilePath_HasErrors(bool expected, Title title)
        {
            Assert.Equal(expected, title.HasErrors);
        }

        [Theory]
        [InlineData(nameof(Application.IsDemo), 1, true)]
        [InlineData(nameof(Application.IsDemo), 0, false)]
        [InlineData(nameof(Application.IsRetailInteractiveDisplay), 1, true)]
        [InlineData(nameof(Application.IsRetailInteractiveDisplay), 0, false)]
        [InlineData(nameof(Application.IsDownloadPlay), 1, true)]
        [InlineData(nameof(Application.IsDownloadPlay), 0, false)]
        public void Attribute(string propertyName, int length, bool isFlagUse)
        {
            using (var a = new Application {DiContainer = _context.DiContainer})
            {
                var property = a.GetType().GetProperty(propertyName);
                property.SetValue(a, isFlagUse);

                Assert.NotNull(a.Attribute);
                Assert.Equal(length, a.Attribute.Length);
            }
        }

        [Theory]
        [InlineData(true, "Demo", nameof(Application.IsDemo))]
        [InlineData(true, "DownloadPlay", nameof(Application.IsDownloadPlay))]
        [InlineData(true, "RetailInteractiveDisplay", nameof(Application.IsRetailInteractiveDisplay))]
        [InlineData(false, null, "")]
        [InlineData(false, "", "")]
        [InlineData(false, "unknown", "")]
        public void Attribute_Set(bool expected, string member, string propertyName)
        {
            using (var a = new Application {DiContainer = _context.DiContainer})
            {
                a.Attribute = new[] {member};

                if (string.IsNullOrEmpty(propertyName) == false)
                {
                    var property = a.GetType().GetProperty(propertyName);
                    var propertyValue = property.GetValue(a);
                    Assert.Equal(expected, propertyValue);
                }
            }
        }

        [Fact]
        public void Attribute_SetNull()
        {
            using (var a = new Application {DiContainer = _context.DiContainer})
            {
                a.Attribute = null;
                Assert.False(a.IsDemo);
                Assert.False(a.IsDownloadPlay);
                Assert.False(a.IsRetailInteractiveDisplay);
                Assert.NotNull(a.Attribute);
                Assert.Equal(0, a.Attribute.Length);
            }
        }

        [Fact]
        public void SupportedLanguage()
        {
            using (var a = new Application {DiContainer = _context.DiContainer})
            {
                a.SupportedLanguage = null;
                Assert.NotNull(a.SupportedLanguage);
                Assert.Equal(0, a.SupportedLanguage.Length);

                a.SupportedLanguage = new [] {"AmericanEnglish", "Japanese"};
                Assert.Equal(2, a.SupportedLanguage.Length);
                Assert.Equal("AmericanEnglish", a.SupportedLanguage[0]);
                Assert.Equal("Japanese", a.SupportedLanguage[1]);
            }
        }

        [Fact]
        public void LocalCommunicationId()
        {
            using (var a = new Application {DiContainer = _context.DiContainer})
            {
                a.LocalCommunicationId = null;
                Assert.NotNull(a.LocalCommunicationId);
                Assert.Equal(0, a.LocalCommunicationId.Length);
            }
        }

        [Fact]
        public void HasErrorNonPublicProperties()
        {
            using (var a = new Application {DiContainer = _context.DiContainer})
            {
                Assert.False(a.HasErrorNonPublicProperties);
            }
        }

        [Theory]
        [InlineData(false, null)]
        [InlineData(false, "")]
        [InlineData(false, "012345678901234567890123456789012345")]
        [InlineData(true, "0123456789012345678901234567890123456")]
        public void HasErrorNonPublicProperties_Isbn(bool result, string data)
        {
            using (var a = new Application {DiContainer = _context.DiContainer})
            {
                a.Isbn = data;
                Assert.Equal(result, a.HasErrorNonPublicProperties);
            }
        }

        [Theory]
        [InlineData(false, null)]
        [InlineData(true, "")]
        [InlineData(false, "123")]
        [InlineData(false, "0x132")]
        public void HasErrorNonPublicProperties_SaveDataOwnerId(bool result, string data)
        {
            using (var a = new Application {DiContainer = _context.DiContainer})
            {
                a.SaveDataOwnerId = data;
                Assert.Equal(result, a.HasErrorNonPublicProperties);
            }
        }

        [Theory]
        [InlineData(false, 0, Application.BcatDeliveryCacheStorageSizeValidationType.Ok)]
        [InlineData(false, 123, Application.BcatDeliveryCacheStorageSizeValidationType.Ok)]
        [InlineData(true, 0, Application.BcatDeliveryCacheStorageSizeValidationType.SizeError)]
        [InlineData(true, 123, Application.BcatDeliveryCacheStorageSizeValidationType.AlignError)]
        [InlineData(true, 0x132, Application.BcatDeliveryCacheStorageSizeValidationType.AlignError)]
        [InlineData(true, 0x100000, Application.BcatDeliveryCacheStorageSizeValidationType.SizeError)]
        [InlineData(true, 0x500000, Application.BcatDeliveryCacheStorageSizeValidationType.Ok)]
        [InlineData(true, 0x100000 * 64, Application.BcatDeliveryCacheStorageSizeValidationType.Ok)]
        [InlineData(true, 0x100000 * 65, Application.BcatDeliveryCacheStorageSizeValidationType.SizeError)]
        [InlineData(true, 0x100000 * 64 - 1, Application.BcatDeliveryCacheStorageSizeValidationType.AlignError)]
        public void BcatDeliveryCacheStorageSize(bool isUseBcat, long size, Application.BcatDeliveryCacheStorageSizeValidationType expected)
        {
            using (var a = new Application {DiContainer = _context.DiContainer})
            {
                a.IsUseBcat = isUseBcat;
                a.BcatDeliveryCacheStorageSize = size;
                Assert.Equal(expected, a.ValidationBcatDeliveryCacheStorageSize);
            }
        }

        [Theory]
        [InlineData(false, "", Application.BcatPassphraseValidationType.Ok)]
        [InlineData(false, null, Application.BcatPassphraseValidationType.Ok)]
        [InlineData(false, "abcd", Application.BcatPassphraseValidationType.Ok)]
        [InlineData(true, null, Application.BcatPassphraseValidationType.FormatError)]
        [InlineData(true, "", Application.BcatPassphraseValidationType.FormatError)]
        [InlineData(true, "abcd", Application.BcatPassphraseValidationType.FormatError)]
        [InlineData(true, "2be7b01684f98f8a668d9f474c6583a44c418bb72be7b01684f98f8a668d9f40", Application.BcatPassphraseValidationType.Ok)]
        [InlineData(true, "2be7b01684f98f8a668d9f474c6583a44c418bb72be7b01684f98f8a668d9f401", Application.BcatPassphraseValidationType.FormatError)]
        public void BcatPassphrase(bool isUseBcat, string passPhrase, Application.BcatPassphraseValidationType expected)
        {
            using (var a = new Application { DiContainer = _context.DiContainer })
            {
                a.IsUseBcat = isUseBcat;
                a.BcatPassphrase = passPhrase;
                Assert.Equal(expected, a.ValidationBcatPassphrase);
            }
        }

        [Theory]
        [InlineData(false, null)]
        [InlineData(true, "")]
        [InlineData(false, "123")]
        [InlineData(false, "0x132")]
        public void HasErrorNonPublicProperties_DeviceSaveDataSize(bool result, string data)
        {
            using (var a = new Application { DiContainer = _context.DiContainer })
            {
                a.DeviceSaveDataSizeHex = data;
                Assert.Equal(result, a.HasErrorNonPublicProperties);
            }
        }

        [Theory]
        [InlineData(false, null)]
        [InlineData(true, "")]
        [InlineData(false, "123")]
        [InlineData(false, "0x132")]
        public void HasErrorNonPublicProperties_DeviceSaveDataJournalSize(bool result, string data)
        {
            using (var a = new Application {DiContainer = _context.DiContainer})
            {
                a.DeviceSaveDataJournalSizeHex = data;
                Assert.Equal(result, a.HasErrorNonPublicProperties);
            }
        }

        public static IEnumerable<object> ValidationHtmlDocumentPathTestData => new[]
        {
            new object[] {Application.HtmlDocumentPathValidationType.PathIsEmpty, "", true},
            new object[] {Application.HtmlDocumentPathValidationType.DirectoryNotFound, "AAA", true},
            new object[] {Application.HtmlDocumentPathValidationType.Ok, ".", true},
        };

        [Theory]
        [MemberData(nameof(ValidationHtmlDocumentPathTestData))]
        public void ValidationHtmlDocumentDirectoryPath(Application.HtmlDocumentPathValidationType result,
            string path, bool isReplace)
        {
            using (var a = new Application { DiContainer = _context.DiContainer })
            {
                a.IsReplaceHtmlDocumentPath = isReplace;
                a.HtmlDocumentPath = path;
                Assert.Equal(result, a.ValidationHtmlDocumentPath);
            }
        }

        public static IEnumerable<object> ValidationAccessibleUrlsFilePathTestData => new[]
        {
            new object[] {Application.AccessibleUrlsFilePathValidationType.PathIsEmpty, "", true},
            new object[] {Application.AccessibleUrlsFilePathValidationType.FileNotFound, "AAA", true},
            new object[] {Application.AccessibleUrlsFilePathValidationType.Ok, "", false},
            new object[] {Application.AccessibleUrlsFilePathValidationType.Ok, ".", false},
        };


        [Theory]
        [MemberData(nameof(ValidationAccessibleUrlsFilePathTestData))]
        public void ValidationAccessibleUrlsDirectoryPath(
            Application.AccessibleUrlsFilePathValidationType result, string path, bool isReplace)
        {
            using (var a = new Application { DiContainer = _context.DiContainer })
            {
                a.IsReplaceAccessibleUrlsFilePath = isReplace;
                a.AccessibleUrlsFilePath = path;
                Assert.Equal(result, a.ValidationAccessibleUrlsFilePath);
            }
        }

        public static IEnumerable<object> ValidationLegalInformationFilePathTestData => new[]
        {
            new object[] {Application.LegalInformationFilePathValidationType.EmptyError, "", true},
            new object[] {Application.LegalInformationFilePathValidationType.FileNotFound, "AAA", true},
            new object[] {Application.LegalInformationFilePathValidationType.Ok, "", false},
        };

        [Theory]
        [MemberData(nameof(ValidationLegalInformationFilePathTestData))]
        public void ValidationLegalInformationFilePath(Application.LegalInformationFilePathValidationType result,
            string path, bool isReplace)
        {
            using (var a = new Application { DiContainer = _context.DiContainer })
            {
                a.IsReplaceLegalInformationFilePath = isReplace;
                a.LegalInformationFilePath = path;
                Assert.Equal(result, a.ValidationLegalInformationFilePath);
            }
        }
    }
}
