﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Authoring.AuthoringEditor.Core.Test.Fixtures;
using Nintendo.Authoring.AuthoringEditor.Foundation;
using Xunit;

namespace Nintendo.Authoring.AuthoringEditor.Core.Test
{
    public class AppTest : IDisposable, IClassFixture<BuildTestAppNspFixture>
    {
        private readonly TestContext _context = new TestContext();

        public void Dispose()
        {
            _context.Dispose();
        }

        private readonly BuildTestAppNspFixture _fixture;

        public AppTest(BuildTestAppNspFixture fixture)
        {
            _fixture = fixture.Initialize();
        }

        [Fact]
        public void DefaultCtor()
        {
            var app = _context.GetInstance<App>();

            Assert.NotNull(app.Project);
            Assert.Equal(string.Empty, app.Project.TargetFilePath);
            Assert.NotEqual(string.Empty, app.OriginalProject);
            Assert.True(app.HasErrors);
            Assert.False(app.IsChangedTarget);
        }

        [Fact]
        public void TempDir()
        {
            string dir;

            using (var context = new TestContext())
            {
                var app = context.GetInstance<App>();

                dir = Path.Combine(context.TempDirPath, "TEMP");
                Directory.CreateDirectory(dir);
                app.AddTempDir(dir);
            }

            Assert.False(Directory.Exists(dir));
        }

        [Fact]
        public void TempDirWithFile()
        {
            string dir;

            using (var context = new TestContext())
            {
                var app = context.GetInstance<App>();

                dir = Path.Combine(context.TempDirPath, "TEMP");
                Directory.CreateDirectory(dir);
                app.AddTempDir(dir);

                File.WriteAllBytes(Path.Combine(dir, "file.bin"), new byte[0]);
            }

            Assert.False(Directory.Exists(dir));
        }

        [Fact]
        public void EmptyProject()
        {
            var app = _context.GetInstance<App>();

            app.Project = _context.GetInstance<Project>();
            Assert.False(app.IsChangedTarget);
        }

        [Fact]
        public void TargetFilePath()
        {
            var app = _context.GetInstance<App>();

            app.Project.TargetFilePath = "ABC";
        }

        [Fact]
        public void CreateNewProject()
        {
            var app = _context.GetInstance<App>();

            app.Project.TargetFilePath = "ABC";

            app.CreateNewProject();

            Assert.Equal(string.Empty, app.Project.TargetFilePath);
            Assert.False(app.IsChangedTarget);
        }

        public static IEnumerable<object[]> AllAppModeTypes =>
            Enum.GetValues(typeof(ExportableFileType)).Cast<ExportableFileType>().Select(t => new object[] {t});

        [Theory]
        [MemberData(nameof(AllAppModeTypes))]
        public void SaveTargetFilePath(ExportableFileType type)
        {
            var app = _context.GetInstance<App>();

            app.Project.TargetFilePath = Path.Combine(_context.TempDirPath, "abc");

            app.Save(type);

            Assert.True(File.Exists(app.Project.TargetFilePath));
            Assert.False(app.IsChangedTarget);
        }

        [Theory]
        [MemberData(nameof(AllAppModeTypes))]
        public void Save(ExportableFileType type)
        {
            var app = _context.GetInstance<App>();

            app.Project.TargetFilePath = "AAA";

            app.Project.Meta.Application.Titles.Add(new Title
            {
                IconFilePath =
                    Path.Combine(TestContext.TestDataDirPath, "icon_testdata/1024x1024x24.bmp"),
                NxIconFilePath =
                    Path.Combine(TestContext.TestDataDirPath, "nxicon_testdata/256x256.jpg")
            });

            var filePath = Path.Combine(_context.TempDirPath, "abc");

            Assert.True(app.IsChangedTarget);

            var i = app.Save(type, filePath);

            Assert.True(i);
            Assert.False(app.IsChangedTarget);
            Assert.True(File.Exists(filePath));
            Assert.Equal(filePath, app.Project.TargetFilePath);
        }

        [Theory]
        [MemberData(nameof(AllAppModeTypes))]
        public void Save_Error(ExportableFileType type)
        {
            var app = _context.GetInstance<App>();

            app.Project.TargetFilePath = "AAA";

            const string filePath = @"zz:\zzz";

            var i = app.Save(type, filePath);

            Assert.False(i);
            Assert.False(File.Exists(filePath));
        }

        public static IEnumerable<object> OpenTestData
        {
            get
            {
                yield return new object[]
                {
                    ImportableFileType.Project,
                    AppModeType.Project,
                    Path.Combine(TestContext.TestDataDirPath, @"project_testdata\test.narp")
                };

                yield return new object[]
                {
                    ImportableFileType.Meta,
                    AppModeType.ApplicationMeta,
                    Path.Combine(TestContext.TestDataDirPath, @"project_testdata\exported.nmeta")
                };
            }
        }

        [Theory]
        [MemberData(nameof(OpenTestData))]
        public void Open(ImportableFileType type, AppModeType appMode, string path)
        {
            _context.GetInstance<AppProfile>().AppMode = appMode;

            var app = _context.GetInstance<App>();

            app.Open(type, path);

            Assert.False(app.IsChangedTarget);
        }

        [Fact]
        public void Open_Nsp()
        {
            var app = _context.GetInstance<App>();

            app.Open(ImportableFileType.Nsp, _fixture.NspFilePath);

            Assert.False(app.IsChangedTarget);
        }

        [Theory]
        [InlineData(ImportableFileType.Project)]
        [InlineData(ImportableFileType.Meta)]
        [InlineData(ImportableFileType.Nsp)]
        public void Open_Error(ImportableFileType type)
        {
            var app = _context.GetInstance<App>();

            Assert.Throws<FileNotFoundException>(() => app.Open(type, "ABC"));

            Assert.False(app.IsChangedTarget);
        }

        public static IEnumerable<object> AllOutputTypes =>
            Enum.GetValues(typeof(OutputTypes)).Cast<object>().Select(x => new[] {x});

        [Theory]
        [MemberData(nameof(AllOutputTypes))]
        public async Task BuildNsp(OutputTypes type)
        {
            var app = _context.GetInstance<App>();

            using (var project = Project.CreateNew(_context.DiContainer))
            {
                app.Project = project;

                project.TargetFilePath = _context.TempDirPath;
                project.ProgramCodeDirectoryPath = _fixture.CodeDirPath;
                project.ProgramDataDirectoryPath = _fixture.DataDirPath;

                var cts = new CancellationTokenSource();
                var result = await app.BuildNspAsync(type, Path.Combine(_context.TempDirPath, $"test.{type.ToString().ToLower()}"), cts.Token).ConfigureAwait(false);

                Assert.Equal(false, cts.IsCancellationRequested);
                Assert.Equal(NspHandleResult.Ok, result.Result);
                Assert.Equal(100, app.WorkingProgress);
                Assert.True(app.IsChangedTarget);
            }
        }

        [Fact]
        public async Task ReplaceNsp()
        {
            var app = _context.GetInstance<App>();

            using (var project = Project.CreateNew(_context.DiContainer))
            {
                app.Project = project;

                var cts = new CancellationTokenSource();

                var outputPath = Path.Combine(_context.TempDirPath, "test.nsp");
                var inputPath = _fixture.NspFilePath;

                project.Meta.Application.DisplayVersion = "1.0.0";

                var output = await app.SaveNspAsync(outputPath, inputPath, cts.Token).ConfigureAwait(false);

                Assert.Equal(NspHandleResult.Ok, output.Result);
                Assert.False(app.IsChangedTarget);
            }
        }
    }
}
