﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using _3dToolsTestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.G3dTool.Entities;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using TextureCompositor;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Reflection;
using System.IO;
using System.Text;

namespace _3dIntermediateFileTextureCompositorTest
{
    [TestClass]
    public class CompositeTest
    {
        [TestMethod]
        public void TextureCompositorProgramMainTest()
        {
            string sampleRoot = Path.Combine(IoUtility.GetG3dDemoRootPath(), "../G3dSandbox/Resources/TextureCompositionSamples");
            string fscPath = Path.Combine(sampleRoot, "town.fsca");
            using (var dirCreator = new ScopedDirectoryCreator())
            {
#if !DEBUG
                string outputFolder = dirCreator.DirectoryPath;
#else
                string outputFolder = "./";
#endif
                string fsdPath = Path.Combine(outputFolder, "town.fsdb");
                ConvertShaderConfig(fscPath, fsdPath);

                string sourceMaleModelPath = Path.Combine(sampleRoot, "Models/MaleA.fmdb");
                string tempModelFolder = IoUtility.GetTempFolderPath();
                using (var modelRoot = new ScopedDirectoryCreator())
                {
                    string testModelPath = Path.Combine(modelRoot.DirectoryPath, Path.GetFileName(sourceMaleModelPath));

                    // テストデータの作成
                    {
                        var modelFile = IfReadUtility.ReadIntermediateFile(sourceMaleModelPath, IoUtility.GetXsdBasePath());
                        var model = modelFile.GetRootEntity<Model>();

                        // 必要ならモデルの書き換え

                        IfWriteUtility.WriteIntermediateFile(modelFile, testModelPath, IoUtility.GetXsdBasePath());

                        string texFolder = Path.Combine(modelRoot.DirectoryPath, "textures");
                        Directory.CreateDirectory(texFolder);
                        foreach (var file in Directory.EnumerateFiles(Path.Combine(sampleRoot, "Models/textures"), "*.ftxb"))
                        {
                            File.Copy(file, Path.Combine(texFolder, Path.GetFileName(file)));
                        }
                    }

                    string materialEditScriptPath = Path.Combine(IoUtility.GetSdkRootPath(), @"Tests\Tools\Sources\Tests\3dToolsTest\3dIntermediateFileTextureCompositorTest\Resources\ScriptTest.csx");
                    string args = $"{testModelPath} --shader-definition {outputFolder} --output {outputFolder} --script {materialEditScriptPath}";
                    Program.Main(args.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries));

                    Assert.IsTrue(File.Exists(Path.Combine(outputFolder, "MaleA.fmdb")));
                    Assert.IsTrue(File.Exists(Path.Combine(outputFolder, "textures/m_skin__m0.ftxb")));
                }
            }
        }

        [TestMethod]
        public void ExtractCompositeSamplerInfoTest()
        {
            string sampleRoot = Path.Combine(IoUtility.GetG3dDemoRootPath(), "../G3dSandbox/Resources/TextureCompositionSamples");
            string fscPath = Path.Combine(sampleRoot, "town.fsca");
            using (var dirCreator = new ScopedDirectoryCreator())
            {
#if !DEBUG
                string shaderDefRoot = dirCreator.DirectoryPath;
#else
                string shaderDefRoot = sampleRoot;
#endif
                string fsdPath = Path.Combine(shaderDefRoot, "town.fsdb");
                ConvertShaderConfig(fscPath, fsdPath);
                var file = IfReadUtility.ReadIntermediateFile(fsdPath, IoUtility.GetXsdBasePath());
                var shaderDef = file.GetRootEntity<ShaderDefinition>();
                {
                    var info = GlslAnalyzationUtility.ExtractCompositeSamplerInfo(
                        shaderDef,
                        shaderDef.ShadingModels.First(x => x.Name == "town"),
                        new TextureCompositor.Macro[] {
                        new TextureCompositor.Macro() { Name = "ENABLE_MIX_TEX", Value = "1" } });
                    Assert.IsNotNull(info);
                    Assert.AreEqual(1, info.CompositeSamplers.Count);
                    Assert.AreEqual(2, info.CompositeSamplers[0].SourceCount);
                    Assert.AreEqual("_s0", info.CompositeSamplers[0].SourceR.SamplerName);
                    Assert.AreEqual(Channel.R, info.CompositeSamplers[0].SourceR.Channel);
                    Assert.AreEqual("_x0", info.CompositeSamplers[0].SourceA.SamplerName);
                    Assert.AreEqual(Channel.R, info.CompositeSamplers[0].SourceA.Channel);
                }

                // マテリアル入力版のテスト
                {
                    var modelFile = IfReadUtility.ReadIntermediateFile(System.IO.Path.Combine(sampleRoot, "Models/MaleA.fmdb"), IoUtility.GetXsdBasePath());
                    var model = modelFile.GetRootEntity<Model>();

                    // テスト用の設定
                    {
                        foreach (var material in model.Materials)
                        {
                            var option = material.ShaderAssign.ShaderOptions.First(x => x.Id == "enable_mix_tex");
                            option.Value = "0";
                        }

                        var skinMat = model.Materials.First(x => x.Name == "m_skin");
                        skinMat.ShaderAssign.ShaderOptions.First(x => x.Id == "enable_mix_tex").Value = "1";
                    }

                    foreach (var material in model.Materials)
                    {
                        var info = GlslAnalyzationUtility.ExtractCompositeSamplerInfo(
                            shaderDef, material);
                        if (material.Name == "m_skin")
                        {
                            Assert.IsNotNull(info);
                            Assert.AreEqual(1, info.CompositeSamplers.Count);
                            Assert.AreEqual(2, info.CompositeSamplers[0].SourceCount);
                            Assert.AreEqual("_s0", info.CompositeSamplers[0].SourceR.SamplerName);
                            Assert.AreEqual(Channel.R, info.CompositeSamplers[0].SourceR.Channel);
                            Assert.AreEqual("_x0", info.CompositeSamplers[0].SourceA.SamplerName);
                            Assert.AreEqual(Channel.R, info.CompositeSamplers[0].SourceA.Channel);
                        }
                        else
                        {
                            Assert.IsNull(info);
                        }
                    }
                }
            }
        }

        [TestMethod]
        public void PreprocessGlslTest()
        {
            string tmpGlslPath = IoUtility.GetTempFilePath(MethodBase.GetCurrentMethod().Name+".glsl");
            using (ScopedFileCreator mainFile = new ScopedFileCreator("main.glsl"), includeFile = new ScopedFileCreator("include.glsl"))
            {
                StringBuilder mainSource = new StringBuilder();
                StringBuilder includeSource = new StringBuilder();

                includeSource.AppendLine("uniform sampler2D mixTex; // @@ id=\"_m0\"");
                File.WriteAllText(includeFile.FilePath, includeSource.ToString());

                mainSource.AppendLine($"#include <{Path.GetFileName(includeFile.FilePath)}>");
                mainSource.AppendLine($"#if ENABLE_MIX_TEX != 0");
                mainSource.AppendLine($"// @@ sampler_id=\"_m0\" source_r=\"_s0:r\"");
                mainSource.AppendLine($"#endif");
                File.WriteAllText(mainFile.FilePath, mainSource.ToString());

                GlslAnalyzationUtility.ApplyPreprocess(mainFile.FilePath, mainFile.FilePath, new TextureCompositor.Macro[]
                {
                    new TextureCompositor.Macro(){ Name = "ENABLE_MIX_TEX", Value = "1" },
                });

                string expandedSource = File.ReadAllText(mainFile.FilePath);
                Assert.IsTrue(expandedSource.Contains("uniform sampler2D mixTex;"));
                Assert.IsFalse(expandedSource.Contains("#if"));
                Assert.IsTrue(expandedSource.Contains("sampler_id=\"_m0\""));
            }
        }

        [TestMethod]
        public void CompositeSamplerTest()
        {
            string sampleRoot = Path.Combine(IoUtility.GetG3dDemoRootPath(), "../G3dSandbox/Resources/TextureCompositionSamples");

            // テストデータの作成
            var shaderDefFile = ConvertAndReadG3dDemoShaderDefinition("town", MethodBase.GetCurrentMethod().Name);
            var shaderDef = shaderDefFile.GetRootEntity<ShaderDefinition>();
            {
                var shadingModel = shaderDef.ShadingModels.First(x => x.Name == "town");
                var samplerVar = new SamplerVar()
                {
                    Id = "_m0",
                };
                samplerVar.FragmentSymbol.Name = "mergeTex";
                shadingModel.SamplerVars.Add(samplerVar);

                // テスト用のツールデータ付与
                CompositeSamplerInfo compInfo = new CompositeSamplerInfo();
                compInfo.CompositeSamplers.Add(new CompositeSampler()
                {
                    SamplerName = "_m0",
                    SourceR = new CompositeSource() { SamplerName = "_s0", Channel = Channel.R },
                    SourceA = new CompositeSource() { SamplerName = "_x0", Channel = Channel.R },
                });
                shaderDef.ToolData.AddToolDataObject(compInfo);
                IfWriteUtility.WriteIntermediateFile(shaderDefFile, IoUtility.GetXsdBasePath());
            }

            var modelFile = IfReadUtility.ReadIntermediateFile(System.IO.Path.Combine(sampleRoot, "Models/MaleA.fmdb"), IoUtility.GetXsdBasePath());
            var model = modelFile.GetRootEntity<Model>();
            var targetMaterial = model.Materials.First(x => x.Name == "m_skin");
            {
                // テスト用のツールデータ付与
                {
                    SourceSamplerInfo samplerInfo = new SourceSamplerInfo();
                    samplerInfo.SourceSamplers.Add(new SourceSampler()
                    {
                        SamplerName = "_s0",
                        ResizeX = 512,
                        ResizeY = 512,
                    });
                    samplerInfo.SourceSamplers.Add(new SourceSampler()
                    {
                        SamplerName = "_x0",
                        ResizeX = 256,
                        ResizeY = 256,
                    });
                    targetMaterial.ToolData.AddToolDataObject(samplerInfo);
                }
            }

            // 合成
            TextureSamplerCompositor compositor = new TextureSamplerCompositor();
            compositor.OutputFolderPath = IoUtility.GetTempFolderPath();
            compositor.TextureReferencePaths.Add(
                System.IO.Path.Combine(
                    System.IO.Path.GetDirectoryName(modelFile.Path), "textures"));

            compositor.CompositeMaterialSamplers(
                targetMaterial,
                shaderDef.ToolData.FindToolDataObject<CompositeSamplerInfo>(),
                shaderDef);

            // サンプラーが合成されていることをチェック
            var compSampler = targetMaterial.Samplers.First(x => x.Name == "_m0");
            {
                Assert.IsFalse(targetMaterial.Samplers.Any(x => x.Name == "_s0"));
                Assert.IsFalse(targetMaterial.Samplers.Any(x => x.Name == "_x0"));

                var samplerAssign = targetMaterial.ShaderAssign.SamplerAssigns.First(x => x.Id == "_m0");
                Assert.AreEqual("_m0", samplerAssign.SamplerName);
            }

            // 合成後のテクスチャーをチェック
            {
                var compTexFile = IfReadUtility.ReadIntermediateFile($"{Path.Combine(compositor.OutputFolderPath, "textures", compSampler.TexName)}.ftxb", IoUtility.GetXsdBasePath());
                var compTex = compTexFile.GetRootEntity<Texture>();
                Assert.IsNotNull(compTex);
            }
        }

        [TestMethod]
        public void CompositeTextureTest()
        {
            string sampleRoot = Path.Combine(IoUtility.GetG3dDemoRootPath(), "../G3dSandbox/Resources/TextureCompositionSamples");
            CompositeTextureArgs args = new CompositeTextureArgs();
            args.DestinationTexPath = "composited.ftxb";
            args.Width = 1024;
            args.Height = 1024;
            args.SourceTextureR = new SourceTexture()
            {
                TexturePath = IoUtility.NormalizeFilePath(System.IO.Path.Combine(sampleRoot, "Models/textures/MaleA_skin_spm.ftxb")),
                Channel = Channel.R
            };
            args.SourceTextureG = new SourceTexture()
            {
                TexturePath = IoUtility.NormalizeFilePath(System.IO.Path.Combine(sampleRoot, "Models/textures/MaleA_skin_AO.ftxb")),
                Channel = Channel.R
            };
            Assert.IsTrue(File.Exists(args.SourceTextureR.TexturePath));
            Assert.IsTrue(File.Exists(args.SourceTextureG.TexturePath));
            TextureSamplerCompositor.CompositeTexture(args);
        }

        private IntermediateFile ConvertAndReadG3dDemoShaderDefinition(string shaderArchiveName, string tempFileNameSuffix, bool keepsTempFile = false)
        {
            IntermediateFile file;
            string outputFolder = IoUtility.GetTempFolderPath();
            Directory.CreateDirectory(outputFolder);
            string fsdPath =  Path.Combine(outputFolder, $"{shaderArchiveName}.fsdb");
            string fscPath = System.IO.Path.Combine(IoUtility.GetG3dDemoRootPath(), $"Resources/shader/{shaderArchiveName}.fsca");
            ConvertShaderConfig(fscPath, fsdPath);
            file = IfReadUtility.ReadIntermediateFile(fsdPath, IoUtility.GetXsdBasePath());
            if (!keepsTempFile)
            {
                System.IO.File.Delete(fsdPath);
            }

            return file;
        }

        private static void ConvertShaderConfig(string fscPath, string fsdPath)
        {
            string shaderConverterPath = System.IO.Path.Combine(IoUtility.Get3dToolsRootPath(), "3dShaderConverter.exe");
            ProcessExecutor.Execute(
                shaderConverterPath,
                $"{fscPath} --output {fsdPath}");
        }

        private static string GetTownMaleRoot()
        {
            return System.IO.Path.Combine(IoUtility.GetG3dDemoRootPath(), "Resources/town/MaleA");
        }
    }
}
