﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using _3dIntermediateFileShaderVariationGenerator;
using _3dToolsTestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.G3dTool.Entities;
using Nintendo.G3dTool.Extensions;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Reflection;

namespace _3dIntermediateFileShaderVariationGeneratorTest
{
    [TestClass]
    public class MainFunctionalityTest
    {
        /// <summary>
        /// シェーダーバリエーションジェネレーターを使って、TestBinaries 以下の fsd, fmd, fmt から fsv を作成するテストです。
        /// </summary>
        [TestMethod]
        public void TestExecuteFilterSample()
        {
            string assemblyFolder = System.IO.Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
            string sdkRoot = IoUtility.GetSdkRootPath();
            string testResourcesFolder = System.IO.Path.Combine(sdkRoot, "Externals/TestBinaries/G3d/ShaderVariationGenerator/Resources");
            string referenceResourcesFolder = System.IO.Path.Combine(sdkRoot, "Externals/TestBinaries/G3d/ShaderVariationGenerator/ReferenceResults");

            Dictionary<string, string> testArgumentList = new Dictionary<string, string>();
            testArgumentList.Add(
                "demoBasic.fsva",
                $"{testResourcesFolder} --shader-definition {testResourcesFolder}/demo.fsdb --output demoBasic.fsva");

            testArgumentList.Add(
                "demoScript.fsva",
                $"{testResourcesFolder} --shader-definition {testResourcesFolder}/demo.fsdb --script {assemblyFolder}/Resources/Script.py --output demoScript.fsva");

            testArgumentList.Add(
                "townBasic.fsva",
                $"{testResourcesFolder} --shader-definition {testResourcesFolder}/town.fsdb --output townBasic.fsva");

            testArgumentList.Add(
                "townScript.fsva",
                $"{testResourcesFolder} --shader-definition {testResourcesFolder}/town.fsdb --script {assemblyFolder}/Resources/Script.py --output townScript.fsva");

            foreach (var args in testArgumentList)
            {
                string fsvFile = args.Key;
                string outputFsvPath = System.IO.Path.Combine(assemblyFolder, fsvFile);

                // 実行
                try
                {
                    Console.WriteLine($"Argument: {args.Value}");
                    Program.Execute(args.Value.Split(' '));

                    // リファレンス結果と比較する
                    string referenceFsvPath = System.IO.Path.Combine(referenceResourcesFolder, fsvFile);
                    Assert.IsTrue(CompareFileBinary(outputFsvPath, referenceFsvPath));
                }
                finally
                {
                    if (System.IO.File.Exists(outputFsvPath))
                    {
                        System.IO.File.Delete(outputFsvPath);
                    }
                }
            }
        }

        /// <summary>
        /// シェーダーバリエーション生成の簡易テストです。
        /// </summary>
        [TestMethod]
        public void CreateShaderVariationFileTest()
        {
            string shaderArchiveName = "shaderArchive";
            string shadingModelNameA = "shadingModelA";
            string shadingModelNameB = "shadingModelB";
            string staticOptionName = "staticOption";
            string dynamicOptionName = "dynamicOption";
            ShaderOptionChoice dynamicOptionChoice = new ShaderOptionChoice(0, 3);
            var inputModels = new List<Model>();
            {
                var model = new Model();
                var bone = model.Skeleton.RootBone = new Bone() { Name = "bone" };
                var material = new Material() { Name = "matA" };
                material.ShaderAssign.ShaderArchive = shaderArchiveName;
                material.ShaderAssign.ShadingModel = shadingModelNameA;
                material.ShaderAssign.ShaderOptions.Add(new ShaderOption()
                {
                    Id = staticOptionName,
                    Value = "1",
                });
                model.Materials.Add(material);
                var shape = model.CreateShape(material, bone, new int[] { 0, 0, 0});
                shape.ShapeInfo.MatName = "matA";
                inputModels.Add(model);
            }
            {
                var model = new Model();
                var bone = model.Skeleton.RootBone = new Bone() { Name = "bone" };
                var material = new Material() { Name = "matB" };
                material.ShaderAssign.ShaderArchive = shaderArchiveName;
                material.ShaderAssign.ShadingModel = shadingModelNameA;
                material.ShaderAssign.ShaderOptions.Add(new ShaderOption()
                {
                    Id = staticOptionName,
                    Value = "3",
                });
                model.Materials.Add(material);
                var shape = model.CreateShape(material, bone, new int[] { 0, 0, 0 });
                inputModels.Add(model);
            }

            var shaderDefinition = new ShaderDefinition();
            {
                var shadingModel = new ShadingModel();
                shadingModel.Name = shadingModelNameA;
                shadingModel.OptionVars.Add(new OptionVar()
                {
                    Id = staticOptionName,
                    Default = "0",
                    Type = option_var_typeType.@static,
                });
                shadingModel.OptionVars.Last().Choice.SetIntRange(0, 3);

                shadingModel.OptionVars.Add(new OptionVar()
                {
                    Id = dynamicOptionName,
                    Default = "0",
                    Type = option_var_typeType.dynamic,
                });
                shadingModel.OptionVars.Last().Choice.DeepCopyFrom(dynamicOptionChoice);

                shaderDefinition.ShadingModels.Add(shadingModel);
            }
            {
                var shadingModel = new ShadingModel();
                shadingModel.Name = shadingModelNameB;
                shadingModel.OptionVars.Add(new OptionVar()
                {
                    Id = staticOptionName,
                    Default = "0",
                    Type = option_var_typeType.@static,
                });
                shadingModel.OptionVars.Last().Choice.SetIntRange(0, 3);

                shadingModel.OptionVars.Add(new OptionVar()
                {
                    Id = dynamicOptionName,
                    Default = "0",
                    Type = option_var_typeType.dynamic,
                });
                shadingModel.OptionVars.Last().Choice.DeepCopyFrom(dynamicOptionChoice);

                shaderDefinition.ShadingModels.Add(shadingModel);
            }

            var shaderVariationFile = Program.CreateShaderVariationFile(inputModels, shaderArchiveName, shaderDefinition);
            var shaderVariation = shaderVariationFile.GetRootEntity<ShaderVariation>();

            Assert.AreEqual(shaderArchiveName, shaderVariation.ShaderVariationInfo.ShaderArchive);
            Assert.AreEqual(1, shaderVariation.TargetShaders.Count);
            Assert.AreEqual(shadingModelNameA, shaderVariation.TargetShaders[0].ShadingModelName);
            Assert.AreEqual(2, shaderVariation.TargetShaders[0].ShaderPrograms.Count);
            {
                var program = shaderVariation.TargetShaders[0].ShaderPrograms[0];
                Assert.AreEqual(2, program.Options.Count);
                Assert.AreEqual(staticOptionName, program.Options[0].Id);
                Assert.AreEqual("1", program.Options[0].Choice.CreateSerializableData());
                Assert.AreEqual(dynamicOptionName, program.Options[1].Id);
                Assert.AreEqual(dynamicOptionChoice.CreateSerializableData(), program.Options[1].Choice.CreateSerializableData());
            }
            {
                var program = shaderVariation.TargetShaders[0].ShaderPrograms[1];
                Assert.AreEqual(2, program.Options.Count);
                Assert.AreEqual("3", program.Options[0].Choice.CreateSerializableData());
                Assert.AreEqual(staticOptionName, program.Options[0].Id);
                Assert.AreEqual(dynamicOptionName, program.Options[1].Id);
                Assert.AreEqual(dynamicOptionChoice.CreateSerializableData(), program.Options[1].Choice.CreateSerializableData());
            }
        }

        private bool CompareFileBinary(string filePath, string referencefilePath)
        {
            System.IO.FileStream file = new System.IO.FileStream(filePath, System.IO.FileMode.Open);
            System.IO.FileStream referenceFile = new System.IO.FileStream(referencefilePath, System.IO.FileMode.Open);
            int byte1;
            int byte2;
            bool result = false;

            try
            {
                if (file.Length == referenceFile.Length)
                {
                    do
                    {
                        byte1 = file.ReadByte();
                        byte2 = referenceFile.ReadByte();
                    }
                    while ((byte1 == byte2) && (byte1 != -1));

                    if (byte1 == byte2)
                    {
                        result = true;
                    }
                }
            }
            catch
            {
            }
            finally
            {
                file.Close();
                referenceFile.Close();
            }

            return result;
        }
    }
}
