﻿using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using System.Collections.Generic;
using System.Linq;
using System.Diagnostics;
using _3dToolsTestUtility;

namespace _3dIntermediateFileLibraryTest
{
    /// <summary>
    /// シェイプ圧縮のテストです。
    /// </summary>
    [TestClass]
    public class CompressShapeTest
    {
        /// <summary>
        /// シンプルなシェイプ圧縮のテストです。
        /// </summary>
        [TestMethod]
        public void TestCompressShapeSimple()
        {
            IfModelShapeCompressor shapeCompressor = new IfModelShapeCompressor(false);
            var model = ModelDataCreationUtility.CreateEmptyModel();

            ModelDataCreationUtility.AddBone(model, "boneRoot", string.Empty, -1, -1, new float[] { 0.0f, 0.0f, 0.0f }, new float[] { 1.0f, 1.0f, 1.0f });

            // shapeA index stream
            ModelDataCreationUtility.AddStream(model, "0 1 2", stream_typeType.@int, 3, 3);
            // shapeB index stream
            ModelDataCreationUtility.AddStream(model, "2 0 1", stream_typeType.@int, 3, 3);
            // shapeC index stream
            ModelDataCreationUtility.AddStream(model, "1 2 0", stream_typeType.@int, 3, 3);

            // shapeA pos
            ModelDataCreationUtility.AddStream(model, "1.1 1.2 1.3    1.1 1.2 1.3    1.1 1.2 1.3", stream_typeType.@float, 9, 3);
            // shapeB pos
            ModelDataCreationUtility.AddStream(model, "2.1 2.2 2.3    2.1 2.2 2.3    2.1 2.2 2.3", stream_typeType.@float, 9, 3);
            // shapeC pos
            ModelDataCreationUtility.AddStream(model, "3.1 3.2 3.3    3.1 3.2 3.3    3.1 3.2 3.3", stream_typeType.@float, 9, 3);

            ModelDataCreationUtility.AddVertexAttribute(model, new VertexAttr[]
                {
                    new VertexAttr()
                    {
                        Name = "_p0",
                        Hint = "position0",
                        StreamIndex = 3,
                        Type = vtx_attrib_typeType.float3,
                    },
                });

            ModelDataCreationUtility.AddVertexAttribute(model, new VertexAttr[]
                {
                    new VertexAttr()
                    {
                        Name = "_p0",
                        Hint = "position0",
                        StreamIndex = 4,
                        Type = vtx_attrib_typeType.float3,
                    },
                });

            ModelDataCreationUtility.AddVertexAttribute(model, new VertexAttr[]
                {
                    new VertexAttr()
                    {
                        Name = "_p0",
                        Hint = "position0",
                        StreamIndex = 5,
                        Type = vtx_attrib_typeType.float3,
                    },
                });

            shapeType shapeA = ModelDataCreationUtility.AddShape(model, "shapeA", "matA", "boneRoot", 0, 0);
            shapeType shapeB = ModelDataCreationUtility.AddShape(model, "shapeB", "matA", "boneRoot", 1, 1);
            shapeType shapeC = ModelDataCreationUtility.AddShape(model, "shapeC", "matA", "boneRoot", 2, 2);
            shapeType shapeD = ModelDataCreationUtility.AddShape(model, "shapeD", "matA", "boneRoot", 2, 2);

            // アルファベット順にソートされることをテストするためにoriginal_bone_nameを適当に入れ替え
            shapeA.shape_info.original_bone_name = "boneB";
            shapeB.shape_info.original_bone_name = "boneC";
            shapeC.shape_info.original_bone_name = "boneA";
            shapeD.shape_info.original_bone_name = "boneA";
            shapeA.shape_info.original_material_name = "matD";
            shapeB.shape_info.original_material_name = "matA";
            shapeC.shape_info.original_material_name = "matC";
            shapeD.shape_info.original_material_name = "matB";

            List<G3dStream> streams = new List<G3dStream>();
            foreach (var stream in model.stream_array.stream)
            {
                streams.Add(new G3dStream(stream));
            }

            shapeCompressor.Compress(model, streams);

            Assert.AreEqual(1, model.shape_array.shape.Length);
            Assert.AreEqual("boneRoot", model.shape_array.shape[0].shape_info.bone_name);
            Assert.AreEqual("matA", model.shape_array.shape[0].shape_info.mat_name);

            // original_bone_name、original_material_name はカンマ区切りで複数設定される
            Assert.AreEqual("boneA,boneB,boneC", model.shape_array.shape[0].shape_info.original_bone_name);
            Assert.AreEqual("matA,matB,matC,matD", model.shape_array.shape[0].shape_info.original_material_name);
        }

        /// <summary>
        /// --ignore-skinning-count のテストです。
        /// </summary>
        [TestMethod]
        public void TestIgnoreSkinningCount()
        {
            bool ignoreSkininngCount = true;
            IfModelShapeCompressor shapeCompressor = new IfModelShapeCompressor(ignoreSkininngCount);

            var model = ModelDataCreationUtility.CreateEmptyModel();

            float[] boneTranslate = new float[] { 0.1f, 0.2f, 0.3f };
            float[] boneScale = new float[] { 0.1f, 0.2f, 0.3f };
            ModelDataCreationUtility.AddBone(model, "boneRoot", string.Empty, -1, -1, new float[]{ 0.0f, 0.0f, 0.0f }, new float[] { 1.0f, 1.0f, 1.0f });
            ModelDataCreationUtility.AddBone(model, "boneA", "boneRoot", 0, 2, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneB", "boneRoot", -1, 3, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneC", "boneRoot", 1, -1, boneTranslate, boneScale);

            // shapeA _i0
            ModelDataCreationUtility.AddStream(model, "3 3 3", stream_typeType.@int, 3, 1);
            // shapeB _i0
            ModelDataCreationUtility.AddStream(model, "0 1    0 1    0 1", stream_typeType.@int, 6, 2);
            // shapeB _w0
            ModelDataCreationUtility.AddStream(model, "0.3 0.7    0.4 0.6    0.5 0.5", stream_typeType.@float, 6, 2);
            // shapeC _i0
            ModelDataCreationUtility.AddStream(model, "2 2 2", stream_typeType.@int, 3, 1);

            // shapeA index stream
            ModelDataCreationUtility.AddStream(model, "0 1 2", stream_typeType.@int, 3, 3);
            // shapeB index stream
            ModelDataCreationUtility.AddStream(model, "2 0 1", stream_typeType.@int, 3, 3);
            // shapeC index stream
            ModelDataCreationUtility.AddStream(model, "1 2 0", stream_typeType.@int, 3, 3);

            // shapeA pos
            ModelDataCreationUtility.AddStream(model, "1.1 1.2 1.3    1.1 1.2 1.3    1.1 1.2 1.3", stream_typeType.@float, 9, 3);
            // shapeB pos
            ModelDataCreationUtility.AddStream(model, "2.1 2.2 2.3    2.1 2.2 2.3    2.1 2.2 2.3", stream_typeType.@float, 9, 3);
            // shapeC pos
            ModelDataCreationUtility.AddStream(model, "3.1 3.2 3.3    3.1 3.2 3.3    3.1 3.2 3.3", stream_typeType.@float, 9, 3);

            ModelDataCreationUtility.AddVertexAttribute(model, new VertexAttr[]
                {
                    new VertexAttr()
                    {
                        Name = "_p0",
                        Hint = "position0",
                        StreamIndex = 7,
                        Type = vtx_attrib_typeType.float3,
                    },
                    new VertexAttr()
                    {
                        Name = "_i0",
                        Hint = "blendindex0",
                        StreamIndex = 0,
                        Type = vtx_attrib_typeType.@int,
                    },
                });

            ModelDataCreationUtility.AddVertexAttribute(model, new VertexAttr[]
                {
                    new VertexAttr()
                    {
                        Name = "_p0",
                        Hint = "position0",
                        StreamIndex = 8,
                        Type = vtx_attrib_typeType.float3,
                    },
                    new VertexAttr()
                    {
                        Name = "_i0",
                        Hint = "blendindex0",
                        StreamIndex = 1,
                        Type = vtx_attrib_typeType.uint2,
                    },
                    new VertexAttr()
                    {
                        Name = "_w0",
                        Hint = "blendweight0",
                        StreamIndex = 2,
                        Type = vtx_attrib_typeType.float2,
                    },
                });

            ModelDataCreationUtility.AddVertexAttribute(model, new VertexAttr[]
                {
                    new VertexAttr()
                    {
                        Name = "_p0",
                        Hint = "position0",
                        StreamIndex = 9,
                        Type = vtx_attrib_typeType.float3,
                    },
                    new VertexAttr()
                    {
                        Name = "_i0",
                        Hint = "blendindex0",
                        StreamIndex = 3,
                        Type = vtx_attrib_typeType.@int,
                    },
                });

            ModelDataCreationUtility.AddShape(model, "shapeA", "matA", "boneRoot", 0, 4);
            ModelDataCreationUtility.AddShape(model, "shapeB", "matA", "boneRoot", 1, 5);
            ModelDataCreationUtility.AddShape(model, "shapeC", "matB", "boneRoot", 2, 6);

            List<G3dStream> streams = new List<G3dStream>();
            foreach (var stream in model.stream_array.stream)
            {
                streams.Add(new G3dStream(stream));
            }

            shapeCompressor.Compress(model, streams);

            // 正しく結合されていることをテスト
            Assert.AreEqual(2, model.shape_array.shape.Length);
            Assert.AreEqual(8, streams.Count);
            int streamIndex = 0;
            {
                float[] correctData = new float[]
                {
                    1.1f * boneScale[0] + boneTranslate[0], 1.2f * boneScale[1] + boneTranslate[1], 1.3f * boneScale[2] + boneTranslate[2],
                    1.1f * boneScale[0] + boneTranslate[0], 1.2f * boneScale[1] + boneTranslate[1], 1.3f * boneScale[2] + boneTranslate[2],
                    1.1f * boneScale[0] + boneTranslate[0], 1.2f * boneScale[1] + boneTranslate[1], 1.3f * boneScale[2] + boneTranslate[2],
                    2.1f, 2.2f, 2.3f,
                    2.1f, 2.2f, 2.3f,
                    2.1f, 2.2f, 2.3f,
                };
                Assert.AreEqual(correctData.Length, streams[streamIndex].FloatData.Count);
                for (int i = 0; i < correctData.Length; ++i)
                {
                    Assert.AreEqual(correctData[i], streams[streamIndex].FloatData[i]);
                }
            }

            ++streamIndex;
            {
                // マトリックスインデックス自体が変換されているので、ストリームもそれに追従しているかテスト
                int[] correctBlendIndices = new int[]
                {
                    1, 1,
                    1, 1,
                    1, 1,
                    0, 2,
                    0, 2,
                    0, 2,
                };
                Assert.AreEqual(correctBlendIndices.Length, streams[streamIndex].IntData.Count);
                for (int i = 0; i < correctBlendIndices.Length; ++i)
                {
                    Assert.AreEqual(correctBlendIndices[i], streams[streamIndex].IntData[i]);
                }
            }

            ++streamIndex;
            {
                float[] correctBlendWeights = new float[]
                {
                    1.0f, 0.0f,
                    1.0f, 0.0f,
                    1.0f, 0.0f,
                    0.3f, 0.7f,
                    0.4f, 0.6f,
                    0.5f, 0.5f,
                };
                Assert.AreEqual(correctBlendWeights.Length, streams[streamIndex].FloatData.Count);
                for (int i = 0; i < correctBlendWeights.Length; ++i)
                {
                    Assert.AreEqual(correctBlendWeights[i], streams[streamIndex].FloatData[i]);
                }
            }

            ++streamIndex;
            {
                float[] correctData = new float[]
                {
                    3.1f * boneScale[0] + boneTranslate[0], 3.2f * boneScale[1] + boneTranslate[1], 3.3f * boneScale[2] + boneTranslate[2],
                    3.1f * boneScale[0] + boneTranslate[0], 3.2f * boneScale[1] + boneTranslate[1], 3.3f * boneScale[2] + boneTranslate[2],
                    3.1f * boneScale[0] + boneTranslate[0], 3.2f * boneScale[1] + boneTranslate[1], 3.3f * boneScale[2] + boneTranslate[2],
                };
                Assert.AreEqual(correctData.Length, streams[streamIndex].FloatData.Count);
                for (int i = 0; i < correctData.Length; ++i)
                {
                    Assert.AreEqual(correctData[i], streams[streamIndex].FloatData[i]);
                }
            }

            ++streamIndex;
            {
                int[] correctBlendIndices = new int[]
                {
                    0, 0,
                    0, 0,
                    0, 0,
                };
                Assert.AreEqual(correctBlendIndices.Length, streams[streamIndex].IntData.Count);
                for (int i = 0; i < correctBlendIndices.Length; ++i)
                {
                    Assert.AreEqual(correctBlendIndices[i], streams[streamIndex].IntData[i]);
                }
            }

            ++streamIndex;
            {
                int[] correctDrawIndices = new int[]
                {
                    0, 1, 2,
                    5, 3, 4,
                };
                Assert.AreEqual(correctDrawIndices.Length, streams[streamIndex].IntData.Count);
                for (int i = 0; i < correctDrawIndices.Length; ++i)
                {
                    Assert.AreEqual(correctDrawIndices[i], streams[streamIndex].IntData[i]);
                }
            }

            ++streamIndex;
            {
                int[] correctDrawIndices = new int[]
                {
                    1, 2, 0,
                };
                Assert.AreEqual(correctDrawIndices.Length, streams[streamIndex].IntData.Count);
                for (int i = 0; i < correctDrawIndices.Length; ++i)
                {
                    Assert.AreEqual(correctDrawIndices[i], streams[streamIndex].IntData[i]);
                }
            }

            ++streamIndex;
            {
                float[] correctData = new float[]
                {
                    1.0f, 0.0f,
                    1.0f, 0.0f,
                    1.0f, 0.0f,
                };
                Assert.AreEqual(correctData.Length, streams[streamIndex].FloatData.Count);
                for (int i = 0; i < correctData.Length; ++i)
                {
                    Assert.AreEqual(correctData[i], streams[streamIndex].FloatData[i]);
                }
            }

            // ボーンのマトリックスインデックスが正しく変換されているかをテスト
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneA"));
                Assert.AreEqual(0, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneB"));
                Assert.AreEqual(1, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneC"));
                Assert.AreEqual(2, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
        }

        /// <summary>
        /// --ignore-skinning-count でスキニング数 8 とリジッドスキニングのシェイプを圧縮するテストです。
        /// </summary>
        [TestMethod]
        public void TestCompress8SkinningCountShapeWithIgnoringSkinningCount()
        {
            bool ignoreSkininngCount = true;
            IfModelShapeCompressor shapeCompressor = new IfModelShapeCompressor(ignoreSkininngCount);

            var model = ModelDataCreationUtility.CreateEmptyModel();

            float[] boneTranslate = new float[] { 0.1f, 0.2f, 0.3f };
            float[] boneScale = new float[] { 0.1f, 0.2f, 0.3f };
            ModelDataCreationUtility.AddBone(model, "boneRoot", string.Empty, -1, -1, new float[] { 0.0f, 0.0f, 0.0f }, new float[] { 1.0f, 1.0f, 1.0f });
            ModelDataCreationUtility.AddBone(model, "boneA", "boneRoot", -1, 8, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneB", "boneRoot", 0, -1, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneC", "boneRoot", 1, -1, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneD", "boneRoot", 2, -1, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneE", "boneRoot", 3, -1, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneF", "boneRoot", 4, -1, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneG", "boneRoot", 5, -1, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneH", "boneRoot", 6, -1, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneI", "boneRoot", 7, -1, boneTranslate, boneScale);

            // shapeA _i0
            ModelDataCreationUtility.AddStream(model, "8 8 8", stream_typeType.@int, 3, 1);

            // shapeB _i0
            ModelDataCreationUtility.AddStream(model, "0 1 2 3    0 1 2 3    0 1 2 3", stream_typeType.@int, 12, 4);
            // shapeB _w0
            ModelDataCreationUtility.AddStream(model, "0.1 0.2 0.1 0.2    0.1 0.1 0.1 0.1     0.1 0.2 0.1 0.2", stream_typeType.@float, 12, 4);
            // shapeB _i1
            ModelDataCreationUtility.AddStream(model, "4 5 6 7    4 5 6 7    4 5 6 7", stream_typeType.@int, 12, 4);
            // shapeB _w1
            ModelDataCreationUtility.AddStream(model, "0.1 0.1 0.1 0.1    0.1 0.2 0.1 0.2    0.1 0.1 0.1 0.1", stream_typeType.@float, 12, 4);

            // shapeA index stream
            ModelDataCreationUtility.AddStream(model, "0 1 2", stream_typeType.@int, 3, 3);
            // shapeB index stream
            ModelDataCreationUtility.AddStream(model, "2 0 1", stream_typeType.@int, 3, 3);

            ModelDataCreationUtility.AddVertexAttribute(model, new VertexAttr[]
                {
                    new VertexAttr()
                    {
                        Name = "_i0",
                        Hint = "blendindex0",
                        StreamIndex = 0,
                        Type = vtx_attrib_typeType.@int,
                    },
                });

            ModelDataCreationUtility.AddVertexAttribute(model, new VertexAttr[]
                {
                    new VertexAttr()
                    {
                        Name = "_i0",
                        Hint = "blendindex0",
                        StreamIndex = 1,
                        Type = vtx_attrib_typeType.uint4,
                    },
                    new VertexAttr()
                    {
                        Name = "_w0",
                        Hint = "blendweight0",
                        StreamIndex = 2,
                        Type = vtx_attrib_typeType.float4,
                    },
                    new VertexAttr()
                    {
                        Name = "_i1",
                        Hint = "blendindex1",
                        StreamIndex = 3,
                        Type = vtx_attrib_typeType.uint4,
                    },
                    new VertexAttr()
                    {
                        Name = "_w1",
                        Hint = "blendweight1",
                        StreamIndex = 4,
                        Type = vtx_attrib_typeType.float4,
                    },
                });


            var shapeA = ModelDataCreationUtility.AddShape(model, "shapeA", "matA", "boneRoot", 0, 5);
            shapeA.shape_info.original_bone_name = string.Empty;
            var shapeB = ModelDataCreationUtility.AddShape(model, "shapeB", "matA", "boneRoot", 1, 6);
            shapeB.shape_info.original_bone_name = string.Empty;

            List<G3dStream> streams = new List<G3dStream>();
            foreach (var stream in model.stream_array.stream)
            {
                streams.Add(new G3dStream(stream));
            }

            shapeCompressor.Compress(model, streams);

            // ボーンのマトリックスインデックスが正しく変換されているかをテスト
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneA"));
                Assert.AreEqual(0, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneB"));
                Assert.AreEqual(1, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneC"));
                Assert.AreEqual(2, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneD"));
                Assert.AreEqual(3, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneE"));
                Assert.AreEqual(4, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneF"));
                Assert.AreEqual(5, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneG"));
                Assert.AreEqual(6, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneH"));
                Assert.AreEqual(7, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneI"));
                Assert.AreEqual(8, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }

            // 正しく結合されていることをテスト
            Assert.AreEqual(1, model.shape_array.shape.Length);
            Assert.AreEqual(5, streams.Count);
            int streamIndex = 0;
            {
                int[] correctBlendIndices = new int[]
                {
                    0, 0, 0, 0,
                    0, 0, 0, 0,
                    0, 0, 0, 0,
                    1, 2, 3, 4,
                    1, 2, 3, 4,
                    1, 2, 3, 4,
                };
                Assert.AreEqual(correctBlendIndices.Length, streams[streamIndex].IntData.Count);
                for (int i = 0; i < correctBlendIndices.Length; ++i)
                {
                    Assert.AreEqual(correctBlendIndices[i], streams[streamIndex].IntData[i]);
                }
            }

            ++streamIndex;
            {
                float[] correctBlendWeights = new float[]
                {
                    1.0f, 0.0f, 0.0f, 0.0f,
                    1.0f, 0.0f, 0.0f, 0.0f,
                    1.0f, 0.0f, 0.0f, 0.0f,
                    0.1f, 0.2f, 0.1f, 0.2f,
                    0.1f, 0.1f, 0.1f, 0.1f,
                    0.1f, 0.2f, 0.1f, 0.2f,
                };
                Assert.AreEqual(correctBlendWeights.Length, streams[streamIndex].FloatData.Count);
                for (int i = 0; i < correctBlendWeights.Length; ++i)
                {
                    Assert.AreEqual(correctBlendWeights[i], streams[streamIndex].FloatData[i]);
                }
            }

            ++streamIndex;
            {
                int[] correctBlendIndices = new int[]
                {
                    0, 0, 0, 0,
                    0, 0, 0, 0,
                    0, 0, 0, 0,
                    5, 6, 7, 8,
                    5, 6, 7, 8,
                    5, 6, 7, 8,
                };
                Assert.AreEqual(correctBlendIndices.Length, streams[streamIndex].IntData.Count);
                for (int i = 0; i < correctBlendIndices.Length; ++i)
                {
                    Assert.AreEqual(correctBlendIndices[i], streams[streamIndex].IntData[i]);
                }
            }

            ++streamIndex;
            {
                float[] correctBlendWeights = new float[]
                {
                    0.0f, 0.0f, 0.0f, 0.0f,
                    0.0f, 0.0f, 0.0f, 0.0f,
                    0.0f, 0.0f, 0.0f, 0.0f,
                    0.1f, 0.1f, 0.1f, 0.1f,
                    0.1f, 0.2f, 0.1f, 0.2f,
                    0.1f, 0.1f, 0.1f, 0.1f,
                };
                Assert.AreEqual(correctBlendWeights.Length, streams[streamIndex].FloatData.Count);
                for (int i = 0; i < correctBlendWeights.Length; ++i)
                {
                    Assert.AreEqual(correctBlendWeights[i], streams[streamIndex].FloatData[i]);
                }
            }
        }

        /// <summary>
        /// 連続するシェイプのスキニング方式を揃えるテストです。
        /// </summary>
        [TestMethod]
        public void TestAlignSkinningTypeForContinuousShape()
        {
            var model = ModelDataCreationUtility.CreateEmptyModel();

            List<G3dStream> streams = new List<G3dStream>();

            float[] boneTranslate = new float[] { 0.1f, 0.2f, 0.3f };
            float[] boneScale = new float[] { 0.1f, 0.2f, 0.3f };
            ModelDataCreationUtility.AddBone(model, "boneRoot", string.Empty, -1, -1, new float[] { 0.0f, 0.0f, 0.0f }, new float[] { 1.0f, 1.0f, 1.0f });
            ModelDataCreationUtility.AddBone(model, "boneA", "boneRoot", -1, 8, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneB", "boneRoot", 0, -1, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneC", "boneRoot", 1, -1, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneD", "boneRoot", 2, -1, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneE", "boneRoot", 3, -1, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneF", "boneRoot", 4, -1, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneG", "boneRoot", 5, -1, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneH", "boneRoot", 6, -1, boneTranslate, boneScale);
            ModelDataCreationUtility.AddBone(model, "boneI", "boneRoot", 7, -1, boneTranslate, boneScale);

            var blendIndexStreamShapeA = new G3dStream(new int[]
                {
                    8,
                    8,
                    8,
                }, 1);
            streams.Add(blendIndexStreamShapeA);
            var blendIndexStream0ShapeB = new G3dStream(new int[]
                {
                    0, 1, 2, 3,
                    0, 1, 2, 3,
                    0, 1, 2, 3,
                }, 4);
            streams.Add(blendIndexStream0ShapeB);
            var blendIndexStream1ShapeB = new G3dStream(new int[]
                {
                    4, 5, 6, 7,
                    4, 5, 6, 7,
                    4, 5, 6, 7,
                }, 4);
            streams.Add(blendIndexStream1ShapeB);
            var skinningWeightStream0ShapeB = new G3dStream(new float[]
                {
                    0.1f, 0.2f, 0.1f, 0.2f,
                    0.1f, 0.1f, 0.1f, 0.1f,
                    0.1f, 0.2f, 0.1f, 0.2f,
                }, 4);
            streams.Add(skinningWeightStream0ShapeB);
            var skinningWeightStream1ShapeB = new G3dStream(new float[]
                {
                    0.1f, 0.2f, 0.1f, 0.2f,
                    0.1f, 0.1f, 0.1f, 0.1f,
                    0.1f, 0.2f, 0.1f, 0.2f,
                }, 4);
            streams.Add(skinningWeightStream1ShapeB);
            var indexStreamShapeA = new G3dStream(new int[]
                {
                    0, 1, 2
                }, 3);
            streams.Add(indexStreamShapeA);
            var indexStreamShapeB = new G3dStream(new int[]
                {
                    2, 0, 1
                }, 3);
            streams.Add(indexStreamShapeB);

            model.stream_array = G3dStreamUtility.ToStreamArray(streams);

            ModelDataCreationUtility.AddVertexAttribute(model, new VertexAttr[]
                {
                    new VertexAttr()
                    {
                        Name = "_i0",
                        Hint = "blendindex0",
                        StreamIndex = streams.IndexOf(blendIndexStreamShapeA),
                        Type = vtx_attrib_typeType.@int,
                    },
                });

            ModelDataCreationUtility.AddVertexAttribute(model, new VertexAttr[]
                {
                    new VertexAttr()
                    {
                        Name = "_i0",
                        Hint = "blendindex0",
                        StreamIndex = streams.IndexOf(blendIndexStream0ShapeB),
                        Type = vtx_attrib_typeType.uint4,
                    },
                    new VertexAttr()
                    {
                        Name = "_i1",
                        Hint = "blendindex1",
                        StreamIndex = streams.IndexOf(blendIndexStream1ShapeB),
                        Type = vtx_attrib_typeType.uint4,
                    },
                    new VertexAttr()
                    {
                        Name = "_w0",
                        Hint = "blendweight0",
                        StreamIndex = streams.IndexOf(skinningWeightStream0ShapeB),
                        Type = vtx_attrib_typeType.float4,
                    },
                    new VertexAttr()
                    {
                        Name = "_w1",
                        Hint = "blendweight1",
                        StreamIndex = streams.IndexOf(skinningWeightStream1ShapeB),
                        Type = vtx_attrib_typeType.float4,
                    },
                });

            shapeType shapeA = ModelDataCreationUtility.AddShape(model, "shapeA", "matA", "boneRoot", 0, streams.IndexOf(indexStreamShapeA));
            shapeType shapeB = ModelDataCreationUtility.AddShape(model, "shapeB", "matA", "boneRoot", 1, streams.IndexOf(indexStreamShapeB));

            // まず連続しないシェイプのスキニング方式が揃えられないことをテストする
            shapeA.shape_info.original_bone_name = "boneX";
            shapeB.shape_info.original_bone_name = "boneY";

            IfModelShapeCompressor.AlignSkinningTypeForContinuousShape(model, streams);

            // リジッドスキニングが変換されていないことをテスト
            Assert.AreEqual(1, model.vertex_array.vertex[0].vtx_attrib_array.Items.Length);
            Assert.AreEqual(7, streams.Count);

            // 連続するシェイプのスキニング方式が揃えられる
            shapeA.shape_info.original_bone_name = "boneX";
            shapeB.shape_info.original_bone_name = "boneX";

            IfModelShapeCompressor.AlignSkinningTypeForContinuousShape(model, streams);
            IfModelShapeCompressor.ReassignBoneMatrixIndex(model, streams);

            // リジッドスキニングがスムーススキニングに変換されていることをテスト
            Assert.AreEqual(4, model.vertex_array.vertex[0].vtx_attrib_array.Items.Length);
            Assert.AreEqual(10, streams.Count);

            // ボーンのマトリックスインデックスが正しく変換されているかをテスト
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneA"));
                Assert.AreEqual(0, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneB"));
                Assert.AreEqual(1, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneC"));
                Assert.AreEqual(2, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneD"));
                Assert.AreEqual(3, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneE"));
                Assert.AreEqual(4, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneF"));
                Assert.AreEqual(5, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneG"));
                Assert.AreEqual(6, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneH"));
                Assert.AreEqual(7, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }
            {
                var bone = model.skeleton.bone_array.bone.First(x => x.name.Equals("boneI"));
                Assert.AreEqual(8, bone.matrix_index[0]);
                Assert.AreEqual(-1, bone.matrix_index[1]);
            }

            {
                int streamIndex = 0;
                int[] correctBlendIndices = new int[]
                {
                    0, 0, 0, 0,
                    0, 0, 0, 0,
                    0, 0, 0, 0,
                };
                Assert.AreEqual(correctBlendIndices.Length, streams[streamIndex].IntData.Count);
                for (int i = 0; i < correctBlendIndices.Length; ++i)
                {
                    Assert.AreEqual(correctBlendIndices[i], streams[streamIndex].IntData[i]);
                }
            }

            {
                int streamIndex = 7;
                float[] correctBlendWeights = new float[]
                {
                    1.0f, 0.0f, 0.0f, 0.0f,
                    1.0f, 0.0f, 0.0f, 0.0f,
                    1.0f, 0.0f, 0.0f, 0.0f,
                };
                Assert.AreEqual(correctBlendWeights.Length, streams[streamIndex].FloatData.Count);
                for (int i = 0; i < correctBlendWeights.Length; ++i)
                {
                    Assert.AreEqual(correctBlendWeights[i], streams[streamIndex].FloatData[i]);
                }
            }

            {
                int streamIndex = 8;
                int[] correctBlendIndices = new int[]
                {
                    0, 0, 0, 0,
                    0, 0, 0, 0,
                    0, 0, 0, 0,
                };
                Assert.AreEqual(correctBlendIndices.Length, streams[streamIndex].IntData.Count);
                for (int i = 0; i < correctBlendIndices.Length; ++i)
                {
                    Assert.AreEqual(correctBlendIndices[i], streams[streamIndex].IntData[i]);
                }
            }

            {
                int streamIndex = 9;
                float[] correctBlendWeights = new float[]
                {
                    0.0f, 0.0f, 0.0f, 0.0f,
                    0.0f, 0.0f, 0.0f, 0.0f,
                    0.0f, 0.0f, 0.0f, 0.0f,
                };
                Assert.AreEqual(correctBlendWeights.Length, streams[streamIndex].FloatData.Count);
                for (int i = 0; i < correctBlendWeights.Length; ++i)
                {
                    Assert.AreEqual(correctBlendWeights[i], streams[streamIndex].FloatData[i]);
                }
            }
        }
    }
}
