﻿using System;
using System.Collections.Generic;
using System.Linq;
using _3dToolsTestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace _3dIntermediateFileLibraryTest
{
    [TestClass]
    public class CompressBoneTest
    {
        [TestMethod]
        public void CullBoneCompressTest()
        {
            List<G3dStream> streams = new List<G3dStream>();
            modelType model = ModelDataCreationUtility.CreateEmptyModel();
            skeletal_animType skeletalAnim = AnimDataCreationUtility.CreateEmptySkeletalAnim();
            bone_visibility_animType boneVisAnim = AnimDataCreationUtility.CreateEmptyBoneVisibilityAnim();

            // 以下のようなボーン構造を用意
            //
            // boneRoot
            // |_ bone_E
            //   |_ bone_F(compress_enable=false)
            //     |_ bone_D
            //       |_ bone_C
            //       | |_ bone_A(rigid_body=true)
            //       |   |_ shapeA
            //       |_ bone_B
            //         |_ shapeB
            //
            AddBone(model, skeletalAnim, boneVisAnim, "boneRoot", string.Empty);
            AddBone(model, skeletalAnim, boneVisAnim, "bone_E", "boneRoot");
            AddBone(model, skeletalAnim, boneVisAnim, "bone_F", "bone_E", compressEnable:false);
            AddBone(model, skeletalAnim, boneVisAnim, "bone_D", "bone_F");
            AddBone(model, skeletalAnim, boneVisAnim, "bone_B", "bone_D");
            AddBone(model, skeletalAnim, boneVisAnim, "bone_C", "bone_D");
            AddBone(model, skeletalAnim, boneVisAnim, "bone_A", "bone_C", rigidBody:true);

            ModelDataCreationUtility.AddShape(model, "shapeA", "matA", "bone_A");
            ModelDataCreationUtility.AddShape(model, "shapeB", "matA", "bone_B");

            IfModelBoneCullCompressor.CompressBoneCull(model);
            IfSkeletalAnimBoneCullCompressor.CompressBoneCull(skeletalAnim);
            IfBoneVisibilityAnimBoneCullCompressor.CompressBoneCull(boneVisAnim);

            //
            // 以下のような結果になることをテスト
            //
            // boneRoot
            // |_ bone_E
            //   |_ bone_F(compress_enable=false)
            //     |_ bone_D
            //     | |_ bone_C
            //     |   |_ bone_A(rigid_body=true)
            //     |     |_ shapeA
            //     |
            //     |_ shapeB
            //
            Assert.AreEqual(6, model.skeleton.bone_array.bone.Length);
            Assert.AreEqual(6, skeletalAnim.bone_anim_array.bone_anim.Length);
            Assert.AreEqual(6, boneVisAnim.bone_vis_bone_anim_array.bone_vis_bone_anim.Length);

            Assert.AreEqual("bone_C", model.skeleton.bone_array.bone.First(x => x.name == "bone_A").parent_name);
            Assert.AreEqual("bone_D", model.skeleton.bone_array.bone.First(x => x.name == "bone_C").parent_name);
            Assert.AreEqual("bone_F", model.skeleton.bone_array.bone.First(x => x.name == "bone_D").parent_name);
            Assert.AreEqual("bone_E", model.skeleton.bone_array.bone.First(x => x.name == "bone_F").parent_name);
            Assert.AreEqual("boneRoot", model.skeleton.bone_array.bone.First(x => x.name == "bone_E").parent_name);

            Assert.AreEqual("bone_C", skeletalAnim.bone_anim_array.bone_anim.First(x => x.name == "bone_A").parent_name);
            Assert.AreEqual("bone_D", skeletalAnim.bone_anim_array.bone_anim.First(x => x.name == "bone_C").parent_name);
            Assert.AreEqual("bone_F", skeletalAnim.bone_anim_array.bone_anim.First(x => x.name == "bone_D").parent_name);
            Assert.AreEqual("bone_E", skeletalAnim.bone_anim_array.bone_anim.First(x => x.name == "bone_F").parent_name);
            Assert.AreEqual("boneRoot", skeletalAnim.bone_anim_array.bone_anim.First(x => x.name == "bone_E").parent_name);

            Assert.AreEqual("bone_C", boneVisAnim.bone_vis_bone_anim_array.bone_vis_bone_anim.First(x => x.name == "bone_A").parent_name);
            Assert.AreEqual("bone_D", boneVisAnim.bone_vis_bone_anim_array.bone_vis_bone_anim.First(x => x.name == "bone_C").parent_name);
            Assert.AreEqual("bone_F", boneVisAnim.bone_vis_bone_anim_array.bone_vis_bone_anim.First(x => x.name == "bone_D").parent_name);
            Assert.AreEqual("bone_E", boneVisAnim.bone_vis_bone_anim_array.bone_vis_bone_anim.First(x => x.name == "bone_F").parent_name);
            Assert.AreEqual("boneRoot", boneVisAnim.bone_vis_bone_anim_array.bone_vis_bone_anim.First(x => x.name == "bone_E").parent_name);

            Assert.AreEqual("bone_A", model.shape_array.shape.First(x => x.name == "shapeA").shape_info.bone_name);
            Assert.AreEqual("bone_F", model.shape_array.shape.First(x => x.name == "shapeB").shape_info.bone_name);
        }

        [TestMethod]
        public void MergeBoneCompressTest()
        {
            List<G3dStream> streams = new List<G3dStream>();
            modelType model = ModelDataCreationUtility.CreateEmptyModel();
            skeletal_animType skeletalAnim = AnimDataCreationUtility.CreateEmptySkeletalAnim();
            bone_visibility_animType boneVisAnim = AnimDataCreationUtility.CreateEmptyBoneVisibilityAnim();

            // 以下のようなボーン構造を用意
            //
            // boneRoot
            // |_ bone_E
            //   |_ bone_F(compress_enable=false)
            //     |_ bone_D
            //       |_ bone_C
            //       | |_ bone_A(rigid_body=true)
            //       |   |_ shapeA
            //       |_ bone_B
            //         |_ shapeB
            //
            AddBone(model, skeletalAnim, boneVisAnim, "boneRoot", string.Empty);
            AddBone(model, skeletalAnim, boneVisAnim, "bone_E", "boneRoot");
            AddBone(model, skeletalAnim, boneVisAnim, "bone_F", "bone_E", compressEnable:false);
            AddBone(model, skeletalAnim, boneVisAnim, "bone_D", "bone_F");
            AddBone(model, skeletalAnim, boneVisAnim, "bone_B", "bone_D");
            AddBone(model, skeletalAnim, boneVisAnim, "bone_C", "bone_D");
            AddBone(model, skeletalAnim, boneVisAnim, "bone_A", "bone_C",rigidBody:true);

            ModelDataCreationUtility.AddShape(model, "shapeA", "matA", "bone_A");
            ModelDataCreationUtility.AddShape(model, "shapeB", "matA", "bone_B");

            List<G3dStream> skelAnimStream = new List<G3dStream>();
            List<G3dStream> boneVisAnimStream = new List<G3dStream>();

            IfModelBoneMergeCompressor.CompressBoneMerge(model);

            // TODO: スケルタルアニメーション、ボーンビジビリティーアニメーションもテストする(ストリームを用意しないとテストできない)

            // 以下のような結果になることをテスト
            //
            //    bone_F(compress_enable=false)
            //   |_ bone_A(rigid_body=true)
            //   | |_ shapeA
            //   |
            //   |_ shapeB
            //
            Assert.AreEqual(2, model.skeleton.bone_array.bone.Length);

            Assert.AreEqual("bone_F", model.skeleton.bone_array.bone.First(x => x.name == "bone_A").parent_name);
            Assert.AreEqual(string.Empty, model.skeleton.bone_array.bone.First(x => x.name == "bone_F").parent_name);

            Assert.AreEqual("bone_A", model.shape_array.shape.First(x => x.name == "shapeA").shape_info.bone_name);
            Assert.AreEqual("bone_F", model.shape_array.shape.First(x => x.name == "shapeB").shape_info.bone_name);
        }

        private void AddBone(
            modelType model,
            skeletal_animType skelAnim,
            bone_visibility_animType boneVisAnim,
            string name,
            string parentName,
            bool compressEnable = true,
            bool rigidBody = false,
            int[] matrixIndex = null)
        {
            if (matrixIndex == null)
            {
                matrixIndex = new int[] { -1, -1 };
            }

            var bone = ModelDataCreationUtility.AddBone(model, name, parentName);
            var boneAnim = AnimDataCreationUtility.AddBoneAnim(skelAnim, name, parentName);
            var boneVisBoneAnim = AnimDataCreationUtility.AddBoneVisibilityAnim(boneVisAnim, name, parentName);

            bone.rigid_body = rigidBody;
            bone.matrix_index = matrixIndex;
            bone.compress_enable = compressEnable;

            boneAnim.rigid_body = rigidBody;
            boneAnim.matrix_index = matrixIndex;
            boneAnim.compress_enable = compressEnable;
            boneAnim.binarize_rotate = true;
            boneAnim.binarize_scale = true;
            boneAnim.binarize_translate = true;
            boneAnim.bone_anim_target = new bone_anim_targetType[]
                {
                    new bone_anim_targetType()
                    {
                        target = bone_anim_target_targetType.translate_x,
                    }
                };

            boneVisBoneAnim.rigid_body = rigidBody;
            boneVisBoneAnim.matrix_index = matrixIndex;
            boneVisBoneAnim.compress_enable = compressEnable;
        }
    }
}
