﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Diagnostics;
using System.Reflection;
using nw.g3d.toollib;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib.nw3de;
using _3dToolsTestUtility;
using Nintendo.G3dTool.Entities;
using System.Windows.Forms;

namespace _3dIntermediateFileAssignUtilityTest
{
    /// <summary>
    /// assign-search-path のテストです。
    /// </summary>
    [TestClass]
    public class AssignSearchPathTest
    {
        public AssignSearchPathTest()
        {
            G3dParallel.Job = 1;
            nw.g3d.ifassign.g3difassign.XsdBasePath = G3dToolUtility.GetXsdBasePath();
        }

        [TestMethod]
        public void DisableFileInfoTest()
        {
            var sourceFile = System.IO.Path.Combine(IoUtility.GetG3dDemoRootPath(), "Resources/human/human.fmdb");
            string outputFmdFilePath = IoUtility.GetTempFilePath($"{System.IO.Path.GetFileNameWithoutExtension(sourceFile)}-output.fmdb");
            string inputFilePath = IoUtility.GetTempFilePath($"{System.IO.Path.GetFileNameWithoutExtension(sourceFile)}-input.fmdb");

            // file_info のあるデータを作成
            {
                var file = IfReadUtility.ReadIntermediateFile(sourceFile, nw.g3d.ifassign.g3difassign.XsdBasePath);
                file.FileInfo.Create.ToolName = "hoge";
                file.FileInfo.Modify.ToolName = "fuga";
                IfWriteUtility.WriteIntermediateFile(file, inputFilePath, IoUtility.GetXsdBasePath());
            }

            {
                // file_info が作られたかチェック
                var file = IfReadUtility.ReadIntermediateFile(inputFilePath, nw.g3d.ifassign.g3difassign.XsdBasePath);
                Assert.AreEqual("hoge", file.FileInfo.Create.ToolName);
                Assert.AreEqual("fuga", file.FileInfo.Modify.ToolName);
            }

            try
            {
                nw.g3d.ifassign.g3difassign tool = new nw.g3d.ifassign.g3difassign();

                // --disable-file-info あり
                {
                    tool.Arguments = $"-j 1 assign-search-path {inputFilePath} --disable-file-info --search-path hoge --search-path fuga -o {outputFmdFilePath}";
                    tool.Run();

                    var file = IfReadUtility.ReadIntermediateFile(outputFmdFilePath, nw.g3d.ifassign.g3difassign.XsdBasePath);
                    Assert.AreEqual("fuga", file.FileInfo.Modify.ToolName);
                }

                // サブコマンドより前に --disable-file-info を記述した場合
                {
                    tool.Arguments = $"-j 1 --disable-file-info assign-search-path {inputFilePath} --search-path hoge --search-path fuga -o {outputFmdFilePath}";
                    tool.Run();

                    var file = IfReadUtility.ReadIntermediateFile(outputFmdFilePath, nw.g3d.ifassign.g3difassign.XsdBasePath);
                    Assert.AreEqual("fuga", file.FileInfo.Modify.ToolName);
                }

                // --disable-file-info なし
                {
                    tool.Arguments = $"-j 1 assign-search-path {inputFilePath} --search-path hoge --search-path fuga -o {outputFmdFilePath}";
                    tool.Run();

                    // 出力後の fmdb が正しいかテスト
                    var file = IfReadUtility.ReadIntermediateFile(outputFmdFilePath, nw.g3d.ifassign.g3difassign.XsdBasePath);
                    Assert.AreEqual(Application.ProductName, file.FileInfo.Modify.ToolName);
                }
            }
            finally
            {
                if (System.IO.File.Exists(outputFmdFilePath))
                {
                    System.IO.File.Delete(outputFmdFilePath);
                }
            }
        }

        [TestMethod]
        public void AssignSearchPath()
        {
            {
                var inputFile = System.IO.Path.Combine(IoUtility.GetG3dDemoRootPath(), "Resources/human/human.fmdb");
                string outputFmdFilePath1 = IoUtility.GetTempFilePath($"{System.IO.Path.GetFileNameWithoutExtension(inputFile)}.fmdb");
                string outputFmdFilePath2 = IoUtility.GetTempFilePath($"{System.IO.Path.GetFileNameWithoutExtension(inputFile)}-2.fmdb");

                try
                {
                    nw.g3d.ifassign.g3difassign tool = new nw.g3d.ifassign.g3difassign();

                    // 探索パスの追加テスト
                    {
                        tool.Arguments = $"-j 1 assign-search-path {inputFile} --search-path hoge --search-path fuga -o {outputFmdFilePath1}";
                        tool.Run();

                        // 出力後の fmdb が正しいかテスト
                        var file = IfReadUtility.ReadIntermediateFile(outputFmdFilePath1, nw.g3d.ifassign.g3difassign.XsdBasePath);
                        var searchPath = file.ToolData.FindToolDataObject<nw3de_SearchPath>();
                        Assert.AreEqual(2, searchPath.SearchPaths.Count);
                        Assert.AreEqual("hoge", searchPath.SearchPaths[0].Path);
                        Assert.AreEqual("fuga", searchPath.SearchPaths[1].Path);
                    }

                    // 探索パスの削除テスト
                    {
                        tool.Arguments = $"-j 1 assign-search-path {outputFmdFilePath1} -o {outputFmdFilePath2}";
                        tool.Run();

                        // 出力後の fmdb が正しいかテスト
                        var file = IfReadUtility.ReadIntermediateFile(outputFmdFilePath2, nw.g3d.ifassign.g3difassign.XsdBasePath);
                        Assert.IsNull(file.FileInfo.Modify.CreateSerializableData());
                        var searchPath = file.ToolData.FindToolDataObject<nw3de_SearchPath>();
                        Assert.IsNull(searchPath);
                    }
                }
                finally
                {
                    if (System.IO.File.Exists(outputFmdFilePath1))
                    {
                        System.IO.File.Delete(outputFmdFilePath1);
                    }
                    if (System.IO.File.Exists(outputFmdFilePath2))
                    {
                        System.IO.File.Delete(outputFmdFilePath2);
                    }
                }
            }
        }
    }
}
