﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using G3dCombinerShaderConverter;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.G3dTool.Entities;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Reflection;

namespace _3dCombinerShaderConverterTest
{
    [TestClass]
    public class EmbedCombinerInfoToShaderDefinitionTest
    {
        /// <summary>
        /// --combiner-editor-path オプションのテストです。
        /// </summary>
        [TestMethod]
        public void CombinerEditorPathOptionTest()
        {
            IntermediateFile file = ConvertCombinerShaderDefinition(MethodBase.GetCurrentMethod().Name, true);
            try
            {
                string combinerEditorPath = System.IO.Path.Combine(_3dToolsTestUtility.IoUtility.GetSdkRootPath(), @"Tools\Graphics\CombinerEditor\CombinerEditor.exe");
                string combinerShaderPath = System.IO.Path.Combine(_3dToolsTestUtility.IoUtility.GetG3dDemoRootPath(), "../G3dSandbox/Resources/CombinerSamples/Models/combiners");
                string args = $"{file.Path} --combiner-shader-path {combinerShaderPath}";
                Environment.SetEnvironmentVariable("G3D_COMBINER_EDITOR_PATH", combinerEditorPath);
                CombinerShaderConverter converter = new CombinerShaderConverter(args.Split(' '));
                converter.Execute();
                Assert.IsTrue(System.IO.File.Exists(file.Path));
            }
            finally
            {
                if (System.IO.File.Exists(file.Path))
                {
                    System.IO.File.Delete(file.Path);
                }
            }
        }

        /// <summary>
        /// シェーダー定義にコンバイナー関連の情報を埋め込むテストです。
        /// </summary>
        [TestMethod]
        public void EmbedCombinerInfoIntoFsdTest()
        {
            G3dParallel.Job = 1;
            Environment.SetEnvironmentVariable("NINTENDO_SDK_ROOT", GetSdkRootPath());
            string shaderConverterPath = G3dCombinerShaderConverter.Utility.FindShaderConverterPath();
            IntermediateFile file = ConvertCombinerShaderDefinition(MethodBase.GetCurrentMethod().Name);

            ShaderDefinition shaderDef = file.GetRootEntity<ShaderDefinition>();
            Assert.IsNotNull(shaderDef);

            var basicShadingModel = shaderDef.ShadingModels.FirstOrDefault(x => x.Name == "basic");
            var shaderSource = shaderDef.ShaderSrcs.First(x => x.Stream.Value.Contains("COMBINER_COLOR0"));
            string originalSource = shaderSource.Stream.Value;

            // コンバイナーのオプション情報のみを埋め込み
            {
                CombinerUtility.EmbedCombinerOptionInfoIntoShaderDefition(
                    ref shaderDef,
                    new string[] { });
                string source1 = shaderSource.Stream.Value;

                var combinerColor0 = basicShadingModel.OptionVars.FirstOrDefault(x => x.Id == "combiner_color0");
                Assert.IsNotNull(combinerColor0);
                Assert.AreEqual(ui_item_valueType.combiner, combinerColor0.UiItem.Value);

                var combinerColor1 = basicShadingModel.OptionVars.FirstOrDefault(x => x.Id == "combiner_color1");
                Assert.IsNotNull(combinerColor1);
                Assert.AreEqual(ui_item_valueType.combiner, combinerColor1.UiItem.Value);

                // ソースが書き変わることを確認する(オプション変数用のマクロが追加される)
                Assert.AreNotEqual(originalSource, source1);

                CombinerUtility.EmbedCombinerOptionInfoIntoShaderDefition(
                    ref shaderDef,
                    new string[] { });
                string source2 = shaderSource.Stream.Value;
                // 2 回目以降はソースが一致することを確認する(オプション変数用のマクロが追加されたまま)
                Assert.AreEqual(source2, source1);
            }


            // 与えられたコンバイナーファイル毎のソースコードの埋め込み
            {
                string combinerSamplePath = GetCombinerSampleRootPath();
                string presetFolder = System.IO.Path.Combine(combinerSamplePath, "Models/combiners");

                IEnumerable<string> presetFiles = System.IO.Directory.EnumerateFiles(presetFolder, "*.fcmb", System.IO.SearchOption.TopDirectoryOnly);

                CombinerUtility.EmbedCombinerOptionInfoIntoShaderDefition(
                    ref shaderDef,
                    presetFiles);
                Assert.AreEqual(5, basicShadingModel.OptionVars.First(x => x.Id == "combiner_color0").Choice.Values.Count);
                Assert.IsTrue(ContainsInSourceCodes("choice=\"0,color0_a,color0_b,color1_a,color1_b\"", shaderDef));
                Assert.IsNotNull(
                    basicShadingModel.OptionVars.FirstOrDefault(x => x.Id == "combiner_color0"));
                Assert.IsNotNull(
                    basicShadingModel.OptionVars.FirstOrDefault(x => x.Id == "combiner_color1"));

                // とりあえずすべてのソースコードをコンバイナーオプション定義が書いてある候補ソースコードとする
                CombinerUtility.EmbedCombinerSourceIntoShaderDefinition(
                    ref shaderDef,
                    presetFiles,
                    WriteMessageLine, WriteMessageLine, WriteErrorMessageLine);


                // 埋め込み後のソースコードに分岐コードが埋め込まれているか確認
                // combiner_color0 はユニフォーム分岐
                Assert.IsTrue(ContainsInSourceCodes("if (COMBINER_COLOR0 == (color0_a))", shaderDef));
                Assert.IsTrue(ContainsInSourceCodes("else if (COMBINER_COLOR0 == (color0_b))", shaderDef));
                Assert.IsTrue(ContainsInSourceCodes("else if (COMBINER_COLOR0 == (color1_a))", shaderDef));
                Assert.IsTrue(ContainsInSourceCodes("else if (COMBINER_COLOR0 == (color1_b))", shaderDef));

                // combiner_color1 はマクロ分岐
                Assert.IsTrue(ContainsInSourceCodes("#if COMBINER_COLOR1 == (color0_a)", shaderDef));
                Assert.IsTrue(ContainsInSourceCodes("#elif COMBINER_COLOR1 == (color0_b)", shaderDef));
                Assert.IsTrue(ContainsInSourceCodes("#elif COMBINER_COLOR1 == (color1_a)", shaderDef));
                Assert.IsTrue(ContainsInSourceCodes("#elif COMBINER_COLOR1 == (color1_b)", shaderDef));
            }

            // choice の値をハッシュ値に書き換え
            {
                CombinerUtility.ConvertCombinerEnumChoiceToHashValue(ref shaderDef);

                var combinerColor0 = basicShadingModel.OptionVars.FirstOrDefault(x => x.Id == "combiner_color0");
                Assert.AreEqual(
                    "0,3655551022,447767813,2083525861,3210951118",
                    combinerColor0.Choice.CreateSerializableData());

                var combinerColor1 = basicShadingModel.OptionVars.FirstOrDefault(x => x.Id == "combiner_color1");
                Assert.AreEqual(
                    "0,3655551022,447767813,2083525861,3210951118",
                    combinerColor1.Choice.CreateSerializableData());

                Assert.IsFalse(ContainsInSourceCodes("choice=\"0,color0_a,color0_b,color1_a,color1_b\"", shaderDef));
                Assert.IsTrue(ContainsInSourceCodes($"#define color0_a (3655551022)", shaderDef));
                Assert.IsTrue(ContainsInSourceCodes($"#define color0_b (447767813)", shaderDef));
                Assert.IsTrue(ContainsInSourceCodes($"#define color1_a (2083525861)", shaderDef));
                Assert.IsTrue(ContainsInSourceCodes($"#define color1_b (3210951118)", shaderDef));
            }

            // 埋め込み後の fsd がコンバートできるかテスト
            string embeddedFsdPath = GetTempFilePath(MethodBase.GetCurrentMethod().Name + "-embedded.fsdb");
            using (new ScopedFileCreator(embeddedFsdPath, true))
            {
                IfWriteUtility.Write(file.CreateSerializableData(), embeddedFsdPath, GetXsdBasePath());
                string bfshaPath = GetTempFilePath(MethodBase.GetCurrentMethod().Name + "demo.bfsha");
                ConvertShader(embeddedFsdPath, bfshaPath);
            }
        }


        /// <summary>
        /// コンバイナーサンプルの demo.fsdb を作成
        /// </summary>
        /// <param name="tempFileNameSuffix"></param>
        /// <param name="keepsTempFile"></param>
        /// <returns></returns>
        private IntermediateFile ConvertCombinerShaderDefinition(string tempFileNameSuffix, bool keepsTempFile=false)
        {
            IntermediateFile file;
            string fsdPath = GetTempFilePath(tempFileNameSuffix + ".fsdb");
            string shaderConverterPath = G3dCombinerShaderConverter.Utility.FindShaderConverterPath();
            string fscPath = System.IO.Path.Combine(GetCombinerSampleRootPath(), "demo.fsca");
            G3dCombinerShaderConverter.Utility.ExecuteProcess(
                shaderConverterPath, $"{fscPath} --output {fsdPath}", false, WriteMessageLine, WriteMessageLine);
            file = IfReadUtility.ReadIntermediateFile(fsdPath, GetXsdBasePath());
            if (!keepsTempFile)
            {
                System.IO.File.Delete(fsdPath);
            }

            return file;
        }

        private static bool ContainsInSourceCodes(string text, ShaderDefinition shaderDef)
        {
            foreach (var src in shaderDef.ShaderSrcs)
            {
                if (src.Stream.Value.Contains(text))
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// グラフィックカードがない CI マシンでこけるので、デバッグ時のみ実行されるようにしておく
        /// </summary>
        /// <param name="fsdPath"></param>
        /// <param name="bfshaPath"></param>
        [Conditional("DEBUG")]
        private static void ConvertShader(string fsdPath, string bfshaPath)
        {
            string shaderConverterPath = G3dCombinerShaderConverter.Utility.FindShaderConverterPath();
            G3dCombinerShaderConverter.Utility.ExecuteProcess(
                shaderConverterPath, $"{fsdPath} --output {bfshaPath} --glsl-version 450 --api-type GL --code-type Binary", false, WriteMessageLine, WriteMessageLine);
        }

        private static void WriteMessageLine(string message)
        {
            Debug.WriteLine(message);
        }

        private static void WriteErrorMessageLine(string message)
        {
            Debug.WriteLine("Error: " + message);
        }

        private static string GetCombinerSampleRootPath()
        {
            return System.IO.Path.Combine(GetSdkRootPath(), @"Samples\Sources\Applications\G3dSandbox\Resources\CombinerSamples");
        }

        private static string GetSdkRootPath()
        {
            return System.IO.Path.Combine(
                System.IO.Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location),
                "../../../../../../..");
        }

        private static string GetTempFilePath(string baseFileName)
        {
            return System.IO.Path.Combine(
                System.IO.Path.GetTempPath(),
                $"Nintendo_{Assembly.GetExecutingAssembly().GetName().Name}_{Guid.NewGuid().ToString("N")}_{baseFileName}");
        }

        private static string GetXsdBasePath()
        {
            return System.IO.Path.Combine(
                System.IO.Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location),
                "../../../../../../../Programs/NintendoWare/Sources/Tools/Graphics/3dTools/3dIntermediateFileXsd");
        }
    }
}
