﻿using _3dToolsTestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.G3dTool.Entities;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Runtime.Serialization.Formatters.Binary;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Serialization;

namespace G3dLibraryTest
{
    [TestClass]
    public class ToolDataSerializationTest
    {
        [Serializable]
        [XmlRootAttribute("dummy_tool_data")]
        public class DummyToolData : ToolDataObject
        {
            [XmlElement("string_param")]
            public string StringParam { get; set; }

            [XmlElement("int_param")]
            public int IntParam { get; set; }

            public List<DummyToolDataElement> Elements { get; } = new List<DummyToolDataElement>();
        }

        [Serializable]
        [XmlRootAttribute("dummy_tool_data_element")]
        public class DummyToolDataElement
        {
            [XmlElement("string_param")]
            public string StringParam { get; set; }

            [XmlElement("int_param")]
            public int IntParam { get; set; }
        }


        [TestMethod]
        public void DeepCopyToolData()
        {
            var dummyToolData = new DummyToolData();
            dummyToolData.IntParam = 10;
            dummyToolData.StringParam = "Hoge";
            dummyToolData.Elements.Add(new DummyToolDataElement() { IntParam = 1, StringParam = "Fuga" });

            var cloned = dummyToolData.DeepCopy() as DummyToolData;

            // 値型のディープコピーのチェック
            Assert.AreEqual(dummyToolData.IntParam, cloned.IntParam);
            Assert.AreEqual(dummyToolData.StringParam, cloned.StringParam);

            // 配列型、参照型のディープコピーのチェック
            Assert.AreNotEqual(dummyToolData.Elements, cloned.Elements);
            Assert.AreEqual(dummyToolData.Elements.Count, cloned.Elements.Count);
            Assert.AreNotEqual(dummyToolData.Elements[0], cloned.Elements[0]);
            Assert.AreEqual(dummyToolData.Elements[0].IntParam, cloned.Elements[0].IntParam);
            Assert.AreEqual(dummyToolData.Elements[0].StringParam, cloned.Elements[0].StringParam);
        }

        [TestMethod]
        public void WriteReadToolDataTest()
        {
            G3dParallel.Job = 1;

            string tmpFilePath = $"{MethodBase.GetCurrentMethod().Name}.fmtb";
            try
            {
                {
                    var file = new IntermediateFile(IntermediateFileKind.Material);
                    Material material = file.GetRootEntity<Material>();
                    material.Name = "dummy";
                    var dummyToolData = new DummyToolData();
                    material.ToolData.Elements.Add(new ToolDataElement(dummyToolData));
                    dummyToolData.IntParam = 10;
                    dummyToolData.StringParam = "Hoge";
                    IfWriteUtility.WriteIntermediateFile(file, tmpFilePath, IoUtility.GetXsdBasePath());
                }

                {
                    var file = IfReadUtility.ReadIntermediateFile(tmpFilePath, IoUtility.GetXsdBasePath());
                    var material = file.GetRootEntity<Material>();
                    Assert.AreEqual(1, material.ToolData.Elements.Count);
                    var dummyToolData = material.ToolData.FindToolDataObject<DummyToolData>();
                    Assert.IsNotNull(dummyToolData);
                    Assert.AreEqual(10, dummyToolData.IntParam);
                    Assert.AreEqual("Hoge", dummyToolData.StringParam);
                }
            }
            finally
            {
                if (System.IO.File.Exists(tmpFilePath))
                {
                    System.IO.File.Delete(tmpFilePath);
                }
            }
        }


        [TestMethod]
        public void ConvertToolDataTest()
        {
            var dummyToolData = new DummyToolData();
            var toolDataElem = new ToolDataElement(dummyToolData);
            dummyToolData.IntParam = 10;
            dummyToolData.StringParam = "Hoge";

            {
                ToolData toolData = new ToolData();
                toolData.Elements.Add(toolDataElem);
                var writeData = toolData.CreateSerializableData();
                Assert.AreEqual("dummy_tool_data", writeData.Any[0].Name);
                Assert.AreEqual("string_param", writeData.Any[0].ChildNodes[0].Name);
                Assert.AreEqual(dummyToolData.StringParam, writeData.Any[0].ChildNodes[0].InnerText);
                Assert.AreEqual("int_param", writeData.Any[0].ChildNodes[1].Name);
                Assert.AreEqual(dummyToolData.IntParam.ToString(), writeData.Any[0].ChildNodes[1].InnerText);

                var readToolData = new ToolData(writeData);
                var readDummyToolData = readToolData.FindToolDataObject<DummyToolData>();
                Assert.IsNotNull(readDummyToolData);
                Assert.AreEqual(dummyToolData.IntParam, readDummyToolData.IntParam);
                Assert.AreEqual(dummyToolData.StringParam, readDummyToolData.StringParam);
            }
        }

        [TestMethod]
        public void ShaderDefinitionToolDataTest()
        {
            var file = new IntermediateFile(IntermediateFileKind.ShaderDefinition);
            var shaderDef = file.GetRootEntity<ShaderDefinition>();
            shaderDef.ShaderDefinitionInfo.ConfigPath = "hoge";
            shaderDef.ShadingModels.Add(new ShadingModel() { Name = "hoge" });
            shaderDef.ShaderSrcs.Add(new ShaderSrc() { Path = "hoge" });
            var dummyToolData = new DummyToolData();
            dummyToolData.IntParam = 10;
            dummyToolData.StringParam = "Hoge";
            shaderDef.ToolData.AddToolDataObject(dummyToolData);

            string tmpFilePath = IoUtility.GetTempFilePath($"{MethodBase.GetCurrentMethod().Name}.fsdb");
            try
            {
                IfWriteUtility.WriteIntermediateFile(file, tmpFilePath, IoUtility.GetXsdBasePath());
                var readFile = IfReadUtility.ReadIntermediateFile(tmpFilePath, IoUtility.GetXsdBasePath());
                var readShaderDef = readFile.GetRootEntity<ShaderDefinition>();
                Assert.IsNotNull(readShaderDef.ToolData.FindToolDataObject<DummyToolData>());
            }
            finally
            {
                if (System.IO.File.Exists(tmpFilePath))
                {
                    System.IO.File.Delete(tmpFilePath);
                }
            }
        }

        [TestMethod]
        public void ShaderConfigToolDataTest()
        {
            var file = new IntermediateFile(IntermediateFileKind.ShaderConfig);
            var shaderConfig = file.GetRootEntity<ShaderConfig>();
            var shader = new Shader() { Name = "hoge" };
            shader.VertexShader.Path = "hoge";
            shaderConfig.Shaders.Add(shader);
            var dummyToolData = new DummyToolData();
            dummyToolData.IntParam = 10;
            dummyToolData.StringParam = "Hoge";
            shaderConfig.ToolData.AddToolDataObject(dummyToolData);

            string tmpFilePath = IoUtility.GetTempFilePath($"{MethodBase.GetCurrentMethod().Name}.fsca");
            try
            {
                IfWriteUtility.WriteIntermediateFile(file, tmpFilePath, IoUtility.GetXsdBasePath());
                var readFile = IfReadUtility.ReadIntermediateFile(tmpFilePath, IoUtility.GetXsdBasePath());
                var rootEntity = readFile.GetRootEntity<ShaderConfig>();
                Assert.IsNotNull(rootEntity.ToolData.FindToolDataObject<DummyToolData>());
            }
            finally
            {
                if (System.IO.File.Exists(tmpFilePath))
                {
                    System.IO.File.Delete(tmpFilePath);
                }
            }
        }
    }
}
